/* $Id: cleanup.c,v 1.5 1998/12/17 02:09:54 marcus Exp $
******************************************************************************

   LibGG - Functions for adding and removing cleanup callbacks

   Copyright (C) 1998 Marcus Sundberg	[marcus@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <stdlib.h>
#include <ggi/gg.h>
#include "plat.h"

#ifdef HAVE_SIGNAL_H
#include <signal.h>
#endif

typedef struct funclist {
	ggcleanup_func	*func;
	void		*arg;
	struct funclist	*next;
} funclist;

static funclist *firstfunc = NULL;
static ggcleanup_func *currentfunc = NULL;
static int is_registered = 0;

static void cleanup_function(void)
{
	funclist *curr = firstfunc, *prev;
	
	while (curr != NULL) {
		currentfunc = curr->func;
		curr->func(curr->arg);
		prev = curr;
		curr = curr->next;
		currentfunc = NULL;
		/* Make sure we don't call the function again */
		ggUnregisterCleanup(prev->func, prev->arg);
	}
}

#ifdef HAVE_SIGNAL

typedef void (ggsighandler)(int);

typedef struct {
	int		 sig;
	ggsighandler	*oldhandler;
} gg_siginfo;

static gg_siginfo siglist[] = { { SIGFPE, SIG_ERR },
				{ SIGSEGV, SIG_ERR },
				{ SIGILL, SIG_ERR },
				{ SIGHUP, SIG_ERR },
				{ SIGINT, SIG_ERR },
				{ SIGQUIT, SIG_ERR },
				{ SIGABRT, SIG_ERR },
				{ SIGPIPE, SIG_ERR },
#ifdef SIGBUS
				{ SIGBUS, SIG_ERR },
#endif
				{ SIGTERM, SIG_ERR } };

#define SIGLIST_LEN	sizeof(siglist)/sizeof(gg_siginfo)

static void sighandler(int signum)
{
	int i;
	ggsighandler *oldfunc = NULL;

	for (i = 0; i < SIGLIST_LEN; i++) {
		if (siglist[i].sig == signum) {
			if (siglist[i].oldhandler == SIG_IGN) {
				signal(signum, sighandler);
				return;
			}
			if (siglist[i].oldhandler != SIG_DFL) {
				oldfunc = siglist[i].oldhandler;
				break;
			}
		}
	}
	if (oldfunc) {
		oldfunc(signum);
	}
	fprintf(stderr, "Terminating on signal %d\n", signum);
	exit(signum);
}

static void unregister_sighandler(void)
{
	int i;

	for (i = 0; i < SIGLIST_LEN; i++) {
		ggsighandler *oldfunc;
		if (siglist[i].oldhandler == SIG_ERR) continue;
		oldfunc = signal(siglist[i].sig, siglist[i].oldhandler);
		if (oldfunc != sighandler) {
			/* Someone else has changed this signal handler, so
			   we shouldn't touch it */
			signal(siglist[i].sig, oldfunc);
		}
		siglist[i].oldhandler = SIG_ERR;
	}
}		
		

static void register_sighandler(void)
{
	int i;
	ggsighandler *oldfunc;

	for (i = 0; i < SIGLIST_LEN; i++) {
		oldfunc = signal(siglist[i].sig, sighandler);
		if (oldfunc != SIG_DFL &&
		    oldfunc != SIG_IGN) {
			/* If there's a sighandler registered already we don't
			   have to care */
			if (oldfunc != SIG_ERR) {
				signal(siglist[i].sig, oldfunc);
			}
			continue;
		}
		siglist[i].oldhandler = oldfunc;
	}
}

#else /* HAVE_SIGNAL */

#define register_sighandler() /* empty */
#define unregister_sighandler() /* empty */

#endif /* HAVE_SIGNAL */			

int ggRegisterCleanup(ggcleanup_func *func, void *arg)
{
	int ret = 0;
	funclist *new;

	ggLock(_gg_global_mutex);
	register_sighandler();
	if (!is_registered) {
		if (atexit(cleanup_function) != 0) {
			ret = GGI_EUNKNOWN;
			goto out;
		}
		is_registered = 1;
	}
	if ((new = malloc(sizeof(funclist))) == NULL) {
		ret = GGI_ENOMEM;
		goto out;
	}
	new->func = func;
	new->arg = arg;
	
	/* Insert first in list */
	new->next = firstfunc;
	firstfunc = new;

  out:
	ggUnlock(_gg_global_mutex);

	return ret;
}


int ggUnregisterCleanup(ggcleanup_func *func, void *arg)
{
	funclist *curr = firstfunc, *prev = NULL;

	if (func == currentfunc) {
		/* Don't unregister a function while it is being called */
		return 0;
	}
	ggLock(_gg_global_mutex);
	while (curr != NULL) {
		if (curr->func == func && curr->arg == arg) {
			if (curr == firstfunc) firstfunc = curr->next;
			else prev->next = curr->next;
			free(curr);
			ggUnlock(_gg_global_mutex);
			return 0;
		}
		prev = curr;
		curr = curr->next;
	}
	if (firstfunc == NULL) unregister_sighandler();

	ggUnlock(_gg_global_mutex);
	return GGI_ENOTALLOC;
}
