/* $Id: filter.c,v 1.6 1998/12/15 03:04:46 marcus Exp $
******************************************************************************

   Filter-save - save away an eventstream for later playback

   Copyright (C) 1998 Andreas Beck      [becka@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <ggi/internal/gii-dl.h>

enum outtype { STDOUT,FIL,PIPE };

typedef struct save_state {
	struct gii_input *inp;
	FILE *output;
	enum outtype type;
} save_state;

static void save_do(gii_input *inp,gii_event *event)
{
	save_state   *state=inp->priv;
	
	fwrite(event,event->any.size,1,state->output);
}

static gii_event_mask GII_save_poll(struct gii_input *inp)
{
	gii_event	ev;
	gii_event_mask	rc,rc2;
	struct timeval	time={0,0};

	save_state   *state=inp->priv;

	DPRINT_MISC("filter-save: poll(%p);\n",inp);
	
	rc2=0;
	/* As long as the input we were derived from has something ... */
	while(giiEventPoll(state->inp,emAll,&time))
	{
		/* Get it. */
		giiEventRead(state->inp,&ev,emAll);

		/* Check if it triggers something. */
		save_do(inp,&ev);
		
		/* Check if we want it. */
		if ( !( inp->curreventmask & (rc=(1<<ev.any.type)) ) )
			continue;		/* No. Drop it. */

		rc2|=rc;			/* Remember what it was. */
		_giiEvQueueAdd(inp,&ev);	/* Post it */
	}

	return rc2;
	return 0;
}

static gii_cmddata_getdevinfo stdin_devinfo={
	"Mouse event generator",	/* long device name */
	"fmou",				/* shorthand */
	emPointer,			/* can_generate */
	0,				/* no buttons */
	0				/* no valuators */
};

static int GII_save_close(gii_input *inp)
{
	save_state   *state=inp->priv;

	switch(state->type)
	{ case FIL: fclose(state->output);break;
	  case PIPE: pclose(state->output);break;
	  default: break;
	}
	giiClose(state->inp);	/* Get rid of old inputlib */

	free(state);
	DPRINT_MISC("filter-save: close %p\n",inp);
	return 0;
}

static int GII_save_seteventmask(struct gii_input *inp, gii_event_mask evm)
{
	save_state   *state=inp->priv;

	inp->curreventmask=( evm & inp->targetcan );

	/* Pass it on ! */
	giiSetEventMask(state->inp,evm);

	return GGI_OK;
}

int GIIdlinit(gii_input *inp, const char *args, void *argptr)
{
	gii_event diev;
	save_state   *state;

	DPRINT_MISC("filter-save starting.(args=\"%s\",input=%p)\n",args,argptr);

	if (NULL==argptr)
	{
		DPRINT_MISC("filter-save: filter libs require an argptr.\n");
		return -1;
	}

        if (NULL==(state=inp->priv=malloc(sizeof(save_state))))
	{
		DPRINT_MISC("filter-save: no memory.\n");
		return -1;
	}

	state->inp=argptr;
	state->output=stdout;
	state->type=STDOUT;
	
	if (args) {
		if (*args=='|') {
			state->output=popen(args+1,"w");
			state->type=PIPE;
		}
		else 		{
			state->output=fopen(args  ,"w"); /* Maybe better "a" ? */
			state->type=FIL;
		}
	}

	if (!state->output) {
		fprintf(stderr,"Couldn't open %s. Falling back to stdout.\n",args);
		state->output=stdout;
		state->type=STDOUT;
	}
	
	/* We leave these on the default handlers
	 *	inp->GIIgeteventmask=_GIIstdgeteventmask;
	 *	inp->GIIgetselectfdset=_GIIstdgetselectfd;
	 */

	/* They are already set, so we can as well use them instead of
	 * accessing the curreventmask member directly.
	 */
	inp->targetcan=emAll;
	inp->GIIseteventmask=GII_save_seteventmask;
	inp->GIIeventpoll=GII_save_poll;
	inp->GIIclose	 =GII_save_close;

	inp->GIIseteventmask(inp,giiGetEventMask(state->inp)|emPointer);

	inp->maxfd=0;		/* We got to poll ... We might have polling children ... */
	/* Hmm - we should probably copy the fdset ... 
	   and add a "dummy" polled source to make sure ... FIXME */

	_giiEventBlank(&diev);
	diev.cmd.size=sizeof(gii_cmd_nodata_event)+sizeof(gii_cmddata_getdevinfo);
	diev.cmd.type=evCommand;
	diev.cmd.origin=inp->origin;
	diev.cmd.code=GII_CMDCODE_GETDEVINFO;
	memcpy(diev.cmd.data,&stdin_devinfo,sizeof(gii_cmddata_getdevinfo));
	_giiEvQueueAdd(inp,&diev);

	DPRINT_MISC("filter-save: fully up\n");

	return 0;
}
