/* Yo Emacs, this -*- C++ -*-

  Copyright (C) 1999,2000 Jens Hoefkens
  jens@hoefkens.com

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
*/

#ifndef __KBGENGINE_H 
#define __KBGENGINE_H 

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif 

#include "kbgboard.h"

#include <qtabdialog.h>
#include <qpopupmenu.h>
#include <kaction.h>


/**
 *
 * Abstract class for a generic backgammon engine. Real engines have
 * to inherit this and implement the interfaces.
 *
 */
class KBgEngine : public QObject
{
	Q_OBJECT

public:

	enum Command {Redo, Undo, Roll, Cube, Done, Load};

	/**
	 * Constructor 
	 */
	KBgEngine(QWidget *parent = 0, QString *name = 0);

	/**
	 * Destructor
	 */
	virtual ~KBgEngine();

	/**
	 * Return the menu entry of the engine
	 */
	virtual QPopupMenu *getMenu() = 0;

	/**
	 * Fills the engine-specific page into the notebook
	 */
	virtual void getSetupPages(QTabDialog *nb) = 0;

	/**
	 * About to be closed. Let the engine exit properly.
	 */
	virtual bool queryExit() = 0;

public slots:

	/**
	 * Read user settings from the config file
	 */
	virtual void readConfig() = 0;

	/**
	 * Save user settings to the config file
	 */
	virtual void saveConfig() = 0;
	
        /**
	 * Roll dice for the player w
	 */
        virtual void rollDice(const int w) = 0;
	
	/**
	 * Double the cube of player w
	 */
	virtual void doubleCube(const int w) = 0;

	/**
	 * A move has been made on the board - see the board class
	 * for the format of the string s
	 */
	virtual void handleMove(QString *s) = 0;

	/**
	 * Undo the last move
	 */
        virtual void undo() = 0;

	/**
	 * Redo the last move
	 */
        virtual void redo() = 0;

	/**
	 * Roll dice for whoevers turn it is
	 */
        virtual void roll() = 0;

	/**
	 * Double the cube for whoevers can double right now
	 */
        virtual void cube() = 0;

	/**
	 * Reload the board to the last known sane state
	 */
	virtual void load() = 0;

	/**
	 * Commit a move
	 */
	virtual void done() = 0;

	/**
	 * Process the string cmd
	 */
	virtual void handleCommand(const QString& cmd) = 0;

	/**
	 * Get additional menu entries in the menu
	 */
	virtual void addMenuItems(QPopupMenu *m) = 0;

	/**
	 * Remove additional menu entries from the menu
	 */
	virtual void removeMenuItems(QPopupMenu *m) = 0;

	/*
	 * Create the popup menu and fill in the generic part
	 */
	void createMenu(KActionCollection *collection);

signals:
	
	/**
	 * The text identifies the current game status - could be put
	 * into the main window caption
	 */
	void statText(const QString &msg);
	
	/**
	 * Text that should be displayed in the ongoing message window
	 */
	void infoText(const QString &msg);

	/**
	 * Emit the most recent game state
	 */
	void newState(const KBgBoardStatus *);

	/**
	 * Get the current state of the board written into the
	 * board status object
	 */
	void getState(KBgBoardStatus *);

	/**
	 * Starts/ends the edit mode of the board
	 */
	void setEditMode(const bool f);

	/**
	 * Toggle RO/RW flag of the board
	 */
	void allowMoving(const bool fl);

	/**
	 * Announce that we will accept/reject the command cmd from now on
	 */
	void allowCommand(int cmd, bool f);

	/**
	 * Tell the board to undo the last move
	 */
	void undoMove();

	/**
	 * Tell the board to redo the last undone move
	 */
	void redoMove();

protected:

	/**
	 * Context menu for the board
	 */
	QPopupMenu *menu;
};

#endif // __KBGENGINE_H 
