// File management functions taken from the Fox library
// and modified to add a progress dialog 
// Some helper functions are also added to get large files support
// Also supports a timeout on the stat and lstat function (this is the reason
// why some standard FOX function cannot be used and are rewritten here)

#include "config.h"
#include "i18n.h"

#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <pwd.h>
#include <grp.h>
#include <ctype.h>
#include <unistd.h>
#include <sys/wait.h>
#include <time.h>
#include <utime.h>

// For Sun compatibility
#ifdef __sun
#include <alloca.h>
#endif

#include <fx.h>
#include <fxkeys.h>
#include <FXPNGIcon.h>
#include <FXUTF8Codec.h>

#include "xfedefs.h"
#include "icons.h"
#include "File.h"
#include "OverwriteBox.h"
#include "MessageBox.h"
#include "CommandWindow.h"


// Global variables
#if defined(linux)
extern FXStringDict* mtdevices;
extern FXStringDict* updevices;
#endif

// Startup time (seconds)
#define STARTUP_TIME 3

// Delay before the progress bar should be shown (ms)
#define SHOW_PROGRESSBAR_DELAY 1000

// Progress dialog width
#define PROGRESSDIALOG_WIDTH 150



// Quote a filename against shell substitutions
// Thanks to Glynn Clements <glynnc@users.sourceforge.net>
FXString quote(FXString str)
{
	FXString result = "'";
	const char *p;

	for (p = str.text(); *p; p++)
		if (*p == '\'')
			result += "'\\''";
		else
			result += *p;

	result += '\'';
	return result;
}


// Test if a string (typically a file name) is encoded in UTF-8
// Note that a pure ASCII string is considered here as UTF-8
FXbool isUtf8(const FXchar* string, FXuint length)
{
	FXchar s[4];
	const FXchar BOM[] = { 0xEF, 0xBB, 0xBF, '\0' };
	
	// Keep only length left bytes
	FXString str=string;
	str=str.left(length);
	
	// Convert forth and back to UTF8
	FXUTF8Codec utf8;
	FXString utf8str=utf8.mb2utf(utf8.utf2mb(str));
 
	// Strings are equal => UTF8 
	if (str==utf8str)
		return TRUE;
	
	// Strings not equal => test if BOM is present
	else
	{
		// String too small to contain BOM
		if (length<=2)
			return FALSE;
		
		// Test if string contains BOM
		else
		{
			s[0]=string[0];
			s[1]=string[1];
			s[2]=string[2];
			s[3]='\0';
			
			// String contains BOM => UTF8
			if (::streq(s,BOM))
				return TRUE;
			
			// String don't contain BOM
			else
				return FALSE;
		}
	}
}


// Replacement of the stat function
int statrep(const char *filename, struct stat *buf)
{
#if defined(linux)

	static int ret;

	// It's a mount point
	if(mtdevices->find(filename))
	{
		// Mount point is down
		if(::streq(updevices->find(filename),"down"))
			return -1;
		
		// Mount point is up
		else
		{
			ret=stat(filename,buf);
			if (ret==-1)
			{
				updevices->remove(filename);
				updevices->insert(filename,"down");
			}
			return ret;
		}
	}
		
	// It's not a mount point
	else
#endif
		return stat(filename,buf);
}


// Replacement of the lstat function
int lstatrep(const char *filename, struct stat *buf)
{
#if defined(linux)

	static int ret;

	// It's a mount point
	if(mtdevices->find(filename))
	{
		// Mount point is down
		if(::streq(updevices->find(filename),"down"))
			return -1;
		
		// Mount point is up
		else
		{
			ret=lstat(filename,buf);		
			if (ret==-1)
			{
				updevices->remove(filename);
				updevices->insert(filename,"down");
			}
			return ret;
		}
	}
		
	// It's not a mount point
	else
#endif
		return lstat(filename,buf);
}


#if defined(linux)
// Stat function used to test if a mount point is up or down
// Actually, this is simply the lstat() function
int lstatmt(const char *filename, struct stat *buf)
{
	return lstat(filename,buf);
}
#endif


// Run a command and simulate a startup time
void runcmd(FXString cmd)
{
	// Run the command in background
	cmd += " &";
	system(cmd.text());
	
	// Very ugly simulation of a startup time!!!
	sleep(STARTUP_TIME);
}


// Safe strcpy function (Public domain, by C.B. Falconer)
// The destination string is always null terminated
// Size sz must be equal to strlen(src)+1
size_t strlcpy(char *dst, const char *src, size_t sz)
{
	const char *start = src;

	if (src && sz--)
	{
		while ((*dst++ = *src))
			if (sz--)
				src++;
			else
		 	{
            	*(--dst) = '\0';
            	break;
         	}
   	}
   	if (src)
   	{
      	while (*src++)
			continue;
      	return src - start - 1;
   	}
   	else if (sz)
		*dst = '\0';
	return 0;
}


// Safe strcat function (Public domain, by C.B. Falconer)
// The destination string is always null terminated
size_t strlcat(char *dst, const char *src, size_t sz)
{
   char  *start = dst;

   while (*dst++)      // assumes sz >= strlen(dst)
      if (sz)
	  	sz--;          // i.e. well formed string
   dst--;
   return dst - start + strlcpy(dst, src, sz);
}


// To test if two strings are equal (strcmp replacement, thanks to Francesco Abbate)
int streq(const FXchar *a, const FXchar *b)
{
	if (a == NULL || b == NULL)
		return 0;
	return (strcmp(a, b) == 0);
}


// Convert a string to lower cases and returns the string size
void strlow ( char *str )
{
    while ( *str )
    {
        *str = tolower ( *str );
        ++str;
    }
} 


// Obtain the non recursive size of a directory
unsigned long long dirsize(const char *path)
{
    DIR *dp;
	struct dirent *dirp;
	struct stat statbuf;
	char buf[MAXPATHLEN];
	unsigned long long dsize=0;
	int ret;
			
	if((dp=opendir(path))==NULL)
        return 0;
	
	while((dirp=readdir(dp)))
    {
        if ( ::streq(dirp->d_name, ".")||::streq(dirp->d_name, "..") )
            continue;
		
		if (::streq(path,ROOTDIR))
			snprintf(buf,sizeof(buf)-1,"%s%s",path,dirp->d_name);
		else
			snprintf(buf,sizeof(buf)-1,"%s/%s",path,dirp->d_name);
		
#if defined(linux)
		// Mount points are not processed to improve performances 
		if(mtdevices->find(buf))
			continue;
#endif
		
		ret=lstatrep(buf,&statbuf);
		if(ret>=0)
		{
			if(!S_ISDIR(statbuf.st_mode))
		  		dsize += (unsigned long long) statbuf.st_size;
		}
	}
	if(closedir(dp)<0)
        fprintf(stderr,_("Cannot close folder %s\n"), path);
	return dsize;
}

// Obtain the recursive size of a directory
unsigned long long pathsize(const char *path)
{
	struct stat statbuf;
	struct dirent *dirp;
    char *ptr;
    DIR *dp;
	unsigned long long dsize;
	int ret;

	ret=lstatrep(path,&statbuf);
	if(ret<0)
        return 0;
    dsize = (unsigned long long) statbuf.st_size;
    
	// Not a directory
	if(!S_ISDIR(statbuf.st_mode))
        return dsize;
    
	ptr=(char*)path + strlen(path);
    if(ptr[-1]!='/')
    {
        *ptr++='/';
        *ptr='\0';
    }
    
	if((dp=opendir(path))==NULL)
        return 0;
    
	while((dirp=readdir(dp)))
    {
        if( ::streq(dirp->d_name, ".") || ::streq(dirp->d_name, "..") )
            continue;
        strlcpy(ptr, dirp->d_name, strlen(dirp->d_name)+1);
		
		// Recursive call
        dsize+=pathsize(path);
    }
	
    ptr[-1]='\0'; // ??
	
    if(closedir(dp)<0)
        fprintf(stderr,_("Cannot close folder %s\n"), path);
    return dsize;
}

// Write the file size in human readable form (bytes or Kbytes or Mbytes)
FXString hSize(FXchar* size)
{
    int flag=0;
    FXchar suf[64];
	FXchar buf[128];
	FXString hsize;

    unsigned long long lsize = strtoull(size,NULL,10);
    float fsize=0.0;

	strlcpy(suf,_("bytes"),sizeof(suf)); 
	if(lsize>1000000000)
    {
        fsize=lsize/1073741824.0;
        strlcpy(suf,_("GB"),sizeof(suf));
        flag=1;
    }
	else if(lsize>1000000)
    {
        fsize=lsize/1048576.0;
        strlcpy(suf,_("MB"),sizeof(suf));
        flag=1;
    }
    else if(lsize>1000)
    {
        fsize=lsize/1024.0;
        strlcpy(suf,_("KB"),sizeof(suf));
        flag=1;
    }
  	
	if(flag)
    {
		if(fsize==(int)fsize)
            snprintf(buf,sizeof(buf),"%.0f %s",fsize,suf);
        else
            snprintf(buf,sizeof(buf),"%.1f %s",fsize,suf);
    }
    else
        snprintf(buf,sizeof(buf),"%llu %s",lsize,suf);
	
	hsize=buf;
	return hsize;
}


// Remove terminating '/' on a path string to simplify a file or directory path
// Thus '/bla/bla////' becomes '/bla/bla'
// Special case : '/' stays to '/'
FXString cleanPath(const FXString path)
{
	FXString in=path, out=path;
	while (1)
	{
		if (in[in.length()-1]=='/' && in.length()!=1)
		{
			out=in.trunc(in.length()-1);
			in=out;
		}
		else
			break;
	}	
	return (out);
}

// Return the absolute path, based on the current directory path
// Remove terminating '/' on a path string to simplify a file or directory path
// Thus '/bla/bla////' becomes '/bla/bla'
// Special case : '/' stays to '/'
FXString filePath(const FXString path)
{
	FXString in=path, out=path;
	while (1)
	{
		if (in[in.length()-1]=='/' && in.length()!=1)
		{
			out=in.trunc(in.length()-1);
			in=out;
		}
		else
			break;
	}
	FXString dir=FXSystem::getCurrentDirectory();

	// If absolute path
	if(ISPATHSEP(out[0]))
		return (out);
	else
		return (dir+PATHSEPSTRING+out);
}

// Return the absolute path, based on the specified directory path
// Remove terminating '/' on a path string to simplify a file or directory path
// Thus '/bla/bla////' becomes '/bla/bla'
// Special case : '/' stays to '/'
FXString filePath(const FXString path, const FXString dir)
{
	FXString in=path, out=path;
	while (1)
	{
		if (in[in.length()-1]=='/' && in.length()!=1)
		{
			out=in.trunc(in.length()-1);
			in=out;
		}
		else
			break;
	}
	// If absolute path
	if(ISPATHSEP(out[0]))
		return (out);
	else
		return (dir+PATHSEPSTRING+out);
}


// Obtain a unique file name based on the current time stamp, such as : filename__TRASH__tv_sec-tv_usec
// If filename is a path name, only the modified filename is returned
// Used for file deletion to trash can
FXString stampname(const FXString filename)
{
	struct timeval tv;
	char stamp[48];
	
	gettimeofday(&tv,NULL);
	snprintf(stamp,sizeof(stamp)-1,"__%s__%ld-%ld",_("TRASH"),tv.tv_sec,tv.tv_usec);
	return (FXPath::name(filename)+stamp);
}


// Test if a directory is empty
// Return -1 if not a directory, 1 if empty and 0 if not empty
FXint isEmptyDir(const FXString directory)
{
	FXint ret=-1;
	DIR* dir;
	struct dirent *entry;
	int n=0;
	
	if ((dir=opendir(directory.text()))!=NULL)
	{
		while (n<3)	{entry=readdir(dir); n++;}
		if (entry==NULL)
			ret=1;
		else
			ret=0;
	}
	if (dir)
		closedir(dir);
	return ret;
}
		
// Check if file exists
FXbool exists(const FXString& file)
{
  	struct stat status;
  	return !file.empty() && (lstatrep(file.text(),&status)==0);
}

// Check if the file represents a directory
FXbool isDirectory(const FXString& file)
{
  	struct stat status;
  	return !file.empty() && (statrep(file.text(),&status)==0) && S_ISDIR(status.st_mode);
}

// Check if file represents a file
FXbool isFile(const FXString& file)
{
  	struct stat status;
  	return !file.empty() && (statrep(file.text(),&status)==0) && S_ISREG(status.st_mode);
}

// Check if current user is member of gid
// (thanks to Armin Buehler <abuehler@users.sourceforge.net>)
FXbool isGroupMember(gid_t gid)
{
	static long ngroups = 0;
	static gid_t *gids = NULL;
	long i;

	// First call : initialization of the number of supplementary groups and the group list
	if (ngroups == 0)
	{
		ngroups = getgroups(0, gids);
		gids = new gid_t[ngroups];
		getgroups(ngroups, gids);
	}
	if (ngroups == 0)
		return FALSE;

	// Check if the group id is contained within the group list
	i = ngroups;
	while (i--)
	{
		if (gid == gids[i])
			return TRUE;
	}
	return FALSE;
}


// Check if the file or the link refered file is readable AND executable
// Function used to test if we can enter a directory
// Uses the access() system function 
FXbool isReadExecutable(const FXString& file)
{	
  	struct stat status;

	// File exists and can be stated
	if (!file.empty() && (statrep(file.text(),&status)==0))
	{
		int ret=access(file.text(),R_OK|X_OK);
		if (ret==0)
			return TRUE;
		else
			return FALSE;
	}	
	
	// File doesn't exist
	else
		return FALSE;
}


FXbool isReadable(const FXString& file)
{	
  	struct stat status;

	// File exists and can be stated
	if (!file.empty() && (statrep(file.text(),&status)==0))
	{
		int ret=access(file.text(),R_OK);
		if (ret==0)
			return TRUE;
		else
			return FALSE;
	}	
	
	// File doesn't exist
	else
		return FALSE;
}


FXbool isWritable(const FXString& file)
{	
  	struct stat status;

	// File exists and can be stated
	if (!file.empty() && (statrep(file.text(),&status)==0))
	{
		int ret=access(file.text(),W_OK);
		if (ret==0)
			return TRUE;
		else
			return FALSE;
	}	
	
	// File doesn't exist
	else
		return FALSE;
}


// Check if file represents a link
FXbool isLink(const FXString& file)
{
  	struct stat status;
  	return !file.empty() && (lstatrep(file.text(),&status)==0) && S_ISLNK(status.st_mode);
}

// Get file info (file or link refered file)
FXbool info(const FXString& file,struct stat& inf)
{
  	return !file.empty() && (statrep(file.text(),&inf)==0);
}

// Return permissions string
// (the FOX function FXSystem::modeString() seems to use another format for the mode field) 
FXString permissions(FXuint mode)
{
	FXchar result[11];
	result[0]=S_ISLNK(mode) ? 'l' : S_ISREG(mode) ? '-' : S_ISDIR(mode) ? 'd' : S_ISCHR(mode) ? 'c' : S_ISBLK(mode) ? 'b' : S_ISFIFO(mode) ? 'p' : S_ISSOCK(mode) ? 's' : '?';
	result[1]=(mode&S_IRUSR) ? 'r' : '-';
	result[2]=(mode&S_IWUSR) ? 'w' : '-';
	result[3]=(mode&S_ISUID) ? 's' : (mode&S_IXUSR) ? 'x' : '-';
	result[4]=(mode&S_IRGRP) ? 'r' : '-';
	result[5]=(mode&S_IWGRP) ? 'w' : '-';
	result[6]=(mode&S_ISGID) ? 's' : (mode&S_IXGRP) ? 'x' : '-';
	result[7]=(mode&S_IROTH) ? 'r' : '-';
	result[8]=(mode&S_IWOTH) ? 'w' : '-';
	result[9]=(mode&S_ISVTX) ? 't' : (mode&S_IXOTH) ? 'x' : '-';
	result[10]=0;
	return result;
}

  // Read symbolic link
FXString readlink(const FXString& file)
{
	FXchar lnk[MAXPATHLEN+1];
	FXint len=readlink(file.text(),lnk,MAXPATHLEN);
	if(0<=len)
  		return FXString(lnk,len);
  	else
  		return FXString::null;
}


// Return true if files are identical
// Compare file names and inodes for case insensitive filesystems
FXbool identical(const FXString& file1,const FXString& file2)
{
	if(file1!=file2)
	{
    	struct stat stat1,stat2;
    	return !::lstatrep(file1.text(),&stat1) && !::lstatrep(file2.text(),&stat2) && stat1.st_ino==stat2.st_ino && stat1.st_dev==stat2.st_dev;
    }
  	return TRUE;
}


// Message Map
FXDEFMAP(File) FileMap[]={
                             FXMAPFUNC(SEL_COMMAND,File::ID_CANCEL_BUTTON,File::onCmdCancel),
							 FXMAPFUNC(SEL_TIMEOUT,File::ID_TIMEOUT,File::onTimeout),
                         };

// Object implementation
FXIMPLEMENT(File,DialogBox,FileMap,ARRAYNUMBER(FileMap))

// Construct object
File::File(FXWindow *owner, FXString title, const unsigned int operation):DialogBox(owner,title,DECOR_TITLE|DECOR_BORDER)
{
    // Progress window
	FXPacker *buttons=new FXPacker(this,LAYOUT_SIDE_BOTTOM|LAYOUT_FILL_X,0,0,10,10,PROGRESSDIALOG_WIDTH,PROGRESSDIALOG_WIDTH,5,5);
    new FXHorizontalSeparator(this,LAYOUT_SIDE_BOTTOM|LAYOUT_FILL_X|SEPARATOR_GROOVE);
    FXVerticalFrame *contents=new FXVerticalFrame(this,LAYOUT_SIDE_TOP|FRAME_NONE|LAYOUT_FILL_X|LAYOUT_FILL_Y);
	allowProgressDialog=TRUE;	

    // Cancel Button
    cancelButton=new FXButton(buttons,_("&Cancel"),NULL,this,File::ID_CANCEL_BUTTON,FRAME_RAISED|FRAME_THICK|LAYOUT_CENTER_X,0,0,0,0,20,20);
    cancelButton->setFocus();
    cancelButton->addHotKey(KEY_Escape);
	isCancelled=FALSE;

	// Progress bar
	progressbar=NULL;

	// Progress dialog depends on the file operation
	switch(operation)	
	{
		case COPY:
	    	// Labels and progress bar
			uplabel=new FXLabel(contents,_("Source:"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_X);
    		downlabel=new FXLabel(contents,_("Target:"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_X);
			progressbar=new FXProgressBar(contents,NULL,0,LAYOUT_FILL_X|FRAME_SUNKEN|FRAME_THICK|PROGRESSBAR_PERCENTAGE,0,0,0,0,PROGRESSDIALOG_WIDTH);
		
			// Timer on
			getApp()->addTimeout(this,File::ID_TIMEOUT,SHOW_PROGRESSBAR_DELAY);
			break;

    	case MOVE:
	    	// Labels and progress bar
			uplabel=new FXLabel(contents,_("Source:"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_X);
    		downlabel=new FXLabel(contents,_("Target:"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_X);
			progressbar=new FXProgressBar(contents,NULL,0,LAYOUT_FILL_X|FRAME_SUNKEN|FRAME_THICK|PROGRESSBAR_PERCENTAGE,0,0,0,0,PROGRESSDIALOG_WIDTH);

			// Timer on
			getApp()->addTimeout(this,File::ID_TIMEOUT,SHOW_PROGRESSBAR_DELAY);
			break;

    	case DELETE:
	    	// Labels
			uplabel=new FXLabel(contents,_("Delete:"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_X);
    		downlabel=new FXLabel(contents,_("From:"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_X);

			// Timer on
			getApp()->addTimeout(this,File::ID_TIMEOUT,SHOW_PROGRESSBAR_DELAY);
			break;

    	case CHMOD:
	    	// Labels
			uplabel=new FXLabel(contents,_("Changing permissions..."),NULL,JUSTIFY_LEFT|LAYOUT_FILL_X);
    		downlabel=new FXLabel(contents,_("File:"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_X);

			// Timer on
			getApp()->addTimeout(this,File::ID_TIMEOUT,SHOW_PROGRESSBAR_DELAY);
			break;

    	case CHOWN:
	    	// Labels
			uplabel=new FXLabel(contents,_("Changing owner..."),NULL,JUSTIFY_LEFT|LAYOUT_FILL_X);
    		downlabel=new FXLabel(contents,_("File:"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_X);

			// Timer on
			getApp()->addTimeout(this,File::ID_TIMEOUT,SHOW_PROGRESSBAR_DELAY);
			break;

#if defined(linux)
    	case MOUNT:
	    	// Labels
			uplabel=new FXLabel(contents,_("Mount file system..."),NULL,JUSTIFY_LEFT|LAYOUT_FILL_X);
    		downlabel=new FXLabel(contents,_("Mount the folder:"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_X);
			break;

    	case UNMOUNT:
	    	// Labels
			uplabel=new FXLabel(contents,_("Unmount file system..."),NULL,JUSTIFY_LEFT|LAYOUT_FILL_X);
    		downlabel=new FXLabel(contents,_("Unmount the folder:"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_X);
			break;
#endif

		default: // Other : RENAME, SYMLINK, ARCHIVE, EXTRACT, PKG_INSTALL, PKG_UNINSTALL
			// Progress dialog not used
			uplabel=NULL;
    		downlabel=NULL;
	}
	
	FXbool confirm_overwrite=getApp()->reg().readUnsignedEntry("OPTIONS","confirm_overwrite",TRUE);
	
	// Initialize the overwrite flags
	if (confirm_overwrite)
	{
		overwrite=FALSE;
		overwrite_all=FALSE;
		skip_all=FALSE;
	}
	else
	{
		overwrite=TRUE;
		overwrite_all=TRUE;
		skip_all=FALSE;
	}		    			
}

// Destructor
File::~File()
{
	getApp()->removeTimeout(this,File::ID_TIMEOUT);
    delete progressbar;
}



// Create and initialize
void File::create()
{
	DialogBox::create();
}


// Force check of timeout for progress dialog (to avoid latency problems)
int File::checkTimeout(void)
{
	if (getApp()->hasTimeout(this,File::ID_TIMEOUT))
	{
		if (getApp()->remainingTimeout(this,File::ID_TIMEOUT)==0)
		{
			getApp()->removeTimeout(this,File::ID_TIMEOUT);
			if (allowProgressDialog)
			{

				show(PLACEMENT_OWNER);
			}
				
			getApp()->forceRefresh();
			getApp()->flush();
			return 1;
		}
	}
	return 0;
}


// Force timeout for progress dialog (used before opening dialogs)
int File::forceTimeout(void)
{
	getApp()->removeTimeout(this,File::ID_TIMEOUT);
	if (allowProgressDialog)
		show(PLACEMENT_OWNER);
	getApp()->forceRefresh();
	getApp()->flush();
	return 1;
}


// Read bytes
long File::fullread(int fd,unsigned char *ptr,long len)
{
    long nread;
#ifdef EINTR
    do
        nread=read(fd,ptr,len);
    while(nread<0 && errno==EINTR);
#else
    nread=read(fd,ptr,len);
#endif
    return nread;
}


// Write bytes
long File::fullwrite(int fd,const unsigned char *ptr,long len)
{
    long nwritten,ntotalwritten=0;
    while(len>0)
    {
        nwritten=write(fd,ptr,len);
        if(nwritten<0)
        {
#ifdef EINTR
            if(errno==EINTR)
                continue;
#endif
            return -1;
        }
        ntotalwritten+=nwritten;
        ptr+=nwritten;
        len-=nwritten;
    }
    return ntotalwritten;
}


// Copy ordinary file
FXbool File::copyfile(const FXString& source, const FXString& target, const FXbool preserve_date)
{
	FXString destfile;
	unsigned char buffer[16384];
    struct stat status;
	struct utimbuf timbuf;
    long long nread,nwritten;
	long long size, nread_all=0;
	double pct=0;
    int src,dst;
    FXbool ok=FALSE;

    if((src=::open(source.text(),O_RDONLY))>=0)
    {
        if(statrep(source.text(),&status)==0)
        {
			// If destination is a directory
			if(::isDirectory(target))
				destfile=target+PATHSEPSTRING+FXPath::name(source);
			else
				destfile=target;
				
         	// Copy file block by block
			size=status.st_size; 
			if((dst=::open(destfile.text(),O_WRONLY|O_CREAT|O_TRUNC,status.st_mode))>=0)
            {
                while(1)
                {
                    nread=File::fullread(src,buffer,sizeof(buffer));
                    if(nread<0)
					{
						forceTimeout();
						if (errno)
							MessageBox::error(this,BOX_OK,_("Error"),"Can't copy file '%s': %s",target.text(),strerror(errno));
						else
							MessageBox::error(this,BOX_OK,_("Error"),"Can't copy file '%s'",target.text());
						goto err;
					}
                    if(nread==0)
                        break;
											
					// Force timeout checking for progress dialog
					checkTimeout();

					// Set percentage value for progress dialog
					nread_all+=nread;
					pct=100*(double)nread_all/(double)size;	
					if (progressbar)
						progressbar->setProgress((int)pct);

					// Give cancel button an opportunity to be clicked
					if(cancelButton)
						getApp()->runModalWhileEvents(cancelButton);

					// Set labels for progress dialog
					FXString label=_("Source: ")+source;
					uplabel->setText(label);
					label=_("Target: ")+target;
					downlabel->setText(label);
					getApp()->repaint();

					// If cancel button was clicked, close files and return
					if (isCancelled)
					{
						::close(dst);
						::close(src);
						return FALSE;
					}
                    nwritten=File::fullwrite(dst,buffer,nread);
                    if(nwritten<0)
					{
						forceTimeout();
						if (errno)
							MessageBox::error(this,BOX_OK,_("Error"),"Can't copy file '%s': %s",target.text(),strerror(errno));
						else
							MessageBox::error(this,BOX_OK,_("Error"),"Can't copy file '%s'",target.text());
                        goto err;
					}
                }
                ok=TRUE;
err:
                ::close(dst);
				
				// Keep original date if asked
				if (preserve_date)
				{
					timbuf.actime=status.st_atime;
					timbuf.modtime=status.st_mtime;
					if(utime(destfile.text(),&timbuf)==-1)
					{
						forceTimeout();
						if (errno)
							MessageBox::error(this,BOX_OK,_("Error"),"Can't copy file '%s': %s",target.text(),strerror(errno));
						else
							MessageBox::error(this,BOX_OK,_("Error"),"Can't copy file '%s'",target.text());
					}
				}
            }
        }
        ::close(src);
    }
    return ok;
}


// Copy directory
FXbool File::copydir(const FXString& source,const FXString& target,struct stat& parentstatus,inodelist* inodes, const FXbool preserve_date)
{
    DIR *dirp;
    struct dirent *dp;
    struct stat status;
	struct utimbuf timbuf;
    inodelist *in,inode;
    FXString destfile,oldchild,newchild;

	// If destination is an existing directory
	if(::isDirectory(target))
		destfile=target+PATHSEPSTRING+FXPath::name(source);
	else
		destfile=target;

    // See if visited this inode already
    for(in=inodes; in; in=in->next)
    {
        if(in->st_ino==parentstatus.st_ino)
            return TRUE;
    }

    // Try make directory, if none exists yet
    if(mkdir(destfile.text(),parentstatus.st_mode|S_IWUSR)!=0 && errno!=EEXIST)
        return FALSE;

    // Can we stat it
    if(lstatrep(destfile.text(),&status)!=0 || !S_ISDIR(status.st_mode))
        return FALSE;

    // Try open directory to copy
    dirp=opendir(source.text());
    if(!dirp)
        return FALSE;

    // Add this to the list
    inode.st_ino=status.st_ino;
    inode.next=inodes;

    // Copy stuff
    while((dp=readdir(dirp))!=NULL)
    {
        if(dp->d_name[0]!='.' || (dp->d_name[1]!='\0' && (dp->d_name[1]!='.' || dp->d_name[2]!='\0')))
        {
            oldchild=source;
            if(!ISPATHSEP(oldchild[oldchild.length()-1]))
                oldchild.append(PATHSEP);
            oldchild.append(dp->d_name);
            newchild=destfile;
            if(!ISPATHSEP(newchild[newchild.length()-1]))
                newchild.append(PATHSEP);
            newchild.append(dp->d_name);
			if(!copyrec(oldchild,newchild,&inode,preserve_date))
            {
				// If the cancel button was pressed
				if (isCancelled)
				{
                	closedir(dirp);
                	return FALSE;
				}
				
				// Or a permission problem occured
				else
				{
					FXString str;
					if (::isDirectory(oldchild))
						str=_("Can't copy folder ") + oldchild + _(": Permission denied");
					else
						str=_("Can't copy file ") + oldchild + _(": Permission denied");
       				forceTimeout();
					MessageBox box(this,_("Error"),str,errorbigicon,BOX_OK_CANCEL|DECOR_TITLE|DECOR_BORDER);   
					FXuint answer=box.execute(PLACEMENT_SCREEN);
					if(answer == BOX_CLICKED_CANCEL)
					{
               			closedir(dirp);
						isCancelled=TRUE;
                		return FALSE;
					}					
				}
            }
        }
    }

    // Close directory
    closedir(dirp);

	// Keep original date if asked
	if (preserve_date)
	{
		if(lstatrep(source.text(),&status)==0)
		{
			timbuf.actime=status.st_atime;
			timbuf.modtime=status.st_mtime;
			if(utime(destfile.text(),&timbuf)==-1)
			{
				forceTimeout();
				if (errno)
					MessageBox::error(this,BOX_OK,_("Error"),"Can't copy folder '%s': %s",target.text(),strerror(errno));
				else
					MessageBox::error(this,BOX_OK,_("Error"),"Can't copy folder '%s'",target.text());
			}
		}
	}
	
    // Success
    return TRUE;
}


// Recursive copy
FXbool File::copyrec(const FXString& source,const FXString& target,inodelist* inodes, const FXbool preserve_date)
{
    struct stat status1,status2;
	
    // Source file or directory does not exist
    if(lstatrep(source.text(),&status1)!=0)
        return FALSE;

    // If target is not a directory, remove it if allowed
    if(lstatrep(target.text(),&status2)==0)
    {
        if(!S_ISDIR(status2.st_mode))
        {
            if(!(overwrite|overwrite_all))
                return FALSE;
            if(unlink(target.text())!=0)
                return FALSE;
        }
    }

	// Disabled in version 1.06 because of problems with broken links
	// Source is not readable for the current user
	
	
	//if (!isReadable(source))
	//{
		//forceTimeout();
		//MessageBox::error(this,BOX_OK,_("Error"),_("Source %s is not readable"),source.text());
		//show();
		//return FALSE;
	//}
	
	
		
	// Source is directory: copy recursively
    if(S_ISDIR(status1.st_mode))
        return File::copydir(source,target,status1,inodes,preserve_date);

    // Source is regular file: copy block by block
    if(S_ISREG(status1.st_mode))
        return File::copyfile(source,target,preserve_date);

    // Source is fifo: make a new one
    if(S_ISFIFO(status1.st_mode))
        return mkfifo(target.text(),status1.st_mode);

    // Source is device: make a new one
    if(S_ISBLK(status1.st_mode) || S_ISCHR(status1.st_mode) || S_ISSOCK(status1.st_mode))
        return mknod(target.text(),status1.st_mode,status1.st_rdev)==0;

    // Source is symbolic link: make a new one
    if(S_ISLNK(status1.st_mode))
    {
        FXString lnkfile=readlink(source);
        return ::symlink(lnkfile.text(),target.text())==0;
    }

    // This shouldn't happen
    return FALSE;
}



// Copy file
FXbool File::copy(const FXString& source, const FXString& target, const FXbool confirm_dialog, const FXbool preserve_date)
{
	FXString targetfile;

	// Source doesn't exist
	if(!::exists(source))
	{
		forceTimeout();
		MessageBox::error(this,BOX_OK,_("Error"),_("Source %s doesn't exist"), source.text());
		return FALSE;
	}

	// Source and target are identical
	if (::identical(target,source))
	{
		forceTimeout();
		MessageBox::error(this,BOX_OK,_("Error"),_("Source %s is identical to target"), target.text());
		return FALSE;
	}

	// Target is an existing directory
	if (::isDirectory(target))
		targetfile=target+PATHSEPSTRING+FXPath::name(source);
	else
		targetfile=target;

	// Source and target file are identical
	if (::identical(targetfile,source))
	{
		forceTimeout();
		MessageBox::error(this,BOX_OK,_("Error"),_("Source %s is identical to target"), targetfile.text());			
		return FALSE;
	}

	// Target already exists
	if(::exists(targetfile))
	{
		// Overwrite dialog if necessary
		if (!(overwrite_all | skip_all) & confirm_dialog)
    	{
			FXString label=_("Source: ")+source;
			uplabel->setText(label);
			label=_("Target: ")+targetfile;
			downlabel->setText(label);
			getApp()->repaint();
			forceTimeout();
			FXString msg=_("File ")+targetfile+_(" already exists. Overwrite?");
       		ConfirmOverWriteBox* dlg = new ConfirmOverWriteBox(this,_("Confirm Overwrite"),msg);
       		FXuint answer=dlg->execute();
			delete dlg;
       		switch(answer)
       		{
       			case 1:
           			overwrite = TRUE;
           			break;
       			case 2:
           			overwrite_all = TRUE;
           			break;
       			case 3:
					overwrite = FALSE;
           			break;
       			case 4:
           			skip_all = TRUE;
					break;
       		}
		}
		if(!(overwrite | overwrite_all) | skip_all)
			return TRUE;

		// Use of the standard C function to remove the target
		// avoids to overwrite a non empty directory
		if (::remove(targetfile.text())==-1)
		{
			forceTimeout();
			MessageBox::error(this,BOX_OK,_("Error"),_("Cannot overwrite the non empty folder %s"), targetfile.text());			
			return FALSE;
		}
	}

	// Copy file or directory
	return File::copyrec(source,targetfile,NULL,preserve_date);
}


// Remove file or directory
FXbool File::remove(const FXString& file)
{
	FXString dirname;
    struct stat status;
	static FXbool ISDIR=FALSE;

    if(lstatrep(file.text(),&status)==0)
    {
        // It is a directory
		if(S_ISDIR(status.st_mode))
        {
            DIR *dirp=opendir(file.text());
            if(dirp)
            {			
                struct dirent *dp;
                FXString child;
				
				// Used to display only one progress dialog when deleting a directory
				ISDIR=TRUE;
							
				// Force timeout checking for progress dialog
				checkTimeout();

				// Give cancel button an opportunity to be clicked
				if(cancelButton)
					getApp()->runModalWhileEvents(cancelButton);

				// Set labels for progress dialog
				FXString label=_("Delete folder: ")+file;
				if (uplabel)
					uplabel->setText(label);
				dirname=FXPath::directory(FXPath::absolute(file));
				label=_("From: ")+dirname;
				if (downlabel)
					downlabel->setText(label);
				getApp()->repaint();

				// If cancel button was clicked, return
				if (isCancelled)
					return FALSE;

                while((dp=readdir(dirp))!=NULL)
                {
					if(dp->d_name[0]!='.' || (dp->d_name[1]!='\0' && (dp->d_name[1]!='.' || dp->d_name[2]!='\0')))
                    {
                        child=file;
                        if(!ISPATHSEP(child[child.length()-1]))
                            child.append(PATHSEP);
                        child.append(dp->d_name);
                        if(!File::remove(child))
						{
							closedir(dirp);
							return FALSE;
						}
                    }
                }
                closedir(dirp);
            }
			if (rmdir(file.text())==-1)
			{
				forceTimeout();
				MessageBox::error(this,BOX_OK,_("Error"),_("Cannot delete folder %s"), file.text());			
				return FALSE;
			}
			else
				return TRUE;
        }
        else
        {
			// If it was not a directory
			if (!ISDIR)
			{
				// Force timeout checking for progress dialog
				checkTimeout();

				// Give cancel button an opportunity to be clicked
				if(cancelButton)
					getApp()->runModalWhileEvents(cancelButton);

				// Set labels for progress dialog
				FXString label=_("Delete: ")+file;
				if (uplabel)
					uplabel->setText(label);
				dirname=FXPath::directory(FXPath::absolute(file));
				label=_("From: ")+dirname;
				if (downlabel)
					downlabel->setText(label);
				getApp()->repaint();

				// If cancel button was clicked, return
				if (isCancelled)
					return FALSE;
			}
			if (unlink(file.text())==-1)
			{
				forceTimeout();
				MessageBox::error(this,BOX_OK,_("Error"),_("Cannot delete file %s"), file.text());			
				return FALSE;
			}
			else
				return TRUE;
        }
    }
    return FALSE;
}


// Rename a file or a directory
FXbool File::rename(const FXString& source, const FXString& target)
{
	// Source doesn't exist
    if(!::exists(source))
	{
		MessageBox::error(this,BOX_OK,_("Error"),_("Source %s doesn't exist"), source.text());
		return FALSE;
	}
			
	// Source and target are identical
	if (::identical(target,source))
	{
		MessageBox::error(this,BOX_OK,_("Error"),_("Source %s is identical to target"), target.text());
		return FALSE;
	}

	// Target already exists
	if(::exists(target))
	{
		// Overwrite dialog if necessary
		if (!(overwrite_all | skip_all))
    	{
			FXString msg=_("File ")+target+_(" already exists. Overwrite?");
       		ConfirmOverWriteBox* dlg = new ConfirmOverWriteBox(this,_("Confirm Overwrite"),msg);
       		FXuint answer=dlg->execute();
			delete dlg;
       		switch(answer)
       		{
       			case 1:
           			overwrite = TRUE;
           			break;
       			case 2:
           			overwrite_all = TRUE;
           			break;
       			case 3:
					overwrite = FALSE;
           			break;
       			case 4:
           			skip_all = TRUE;
					break;
       		}
		}
		if(!(overwrite | overwrite_all) | skip_all)
			return TRUE;
	}
       		
	// Rename file using the standard C function
	// This should only work for files that are on the same file system
	if (::rename(source.text(),target.text())==0)
		return TRUE;
	if(errno!=EXDEV)
	{
		MessageBox::error(this,BOX_OK,_("Error"),_("Cannot rename to target %s"), target.text());	
		return FALSE;
	}

	// If files are on different file systems, use the copy/delete scheme and preserve the original date
	FXbool ret=this->copy(source,target,FALSE,TRUE);
	if (ret)
		return (remove(source.text())==TRUE);
	else
		return FALSE;
}


// Move files
FXbool File::move(const FXString& source,const FXString& target)
{
	// Source doesn't exist
	if(!::exists(source))
	{
		forceTimeout();
		MessageBox::error(this,BOX_OK,_("Error"),_("Source %s doesn't exist"), source.text());
		return FALSE;
	}

	// Source and target are identical
	if (identical(target,source))
	{
		forceTimeout();
		MessageBox::error(this,BOX_OK,_("Error"),_("Source %s is identical to target"), target.text());
		return FALSE;
	}

	// Target is an existing directory
	FXString targetfile;
	if (::isDirectory(target))
		targetfile=target+PATHSEPSTRING+FXPath::name(source);
	else
		targetfile=target;

	// Source and target file are identical
	if (::identical(targetfile,source))
	{
		forceTimeout();
		MessageBox::error(this,BOX_OK,_("Error"),_("Source %s is identical to target"), target.text());
		return FALSE;
	}

	// Force timeout checking for progress dialog
	checkTimeout();

	// Give cancel button an opportunity to be clicked
	if(cancelButton)
		getApp()->runModalWhileEvents(cancelButton);

	// Set labels for progress dialog
	FXString label=_("Source: ")+source;
	uplabel->setText(label);
	label=_("Target: ")+target;
	downlabel->setText(label);
	getApp()->repaint();

	// Target file already exists
	if(::exists(targetfile))
    {
		// Overwrite dialog if necessary
		if (!overwrite_all & !skip_all)
		{
			forceTimeout();
			FXString msg=_("File ")+targetfile+_(" already exists. Overwrite?");
       		ConfirmOverWriteBox* dlg = new ConfirmOverWriteBox(this,_("Confirm Overwrite"),msg);
       		FXuint answer=dlg->execute();
			delete dlg;
       		switch(answer)
       		{
       			case 1:
           			overwrite = TRUE;
           			break;
       			case 2:
           			overwrite_all = TRUE;
           			break;
       			case 3:
           			overwrite = FALSE;
					break;
       			case 4:
           			skip_all = TRUE;
					break;
       		}
			
		}
		if(!(overwrite | overwrite_all) | skip_all)
			return TRUE;
	}

	// Set the progress to 100% for move file operation
	if (progressbar)
		progressbar->setProgress(100);

	// Rename file using the standard C function
	// This should only work for files that are on the same file system
	if (::rename(source.text(),targetfile.text())==0)
		return TRUE;
	if(errno!=EXDEV)
	{
		forceTimeout();
		MessageBox::error(this,BOX_OK,_("Error"),_("Cannot rename to target %s"), targetfile.text());	
		return FALSE;
	}

	// If files are on different file systems, use the copy/delete scheme and preserve the original date
	FXbool ret=this->copy(source,targetfile,FALSE,TRUE);
	if (ret)
		return (remove(source.text())==TRUE);
	else
		return FALSE;
}


// Symbolic Link file
FXbool File::symlink(const FXString& source,const FXString& target)
{
	// Source doesn't exist
    if(!::exists(source))
	{
		MessageBox::error(this,BOX_OK,_("Error"),_("Source %s doesn't exist"), source.text());
		return FALSE;
	}
			
	// Source and target are identical
	if (::identical(target,source))
	{
		MessageBox::error(this,BOX_OK,_("Error"),_("Source %s is identical to target"), target.text());
		return FALSE;
	}

	// Target is an existing directory
	FXString targetfile;
	if (::isDirectory(target))
		targetfile=target+PATHSEPSTRING+FXPath::name(source);
	else
		targetfile=target;

	// Target already exists
	if(::exists(targetfile))
	{
		// Overwrite dialog if necessary
		if (!(overwrite_all | skip_all))
    	{
			FXString msg=_("File ")+targetfile+_(" already exists. Overwrite?");
       		ConfirmOverWriteBox* dlg = new ConfirmOverWriteBox(this,_("Confirm Overwrite"),msg);
       		FXuint answer=dlg->execute();
			delete dlg;
       		switch(answer)
       		{
       			case 1:
           			overwrite = TRUE;
           			break;
       			case 2:
           			overwrite_all = TRUE;
           			break;
       			case 3:
					overwrite = FALSE;
           			break;
       			case 4:
           			skip_all = TRUE;
					break;
       		}
		}
		if(!(overwrite | overwrite_all) | skip_all)
			return TRUE;
	}
    
	// Create symbolic link using the standard C function
	FXint ret=::symlink(source.text(),targetfile.text());	
	if (ret==0)
		return TRUE;
	else
	{
		if (errno)
			MessageBox::error(this,BOX_OK,_("Error"),"Can't symlink '%s': %s",target.text(),strerror(errno));
		else
			MessageBox::error(this,BOX_OK,_("Error"),"Can't symlink '%s'",target.text());
		return FALSE;
	}		
}


// Chmod a file or directory, recursively or not
// Note : the variable file returns the last processed file
// It can be different from the initial path, if recursive chmod is used
// (Used to fill an error message, if needed) 
int File::chmod(char *path, char *file, mode_t mode, FXbool rec, const FXbool dironly, const FXbool fileonly)
{			
    struct stat stats;

	// Initialise the file variable with the initial path
	strlcpy(file, path, strlen(path)+1);

    if (lstatrep(path, &stats)) //if it doesn't exist
        return -1;

    if (!S_ISDIR(stats.st_mode)) // file
	{
        if(dironly)
            return 0;
			
		// Force timeout checking for progress dialog
		checkTimeout();

		// Give cancel button an opportunity to be clicked
		if(cancelButton)
			getApp()->runModalWhileEvents(cancelButton);

		// Set labels for progress dialog
		FXString label=_("Changing permissions...");
		uplabel->setText(label);
		label=_("File: ")+FXString(path);
		downlabel->setText(label);
		getApp()->repaint();

		// If cancel button was clicked, return
		if (isCancelled)
			return -1;

        return ::chmod(path,mode);
    }
    else // directory
    {
		if(rec == FALSE && !fileonly)
        {
			// Force timeout checking for progress dialog
			checkTimeout();

			// Give cancel button an opportunity to be clicked
			if(cancelButton)
				getApp()->runModalWhileEvents(cancelButton);

			// Set labels for progress dialog
			FXString label=_("Changing permissions...");
			uplabel->setText(label);
			label=_("Folder: ")+FXString(path);
			downlabel->setText(label);
			getApp()->repaint();

			// If cancel button was clicked, return
			if (isCancelled)
				return -1;

			if(::chmod(path,mode)) // do not change recursively
				return -1;
        }
        else
            return rchmod(path,file,mode,dironly,fileonly); // recursive change
    }
    return 0;

}


// Recursive chmod for a directory
int File::rchmod(char *path, char *file, mode_t mode, const FXbool dironly, const FXbool fileonly)
{
    struct stat stats;

	// Initialise the file variable with the initial path
	strlcpy(file, path, strlen(path)+1);

    if (lstatrep(path, &stats)) //if it doesn't exist
        return -1;

    if (!S_ISDIR(stats.st_mode)) // file
    {
        if(dironly)
            return 0;

		// Force timeout checking for progress dialog
		checkTimeout();

		// Give cancel button an opportunity to be clicked
		if(cancelButton)
			getApp()->runModalWhileEvents(cancelButton);

		// Set labels for progress dialog
		FXString label=_("Changing permissions...");
		uplabel->setText(label);
		label=_("File: ")+FXString(path);
		downlabel->setText(label);
		getApp()->repaint();

		// If cancel button was clicked, return
		if (isCancelled)
			return -1;

        return ::chmod(path,mode);
    }

	DIR *dir;
    struct dirent *entry;
    int i, pl = strlen(path);

    if (!(dir = opendir(path)))
        return -1;

    for(i = 0; (entry = readdir(dir)); i++)
	{	
        if (entry->d_name[0] != '.' || (entry->d_name[1] != '\0'
                                        && (entry->d_name[1] != '.' ||
                                            entry->d_name[2] != '\0')))
        {
            int pl1 = pl, l = strlen(entry->d_name);
            char *path1 = (char *)alloca(pl1+l+2);

            strlcpy(path1, path, strlen(path)+1);
            if (path1[pl1-1] != '/')
                path1[pl1++] = '/';
            strlcpy(path1+pl1, entry->d_name, strlen(entry->d_name)+1);

			// Modify the file variable with the new path
			strlcpy(file, path1, strlen(path1)+1);
            if (rchmod(path1,file,mode,dironly,fileonly))
                return -1;
        }
	}

    if (closedir(dir))
        return -1;

	if (fileonly)
		return 0;
	else
		return ::chmod(path,mode);
}


// Chown a file or directory, recursively or not
// Note : the variable file returns the last processed file
// It can be different from the initial path, if recursive chmod is used
// (Used to fill an error message, if needed) 
int File::chown(char *path, char *file, uid_t uid, gid_t gid, const FXbool rec, const FXbool dironly, const FXbool fileonly)
{
    struct stat stats;

	// Initialise the file variable with the initial path
	strlcpy(file, path, strlen(path)+1);

    if (lstatrep(path, &stats)) //if it doesn't exist
        return -1;

    if (!S_ISDIR(stats.st_mode)) // file
    {
        if(dironly)
            return 0;

		// Force timeout checking for progress dialog
		checkTimeout();

		// Give cancel button an opportunity to be clicked
		if(cancelButton)
			getApp()->runModalWhileEvents(cancelButton);

		// Set labels for progress dialog
		FXString label=_("Changing owner...");
		uplabel->setText(label);
		label=_("File: ")+FXString(path);
		downlabel->setText(label);
		getApp()->repaint();

		// If cancel button was clicked, return
		if (isCancelled)
			return -1;

        if(::chown(path,uid,gid))
            return -1;
    }
    else // directory
    {	
		if(rec == FALSE && !fileonly)
        {
			// Force timeout checking for progress dialog
			checkTimeout();

			// Give cancel button an opportunity to be clicked
			if(cancelButton)
				getApp()->runModalWhileEvents(cancelButton);

			// Set labels for progress dialog
			FXString label=_("Changing owner...");
			uplabel->setText(label);
			label=_("Folder: ")+FXString(path);
			downlabel->setText(label);
			getApp()->repaint();

			// If cancel button was clicked, return
			if (isCancelled)
				return -1;

            if(::chown(path,uid,gid)) // do not change recursively
                return -1;
        }
        else
            if(rchown(path,file,uid,gid,dironly,fileonly)) // recursive change
                return -1;
    }
    return 0;

}


// Recursive chown for a directory
int File::rchown(char *path, char *file, uid_t uid, gid_t gid, const FXbool dironly, const FXbool fileonly)
{
    struct stat stats;

	// Initialise the file variable with the initial path
	strlcpy(file, path, strlen(path)+1);

    if (lstatrep(path, &stats)) //if it doesn't exist
        return -1;

    if (!S_ISDIR(stats.st_mode)) // file
    {
		if(dironly)
			return 0;

		// Force timeout checking for progress dialog
		checkTimeout();

		// Give cancel button an opportunity to be clicked
		if(cancelButton)
			getApp()->runModalWhileEvents(cancelButton);

		// Set labels for progress dialog
		FXString label=_("Changing owner...");
		uplabel->setText(label);
		label=_("File: ")+FXString(path);
		downlabel->setText(label);
		getApp()->repaint();

		// If cancel button was clicked, return
		if (isCancelled)
			return -1;

		return ::chown(path,uid,gid);
    }
	
    DIR *dir;
    struct dirent *entry;
    int i, pl = strlen(path);

    if (!(dir = opendir(path)))
        return -1;

    for(i = 0; (entry = readdir(dir)); i++)
        if (entry->d_name[0] != '.' || (entry->d_name[1] != '\0'
                                        && (entry->d_name[1] != '.' ||
                                            entry->d_name[2] != '\0')))
        {
            int pl1 = pl, l = strlen(entry->d_name);
            char *path1 = (char *)alloca(pl1+l+2);

            strlcpy(path1, path, strlen(path)+1);
            if (path1[pl1-1] != '/')
                path1[pl1++] = '/';
            strlcpy(path1+pl1, entry->d_name, strlen(entry->d_name)+1);
			strlcpy(file, path1, strlen(path1)+1);
            if (rchown(path1,file,uid,gid,dironly,fileonly))
                return -1;
        }

    if (closedir(dir))
        return -1;
 
	if (fileonly)
		return 0;
	else
		return ::chown(path,uid,gid);
}


// Extract an archive in a specified directory
int File::extract(const FXString name, const FXString dir, const FXString cmd)
{
	// Change to the specified directory
    FXString currentdir=FXSystem::getCurrentDirectory();
	chdir(dir.text());

    // Make and show command window
	CommandWindow *cmdwin=new CommandWindow(getApp(),_("Extract archive"),cmd,30,80);			
	cmdwin->create();
	cmdwin->setIcon(archexticon);

	// The command window object deletes itself after closing the window!

	// Return to initial directory
	chdir(currentdir.text());
	
    return 1;
}


// Create an archive
int File::archive(const FXString name, const FXString cmd)
{
	// Target file already exists
	if(::exists(FXPath::dequote(name)))
    {
		// Overwrite dialog if necessary
		if (!overwrite_all & !skip_all)
		{
			FXString msg=_("File ")+name+_(" already exists. Overwrite?");
       		ConfirmOverWriteBox* dlg = new ConfirmOverWriteBox(this,_("Confirm Overwrite"),msg);
       		FXuint answer=dlg->execute();
			delete dlg;
       		switch(answer)
       		{
       			case 1:
           			overwrite = TRUE;
           			break;
       			case 2:
           			overwrite_all = TRUE;
           			break;
       			case 3:
           			overwrite = FALSE;
					break;
       			case 4:
           			skip_all = TRUE;
					break;
       		}
		}
		if(!(overwrite | overwrite_all) | skip_all)
			return TRUE;
	}

    // Make and show command window
	CommandWindow *cmdwin=new CommandWindow(getApp(),_("Add to archive"),cmd,30,80);			
	cmdwin->create();
	cmdwin->setIcon(archaddicon);

	// The command window object deletes itself after closing the window!

    return 1;
}


#if defined(linux)
int File::mount(const FXString dir, const FXString msg, const FXString cmd, const unsigned int op)
{
	FXbool mount_messages=getApp()->reg().readUnsignedEntry("OPTIONS","mount_messages",TRUE);

	// Show progress dialog (no timer here)
	show(PLACEMENT_OWNER);
	getApp()->forceRefresh();
	getApp()->flush();

	// Set labels for progress dialog
	uplabel->setText(msg);				
	downlabel->setText(dir.text());
	getApp()->repaint();

	// Give cancel button an opportunity to be clicked
	if(cancelButton)
		getApp()->runModalWhileEvents(cancelButton);

	// If cancel button was clicked, return
	if (isCancelled)
		return -1;

	// Perform the mount/unmount command
	FILE *pcmd=popen(cmd.text(),"r");
	if(!pcmd)
	{
		MessageBox::error(this,BOX_OK,_("Error"),_("Failed command: %s"),cmd.text());
		return -1;
	}

	// Get error message if any
	char text[10000]={0};
	FXString buf;
	while(fgets(text,sizeof(text),pcmd))
		buf+=text;
	snprintf(text,sizeof(text)-1,"%s",buf.text());
	
	// Close the stream
	if(pclose(pcmd))
	{
		MessageBox::error(this,BOX_OK,_("Error"),"%s",text);
		return -1;
	}
	// Hide progress dialog
	hide();
	
	// Success message, eventually
	if(mount_messages)
	{
		if (op==MOUNT)
			MessageBox::information(this,BOX_OK,_("Success"),_("Folder %s was successfully mounted."),dir.text());
		else
			MessageBox::information(this,BOX_OK,_("Success"),_("Folder %s was successfully unmounted."),dir.text());
	}
	return 1;
}


// Install / Upgrade package
int File::pkgInstall(const FXString name, const FXString cmd)
{
    // Make and show command window
	CommandWindow *cmdwin=new CommandWindow(getApp(),_("Install/Upgrade package"),cmd,10,80);			
	cmdwin->create();

	FXString msg=_("Installing package: ") + name + "\n"; 
	cmdwin->appendText(msg.text());

	// The command window object deletes itself after closing the window!

    return 1;
}

// Uninstall package
int File::pkgUninstall(const FXString name, const FXString cmd)
{
    // Make and show command window
	CommandWindow *cmdwin=new CommandWindow(getApp(),_("Uninstall package"),cmd,10,80);			
	cmdwin->create();
	
	FXString msg=_("Uninstalling package: ") + name + "\n"; 
	cmdwin->appendText(msg.text());

	// The command window object deletes itself after closing the window!

    return 1;
}
#endif


// Handle cancel button in progress bar dialog
long File::onCmdCancel(FXObject*, FXSelector,void*)
{
	isCancelled=TRUE;
	return 1;
}


// Handle timeout for progress bar
long File::onTimeout(FXObject*, FXSelector,void*)
{
	if (allowProgressDialog)
		show(PLACEMENT_OWNER);
    getApp()->forceRefresh();
    getApp()->flush();
    return 1;
}

