#include "xml.h"

#include <qmessagebox.h>
#include <qcolor.h>
#include <qstylesheet.h>
#include <iostream>

#include "misc.h"
#include "settings.h"
#include "linkablemapobj.h"

#include "version.h"

static BranchObj *lastBranch;
static FloatObj *lastFloat;
static OrnamentedObj *lastOO;

extern Settings settings;

mapBuilderHandler::mapBuilderHandler() {}

mapBuilderHandler::~mapBuilderHandler() {}

QString mapBuilderHandler::errorProtocol() { return errorProt; }

bool mapBuilderHandler::startDocument()
{
    errorProt = "";
    state = StateInit;
    laststate = StateInit;
    branchDepth=0;
	htmldata="";
	isVymPart=false;
    return true;
}


QString mapBuilderHandler::parseHREF(QString href)
{
	QString type=href.section(":",0,0);
	QString path=href.section(":",1,1);
	if (!tmpDir.endsWith("/"))
		return tmpDir + "/" + path;
	else	
		return tmpDir + path;
}

bool mapBuilderHandler::startElement  ( const QString&, const QString&,
                    const QString& eName, const QXmlAttributes& atts ) 
{
    QColor col;
	//cout << "startElement <"<<eName<<">  state="<<state <<"  laststate="<<laststate<<"   loadMode="<<loadMode<<endl;
    if ( state == StateInit && (eName == "vymmap")  ) 
	{
        state = StateMap;
		if (!atts.value( "version").isEmpty() ) 
		{
			mc->setVersion(atts.value( "version" ));
			if (!mc->checkVersion())
				QMessageBox::warning( 0, "Warning: Version Problem" ,
				   "<h3>Map is newer than VYM</h3>"
				   "<p>The map you are just trying to load was "
				   "saved using vym " +atts.value("version")+". "
				   "The version of this vym is " __VYM_VERSION
				   ". If you run into problems after pressing "
				   "the ok-button below, updating vym should help.");

		}
		if (loadMode==NewMap)
		{
			if (!atts.value( "author").isEmpty() )
			{
				mc->setAuthor(atts.value( "author" ) );
			}
			if (!atts.value( "comment").isEmpty() )
			{
				mc->setComment (atts.value( "comment" ) );
			}
			if (!atts.value( "backgroundColor").isEmpty() )
			{
				col.setNamedColor(atts.value("backgroundColor"));
				mc->getCanvas()->setBackgroundColor(col);
			}	    
			if (!atts.value( "linkColorHint").isEmpty() ) 
			{
				if (atts.value("linkColorHint")=="HeadingColor")
					me->setLinkColorHint(HeadingColor);
				else
					me->setLinkColorHint(DefaultColor);
			}
			if (!atts.value( "linkStyle").isEmpty() ) 
			{
				QString s=atts.value("linkStyle");
				if (s=="StyleLine")
					me->setLinkStyle(StyleLine);
				else	
					if (s=="StyleParabel")
						me->setLinkStyle(StyleParabel);
					else	
						if (s=="StylePolyLine")
							me->setLinkStyle(StylePolyLine);
						else	
							me->setLinkStyle(StylePolyParabel);
			}	
			if (!atts.value( "linkColor").isEmpty() ) 
			{
				col.setNamedColor(atts.value("linkColor"));
				me->setLinkColor(col);
			}	
			if (!atts.value( "defXLinkColor").isEmpty() ) 
			{
				col.setNamedColor(atts.value("defXLinkColor"));
				me->setDefXLinkColor(col);
			}	
			if (!atts.value( "defXLinkWidth").isEmpty() ) 
			{
				me->setDefXLinkWidth(atts.value("defXLinkWidth").toInt ());
			}	
		}	
	} else if ( eName == "select" && state == StateMap ) 
	{
		state=StateMapSelect;
	} else if ( eName == "setting" && state == StateMap ) 
	{
		state=StateMapSetting;
		if (loadMode==NewMap)
			readSettingAttr (atts);
	} else if ( eName == "mapcenter" && state == StateMap ) 
	{
		state=StateMapCenter;
		if (loadMode==NewMap)
		{	
			// Really use the found mapcenter as MCO in a new map
			lastBranch=mc;	// avoid empty pointer
		} else
		{
			// Treat the found mapcenter as a branch 
			// in an existing map
			LinkableMapObj* lmo=me->getSelection();
			if (lmo && (typeid(*lmo) == typeid(BranchObj) ) 
			        || (typeid(*lmo) == typeid(MapCenterObj) ) )
			{
				lastBranch=(BranchObj*)lmo;
				if (loadMode==ImportAdd)
				{
					lastBranch->addBranch();
					lastBranch=lastBranch->getLastBranch();
				} else
					lastBranch->clear();
			} else
				return false;
		}
		readBranchAttr (atts);
	} else if ( (eName == "standardflag" ||eName == "standardFlag") && state == StateMapCenter) 
	{
		state=StateMapCenterStandardFlag;
	} else if ( eName == "heading" && state == StateMapCenter) 
	{
		state=StateMapCenterHeading;
		if (!atts.value( "textColor").isEmpty() ) 
		{
			col.setNamedColor(atts.value("textColor"));
			lastBranch->setColor(col );
		}	    
	} else if ( eName == "note" && state == StateMapCenter) 
	{	// only for backward compatibility (<1.4.6). Use htmlnote now.
		state=StateMapCenterNote;
		if (!readNoteAttr (atts) ) return false;
	} else if ( eName == "htmlnote" && state == StateMapCenter) 
	{
		laststate=state;
		state=StateHtmlNote;
    } else if ( eName == "floatimage" && state == StateMapCenter ) 
	{
		state=StateMapCenterFloatImage;
        lastBranch->addFloatImage();
		lastFloat=lastBranch->getLastFloatImage();
		if (!readFloatImageAttr(atts)) return false;
	} else if ( (eName == "branch"||eName=="floatimage") && state == StateMap) 
	{
		// This is used in vymparts, which have no mapcenter!
		isVymPart=true;
		LinkableMapObj* lmo=me->getSelection();
		if (!lmo)
		{
			// If a vym part is _loaded_ (not imported), 
			// selection==lmo==NULL
			// Treat it like ImportAdd then...
			loadMode=ImportAdd;
			lmo=mc;
		}	
		if (lmo && (typeid(*lmo) == typeid(BranchObj) ) 
				|| (typeid(*lmo) == typeid(MapCenterObj) ) )
		{
			lastBranch=(BranchObj*)(lmo);
			if (eName=="branch")
			{
				state=StateBranch;
				if (loadMode==ImportAdd)
				{
					lastBranch->addBranch();
					lastBranch=lastBranch->getLastBranch();
					
				} else
					lastBranch->clear();
				branchDepth=1;
				readBranchAttr (atts);
			} else if (eName=="floatimage")
			{
				state=StateFloatImage;
				lastBranch->addFloatImage();
				lastFloat=lastBranch->getLastFloatImage();
				if (!readFloatImageAttr(atts)) return false;
			} else return false;
		} else return false;
	} else if ( eName == "branch" && state == StateMapCenter) 
	{
		state=StateBranch;
		branchDepth=1;
		lastBranch->addBranch();
		lastBranch=lastBranch->getLastBranch();
		readBranchAttr (atts);
	} else if ( (eName=="standardflag" ||eName == "standardFlag") && state == StateBranch) 
	{
		state=StateBranchStandardFlag;
	} else if ( eName == "heading" && state == StateBranch) 
	{
		state=StateBranchHeading;
		if (!atts.value( "textColor").isEmpty() ) 
		{
			col.setNamedColor(atts.value("textColor"));
			lastBranch->setColor(col );
		}	    
    } else if ( eName == "note" && state == StateBranch) 
	{
        state=StateBranchNote;
		if (!readNoteAttr (atts) ) return false;
	} else if ( eName == "htmlnote" && state == StateBranch) 
	{
		laststate=state;
		state=StateHtmlNote;
		no.clear();
		if (!atts.value( "fonthint").isEmpty() ) 
			no.setFontHint(atts.value ("fonthint") );
    } else if ( eName == "floatimage" && state == StateBranch ) 
	{
		state=StateBranchFloatImage;
        lastBranch->addFloatImage();
		lastFloat=lastBranch->getLastFloatImage();
		if (!readFloatImageAttr(atts)) return false;
    } else if ( eName == "xlink" && state == StateBranch ) 
	{
		state=StateBranchXLink;
		if (!readXLinkAttr (atts)) return false;
    } else if ( eName == "branch" && state == StateBranch ) 
	{
        lastBranch->addBranch();
		lastBranch=lastBranch->getLastBranch();		
        branchDepth++;
		readBranchAttr (atts);
    } else if ( eName == "html" && state == StateHtmlNote ) 
	{
		state=StateHtml;
		htmldata="<"+eName;
		readHtmlAttr(atts);
		htmldata+=">";
    } else if ( state == StateHtml ) 
	{
		// accept all while in html mode,
		htmldata+="<"+eName;
		readHtmlAttr(atts);
		htmldata+=">";
    } else
        return false;   // Error
    return true;
}

bool mapBuilderHandler::endElement  ( const QString&, const QString&, const QString &eName)
{
//	cout << "endElement </"<<eName<<">  state="<<state <<"  laststate="<<laststate<<endl;
    switch ( state ) 
	{
        case StateMapSelect: state=StateMap;  return true;
        case StateMapSetting: state=StateMap;  return true;
        case StateMapCenter: state=StateMap;  return true;
        case StateMapCenterStandardFlag: state=StateMapCenter;  return true;
        case StateMapCenterHeading: state=StateMapCenter;  return true;
        case StateMapCenterNote: state=StateMapCenter;  return true;
        case StateMapCenterFloatImage: state=StateMapCenter;  return true;
        case StateFloatImage: state=StateMap; return true;
        case StateBranch: 
            if (branchDepth>1) 
			{
                branchDepth--;
                state=StateBranch;
            } else  
			{
                branchDepth=0;
				if (isVymPart)
					state=StateMap;
				else
					state=StateMapCenter;
            }   
			lastBranch=(BranchObj*)(lastBranch->getParObj());
             return true;
        case StateBranchStandardFlag: state=StateBranch;  return true;
        case StateBranchHeading: state=StateBranch;  return true;
        case StateBranchNote: state=StateBranch; return true;
        case StateBranchFloatImage: state=StateBranch;  return true;
        case StateBranchXLink: state=StateBranch;  return true;
        case StateHtmlNote: state=laststate; return true;
        case StateHtml: 
			htmldata+="</"+eName+">";
			if (eName=="html")
			{
				state=StateHtmlNote;  
				htmldata.replace ("<br></br>","<br />");
				no.setNote (htmldata);
				lastBranch->setNote (no);
				return true;
			}	else
			{
				return true;
			}	
        case StateMap: state=StateInit;  return true;
        default : 
			// even for HTML includes, this should never be reached
			return false;
    }   
}

bool mapBuilderHandler::characters   ( const QString& ch)
{
	//cout << "characters \""<<ch<<"\"  state="<<state <<"  laststate="<<laststate<<endl;

	QString ch_org=quotemeta (ch);
    QString ch_simplified=ch.simplifyWhiteSpace();
    if ( ch_simplified.isEmpty() ) return true;

    switch ( state ) 
    {
        case StateInit: break;
        case StateMap: break; 
		case StateMapSelect:
			me->select(ch_simplified);
			break;
		case StateMapSetting:break;
        case StateMapCenter: break;
        case StateMapCenterStandardFlag: 
            lastBranch->activateStandardFlag(ch_simplified); 
            break;
        case StateMapCenterHeading: 
            lastBranch->setHeading(ch_simplified); 
            break;
        case StateMapCenterNote:
			lastBranch->setNote(ch_simplified);
			break;
        case StateBranch: break;
        case StateBranchStandardFlag: 
            lastBranch->activateStandardFlag(ch_simplified); 
            break;
        case StateBranchHeading: 
            lastBranch->setHeading(ch_simplified);
            break;
        case StateBranchNote: 
			lastBranch->setNote(ch_simplified);
			break;
        case StateBranchFloatImage: break;
        case StateHtmlNote: break;
        case StateHtml:
			htmldata+=ch_org;
			break;
        default: 
			return false;
    }
    return true;
}

QString mapBuilderHandler::errorString() 
{
    return "the document is not in the VYM file format";
}

bool mapBuilderHandler::fatalError( const QXmlParseException& exception ) 
{
    errorProt += QString( "Fatal parsing error: %1 in line %2, column %3\n")
    .arg( exception.message() )
    .arg( exception.lineNumber() )
    .arg( exception.columnNumber() );
	// Try to read the bogus line
	errorProt+=QString("File is: %1\n").arg(inputFile);
	QString s;
	if (loadStringFromDisk (inputFile,s))
	{
		QStringList sl=QStringList::split ("\n",s);
		int i=1;
		QStringList::Iterator it = sl.begin();
		while (i<exception.lineNumber()-1)
		{
			it++;
			i++;
		}
		s=*it;
		s.insert (exception.columnNumber()-1,"<ERROR>");
		errorProt+=s;
    }
    return QXmlDefaultHandler::fatalError( exception );
}

void mapBuilderHandler::setMapEditor (MapEditor* e)
{
    me=e;
	mc=me->getMapCenter();
}

void mapBuilderHandler::setTmpDir (QString tp)
{
	tmpDir=tp;
}

void mapBuilderHandler::setInputFile (QString f)
{
	inputFile=f;
}

void mapBuilderHandler::setLoadMode (const LoadMode &lm)
{
	loadMode=lm;
}

bool mapBuilderHandler::readBranchAttr (const QXmlAttributes& a)
{
	lastOO=lastBranch;
	if (!readOOAttr(a)) return false;

	if (!a.value( "scrolled").isEmpty() )
		lastBranch->toggleScroll();
	if (!a.value( "frameType").isEmpty() ) 
		lastBranch->setFrameType (a.value("frameType"));

	if (!a.value( "incImgV").isEmpty() ) 
	{	
		if (a.value("incImgV")=="true")
			lastBranch->setIncludeImagesVer(true);
		else	
			lastBranch->setIncludeImagesVer(false);
	}	
	if (!a.value( "incImgH").isEmpty() ) 
	{	
		if (a.value("incImgH")=="true")
			lastBranch->setIncludeImagesHor(true);
		else	
			lastBranch->setIncludeImagesHor(false);
	}	
	return true;	
}

bool mapBuilderHandler::readOOAttr (const QXmlAttributes& a)
{
	if (lastOO)
	{
		bool okx,oky;
		int x,y;
		if (!a.value( "absPosX").isEmpty() && loadMode==NewMap && branchDepth<2) 
		{
			if (!a.value( "absPosY").isEmpty() ) 
			{
				x=a.value("absPosX").toInt (&okx, 10);
				y=a.value("absPosY").toInt (&oky, 10);
				if (okx && oky  )
					lastOO->move(x,y);
				else
					return false;   // Couldn't read absPos
			}           
		}           
		if (!a.value( "url").isEmpty() ) 
			lastOO->setURL (a.value ("url"));
		if (!a.value( "vymLink").isEmpty() ) 
			lastOO->setVymLink (a.value ("vymLink"));
		if (!a.value( "hideInExport").isEmpty() ) 
			if (a.value("hideInExport")=="true")
				lastOO->setHideInExport(true);

		if (!a.value( "hideLink").isEmpty()) 
		{
			if (a.value ("hideLink") =="true")
				lastOO->setHideLinkUnselected(true);
			else	
				lastOO->setHideLinkUnselected(false);
		}	
	}
	return true;	
}

bool mapBuilderHandler::readNoteAttr (const QXmlAttributes& a)
{	// only for backward compatibility (<1.4.6). Use htmlnote now.
	no.clear();
	QString fn;
	if (!a.value( "href").isEmpty() ) 
	{
		// Load note
		fn=parseHREF(a.value ("href") );
		QFile file (fn);
		QString s;						// Reading a note

		if ( !file.open( IO_ReadOnly) )
		{
			qWarning ("mapBuilderHandler::readNoteAttr:  Couldn't load "+fn);
			return false;
		}	
		QTextStream stream( &file );
		QString lines;
		while ( !stream.eof() ) {
			lines += stream.readLine()+"\n"; 
		}
		file.close();
		// Convert to richtext
		if ( !QStyleSheet::mightBeRichText( lines ) )
		{
			// Here we are workarounding the QT conversion method:
			// convertFromPlainText does not generate valid xml, needed
			// for the parser, but just <p> and <br> without closing tags.
			// So we have to add those by ourselves
			//lines=quotemeta (lines);
			lines = QStyleSheet::convertFromPlainText( lines, QStyleSheetItem::WhiteSpaceNormal );
			lines.replace ("<br>","<br />");
		}	

		lines ="<html><head><meta name=\"qrichtext\" content=\"1\" /></head><body>"+lines + "</p></body></html>";
		no.setNote (lines);
	}		
	if (!a.value( "fonthint").isEmpty() ) 
		no.setFontHint(a.value ("fonthint") );
	if (state == StateMapCenterNote) 	
		mc->setNote(no);
	else
		lastBranch->setNote(no);
	return true;
}

bool mapBuilderHandler::readFloatImageAttr (const QXmlAttributes& a)
{
	lastOO=lastFloat;
	
	//if (!readOOAttr(a)) return false;

	if (!a.value( "useOrientation").isEmpty() ) 
	{
		if (a.value ("useOrientation") =="true")
			lastFloat->setUseOrientation (true);
		else	
			lastFloat->setUseOrientation (false);
	}	
	if (!a.value( "href").isEmpty() )
	{
		// Load FloatImage
		if (!lastFloat->load (parseHREF(a.value ("href") ) ))
		{
			QMessageBox::warning( 0, "Warning: " ,
				"Couldn't load float image\n"+parseHREF(a.value ("href") ));
			lastBranch->removeFloatImage(((FloatImageObj*)(lastFloat)));
			lastFloat=NULL;
			return true;
		}
		
	}	
	if (!a.value( "floatExport").isEmpty() ) 
	{
		// Only for compatibility. THis is not used since 1.7.11 
		if (a.value ("floatExport") =="true")
			lastFloat->setFloatExport(true);
		else	
			lastFloat->setFloatExport (false);
	}	
	if (!a.value( "zPlane").isEmpty() ) 
		lastFloat->setZ (a.value("zPlane").toInt ());
    int x,y;
    bool okx,oky;
	if (!a.value( "relPosX").isEmpty() ) 
	{
		if (!a.value( "relPosY").isEmpty() ) 
		{
			// read relPos
			x=a.value("relPosX").toInt (&okx, 10);
			y=a.value("relPosY").toInt (&oky, 10);
			if (okx && oky) 
				
				{
					lastFloat->setRelPos (QPoint (x,y) );
					// make sure floats in mapcenter are repositioned to relative pos
					if (mc==lastBranch) mc->positionContents();
				}
			else
				// Couldn't read relPos
				return false;  
		}           
	}	
	
	if (!readOOAttr(a)) return false;

	if (!a.value ("orgName").isEmpty() )
	{
		((FloatImageObj*)(lastFloat))->setOriginalFilename (a.value("orgName"));
	}
	return true;
}

bool mapBuilderHandler::readXLinkAttr (const QXmlAttributes& a)
{
	QColor col;
	bool okx;
	bool success=false;
	XLinkObj *xlo=new XLinkObj (mc->getCanvas());
	if (!a.value( "color").isEmpty() ) 
	{
		col.setNamedColor(a.value("color"));
		xlo->setColor (col);
	}

	if (!a.value( "width").isEmpty() ) 
	{
		xlo->setWidth(a.value ("width").toInt (&okx, 10));
	}

	if (!a.value( "beginBranch").isEmpty() ) 
	{
		if (!a.value( "endBranch").isEmpty() ) 
		{
			LinkableMapObj *lmo=mc->findObjBySelect (a.value( "beginBranch"));
			if (lmo && typeid (*lmo)==typeid (BranchObj))
			{
				xlo->setBegin ((BranchObj*)(lmo));
				lmo=mc->findObjBySelect (a.value( "endBranch"));
				if (lmo && typeid (*lmo)==typeid (BranchObj))
				{
					xlo->setEnd ((BranchObj*)(lmo));
					xlo->activate();
				}
			}
			success=true; // Not all branches there yet, no error
		}           
	}	
	if (!success) delete (xlo);
	return success;
}

bool mapBuilderHandler::readHtmlAttr (const QXmlAttributes& a)
{
	for (int i=1; i<=a.count(); i++)
		htmldata+=" "+a.localName(i-1)+"=\""+a.value(i-1)+"\"";
	return true;
}

bool mapBuilderHandler::readSettingAttr (const QXmlAttributes& a)
{
	if (!a.value( "key").isEmpty() ) 
	{
		if (!a.value( "value").isEmpty() ) 
			settings.setLocalEntry (me->getDestPath(), a.value ("key"), a.value ("value"));
		else
			return false;
		
	} else
		return false;
	
	return true;
}
