/* DCTC - a Direct Connect text clone for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * main.c: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <sys/utsname.h>
#include <sys/un.h>
#include <linux/sem.h>     /* for the value of SEMVMX */
#include <errno.h>
#include <getopt.h>
#include <string.h>
#include <glib.h>
#include <pthread.h>

#include "var.h"
#include "display.h"
#include "action.h"
#include "network.h"
#include "dc_com.h"
#include "key.h"
#include "typical_action.h"
#include "macro.h"
#include "dc_manage.h"
#include "keyboard.h"
#include "db.h"
#include "hl_locks.h"
#include "main.h"
#include "gts.h"
#include "user_manage.h"
#include "timed_out_string.h"
#include "sema.h"
#include "gdl.h"
#include "uaddr.h"

/*****************************************************/
/* this mutex locks all pointers of user description */
/*****************************************************/
/* nickname, user_desc, cnx_type, email, host_ip */
/*************************************************/
HL_MUTEX user_info=HL_MUTEX_INIT;		/* this is a high level mutex allowing multiple readers or 1 writer */

/*******************************/
/* user description for DC hub */
/*******************************/
char *nickname=NULL;					/* user nickname */
char *user_desc=NULL;				/* user description (can be NULL) */
char *cnx_type=NULL;					/* connection type. (ex: DSL,Cable,...) */
char cnx_opt=0x1;                /* user status */
											/* bit 0 always set. */
                                 /* bit 1 ? ==0, user here, ==1, user away */
											/* bit 2 ? ==0, normal, ==1, is server */
											/*         according to DC, a server is a client online */
											/*	        for more than 2 hours.                           */
											/* bit 3 ? ==0, normal, ==1, fast upload */
											/*         according to DC, a fast upload is a client */
											/*         having an upload speed greater than 100KB/s. */
char *email=NULL;						/* user e-mail (can be NULL) */
double sizeof_data=0;			 	/* size of the data shared by this client */
double offset_sizeof_data=0;     /* when the client has to return the size of data it shares, this value */
                                 /* is added to the previous one */

int free_dl_slot;						/* number of free download slot */
int ttl_dl_slot;						/* number of existing download slot */
int dl_on;								/* this flag enables or disables the upload capability */
											/* if ==1, other users can download from you */
											/* if ==0, other users cannot download from you */
											/* this flag doesn't affect free_dl_slot/ttl_dl_slot */
											/* you still return the original value. If someone want to download, */
											/* the client will just return a "no more slot" message. */

char *nick_passwd=NULL;				/* when the hub wants a password. We will use the one here */
											/* if ==NULL, the client waits one before resuming */

/**************************************/
/* description of the connection type */
/**************************************/
/* (BHF means behind firewall) */
/*******************************/
int behind_fw=0;        			/* !=0, host is behind firewall */
unsigned short com_port;	      /* port available, to receive data if not behind fw */
											/* default value is 412 but it is not very good     */
											/* because port below 1024 are not allowed for      */
											/* non-privileged users under Un*x                  */
char *host_ip;					      /* ip of this host */



GString *org_hubip=NULL;	/* strings received from the -g flag */
									/* don't use it except you need the original -g parameter */
GString *hubip=NULL;			/* IP of the hub where we are connected */
unsigned int hub_port=411;	/* port of the hub where we are connected (411 is the default port) */
GString *hubname=NULL;		/* Name of the hub where we are connected (filed when connected) */

unsigned int recon_delay=30;	/* number of seconds before the hub connection loss and the /RECON */
unsigned int auto_rebuild_delay=15*60;	/* number of seconds between 2 rebuilds of the shared database */
unsigned int auto_scan_delay=10*60;	/* number of seconds between 2 GDL autoscan */

int follow_force_move=0;	/* does the client accept $ForceMove and $OpForceMove */

int max_wait=30;				/* a task can be queued up to 30 secondes before destruction */
int max_hang=10*60;			/* a running xfer can received nothing during up to 10 minutes before cancellation */
									/* WARNING: Due to the fact that Direct Connect works with 8KByte bloc, */
  									/* a timeout of less than 8 minutes is not recommended else a timeout may occur */
  									/* if speed goes beyond 1KB/s. */


int main_sck=-1;				/* it is the TCP socket connected to the hub */
int cnx_in_progress=0;		/* this flag is set when main_sck is !=-1 but the connect stage is not yet done */

int in_sck=-1;					/* it is the TCP socket to use as com port */

int srch_sck=-1;				/* it is the UDP socket used to receive active search result */
									/* it uses the same port number has in_sck */

int keyb_fd=0;					/* it is the keyboard fd */
									/* must be ==0. If the client loses its term, goes to -1 */

int when_done=0;				/* when a download is done, ==0, do nothing */
									/* ==1, move the file into the done/ directory */

#define MAX_ALLOWED_EMODE 1
int wanna_emode=0;			/* when the --mdc flag is used, this variable contains the flag value */
int having_emode=0;			/* depending of the connected hub, the current emode may differ from the */
									/* wanted emode */

int with_md5sum=1;			/* enable/disable md5sum for content search */

int with_ddl=0;				/* enable/disable dDL capability */

/************************************************************/
/* debug mode. ==0, no debug. !=0, debug mode               */
/* if debug_mode is set, all debug messages will be printed */
/************************************************************/
int debug_mode=0;

int force_quit=0;				/* if ==1, when /QUIT is called, it doesn't wait end of xfers */

static int user_wants_to_quit=0;	/* this flag is changed when the user hit ctrl-c to quit */

/*******************************************************************/
/* the following vars are used to manage the local AF_UNIX sockets */
/*******************************************************************/
GString *local_dctc_sock_path=NULL;					/* local name */
int local_sck=-1;											/* main local socket (to accept connection) */

GArray *local_client_socket=NULL;					/* array of all locally connected socket */
G_LOCK_DEFINE(local_client_socket);

GString *dctc_dir=NULL;									/* == $(HOME)/.dctc */

const char *dc_version="1,0091";								/* DC version to use */

/*****************************************/
/* information to limit upload bandwidth */
/*****************************************/
int bl_semid=-1;								
GString *bl_fname=NULL;

time_t client_start;

G_LOCK_DEFINE(inet_ntoa);		/* inet_ntoa is not thread save, this prevents problem */

/*****************************/
/* display connection status */
/*****************************/
void display_cnx_status(void)
{
	char buf[512];
	int val;

	/* etat de la connexion: bit0: 0= pas de socket, 1= socket vers hub */
	/*							  bit1: (si bit0!=0) 0= connexion en cours, 2= connection etablie */

	val=(main_sck!=-1)?1:0;
	if(cnx_in_progress==0)
		val+=2;

	sprintf(buf,"%d",val);
	disp_msg(VAR_MSG,NULL,"cnx_status",buf,NULL);
}

/**************************************************************/
/* this function is called when the hub closes its connection */
/*********************************************************************************/
/* if exit_flag==DO_EXIT, the client will not attempt to automatically reconnect */
/*********************************************************************************/
void hub_disconnect(HUBDISC_FLAG exit_flag)
{  
	int i;

	disp_msg(INFO_MSG,"hub_disconnect","in",NULL);

	if(exit_flag==DO_RECONNECT)
	{
		/* close current socket */
		if(main_sck!=-1)
		{
			close(main_sck);
			main_sck=-1;
		}

		reset_hub_user_list();		/* else /ULIST will return the old user list */
		display_cnx_status();

		add_new_sim_input(recon_delay,"/RECON");
		add_new_sim_input(0,"/VARS");
		
		return;
	}

	exit_gts();

	if((waiting_action!=NULL)&&(force_quit==0))
	{
		/* wait all thread done */
		char tmp[512];
		
		disp_msg(INFO_MSG,"hub_disconnect","have xfer ?",NULL);
		G_LOCK(waiting_action);
		
		sprintf(tmp,"%d xfer",waiting_action->len);
		disp_msg(INFO_MSG,NULL,tmp,NULL);
		while(waiting_action->len!=0)		/* all thread done ? */
		{
			G_UNLOCK(waiting_action);
			sleep(1);
			G_LOCK(waiting_action);
			sprintf(tmp,"%d xfer",waiting_action->len);
			disp_msg(INFO_MSG,NULL,tmp,NULL);

			if(user_wants_to_quit)		/* wait except if the user wants to leave now */
				break;
		}
		disp_msg(INFO_MSG,"hub_disconnect","no more xfer",NULL);
	}
	
	disp_msg(VAR_MSG,NULL,"cnx_status","0",NULL);	/* we are disconnected */

	/* close all connections of waiting_action else, threads don't leave properly */
	i=0;
	while(i<waiting_action->len)
	{
		WAIT_ACT *p;

		p=g_ptr_array_index(waiting_action,i);
		if((p!=NULL)&&(p->sock_fd!=-1))
		{
			shutdown(p->sock_fd,2);
			close(p->sock_fd);
		}
		i++;
	}

	/* and end */
	disp_msg(INFO_MSG,"hub_disconnect","end",NULL);
	if(local_dctc_sock_path!=NULL)
		unlink(local_dctc_sock_path->str);
	sleep(1);		/* without this sleep, some of the latest disp_msg doesn't work */
	exit(1);
}

/**************************************************************/
/* xfers queued for a too long time are automatically expired */
/**************************************************************/
static void	expire_old_xfer(void)
{
	int i;
	time_t cur_time;
	int force_refresh=0;

	disp_msg(DEBUG_MSG,"expire_old_xfer","in",NULL);
	cur_time=time(NULL);

	G_LOCK(waiting_revcon);
	if((waiting_revcon!=NULL)&&(waiting_revcon->len!=0))
	{
		for(i=waiting_revcon->len-1;i>=0;i--)
		{
			WAIT_REVCON *ptr;

			ptr=g_ptr_array_index(waiting_revcon,i);
			
			if( (cur_time-ptr->last_touch)> max_wait)
			{	/* too old ? */

				/* remove action from the list */
				g_ptr_array_remove_index(waiting_revcon,i);

				if(!strncmp(ptr->action_to_do->str,"DL/",3))
				{
					/* we will requeue this xfer inside sim_input */
					{
						GString *sim;
						GString *local;
						GString *org_remote;
						char *t;
						char sep;

						sep=ptr->action_to_do->str[sizeof("DL/")-1]; /* get the separator */

						local=g_string_new(ptr->action_to_do->str + sizeof("DL/") -1 + 1 /* ignore the separator */ );
						t=strrchr(local->str,sep);		 /* can't fail */
						local=g_string_truncate(local,t-local->str);	 /* truncate the size of the string in local */

						t=strrchr(local->str,sep);		 /* can't fail */
						org_remote=g_string_new(t+1);
						local=g_string_truncate(local,t-local->str);	 /* truncate the remote path of the string in local */

						sim=g_string_new("");

						/* we use | as separator because it cannot appear anywhere */
						g_string_sprintf(sim,"/DL |%s|%s|%s|",
											ptr->remote_nick->str, local->str, org_remote->str);
						/* @@@ add the xtra information at the end of the /DL */
						{
							UCNX tp;
							GString *xt;

							xt=get_xtra_information_cnx(ptr->remote_nick->str,&tp);
							if((tp==ACTIVE)&&(xt!=NULL)&&(xt->len!=0))
							{
								g_string_sprintfa(sim,"|%s",xt->str);
							}
							if(xt!=NULL)
								g_string_free(xt,TRUE);
						}

						/* wait 30 seconds before retrying */
						if(add_gts_entry(ptr->remote_nick->str,sim->str,60))
            			add_new_sim_input(60,sim->str);     /* fail to queue in the GTS, use sim_input instead */

						g_string_free(sim,TRUE);
						g_string_free(local,TRUE);
						g_string_free(org_remote,TRUE);
					}
					disp_msg(XFER_DL_UNQUEUED,NULL, ptr->remote_nick->str,ptr->action_to_do->str,NULL);
				}
				else if(!strncmp(ptr->action_to_do->str,"LS/",3))
				{
					/* we will requeue this xfer inside sim_input */
					{
						GString *sim;

						sim=g_string_new("");

						g_string_sprintf(sim,"/LS %s", ptr->remote_nick->str);

						/* wait 30 seconds before retrying */
						if(add_gts_entry(ptr->remote_nick->str,sim->str,30))
            			add_new_sim_input(30,sim->str);     /* fail to queue in the GTS, use sim_input instead */

						g_string_free(sim,TRUE);
					}

					disp_msg(XFER_LS_UNQUEUED,NULL, ptr->remote_nick->str,ptr->action_to_do->str,NULL);
				}
				else if(!strncmp(ptr->action_to_do->str,"XDL|",4))
				{
					/* notify to GDL the transfer never start */
					gchar **fields;
					fields=g_strsplit(ptr->action_to_do->str,"|",3);		/* there is always 3 fields, never less, never more */
					do_gdl_abort(strtoul(fields[1],NULL,10),fields[2]);
					g_strfreev(fields);
					disp_msg(DEBUG_MSG,NULL, "expiring XDL","|lu",(unsigned long)ptr,NULL);
				}
				/* free structure */
				free_action_to_do(ptr,0);
							/* don't use 1 here because waiting_revcon is already locked */
				force_refresh=1;
			}
		}
	}
	G_UNLOCK(waiting_revcon);

	if(force_refresh)
		disp_msg(REFRESH_MSG,NULL,NULL);

	disp_msg(DEBUG_MSG,"expire_old_xfer","out",NULL);
}

/******************************************************************************/
/* expire running xfer having not received anything since the last 10 minutes */
/******************************************************************************/
static void	expire_hanging_xfer(void)
{
	int i;
	time_t cur_time;

	disp_msg(DEBUG_MSG,"expire_hanging_xfer","in",NULL);

	cur_time=time(NULL);

	G_LOCK(waiting_action);
	if((waiting_action!=NULL)&&(waiting_action->len!=0))
	{
		for(i=waiting_action->len-1;i>=0;i--)
		{
			WAIT_ACT *ptr;

			ptr=g_ptr_array_index(waiting_action,i);
			
			if( (cur_time-ptr->last_touch)> max_hang)
			{	/* too old ? */
				disp_msg(DEBUG_MSG,"expire_hanging_xfer","|lu",(unsigned long)ptr,"|lu",(unsigned long)(ptr->thread_id));
				shutdown(ptr->sock_fd,2);						/* simulate a network connection close, like a kill */
			}
		}
	}
	G_UNLOCK(waiting_action);

	disp_msg(DEBUG_MSG,"expire_hanging_xfer","out",NULL);
}

/**************************************************/
/* check if something is "queued" on the keyboard */
/**************************************************/
static void check_sim_input(int sck)
{
	unsigned int i;
	time_t cur_time;
	GArray *temp_sim;

	disp_msg(DEBUG_MSG,"check_sim_input","in",NULL);

	/* we cannot call keyboard functions if sim_input is locked */
	/* because some of these functions also want to lock it */
	/* to avoid a race problem, we lock sim_input, extract the entries we want, */
	/* unlock it and then call the functions */
	temp_sim=g_array_new(FALSE,FALSE,sizeof(SIM_INPUT));

	G_LOCK(sim_input);

	cur_time=time(NULL);

	i=0;

	while(i<sim_input->len)
	{
		SIM_INPUT *ptr;

		ptr=&(g_array_index(sim_input,SIM_INPUT,i));

		if(ptr->min_start_time>cur_time)
			i++;				/* time not yet reached */
		else
		{
			/* copy the entry in the temp array */
			temp_sim=g_array_append_val(temp_sim,*ptr);

			/* and remove it */
			sim_input=g_array_remove_index(sim_input,i);
			/* don't add 1 to i, we have remove the current entry, so we already are on the next entry */
		}
	}

	G_UNLOCK(sim_input);

	/* now we will call the function */
	for(i=0;i<temp_sim->len;i++)
	{
		SIM_INPUT *ptr;

		ptr=&(g_array_index(temp_sim,SIM_INPUT,i));

		disp_msg(ASTART_MSG,NULL,ptr->keyb_string->str,NULL);

		keyboard_input(sck,ptr->keyb_string->str);

		/* the g_string is no more useful */
		g_string_free(ptr->keyb_string,TRUE);
	}

	/* g_strings are already freed */
	g_array_free(temp_sim,TRUE);

	disp_msg(DEBUG_MSG,"check_sim_input","out",NULL);
}

/********************************************/
/* someone connect to the local unix socket */
/********************************************/
static void manage_local_socket(int local_sck)
{
	int nw;
	int obuf=128*1024;

	nw=accept(local_sck,NULL,NULL);
	if(nw==-1)
		return;

	if(setsockopt(nw,SOL_SOCKET,SO_SNDBUF,&obuf,sizeof(obuf))<0)
	{
		perror("setsockopt");
	}

	G_LOCK(local_client_socket);
	local_client_socket=g_array_append_val(local_client_socket,nw);
	G_UNLOCK(local_client_socket);
}

/*******************************************************************/
/* read data from unix_socket and send it to the keyboard function */
/*******************************************************************/
static void	manage_unix_socket(int main_sck, int sck, int num)
{
	GString *s;
	int n;
	char c;

	if(sck==-1)	 /* sometimes, after a network error, it is possible to come here with an invalid socket descriptor */
		return;

	s=g_string_sized_new(512);

	do
	{
		n=read(sck,&c,1);
		if(n!=1)
		{
			g_string_free(s,TRUE);
			/*  close sck and remove it from local_client_socket */
			shutdown(sck,2);
			close(sck);
			G_LOCK(local_client_socket);
			local_client_socket=g_array_remove_index_fast(local_client_socket,num);
			G_UNLOCK(local_client_socket);
			return;
		}

		s=g_string_append_c(s,c);
	}while(c!='\n');

	s->str[s->len-1]='\0';		/* replace \n by \0 */

	printf("unix_socket: %s\n",s->str);

	keyboard_input(main_sck,s->str);
	
	g_string_free(s,TRUE);
}

/*******************************************************************/
/* this function is called when the hub socket is ready to be used */
/* (2nd stage of the connection). We can have 2 cases:             */
/* - the socket is successfully connected.                         */
/* - the socket connection has failed.                             */
/*******************************************************************/
static void	do_login_stage(void)
{
	int buffer[512/sizeof(int)];
	int blen=512;
	int i;

	if(getsockopt(main_sck,SOL_SOCKET,SO_ERROR,buffer,&blen)==-1)
	{
		disp_msg(HUB_DISCONNECT,"","Fail to establish connection with hub.",NULL);
		hub_disconnect(DO_RECONNECT);
		return;
	}

	if((blen<sizeof(int))||(buffer[0]!=0))
	{
		disp_msg(GLOB_CHAT_MSG,"","Unable to reach hub",hubip->str,"because",strerror(buffer[0]),NULL);
		hub_disconnect(DO_RECONNECT);
		return;
	}

	cnx_in_progress=0;			/* socket is connected */
	display_cnx_status();
	reset_hub_user_list();

	/* before all, we must pass the hub challenge */
	if((i=unlock_access(main_sck,NULL))!=0)
	{
#if 0
		GString *cur_input;

		cur_input=get_a_dc_line(main_sck);
		while(cur_input!=NULL)
		{
			disp_msg(INFO_MSG,"",cur_input->str,NULL);
			g_string_free(cur_input,TRUE);
			cur_input=get_a_dc_line(main_sck);
		}
#else
		GString *cur_input;

		cur_input=get_a_dc_line(main_sck);
		while(cur_input!=NULL)
		{
			disp_msg(INFO_MSG,"",cur_input->str,NULL);
			process_incoming_dc_data(main_sck,cur_input);
			g_string_free(cur_input,TRUE);
			cur_input=get_a_dc_line(main_sck);
		}

#endif
		disp_msg(HUB_DISCONNECT,"","Hub has closed its connection.",NULL);
		hub_disconnect(DO_RECONNECT);
		return;
	}

	disp_msg(HUB_RECONNECT,"",hubip->str,NULL);

	/* do the login stage */
	LOCK_READ(user_info);
	send_dc_line(main_sck,"$ValidateNick",nickname,NULL);		/* send our nickname */
#if 0
	send_dc_line(main_sck,"$Version",dc_version,NULL);					/* nothing is returned */
	send_dc_line(main_sck,"$GetNickList",NULL);					/* get list of all nickname */
	set_user_info(main_sck,nickname,user_desc,cnx_type,cnx_opt,email,sizeof_data);
#endif
	UNLOCK_READ(user_info);
}

static void check_connection(void)
{
	if(cnx_in_progress==0)		/* connection in progress ? no, send the command */
	{
		time_t now;

		now=time(NULL);
		if((now-last_cmd_time)>(5*60))
		{
			LOCK_WRITE(user_info);
			cnx_opt|=2;						/* set the away flag */
			UNLOCK_WRITE(user_info);
			LOCK_READ(user_info);
			set_user_info(main_sck,nickname, user_desc, cnx_type,cnx_opt,email,sizeof_data);
			UNLOCK_READ(user_info);
		}
	}
}

/***********************************************/
/* main loop of the program                    */
/* this loop is the running in the main thread */
/***********************************************/
static void main_loop()
{
	int ln;
	int n;
	fd_set rd,wd;
	struct timeval tv;
	int i;

	while(1)
	{
		n=-1;

		FD_ZERO(&rd);
		FD_ZERO(&wd);

		if(main_sck!=-1)
		{
			if(cnx_in_progress==1)
				FD_SET(main_sck,&wd);		/* during login stage, we want to be able to write */
			else
				FD_SET(main_sck,&rd);		/* network input */
			n=max(n,main_sck);
		}

		if(keyb_fd==0)
		{
			FD_SET(0,&rd);			/* keyboard input */
			n=max(n,0);
		}

		if(in_sck!=-1)			/* com port exists ? */
		{
			FD_SET(in_sck,&rd);
			n=max(n,in_sck);
		}

		if(srch_sck!=-1)
		{
			FD_SET(srch_sck,&rd);
			n=max(n,srch_sck);
		}

		/* local socket */
		if(local_sck!=-1)
		{
			FD_SET(local_sck,&rd);
			n=max(n,local_sck);
		}

		/* add unix client socket */
		G_LOCK(local_client_socket);
		for(i=0;i<local_client_socket->len;i++)
		{
			int hd;

			hd=g_array_index(local_client_socket,int,i);
			FD_SET(hd,&rd);
			n=max(n,hd);
		}
		G_UNLOCK(local_client_socket);

		/* wait at most 1 second */
		tv.tv_sec=1;
		tv.tv_usec=0;

		disp_msg(DEBUG_MSG,"main_loop","1",NULL);

		ln=select(n+1,&rd,&wd,NULL,&tv);
		if(ln>0)
		{
			if((keyb_fd==0)&&(FD_ISSET(0,&rd)))
			{	/* something from the keyboard ? */
				keyboard_input(main_sck,NULL);
			}
			else if( (main_sck!=-1) && (cnx_in_progress==1) && (FD_ISSET(main_sck,&wd)) )
			{	/* the main_socket is ready for writing */
				do_login_stage();
			}
			else if( (main_sck!=-1) && (cnx_in_progress==0) && (FD_ISSET(main_sck,&rd)) )
			{	/* something from the hub ? */
				get_dc_line_and_process(main_sck);
			}
			else if((in_sck!=-1)&&(FD_ISSET(in_sck,&rd)))
			{	/* someone wants to connect to this client ? */
				manage_com_port(in_sck);
			}
			else if((srch_sck!=-1)&&(FD_ISSET(srch_sck,&rd)))
			{
				manage_srch_port(srch_sck,main_sck);
			}
			else if((local_sck!=-1)&&(FD_ISSET(local_sck,&rd)))
			{
				/* someone tries to reach the client from unix socket */
				manage_local_socket(local_sck);
			}
			else
			{
				/* check unix client socket */
				G_LOCK(local_client_socket);
				for(i=0;i<local_client_socket->len;i++)
				{
					int hd;

					hd=g_array_index(local_client_socket,int,i);
					if(FD_ISSET(hd,&rd))
					{
						G_UNLOCK(local_client_socket);
						manage_unix_socket(main_sck,hd,i);
						G_LOCK(local_client_socket);
					}
				}
				G_UNLOCK(local_client_socket);
			}
		}
		else if(ln==-1)
		{
			if(errno!=EINTR)
				break;
		}
		/* be careful, after calling keyboard_input or get_dc_line_and_process     */
		/* main_sck can become invalid (==-1) and you are deconnected from the hub */
		/* in such case, hub_disconnect has been called and a /RECON should be in  */
		/* the sim_input queue. */

		disp_msg(DEBUG_MSG,"main_loop","2",NULL);
		expire_old_xfer();
		expire_hanging_xfer();
		check_sim_input(main_sck);
		timeout_tos();
		check_connection();

		check_sema_master(bl_semid);

		/* if we are connected to the hub */
		if ((main_sck!=-1) && (cnx_in_progress==0))
			gts_action(hub_user_list);

		if(with_ddl)
		{
			uaddr_action();
		}

		if(user_wants_to_quit)
		{
			disp_msg(ERR_MSG,NULL,"use /QUIT to quit",NULL);
			user_wants_to_quit=0;
		}

		/* handle SERVER status */
		LOCK_READ(user_info);
		if( ((cnx_opt&4)==0) 				/* file is not yet a server */
			&& ((sizeof_data+offset_sizeof_data)>(2.0*1024.0*1024.0*1024.0))		/* sharing more than 2GB */
			&& ( (time(NULL)-client_start) > (2*3600)) )									/* up for more than 2 hours */
		{
			UNLOCK_READ(user_info);

			LOCK_WRITE(user_info);
			cnx_opt|=4;						/* set the server flag */
			UNLOCK_WRITE(user_info);

			LOCK_READ(user_info);
			set_user_info(main_sck,nickname, user_desc, cnx_type,cnx_opt,email,sizeof_data);
		}
		UNLOCK_READ(user_info);
	}
}

static void catch_sig(int sig)
{
	switch(sig)
	{
		case SIGTERM:
		case SIGQUIT:
		case SIGINT:
						user_wants_to_quit=1;
						break;
	}
}

/************************************/
/* we don't want to receive SIGPIPE */
/************************************/
static void set_sig(void)
{
	struct sigaction act;
	sigset_t set;

	/* ignore SIGPIPE */
	sigemptyset(&set);
	sigaddset(&set,SIGPIPE);
	sigaddset(&set,SIGHUP);
	act.sa_handler=SIG_IGN;
	act.sa_mask=set;
	act.sa_flags=SA_RESTART;

	sigaction(SIGPIPE,&act,NULL);
	sigaction(SIGHUP,&act,NULL);
	sigprocmask(SIG_UNBLOCK,&set,NULL);

	/* and catch some other sig */
	act.sa_handler=catch_sig;
	act.sa_mask=set;
	act.sa_flags=SA_RESTART;	
	sigaction(SIGINT,&act,NULL);
	sigaction(SIGQUIT,&act,NULL);
	sigaction(SIGTERM,&act,NULL);
	sigprocmask(SIG_UNBLOCK,&set,NULL); /* supprime les signaux */
}

/*************************/
/* display program usage */
/*************************/
static void display_usage(char *fname)
{
	fprintf(stderr,"Usage: %s [options]\n"
						"Options:\n"
						"  -h, --help                      Display this help\n"
						"  -n, --nick=NICKNAME             user nickname (default: 'Noname')\n"
						"  -i, --info=INFO                 user description (default: none)\n"
						"  -c, --cnx=CNX                   user connection type (default: 'Cable')\n"
						"  -e, --email=EMAIL               user e-mail (default: none)\n"
						"  -d, --dlslot=NUM                number of download slot (default: 3)\n"
						"  -s, --share=DIR                 shared directory (default: none)\n"
						"  -o, --offset=NUM                add this value to size real size of shared\n"
						"                                  data (default: 0)\n"
						"  -a, --hostaddr=HOSTIP           IP of the host running this client (mandatory\n"
						"                                  if using active mode)\n"
						"                                  (default: IP of the first network interface)\n"
						"  -p, --port=PORT                 localhost port to accept incoming connection\n"
						"                                  (only in active mode)\n"
						"                                  (default: 412 (like Direct Connect)). Be\n"
						"                                  careful, value below 1024 is only allowed to\n"
						"                                  privileged users (root)\n"
						"  -g, --hubip=HUBIP[:PORT]        IP of the hub to connect to. If no port is given\n"
						"                                  the default port (411) is used\n"
						"  -f, --firewall                  enable passive mode because host is behind a\n"
						"                                  firewall (default: not behind a firewall)\n"
						"  -x, --no_xfer                   Remote users cannot download (same as /DLOFF)\n"
						"  -w, --when_done                 When download is done, move the file into\n"
						"                                  done/ directory (same as /DONE).\n"
						"  -t, --no_tty                    Detach from the tty used by the shell\n"
						"                                  starting the program\n"
						"  -l, --link                      Wait a connection on the unix socket before\n"
						"                                  starting (only useful with a GUI)\n"
						"  -v, --version=VERSION           Force DC version. Modifying this number\n"
						"                                  doesn't affect DCTC but it allows you to enter\n"
						"                                  a hub refusing you just because you don't have\n"
						"                                  the latest version of DC. This DCTC supports\n"
						"                                  DC protocol version %s\n"
						"  -u, --upload=NUM                Upload bandwidth limit. NUM is the number of\n"
						"                                  512 bytes per second allowed to be sent\n"
						"                                  (default: unlimited [in fact, it is 16MB/s])\n"
						"                                  Note: this flag is only taken into account\n"
						"                                  if this client creates the bandwidth limit\n"
						"                                  semaphore, use /UBL to change the value later\n"
						"  -m, --mdc=NUM                   enable non DC hub compatible command\n"
						"                                  this flag is for future usage. Current default\n"
						"                                  value is 0.\n"
						"  -b, --precmd=COMMAND            perform the given command BEFORE connecting to\n"
						"                                  the hub. Not all / commands can be used here,\n"
						"                                  look at the command list to know which are.\n"
						"                                  You can run more than one command by using\n"
						"                                  --precmd for each one. When multiple --precmd\n"
						"                                  are given, commands are performed in the given\n"
						"                                  order.\n"
						"  -5, --nomd5sum                  Do not compute MD5 sum for each shared file\n"
						"                                  This sum is used for content search.\n"
						"                                  Default: enabled\n"
						"\n"
						"Be careful, most of the information you provide can't contain the following\n"
						"characters because Direct Connect uses them internally: | $\n"
						,fname,dc_version);
}

/************************************************************/
/* check if the given string contains invalid DC characters */
/************************************************************/
void check_string(char *string)
{
	while(*string!='\0')
	{
		if((*string=='|')||(*string=='$'))
		{
			fprintf(stderr,"Invalid character ($ or |): %s\n",string);
			exit(2);
		}
		string++;
	}
}

static const char *lst_cnx_type[]=
									{
										"56Kbps",
										"33.6Kbps",
										"28.8Kbps",
										"Satellite",		/* according to DC, passive mode is required here */
										"ISDN",
										"DSL",
										"Cable",
										"LAN(T1)",
										"LAN(T3)",
										NULL
									};

/*****************************************************************************/
/* verify if the given string is inside the list of existing connection type */
/*****************************************************************************/
/* output: 1=ok, 0=error */
/*************************/
int is_a_valid_cnx(char *string)
{
	int i;

	i=0;
	while(lst_cnx_type[i]!=NULL)
	{
		if(!strcmp(lst_cnx_type[i],string))
			return 1;
		i++;
	}

	return 0;
}

/***********************************************/
/* check if the given connection type is valid */
/***********************************************/
static void verify_cnx_type(char *string)
{
	int i;

	if(is_a_valid_cnx(string))
		return;

	fprintf(stderr,"Unknown connection type: %s\n",string);
	fprintf(stderr,"Allowed values are: ");
	
	i=0;
	while(lst_cnx_type[i]!=NULL)
	{
		if(i!=0)
			fprintf(stderr,",");
		fprintf(stderr,"'%s'",lst_cnx_type[i]);
		i++;
	}
	
	fprintf(stderr,"\n");
	exit(1);
}

/****************************************************/
/* rebuild the shared file database periodically    */
/* The rebuilding is performed in a thread to avoid */
/* hanging of the client.                           */
/****************************************************/
static void *periodic_rebuild_db(void *nothing)
{
	time_t last_update_time=time(NULL);

	while(1)
	{
		sleep(60);
		if(auto_rebuild_delay!=0)
		{
			time_t cur_time;

			cur_time=time(NULL);

			if((last_update_time+auto_rebuild_delay)<cur_time)
			{
				last_update_time=cur_time;
				rebuild_database();
				add_new_sim_input(0,"/REUINFO");			/* force refresh of the user information on the hub */
			}
		}
	}
   pthread_exit(NULL);
}

static void start_rebuild_db_thread(void)
{
	pthread_t thread_id;
	pthread_attr_t thread_attr;

	pthread_attr_init (&thread_attr);
	pthread_attr_setdetachstate(&thread_attr, PTHREAD_CREATE_DETACHED);
	if(pthread_create(&thread_id,&thread_attr, (void*)periodic_rebuild_db,NULL)!=0)
	{
		disp_msg(ERR_MSG,"start_rebuild_db_thread","pthread_create failed","Periodic rebuilding of shared file database disabled",NULL);
	}
}


/* beginning of the code */
int main(int argc,char **argv)
{
	static struct option optab[]=	{
												{"help",no_argument,NULL,'h'},				/* get help */
												{"nick",required_argument,NULL,'n'},		/* nickname */
												{"info",required_argument,NULL,'i'},		/* user description */
												{"cnx",required_argument,NULL,'c'},			/* connection description */
												{"email",required_argument,NULL,'e'},		/* user email */
												{"dlslot",required_argument,NULL,'d'},		/* number of download slot */
												{"share",required_argument,NULL,'s'},		/* shared directory */
												{"offset",no_argument,NULL,'o'},			   /* alter the true size of shared data */
												{"hostip",required_argument,NULL,'a'},		/* IP of the host running client */
												{"port",required_argument,NULL,'p'},		/* port to use to accept connection */
												{"hubip",required_argument,NULL,'g'},		/* hub to connect to */
												{"firewall",no_argument,NULL,'f'},			/* host behind firewall ? */
												{"no_xfer",no_argument,NULL,'x'},			/* disable remote xfer ? */
												{"when_done",no_argument,NULL,'w'},			/* move file when download over */
												{"no_tty",no_argument,NULL,'t'},				/* detach from start shell tty */
												{"link",no_argument,NULL,'l'},				/* wait someone connection before starting */
												{"version",required_argument,NULL,'v'},	/* force change of version number */
												{"upload",required_argument,NULL,'u'},		/* upload bandwidth limit */
												{"mdc",required_argument,NULL,'m'},			/* enable non compatible DC command */
												{"precmd",required_argument,NULL,'b'},		/* send a command to execute before hub connection */
												{"nomd5sum",no_argument,NULL,'5'},	/* don't compute md5 sum for each file of the database */
												{NULL,0,NULL,'\0'}					/* last option */
											};
	static const char *short_opt="hn:i:c:e:d:s:o:a:p:g:fxwtlv:u:m:b:5";
										
	int ch;
	int detach_from_tty=0;
	int wait_lnk=0;
	int spd_limit=SEMVMX;

	/* the 2 sh_* are used to stored -s directories */
	GStringChunk *sh_gsc;
	GPtrArray *sh_gpa;

	/* the 2 pre_* are used to stored --precmd commands */
	GStringChunk *pre_gsc;
	GPtrArray *pre_gpa;

	disp_msg(INFO_MSG,NULL,"Direct Connect Text Client v" VERSION ,NULL);

	if(argc==1)
	{
		display_usage(argv[0]);
		exit(0);
	}

	/* set default values */
	nickname=strdup("Noname");
	user_desc=NULL;
	cnx_type=strdup("Cable");
	email=NULL;
	sizeof_data=0;
	offset_sizeof_data=0;
	ttl_dl_slot=3;
	dl_on=1;

	/* to be able to accept connection, we must set few values */
	behind_fw=0;
	host_ip=NULL;
	com_port=412;

	sh_gsc=g_string_chunk_new(128);
	sh_gpa=g_ptr_array_new();

	pre_gsc=g_string_chunk_new(128);
	pre_gpa=g_ptr_array_new();

	while((ch=getopt_long(argc,argv,short_opt,optab,NULL))!=EOF)
	{
		switch(ch)
		{
			case 'h':	display_usage(argv[0]);
							exit(0);

			case 'n':	if(nickname!=NULL)
								free(nickname);
							nickname=strdup(optarg);
							check_string(nickname);
							break;

			case 'i':	if(user_desc!=NULL)
								free(user_desc);
							user_desc=strdup(optarg);
							check_string(nickname);
							break;

			case 'c':	if(cnx_type!=NULL)
								free(cnx_type);
							cnx_type=strdup(optarg);
							verify_cnx_type(cnx_type);
							break;

			case 'e':	if(email!=NULL)
								free(email);
							email=strdup(optarg);
							check_string(email);
							break;

			case 'd':	ttl_dl_slot=atoi(optarg);
							if(ttl_dl_slot<0)
								ttl_dl_slot=0;
							break;

			case 'm':	wanna_emode=atoi(optarg);
							if(wanna_emode<0)
								wanna_emode=0;
							if(wanna_emode>MAX_ALLOWED_EMODE)
								wanna_emode=0;
							break;

			case 'a':	if(host_ip!=NULL)
								free(host_ip);
							host_ip=strdup(optarg);
							break;

			case 'p':	com_port=strtoul(optarg,NULL,10);
							if(com_port==0)
							{
								fprintf(stderr,"You can't use 0 as com port.\n");
								exit(1);
							}
							break;

			case 's':
							/* because adding a directory can be a quite long operation */
							/* dir name are now stored and will be processed later, when the */
							/* unix socket is opened. Thus, waiting UI will know the client is */
							/* running */
							{
								GString *tmp;

								tmp=g_string_new(optarg);
								if(tmp->len!=0)
								{
									gchar *p;
									/* remote the trailing / if exists */
									if((tmp->len>1)&&(tmp->str[tmp->len-1]=='/'))
										tmp=g_string_truncate(tmp,tmp->len-1);

									p=g_string_chunk_insert(sh_gsc,tmp->str);
									if(p==NULL)
									{
										fprintf(stderr,"out of memory while processing -s.\n");
										exit(1);
									}
									
									g_ptr_array_add(sh_gpa,p);
								}
								else
								{
									fprintf(stderr,"-s with empty parameter is not allowed\n");
									exit(1);
								}

								g_string_free(tmp,TRUE);
							}
							break;
							
			case 'o':	offset_sizeof_data=strtod(optarg,NULL);
							break;
							
			case 'g':
							if(hubip==NULL)
								hubip=g_string_new(optarg);
							else
								hubip=g_string_assign(hubip,optarg);

							if(org_hubip==NULL)
								org_hubip=g_string_new(hubip->str);
							else
								org_hubip=g_string_assign(org_hubip,hubip->str);

							{
								char *dport;
								dport=strchr(hubip->str,':');
								if(dport==NULL)
									hub_port=411;
								else
								{
									hub_port=atoi(dport+1);
									hubip=g_string_truncate(hubip,dport-hubip->str);

									if((hub_port<1)||(hub_port>65535))
									{
										fprintf(stderr,"port in -g option is invalid\n");
										exit(1);
									}
								}
							}
							printf("hubip: %s hubport: %hu\n",hubip->str,hub_port);
							break;

			case 'f':	behind_fw=1;
							break;

			case 'x':	dl_on=0;
							break;

			case 'w':	when_done=1;
							break;

			case 't':	detach_from_tty=1;
							break;

			case 'l':	wait_lnk=1;
							break;

			case 'v':	dc_version=strdup(optarg);
							break;

			case 'u':	spd_limit=atoi(optarg);
							if((spd_limit<0)||(spd_limit>SEMVMX))
								spd_limit=SEMVMX;
							break;

			case 'b':	g_ptr_array_add(pre_gpa,g_string_chunk_insert(pre_gsc,optarg));
							break;

			case '5':	with_md5sum=0;
							break;

			default:
							fprintf(stderr,"Unknown option: %c\n",ch);
							exit(1);
		}
	}

	/* at the beginning, all download slots are free */
	free_dl_slot=ttl_dl_slot;

	/* check if hub address seems to be ok */
	if( (hubip==NULL) || (hubip->len<4) )
	{
		fprintf(stderr,"You must provide a valid hub address\n");
		exit(1);
	}

	/* no IP ? set it to the first network interface IP */
	if(host_ip==NULL)
	{
		GString *hip;

		hip=get_default_host_ip();
		if(hip==NULL)
		{
			fprintf(stderr,"Unable to obtain localhost IP.\n");
			exit(1);
		}
		host_ip=hip->str;

		g_string_free(hip,FALSE);
		fprintf(stderr,"Using %s as localhost IP.\n",host_ip);
	}

	g_thread_init(NULL);				/* initialize glib thread functions */

	client_start=time(NULL);

	/* ignore some signals */
	set_sig();

	if(detach_from_tty)
	{
		int a;

		a=open("/dev/null",O_WRONLY);
		if(a==-1)
			exit(fprintf(stderr,"unable to open /dev/null"));

		dup2(a,0);
		dup2(a,1);
		dup2(a,2);
		close(a);
	
		keyb_fd=-1;
	}

	/* allocated running task array and queued task array */
	waiting_action=g_ptr_array_new();
	waiting_revcon=g_ptr_array_new();
	sim_input=g_array_new(FALSE,FALSE,sizeof(SIM_INPUT));

	/* run precmd commands */
	{
		int i;

		for(i=0;i<pre_gpa->len;i++)
		{
			process_precmd_command(g_ptr_array_index(pre_gpa,i));
		}
		g_ptr_array_free(pre_gpa,TRUE);
		g_string_chunk_free(pre_gsc);
	}

	/* hub communication socket */
	hub_logged=0;
	if(wanna_emode)
	{
		/* if the emode is enabled, first try to open the emode socket */
		main_sck=create_and_open_sock_on(hubip->str,hub_port+1,0);
		if(main_sck>=0)
			goto emode_connected;
		having_emode=wanna_emode;
	}

	main_sck=create_and_open_sock_on(hubip->str,hub_port,0);
	if(main_sck<0)
	{
		disp_msg(HUB_DISCONNECT,"","Hub has closed its connection.",NULL);
		hub_disconnect(DO_EXIT);
		exit(1);			/* never reached, hub_disconnect never returns */
	}

	emode_connected:
	cnx_in_progress=1;

	/* create com port, even if behind firewall else we cannot return search result */
	{
		do
		{
			do
			{
				in_sck=_x_tcp(com_port);
				if(in_sck==-1)
					com_port++;
			} while(in_sck==-1);

			listen(in_sck,64);
	
			srch_sck=_x_udp(com_port);
			if(srch_sck==-1)
			{
				close(in_sck);
				com_port++;
			}
		}while(srch_sck==-1);

		listen(in_sck,64);
	}

	/* create array for local client sockets */
	local_client_socket=g_array_new(FALSE,FALSE,sizeof(int));

	/* create local unix socket */
	{
		char *path;
		struct stat st;

		path=getenv("HOME");
		local_dctc_sock_path=g_string_new(NULL);
		g_string_sprintf(local_dctc_sock_path,"%s/.dctc",(path!=NULL)?path:".");

		if(stat(local_dctc_sock_path->str,&st))
		{
			if(mkdir(local_dctc_sock_path->str,0777))
			{
				perror("mkdir");
				exit(1);
			}
		}

		dctc_dir=g_string_new(local_dctc_sock_path->str);

		local_dctc_sock_path=g_string_append(local_dctc_sock_path,"/running");
		if(stat(local_dctc_sock_path->str,&st))
		{
			if(mkdir(local_dctc_sock_path->str,0777))
			{
				perror("mkdir");
				exit(1);
			}
		}

		g_string_sprintfa(local_dctc_sock_path,"/dctc-%08X-%s",getpid(),org_hubip->str);

		local_sck=socket(AF_UNIX,SOCK_STREAM,0);
		if(local_sck==-1)
		{
			perror("local_sck - socket");
			exit(1);
		}
		
		{
			struct sockaddr_un name;

			name.sun_family=AF_UNIX;
			strcpy(name.sun_path,local_dctc_sock_path->str);
			if(bind(local_sck,(void *)&name,sizeof(struct sockaddr_un)))
			{
				perror("local_sck - bind");
				exit(1);
			}

			listen(local_sck,3);
		}

		if(wait_lnk)
		{
			manage_local_socket(local_sck);
		}
	}

	last_cmd_time=time(NULL);
	display_cnx_status();

	init_gts();
	init_uaddr();

	/* perform bandwidth limitation semaphore init */
	{
		struct utsname unm;

		uname(&unm);

		bl_fname=g_string_new(dctc_dir->str);
		bl_fname=g_string_append_c(bl_fname,'/');
		bl_fname=g_string_append(bl_fname,unm.nodename);
		if(do_sema_init(bl_fname->str,&bl_semid,spd_limit))
		{
			fprintf(stderr,"Unable to initialize Bandwidth limitation semaphore.\n");
			exit(1);
		}
	}

	/* its time to add the shared directories to the list of shared directories */
	{
		int i;
		float cur_pos=0;
		float step=100.0/(sh_gpa->len);

		disp_msg(PROGRESS_BAR,"","init_share","0","Initialize shared file database",NULL);
		
		for(i=0;i<sh_gpa->len;i++)
		{
			gchar *p;

			p=g_ptr_array_index(sh_gpa,i);
			if(p!=NULL)
			{
				char tmp_str[512];
				sprintf(tmp_str,"%.2f",cur_pos);
				disp_msg(PROGRESS_BAR,"","init_share",tmp_str,"Initialize shared file database",p,NULL);
				add_shared_directory(p);
			}
			cur_pos+=step;
		}
		disp_msg(PROGRESS_BAR,"","init_share","100","Shared file database Initialized",NULL);

		/* now, we can free no more useful structures */
		g_ptr_array_free(sh_gpa,TRUE);
		g_string_chunk_free(sh_gsc);
	}

	start_rebuild_db_thread();

	start_gdl_thread();

	main_loop();

	exit_gts();
	exit_uaddr();

	/* close internet socket */
	close(main_sck);
	main_sck=-1;

	/* close unix sockets */
	close(local_sck);
	local_sck=-1;

	G_LOCK(local_client_socket);
	while(local_client_socket->len!=0)
	{
		int hd;

		hd=g_array_index(local_client_socket,int,0);
		shutdown(hd,2);
		close(hd);
		local_client_socket=g_array_remove_index_fast(local_client_socket,0);
	}
	G_UNLOCK(local_client_socket);
	
	unlink(local_dctc_sock_path->str);

	disp_msg(INFO_MSG,"main","end of client",NULL);
	return 0;
}

