/*
	Author: Marco Costalba (C) 2005-2006

	Copyright: See COPYING file that comes with this distribution


 Definitions of complex namespace constants

 Complex constant objects are not folded in like integral types, so they
 are declared 'extern' in namespace to avoid duplicating them as file scope
 data in each file where QGit namespace is included.

*/
#include <sys/types.h> // used by chmod()
#include <sys/stat.h>  // used by chmod()
#include <qsettings.h>
#include <qfile.h>
#include "common.h"

// minimum git version required
const QString QGit::GIT_VERSION = "1.4.4";

// colors
const QColor QGit::BROWN       = QColor(150, 75, 0);
const QColor QGit::ORANGE      = QColor(255, 160, 50);
const QColor QGit::DARK_ORANGE = QColor(216, 144, 0);
const QColor QGit::LIGHT_ORANGE = QColor(255, 221, 170);
const QColor QGit::LIGHT_BLUE  = QColor(85, 255, 255);
const QColor QGit::PURPLE      = QColor(221, 221, 255);
const QColor QGit::DARK_GREEN  = QColor(0, 205, 0);

// initialized at startup according to system wide settings
QColor QGit::ODD_LINE_COL;
QColor QGit::EVEN_LINE_COL;

/*
   Default QFont c'tor calls static method QApplication::font() that could
   be still NOT initialized at this time, so set a dummy font family instead,
   it will be properly changed later, at startup
*/
QFont QGit::TYPE_WRITER_FONT("Helvetica");

// patches drag and drop
const QString QGit::PATCHES_DIR  = "/.qgit_patches_copy";
const QString QGit::PATCHES_NAME = "qgit_import";

// git index parameters
const QString QGit::ZERO_SHA        = "0000000000000000000000000000000000000000";
const QString QGit::CUSTOM_SHA      = "CUSTOM";
const QString QGit::ALL_MERGE_FILES = "ALL_MERGE_FILES";

// settings keys
const QString QGit::APP_KEY         = "/qgit/";
const QString QGit::FP_DIR_KEY      = "format_patch_last_dir";
const QString QGit::FPATCH_ARGS_KEY = "format_patch_args";
const QString QGit::FLAGS_KEY       = "patch_flags";
const QString QGit::CMT_GEOM_KEY    = "commit_viewer_geometry";
const QString QGit::CMT_SPLIT_KEY   = "commit_viewer_splitter_sizes";
const QString QGit::CMT_TEMPL_KEY   = "commit_template_file_path";
const QString QGit::CMT_ARGS_KEY    = "commit_args";
const QString QGit::EX_KEY          = "exclude_file_path";
const QString QGit::EX_PER_DIR_KEY  = "exclude_per_directory_file_name";
const QString QGit::EXT_DIFF_KEY    = "external_diff_viewer";
const QString QGit::REC_REP_KEY     = "recent_open_repos";
const QString QGit::MCR_NAME_KEY    = "macro_name";
const QString QGit::MCR_TEXT_KEY    = "commands";
const QString QGit::MCR_LIST_KEY    = "macro_list";
const QString QGit::FONT_KEY        = "typewriter_font";

// settings default values
const QString QGit::CMT_GEOM_DEF    = "290,140,495,540";
const QString QGit::CMT_SPLIT_DEF   = "155,342";
const QString QGit::CMT_TEMPL_DEF   = ".git/commit-template";
const QString QGit::EX_DEF          = ".git/info/exclude";
const QString QGit::EX_PER_DIR_DEF  = ".gitignore";
const QString QGit::EXT_DIFF_DEF    = "kompare";
const QString QGit::MCR_NAME_DEF    = "New macro";

// cache file
const QString QGit::BAK_EXT          = ".bak";
const QString QGit::C_DAT_FILE       = "/qgit_cache.dat";

// misc
const QString QGit::QUOTE_CHAR = "$";

using namespace QGit;

// settings helpers
uint QGit::flags(SCRef group) {

	QSettings settings;
	return settings.readNumEntry(APP_KEY + group + FLAGS_KEY, FLAGS_DEF);
}

bool QGit::testFlag(uint f, SCRef group) {

	return (flags(group) & f);
}

void QGit::setFlag(uint f, bool b, SCRef group) {

	QSettings settings;
	int flags = settings.readNumEntry(APP_KEY + group + FLAGS_KEY, FLAGS_DEF);
	flags = (b) ? flags | f : flags & ~f;
	settings.writeEntry(APP_KEY + group + FLAGS_KEY, flags);
}

void QGit::writeSetting(SCRef key, SCRef value, SCRef group) {

	QSettings settings;
	settings.writeEntry(APP_KEY + group + key, value);
}

// misc helpers
bool QGit::stripPartialParaghraps(const QByteArray& ba, QString* dst, QString* prev) {

	const QString src(ba);
	int idx = src.findRev('\n');
	if (idx == -1) {
		prev->append(src);
		*dst = "";
		return false;
	}
	*dst = src.left(idx).prepend(*prev); // strip trailing '\n'
	*prev = src.mid(idx + 1); // src[idx] is '\n', skip it
	return true;
}

bool QGit::writeToFile(SCRef fileName, SCRef data, bool setExecutable) {

	QFile file(fileName);
	if (!file.open(IO_WriteOnly)) {
		dbp("ERROR: unable to write file %1", fileName);
		return false;
	}
	QTextStream stream(&file);
	stream << data;
	file.close();
	if (setExecutable)
		chmod(fileName, 0755);
	return true;
}

bool QGit::writeToFile(SCRef fileName, const QByteArray& data, bool setExecutable) {

	QFile file(fileName);
	if (!file.open(IO_WriteOnly)) {
		dbp("ERROR: unable to write file %1", fileName);
		return false;
	}
	QDataStream stream(&file);
	stream.writeRawBytes(data.data(), data.size());
	file.close();
	if (setExecutable)
		chmod(fileName, 0755);
	return true;
}

bool QGit::readFromFile(SCRef fileName, QString& data) {

	data = "";
	QFile file(fileName);
	if (!file.open(IO_ReadOnly)) {
		dbp("ERROR: unable to read file %1", fileName);
		return false;
	}
	QTextStream stream(&file);
	data = stream.read();
	file.close();
	return true;
}

void QGit::baAppend(QByteArray** baPtr, const char* ascii, int len) {

	QByteArray* ba = *baPtr;
	uint oldSize = 0;
	if (ba) {
		oldSize = ba->size();
		ba->resize(oldSize + len, QGArray::SpeedOptim);
	} else {
		ba = new QByteArray(len);
		*baPtr = ba;
	}
	// no detach here, no copy on write
	memcpy(ba->data() + oldSize, ascii, len);
}

void QGit::baAppend(QByteArray& ba, const QByteArray& src) {

	QByteArray* baPtr = &ba;
	baAppend(&baPtr, src.data(),  src.size());
}
