<?php
// $Horde: imp/lib/Identity/IMP.php,v 1.3 2001/11/06 13:21:52 chuck Exp $

require_once HORDE_BASE . '/lib/Identity.php';

/**
 * This class provides an IMP-specific interface to all identities a
 * user might have. Its methods take care of any site-specific
 * restrictions configured in prefs.php and conf.php.
 * 
 * @author  Jan Schneider <jan@ammma.de>
 * @author  Chuck Hagenbuch <chuck@horde.org>
 * @version $Revision: 1.3 $
 * @since   IMP 2.3.7
 * @package horde.identity
 */
class Identity_IMP extends Identity {
    
    /**
     * Array containing all of the properties in this identity,
     * excluding the id.
     * @var array $properties
     */
    var $properties = array('fullname', 'from_addr', 'replyto_addr',
                            'signature', 'sig_first', 'sig_dashes',
                            'save_sent-mail', 'sent_mail_folder');
    
    /**
     * Reads all the user's identities from the prefs object or builds
     * a new identity from the standard values given in prefs.php.
     */
    function Identity_IMP()
    {
        parent::Identity();
    }
    
    /**
     * Returns a complete From: header based on all relevent factors (fullname,
     * from address, input fields, locks etc.)
     * @param integer $ident        (optional) The identity to retrieve the values from
     * @return string               A full From: header in the format 'Fullname <user@example.com>'
     */
    function getFromLine($ident = null)
    {
        global $conf, $imp, $prefs;
    
        /* Cache the value of $_from for future checks. */
        static $froms;
        if (isset($froms[$ident])) {
            return $froms[$ident];
        }
    
        if (!isset($ident)) {
            $ident = Horde::getFormData('identity');
            /* Attempt to retrieve the From: address from the form. */
            $address = Horde::getFormData('from');
        }
        
        if (empty($address) || $prefs->isLocked('from_addr')) {
            $address = $this->getFromAddress($ident);
            $name = $this->getFullname($ident);
        }
        if (!empty($address)) {
            $ob = imap_rfc822_parse_adrlist($address, $imp['maildomain']);
        }
    
        if (empty($name)) {
            if (!empty($ob[0]->personal)) {
                $name = $ob[0]->personal;
            } else {
                $name = $this->getFullname($ident);
            }
        }
    
        if (!empty($ob[0]->host) && !empty($ob[0]->mailbox)) {
            $from = MIME::trimEmailAddress(imap_rfc822_write_address($ob[0]->mailbox, $ob[0]->host, $name));
        } else {
            $info = explode('@', IMP::getPlainAddress());
            $from = MIME::trimEmailAddress(imap_rfc822_write_address($info[0], $info[1], $name));
        }
        
        $froms[$ident] = $from;
        return $from;
    }
    
    /**
     * Returns an array with From: headers from all identities
     * @return array        The From: headers from all identities
     */
    function getAllFromLines()
    {
        foreach ($this->identities as $ident => $dummy) {
            $list[$ident] = $this->getFromAddress($ident);
        }
        return $list;
    }
    
    /**
     * Returns an array with the necessary values for the identity select
     * box in the IMP compose window.
     * @return array        The array with the necessary strings
     */
    function getSelectList()
    {
        $froms = $this->getAll('from_addr');
        $ids = $this->getAll('id');
        foreach ($ids as $key => $id) {
            $list[$key] = $froms[$key] . ' (' . $id . ')';
        }
        return $list;
    }
    
    /**
     * Returns true if the given address belongs to one of the identities.
     * @param string $address           The address to search for in the identities
     * @return boolean                  True if the address was found
     */
    function hasAddress($address)
    {
        $list = $this->getAllFromAddresses();
        foreach ($list as $from) {
            if (strpos(strtolower($address), strtolower($from)) !== false) {
                return true;
            }
        }
        return false;
    }
    
    /**
     * Returns the from address based on the chosen identity and any
     * specified hook functions. If no address can be found it is build
     * from the current user name and the specified maildomain.
     * @param integer           (optional) The identity to retrieve the address from
     * @return string           A valid from address
     */
    function getFromAddress($ident = null)
    {
        global $conf, $imp;
        static $froms;
        
        if (isset($froms[$ident])) {
            return $froms[$ident];
        }
        
        $val = $this->getValue('from_addr', $ident);
        
        if (!empty($conf['hooks']['from']) && function_exists($conf['hooks']['from'])) {
            $val = call_user_func($conf['hooks']['from'], $imp);
        } elseif (empty($val)) {
            $val = $imp['user'];
        }
        if (!strstr($val, '@')) {
            $val .= '@' . $imp['maildomain'];
        }
        
        $froms[$ident] = $val;
        return $val;
    }
    
    /**
     * Returns an array with all identities' from addresses
     * @return array            The array with the from addresses
     */
    function getAllFromAddresses()
    {
        static $list;
        
        if (isset($list)) {
            return $list;
        }
        
        foreach ($this->identities as $key => $identity) {
            $list[$key] = $this->getFromAddress($key);
        }
        
        return $list;
    }
    
    /**
     * Returns the users full name and takes care of any hook functions specified.
     * @param integer $ident            (optional) The identity to retrieve the name from
     * @return string                   The users full name
     */
    function getFullname($ident = null)
    {
        global $conf, $imp;
        static $names;
        
        if (isset($names[$ident])) {
            return $names[$ident];
        }
        
        $val = $this->getValue('fullname', $ident);
        
        if (!empty($conf['hooks']['fullname']) && function_exists($conf['hooks']['fullname'])) {
            $val = call_user_func($conf['hooks']['fullname'], $imp);
        }
        
        $names[$ident] = $val;
        return $val;
    }
    
    /**
     * Returns the full signature based on the current settings for the
     * signature itself, the dashes and the position and any specified
     * hook function.
     * @param integer $ident            The identity to retrieve the signature from
     * @return  string                  The full signature
     */
    function getSignature($ident = null)
    {
        global $conf, $imp;
        static $signatures;
        
        if (isset($signatures[$ident])) {
            return $signatures[$ident];
        }
        
        $val = $this->getValue('signature', $ident);
        
        if (isset($val)) {
            $sig_first = $this->getValue('sig_first', $ident);
            $sig_dashes = $this->getValue('sig_dashes', $ident);
            $val = str_replace("\r\n", "\n", $val);
            if ($sig_dashes) {
                $val = "-- \n$val";
            }
            if (isset($sig_first) && $sig_first) {
                $val = "\n" . $val . "\n\n\n";
            } else {
                $val = "\n" . $val;
            }
        }
        if (!empty($conf['hooks']['signature']) && function_exists($conf['hooks']['signature']) ) {
            $val = call_user_func($conf['hooks']['signature'], $imp, $val);
        }
        
        $signatures[$ident] = $val;
        return $val;
    }
    
    /**
     * Returns an array with the signatures from all identities
     * @return array            The array with all the signatures
     */
    function getAllSignatures()
    {
        static $list;
        
        if (isset($list)) {
            return $list;
        }
        
        foreach ($this->identities as $key => $identity) {
            $list[$key] = $this->getSignature($key);
        }
        
        return $list;
    }
    
    /**
     * Returns the sent-mail folder name. Removes any namespaces or
     * folder prefixes before returning the value.
     * @param integer $ident            (optional) The identity to retrive the folder name from
     * @return string                   The folder name
     */
    function getSentmailFolder($ident = null)
    {
        global $prefs, $imp;
        
        if (!isset($ident)) {
            $folder = Horde::getFormData('sent_mail_folder');
        }
        if (empty($folder) || $prefs->isLocked('sent_mail_folder')) {
            $folder = $this->getValue('sent_mail_folder', $ident);
        }
        
        if (strlen($imp['namespace']) > 0) {
            $folder = str_replace($imp['namespace'], '', $folder);
        }
        if (strlen($imp['folders']) > 0) {
            $folder = str_replace($imp['folders'], '', $folder);
        }
        
        return $folder;
    }
    
    /**
     * Returns an array with the sent-mail folder names from all the identities.
     * @return array            The array with the folder names
     */
    function getAllSentmailFolders()
    {
        foreach ($this->identities as $key => $identity) {
            $list[$key] = $this->getSentmailFolder($key);
        }
        return $list;
    }
    
    /**
     * Returns true if the mail should be saved and is allowed to
     * @param integer $ident            The identity to retrieve the setting from
     * @return boolean                  True if the sent mail should be saved.
     */
    function saveSentmail($ident = null)
    {
        global $conf;
        
        if (!$conf['user']['allow_folders']) {
            return false;
        }
        
        return Horde::getFormData('save_sent_mail') || $this->getValue('save_sent_mail', $ident);
    }
    
}
?>
