<?php
// $Horde: imp/lib/IMP.php,v 1.198.2.4 2001/11/18 16:17:24 chuck Exp $

/** @const IMP_NAME The app name. */ define('IMP_NAME', 'Internet Messaging Program (IMP)');

if (!defined('IMP_BASE')) {
    /** @const IMP_BASE The imp fileroot. */ define('IMP_BASE', dirname(__FILE__) . '/..');
}
require_once HORDE_BASE . '/lib/Prefs.php';

// Actions
/** @const DELETE_MESSAGES Delete messages.            */ define('DELETE_MESSAGES', 100);
/** @const UNDELETE_MESSAGES Undeleted messages.       */ define('UNDELETE_MESSAGES', 101);
/** @const MOVE_MESSAGES Move messages.                */ define('MOVE_MESSAGES', 102);
/** @const COPY_MESSAGES Copy messages.                */ define('COPY_MESSAGES', 103);
/** @const EXPUNGE_MAILBOX Expunge mailbox.            */ define('EXPUNGE_MAILBOX', 104);
/** @const IMP_LOGIN Run login attempt.                */ define('IMP_LOGIN', 105);
/** @const REPLY Reply to an email.                    */ define('REPLY', 106);
/** @const REPLY_ALL Reply to all.                     */ define('REPLY_ALL', 107);
/** @const FORWARD Forward an email.                   */ define('FORWARD', 108);
/** @const DELETE_FOLDER Delete a folder.              */ define('DELETE_FOLDER', 109);
/** @const CREATE_FOLDER Create a folder.              */ define('CREATE_FOLDER', 110);
/** @const RENAME_FOLDER Rename a folder.              */ define('RENAME_FOLDER', 111);
/** @const DOWNLOAD_ATTACH Download a part.            */ define('DOWNLOAD_ATTACH', 112);
/** @const VIEW_ATTACH View a part.                    */ define('VIEW_ATTACH', 113);
/** @const SEND_MESSAGE Send a message.                */ define('SEND_MESSAGE', 114);
/** @const VIEW_SOURCE View a raw email.               */ define('VIEW_SOURCE', 115);
/** @const SAVE_MESSAGE Save message disk.             */ define('SAVE_MESSAGE', 116);
/** @const SUBSCRIBE_FOLDER Subscribe folder.          */ define('SUBSCRIBE_FOLDER', 117);
/** @const UNSUBSCRIBE_FOLDER Unsubscribe folder.      */ define('UNSUBSCRIBE_FOLDER', 118);
/** @const BOUNCE_MESSAGE Bounce message.              */ define('BOUNCE_MESSAGE', 119);
/** @const BOUNCE_COMPOSE Display bounce form.         */ define('BOUNCE_COMPOSE', 120);
/** @const SPAM_REPORT Report an email as spam.        */ define('SPAM_REPORT', 121);
/** @const ADD_ATTACHMENT Add an attachment.           */ define('ADD_ATTACHMENT', 122);
/** @const DELETE_ATTACHMENT Remove an attachment.     */ define('DELETE_ATTACHMENT', 123);
/** @const MAILTO Send a message to an address.        */ define('MAILTO', 124);
/** @const COMPOSE Display the compose form.           */ define('COMPOSE', 125);
/** @const DRAFT Resume a message saved as a draft.    */ define('DRAFT', 126);
/** @const SAVE_DRAFT. Save a draft.                   */ define('SAVE_DRAFT', 127);
/** @const HIDE_DELETED Don't show deleted messages.   */ define('HIDE_DELETED', 128);
/** @const LOGIN_COMPOSE Go directly to compose.       */ define('LOGIN_COMPOSE', 129);
/** @const SEARCH Run a search on one or more folders. */ define('SEARCH', 130);
/** @const SPELL_CHECK Spell check a message.          */ define('SPELL_CHECK', 131);
/** @const SPELL_CHECK_FORWARD Spell check next words. */ define('SPELL_CHECK_FORWARD', 132);
/** @const SPELL_CHECK_CANCEL Discard spell check.     */ define('SPELL_CHECK_CANCEL', 133);
/** @const SPELL_CHECK_DONE Finish spell check.        */ define('SPELL_CHECK_DONE', 134);
/** @const MESSAGE_MISSING Report no such message.     */ define('MESSAGE_MISSING', 135);
/** @const PASSWD_INPUT Password form.                 */ define('PASSWD_INPUT', 136);
/** @const PASSWD_DISALLOW Password changing failed.   */ define('PASSWD_DISALLOW', 137);
/** @const PASSWD_CHANGE Change password.              */ define('PASSWD_CHANGE', 138);
/** @const EXPAND_FOLDER Show any subfolders.          */ define('EXPAND_FOLDER', 139);
/** @const COLLAPSE_FOLDER Hide any subfolders.        */ define('COLLAPSE_FOLDER', 140);
/** @const EXPAND_ALL_FOLDERS Show every subfolder.    */ define('EXPAND_ALL_FOLDERS', 141);
/** @const COLLAPSE_ALL_FOLDERS Hide every subfolder.  */ define('COLLAPSE_ALL_FOLDERS', 142);
/** @const REFRESH_FOLDERS Refresh folder information. */ define('REFRESH_FOLDERS', 143);
/** @const POLL_FOLDER Check folders.                  */ define('POLL_FOLDER', 144);
/** @const NOPOLL_FOLDER Don't check folders.          */ define('NOPOLL_FOLDER', 145);
/** @const TOGGLE_SUBSCRIBED_VIEW Show subscribed/all. */ define('TOGGLE_SUBSCRIBED_VIEW', 146);
/** @const FLAG_MESSAGES Apply an IMAP flag.           */ define('FLAG_MESSAGES', 147);
/** @const PRINT_MESSAGE Display message for printing. */ define('PRINT_MESSAGE', 148);
/** @const FILTER Run filters.                         */ define('FILTER', 149); 
/** @const FILTER_CREATE Create a new mailbox filter.  */ define('FILTER_CREATE', 150);
/** @const FILTER_MODIFY Change an existing filter.    */ define('FILTER_MODIFY', 151);
/** @const FILTER_DELETE Remove a filter.              */ define('FILTER_DELETE', 152);
/** @const FILTER_DOWN Apply a filter later.           */ define('FILTER_DOWN', 153);
/** @const FILTER_UP Apply a filter sooner.            */ define('FILTER_UP', 154);
/** @const EXPAND_ADDRESSES Expand message recipients. */ define('EXPAND_ADDRESSES', 155);
/** @const ADD_ADDRESS Add address to addressbook.     */ define('ADD_ADDRESS', 156);
/** @const DOWNLOAD_FOLDER Download folder as mbox.    */ define('DOWNLOAD_FOLDER', 157);
/** @const IMP_BLACKLIST Blacklist spammers            */ define('IMP_BLACKLIST', 158);

// IMAP Flags
/** @const IMP_ALL Match all IMAP flags.      */ define('IMP_ALL', 0);
/** @const IMP_UNSEEN \\UNSEEN flag.          */ define('IMP_UNSEEN', 1);
/** @const IMP_DELETED \\DELETED flag.        */ define('IMP_DELETED', 2);
/** @const IMP_ANSWERED \\ANSWERED flag.      */ define('IMP_ANSWERED', 4);
/** @const IMP_FLAGGED \\FLAGGED flag.        */ define('IMP_FLAGGED', 8);
/** @const IMP_DRAFT \\DRAFT flag.            */ define('IMP_DRAFT', 16);
/** @const IMP_PERSONAL An email is personal. */ define('IMP_PERSONAL', 32);

/* Set the umask. */
if (isset($conf['umask'])) {
    umask($conf['umask']);
}


/**
 * IMP Base Class.
 *
 * @author Chuck Hagenbuch <chuck@horde.org>
 * @author Jon Parise <jon@horde.org>
 * @version $Revision: 1.198.2.4 $
 * @package imp
 */
class IMP {
    
    /**
     * Take information posted from a login attempt and try setting up
     * an initial IMP session. Handle Horde authentication, if
     * required, and only do enough work to see if the user can log
     * in. This function should only be called once, when the user
     * first logs in.
     *
     * @access public
     *
     * @return mixed True on success, string with the failure reason on failure.
     */
    function createSession()
    {
        global $imp, $conf, $registry, $HTTP_POST_VARS;
        
        if (!isset($HTTP_POST_VARS['imapuser'])
            || !isset($HTTP_POST_VARS['pass'])
            || !isset($HTTP_POST_VARS['server'])) {
            return 'failed';
        }
        
        $imp = array();
        $imp['user'] = trim($HTTP_POST_VARS['imapuser']);
        
        /* Run the username through virtualhost expansion functions if
           necessary. */
        if (!empty($conf['hooks']['vinfo']) && function_exists($conf['hooks']['vinfo'])) {
            $imp['user'] = call_user_func($conf['hooks']['vinfo']);
        }
        
        $imp['pass'] = Secret::write(Secret::getKey('imp'), trim($HTTP_POST_VARS['pass']));
        
        /* We might need to override some of the defaults with
           environmental settings. */
        if ($conf['server']['server_list']) {
            include_once IMP_BASE . '/config/servers.php';
        }

        if (Auth::getAuth()) {
            $imp['uniquser'] = Auth::getAuth();
        } elseif ($conf['server']['server_list'] && !empty($servers[$HTTP_POST_VARS['server']]['realm'])) {
            $imp['uniquser'] = $imp['user'] . '@' . $servers[$HTTP_POST_VARS['server']]['realm'];
        } elseif (!empty($HTTP_POST_VARS['realm'])) {
            $imp['uniquser'] = $imp['user'] . '@' . $HTTP_POST_VARS['realm'];
        } else {
            $imp['uniquser'] = $imp['user'];
        }

        if ($conf['server']['server_list']
            && isset($servers[$HTTP_POST_VARS['server']])
            && is_array($servers[$HTTP_POST_VARS['server']])) {
            $svr = &$HTTP_POST_VARS['server'];
            
            $imp['server'] = $servers[$svr]['server'];
            $imp['port'] = $servers[$svr]['port'];
            $imp['protocol'] = $servers[$svr]['protocol'];
            $imp['maildomain'] = $servers[$svr]['maildomain'];
            $imp['namespace'] = $servers[$svr]['namespace'];

            if (($conf['mailer']['type'] == 'smtp') &&
                !empty($servers[$svr]['smtphost'])) {
                $imp['smtphost'] = $servers[$svr]['smtphost'];
            }
        } else {
            $imp['server'] = Horde::getFormData('server', '');
            $imp['port'] = Horde::getFormData('port', '');
            $imp['protocol'] = Horde::getFormData('protocol', '');
            $imp['maildomain'] = Horde::getFormData('maildomain', '');
            $imp['namespace'] = Horde::getFormData('namespace', '');
        }
        
        $imp['mailbox'] = '';
        if (IMP::authenticate(OP_HALFOPEN) === true) {
            if ($registry->getMethod('auth/login') == $registry->getApp()) {
                include_once HORDE_BASE . '/lib/Auth.php';
                Auth::setAuth($imp['uniquser'], array('password' => $HTTP_POST_VARS['pass']));
                $registry->loadPrefs();
            }
            global $prefs;
            
            if ($conf['server']['server_list']
                && isset($servers[$HTTP_POST_VARS['server']])
                && is_array($servers[$HTTP_POST_VARS['server']])) {
                $svr = &$HTTP_POST_VARS['server'];
                
                $prefs->setValue('folders', $servers[$svr]['folders']);
            } else {
                if (!$prefs->isLocked('folders') && !empty($HTTP_POST_VARS['folders'])) {
                    $prefs->setValue('folders', $HTTP_POST_VARS['folders']);
                }
            }
            
            /* Set the session variables. These are cached. */
            $imp['folders'] = $prefs->getValue('folders');
            $imp['mailbox'] = $prefs->getValue('mailbox');
            IMP::setLabel($imp);
            
            $GLOBALS['HTTP_SESSION_VARS']['imp'] = &$imp;
            session_register('imp');
            
            return true;
        } else {
            return 'failed';
        }
    }
    
    /**
     * Set up the session for use in the page. Retrieve preferences,
     * and make sure to update the mailbox, the sorting information,
     * etc, if any of them have been changed. Also does
     * protocol-specific tasks, such as automatically disabling
     * folders if pop3 is being used.
     *
     * @access public
     *
     * @return mixed True on success, string containing the failure reason on failure.
     */
    function setupSession()
    {
        global $conf, $prefs, $registry;
        
        if (!isset($GLOBALS['HTTP_SESSION_VARS']['imp']) ||
            !is_array($GLOBALS['HTTP_SESSION_VARS']['imp']) ||
            !isset($GLOBALS['HTTP_SESSION_VARS']['imp']['user'])) {
            if (isset($prefs)) {
                $prefs->cleanup(true);
            }
            return 'session';
        } elseif (!isset($GLOBALS['imp'])) {
            $GLOBALS['imp'] = &$GLOBALS['HTTP_SESSION_VARS']['imp'];
        }
        
        if (isset($GLOBALS['HTTP_GET_VARS']['sortby'])) {
            $prefs->setValue('sortby', $GLOBALS['HTTP_GET_VARS']['sortby']);
        }
        if (isset($GLOBALS['HTTP_GET_VARS']['sortdir'])) {
            $prefs->setValue('sortdir', $GLOBALS['HTTP_GET_VARS']['sortdir']);
        }
        
        $mailbox = urldecode(Horde::getFormData('mailbox'));
        if ($mailbox != null) {
            $GLOBALS['imp']['mailbox'] = $mailbox;
        } elseif (!isset($GLOBALS['imp']['mailbox'])) {
            $GLOBALS['imp']['mailbox'] = 'INBOX';
        }
        IMP::setLabel($GLOBALS['imp']);
        
        switch ($GLOBALS['imp']['protocol']) {
        case 'pop3':
            $conf['user']['allow_folders'] = false;
            $prefs->setValue('save_sent_mail', false);
            $prefs->setLocked('save_sent_mail', true);
            $prefs->setLocked('sent_mail_folder', true);
            $prefs->setLocked('drafts_folder', true);
            $prefs->setLocked('trash_folder', true);
            break;
        }
        
        return true;
    }
    
    /**
     * Attempt to open a connection to the IMAP server using the
     * information in the $imp session variable.
     *
     * @access public
     *
     * @param integer $flags  (optional) Flags to pass to IMAP_OPEN. Valid values are:
     * <pre>
     *     OP_READONLY  : Open the mailbox read-only.
     *     OP_ANONYMOUS : (NNTP only) Don't use or update a .newrc file.
     *     OP_HALFOPEN  : (IMAP and NNTP only) Open a connection, but not a specific mailbox.
     *     CL_EXPUNGE   : Expunge the mailbox automatically when the stream is closed.
     * </pre>
     * @param boolean $setup (optional)  Run IMP::setupSession() first.
     *
     * @return mixed True on success, string containing the failure reason on failure.
     */
    function authenticate($flags = 0, $setup = false)
    {
        if ($setup) {
            $retval = IMP::setupSession();
            if ($retval !== true) {
                return $retval;
            }
        }

        global $imp, $conf, $HTTP_SERVER_VARS, $HTTP_POST_VARS;
        if (!(isset($imp) && is_array($imp))) {
            if (isset($GLOBALS['prefs'])) {
                $GLOBALS['prefs']->cleanup(true);
            }
            return 'session';
        }
        
        switch ($imp['protocol']) {
        case 'pop3':
            $connstr = IMP::serverString('pop3');
            $flags &= ~OP_ANONYMOUS;
            $flags &= ~OP_HALFOPEN;
            break;
        default:
            $flags &= ~OP_ANONYMOUS;
            $mailbox = Horde::getFormData('thismailbox', $imp['mailbox']);
            if ($mailbox == '**search') {
                $aindex = Horde::getFormData('array_index');
                if (($aindex !== null) && isset($imp['searchfolders'])) {
                    $tmp = explode(':', $imp['messagefolders']);
                    $connstr = IMP::serverString($imp['protocol']) . imap_utf7_encode($tmp[$aindex]);
                } else {
                    $connstr = IMP::serverString($imp['protocol']);
                    $flags = $flags | OP_HALFOPEN;
                }
            } else {
                $connstr = IMP::serverString($imp['protocol']) . imap_utf7_encode($mailbox);
            }
            break;
        }

        if ($flags == 0) {
            $imp['stream'] = @imap_open($connstr, $imp['user'], Secret::read(Secret::getKey('imp'), $imp['pass']));
        } else {
            $imp['stream'] = @imap_open($connstr, $imp['user'], Secret::read(Secret::getKey('imp'), $imp['pass']), $flags);
        }
        
        if (!$imp['stream']) {
            if (!empty($imp['server']) && !empty($imp['port']) &&
                !empty($imp['protocol']) && !empty($imp['user'])) {
                if (!empty($HTTP_SERVER_VARS['HTTP_X_FORWARDED_FOR'])) {
                    $entry = sprintf('FAILED LOGIN %s (forwarded for [%s]) to %s:%s[%s] as %s',
                                     $HTTP_SERVER_VARS['REMOTE_ADDR'], $HTTP_SERVER_VARS['HTTP_X_FORWARDED_FOR'], $imp['server'], $imp['port'], $imp['protocol'], $imp['user']);
                } else {
                    $entry = sprintf('FAILED LOGIN %s to %s:%s[%s] as %s',
                                     $HTTP_SERVER_VARS['REMOTE_ADDR'], $imp['server'], $imp['port'], $imp['protocol'], $imp['user']);
                }
                Horde::logMessage($entry, __FILE__, __LINE__, LOG_ERR);
            }
            
            $imp = null;
            session_unregister('imp');
            if (isset($GLOBALS['prefs'])) {
                $GLOBALS['prefs']->cleanup(true);
            }
            return 'failed';
        }
        return true;
    }
    
    /**
     * Tack on any prefixes that need to be put at the beginning of
     * the folder path.
     *
     * @return string The prefix, if any.
     */
    function preambleString()
    {
        global $imp;
        return $imp['folders'] . $imp['namespace'];
    }
    
    /**
     * Generate a full c-client server specification string.
     *
     * @param string $protocol (optional) Override the protocol currently being used.
     * @return string          The full spec string.
     */
    function serverString($protocol = null)
    {
        global $imp;
        
        if (!isset($protocol)) {
            $protocol = $imp['protocol'];
        }
        return '{' . $imp['server'] . ':' . $imp['port'] . '/' . $protocol . '}';
    }

    /**
     * Set the plain-text label that is displayed for the current
     * mailbox, replacing '**search' with an appropriate string and
     * removing namespace and folder prefix information from what is
     * shown to the user.
     *
     * @access private
     *
     * @param array &$imp  The hash to pull mailbox information from and put
     *                    the label in.
     */
    function setLabel(&$imp)
    {
        global $conf;
        
        if ($imp['mailbox'] == '**search') {
            $imp['label'] = _("Search Results");
        } else {
            $imp['label'] = $imp['mailbox'];
            if (strcmp($imp['folders'], substr($imp['label'], 0, strlen($imp['folders']))) == 0) {
                $imp['label'] = substr($imp['label'], strlen($imp['folders']));
            }
            if (strcmp($imp['namespace'], substr($imp['label'], 0, strlen($imp['namespace']))) == 0) {
                $imp['label'] = substr($imp['label'], strlen($imp['namespace']));
            }
        }
    }
    
    /**
     * Takes an address object, as returned by imap_header(), and
     * formats it as a string.
     *
     * Object Format
     * For the address: John Doe <john_doe@example.com>
     * The object fields are:
     * <pre>
     *  $object->personal = Personal name ("John Doe")
     *  $object->mailbox  = The user's mailbox ("john_doe")
     *  $object->host     = The host the mailbox is on ("example.com")
     * </pre>
     *
     * @access public
     *
     * @param Object $ob      The address object to be turned into a string
     * @param mixed $filter  (optional) A user@example.com style bare address
     *                        to ignore. Either single string or an array of
     *                        strings. If the address matches $filter,
     *                        an empty string will be returned.
     *
     * @return string $address  The formatted address (Example: John Doe
     *                          <john_doe@example.com>)
     */
    function addrObject2String($ob, $filter = '')
    {
        include_once HORDE_BASE . '/lib/MIME.php';
        
        /* If the personal name is set, decode it. */
        $ob->personal = isset($ob->personal) ? MIME::decode($ob->personal) : '';
        
        /*
         * If both the mailbox and the host are empty, return an empty
         * string.  If we just let this case fall through, the call to
         * imap_rfc822_write_address will end up return just a '@', which
         * is undesirable.
         */
        if (empty($ob->mailbox) && empty($ob->host)) {
            return '';
        }
        
        /* Make sure these two variables have some sort of value. */
        if (!isset($ob->mailbox)) $ob->mailbox = '';
        if (!isset($ob->host)) $ob->host = '';
        
        /* Filter out unwanted addresses based on the $filter string. */
        if (!is_array($filter)) {
            $filter = array($filter);
        }
        foreach ($filter as $f) {
            if (strcasecmp($f, $ob->mailbox . '@' . $ob->host) == 0) {
                return '';
            }
        }
        
        /* Return the trimmed, formatted email address. */
        return Mime::trimEmailAddress(imap_rfc822_write_address($ob->mailbox,
                                                                $ob->host, $ob->personal));
    }
    
    /**
     * Takes an array of address objects, as returned by
     * imap_header(), and passes each of them through
     * IMP::addrObject2String().
     *
     * @access public
     *
     * @param array $addresses   The array of address objects.
     * @param string $filter     (optional) A user@example.com style bare address
     *                           to ignore. If any address matches $filter,
     *                           it will not be included in the final string.
     *
     * @return string $addresses  All of the addresses in a comma-delimited
     *                            string.
     */
    function addrArray2String($addresses, $filter = '')
    {
        if (!is_array($addresses)) return '';
        
        foreach ($addresses as $addr) {
            $val = IMP::addrObject2String($addr, $filter);
            if (!empty($val)) $return[] = $val;
        }
        
        if (isset($return) && is_array($return)) {
            return implode(', ', array_unique($return));
        } else {
            return '';
        }
    }
    
    /**
     * Return an array of folders. This is a wrapper around the
     * IMP_Folder::flist() function which reduces the number of
     * arguments needed if we can assume that IMP's full environment
     * is present.
     *
     * @access public
     *
     * @param array $filter (optional) An array of mailboxes to ignore.
     *
     * @return array $mailboxes   The array of mailboxes returned by
     *                            IMP_Folder::flist().
     */
    function flist($filter = array())
    {
        include_once IMP_BASE . '/lib/Folder.php';
        global $imp, $conf, $prefs;
        return IMP_Folder::flist($imp['stream'], IMP::serverString(), $imp['folders'], $prefs->getValue('subscribe'), $filter, $conf['server']['hierarchies'], $conf['server']['show_dotfiles'], $imp['namespace']);
    }
    
    /**
     * Wrapper around IMP::flist() which generates the body of a
     * &lt;select&gt; form input from the generated folder list. The
     * &lt;select&gt; and &lt;/select&gt; tags are NOT included in the output of
     * this function.
     *
     * @access public
     *
     * @param string $heading   (optional) An optional string to use as
     *                          the label for an empty-value option at
     *                          the top * of the list. Defaults to none.
     * @param boolean $abbrev   (optional) If true, bbreviate long mailbox names
     *                          by replacing the middle of the name with
     *                          '...'. Defaults to true.
     * @param array $filter     (optional) An array of mailboxes to ignore.
     * @param string $selected  (optional) The mailbox to have selected by
     *                          default. Defaults to the first option in the
     *                          list.
     *
     * @return string $options  A string containg <option> elements for each
     *                          mailbox in the list.
     */
    function flistSelect($heading = '', $abbrev = true, $filter = array(), $selected = null)
    {
        $mailboxes = IMP::flist($filter);
        $options = '';
        
        if (strlen($heading) > 0) {
            $options .= '<option value="">' . $heading . "</option>\n";
        }
        
        foreach ($mailboxes as $mbox) {
            $sel = ($mbox['val'] === $selected) ? ' selected="selected"' : '';
            if ($abbrev) {
                $options .= '<option value="' . $mbox['val'] . '"' . "$sel>" . $mbox['abbrev'] . "</option>\n";
            } else {
                $options .= '<option value="' . $mbox['val'] . '"' . "$sel>" . $mbox['label'] . "</option>\n";
            }
        }
        
        return $options;
    }
    
    /**
     * Return an array of unsubscribed folders. This is a wrapper
     * around the IMP_Folder::flistUnsubscribed() function which
     * reduces the number of arguments needed if we can assume that
     * IMP's full environment is present.
     *
     * @access public
     *
     * @param array $filter  (optional) An array of mailboxes to ignore.
     *
     * @return array $mailboxes  The array of mailboxes returned by
     *                           IMP_Folder::flistUnsubscribed().
     */
    function flistUnsubscribed($filter = array())
    {
        include_once IMP_BASE . '/lib/Folder.php';
        global $imp, $conf;
        return IMP_Folder::flistUnsubscribed($imp['stream'], IMP::serverString(), $imp['folders'], $filter, $conf['server']['hierarchies'], $conf['server']['show_dotfiles'], $imp['namespace']);
    }
    
    /**
     * Returns the appropriate link to call the message composition screen.
     *
     * @access public
     *
     * @param mixed $args    (optional) Hash of arguments to pass to compose.php.
     *                       If this is passed in as a string, it will be
     *                       parsed as a toaddress?subject=foo&cc=ccaddress
     *                       (mailto-style) string.
     * @param array $extra   (optional) Hash of extra, non-standard arguments to
     *                       pass to compose.php.
     *
     * @return string $link  The link to the message composition screen.
     */
    function composeLink($args = array(), $extra = array())
    {
        global $prefs, $registry;
        
        /* Make sure the compose window always knows which mailbox
           it's in, for replying, forwarding, marking as answered,
           etc. */
        $extra['thismailbox'] = $GLOBALS['imp']['mailbox'];
        
        if (is_string($args)) {
            $string = $args;
            $args = array();
            if (($pos = strpos($string, '?')) !== false) {
                parse_str(substr($string, $pos + 1), $args);
                $args['to'] = substr($string, 0, $pos);
            } else {
                $args['to'] = $string;
            }
        }
        
        if ($prefs->getValue('compose_popup')) {
            /* This is NOT an elegant way to do it, but %27 needs to
               be escaped or else IE's jscript will interpret it as a
               single quote and refuse to work. */
            return str_replace('%27', '\%27', $registry->link('mail/compose', $args, $extra));
        } else {
            /* Merge the two argument arrays. */
            $args = array_merge($args, $extra);
            
            /* Convert the $args hash into proper URL parameters. */
            $params = '?';
            foreach ($args as $key => $val) {
                if (is_int($val) || !empty($val)) {
                    $key = urlencode($key);
                    $val = urlencode($val);
                    $params .= "$key=$val&";
                }
            }
            
            /* Check for page or message number information to let the
               compose window return to the right place in the
               mailbox. */
            global $msgindex, $page;
            if (isset($msgindex)) {
                $params .= 'start=' . urlencode(($msgindex + 1)) . '&';
            }
            if (isset($page)) {
                $params .= 'page=' . urlencode($page) . '&';
            }
            
            return Horde::url('compose.php' . substr($params, 0, -1));
        }
    }
    
    /**
     * Generate an URL to the logout screen that includes any known
     * information, such as username, server, etc., that can be filled
     * in on the login form.
     *
     * @param string $uri               The page that will process the logout.
     * @param string $reason (optional) The reason for the logout.
     *
     * @return string     The $uri with parameters added.
     */
    function logoutUrl($uri, $reason = null)
    {
        global $conf, $imp;
        
        if (!strstr($uri, '?')) {
            $uri .= '?1=1';
        }
        
        if (!empty($imp['user'])) {
            $uri .= '&imapuser=' . urlencode($imp['user']);
        } elseif (!empty($GLOBALS['HTTP_POST_VARS']['imapuser'])) {
            $uri .= '&imapuser=' . urlencode($GLOBALS['HTTP_POST_VARS']['imapuser']);
        }
        
        if (!empty($GLOBALS['HTTP_POST_VARS']['server'])) {
            $uri .= '&server=' . urlencode($GLOBALS['HTTP_POST_VARS']['server']);
        } elseif (!empty($imp['server'])) {
            $uri .= '&server=' . urlencode($imp['server']);
        }
        
        if (!empty($imp['port'])) {
            $uri .= '&port=' . urlencode($imp['port']);
        }
        if (!empty($imp['protocol'])) {
            $uri .= '&protocol=' . urlencode($imp['protocol']);
        }
        if (!empty($imp['folders'])) {
            $uri .= '&folders=' . urlencode($imp['folders']);
        }
        if (!empty($imp['language'])) {
            $uri .= '&language=' . urlencode($imp['language']);
        }
        if (!empty($reason)) {
            $uri .= '&reason=' . urlencode($reason);
        }
        
        return $uri;
    }
    
    /**
     * Fetch a part of a MIME message.
     *
     * @param object MIME_Part $mime The MIME_Part object describing the part to fetch.
     *
     * @return string The raw MIME part asked for.
     */
    function getPart($mime)
    {
        global $imp;
        return imap_fetchbody($imp['stream'], $mime->index, $mime->imap_id, FT_UID);
    }
    
    /**
     * Fetch part of a MIME message and decode it, if it is base_64 or
     * qprint encoded.
     *
     * @param object MIME_Part $mime The MIME_Part object describing the part to fetch.
     *
     * @return string The raw MIME part asked for.
     */
    function getDecodedPart($mime)
    {
        global $imp, $registry;
        
        if ($mime->encoding == ENCBASE64) {
            $msg = imap_base64(imap_fetchbody($imp['stream'], $mime->index, $mime->imap_id, FT_UID));
        } elseif ($mime->encoding == ENCQUOTEDPRINTABLE) {
            $raw = imap_fetchbody($imp['stream'], $mime->index, $mime->imap_id, FT_UID);
            $data = imap_qprint($raw);
            if (empty($data)) {
                $data = $raw;
            }
            $msg = $data;
        } else {
            $msg = imap_fetchbody($imp['stream'], $mime->index, $mime->imap_id, FT_UID);
        }

        /* Convert Cyrillic character sets. */
        if (stristr($registry->getCharset(), 'windows-1251')) {
            if (Horde::getFormData('charset') == 'koi') {
                $msg = convert_cyr_string($msg, 'k', 'w');
            } else if (Horde::getFormData('charset') == 'iso') {
                $msg = convert_cyr_string($msg, 'i', 'w');
            } else if (Horde::getFormData('charset') == 'mac') {
                $msg = convert_cyr_string($msg, 'm', 'w');
            } else if (Horde::getFormData('charset') == 'dos') {
                $msg = convert_cyr_string($msg, 'a', 'w');
            } else if (stristr($mime->charset, 'koi8-r')) {
                $msg = convert_cyr_string($msg, 'k', 'w');
            }
        }
        if (stristr($registry->getCharset(), 'koi8-r')) {
            if (Horde::getFormData('charset')=='win') {
                $msg = convert_cyr_string($msg, 'w', 'k');
            } else if (Horde::getFormData('charset') == 'iso') {
                $msg = convert_cyr_string($msg, 'i', 'k');
            } else if (Horde::getFormData('charset') == 'mac') {
                $msg = convert_cyr_string($msg, 'm', 'k');
            } else if (Horde::getFormData('charset') == 'dos') {
                $msg = convert_cyr_string($msg, 'a', 'k');
            } else if (stristr($mime->charset, 'windows-1251')) {
                $msg = convert_cyr_string($msg, 'w', 'k');
            }
        }

        return $msg;
    }

    /**
     * Accepts an IMP mime variable and attached a MIME_Part and a MIME_Viewer
     * object onto it.  The actual contents of the part are not filled in here.
     * @param mime Reference to the IMP $mime variable
     */ 
    function resolveMimeViewer(&$mime)
    {
        global $mime_drivers, $mime_drivers_map;
        if (!isset($mime->part) || isset($mime->viewer)) {
            /* Spawn the MIME_Viewer driver */
            $mime->part = new MIME_Part( $mime->TYPE . '/' . $mime->subtype );
            $mime->viewer = &MIME_Viewer::factory($mime->part);
        }
    }

    /**
     * Returns an html table row summarizing a part of a MIME message.
     *
     * @access public
     * @param array $mime    The MIME object to summarize.
     * @return string $row   The html table row summary.
     */
    function partSummary(&$mime, $mode = VIEW_ATTACH)
    {
        global $mime_drivers, $mime_mapping;

        if (!($mime->size > 0)) return ''; // don't display zero-size attachments
        
        global $conf;
        
        include_once HORDE_BASE . '/lib/MIME.php';
        include_once HORDE_BASE . '/lib/SessionCache.php';

        IMP::resolveMimeViewer( $mime );

        $mime_type = $mime->TYPE . '/' . $mime->subtype;
        $icon = MIME_Viewer::getIcon($mime_type);

        // icon column
        $row = '<tr valign="middle"><td><img src="' . $icon . '" height="16" width="16" border="0" alt="" /></td>';
            
        // number column
        $row .= '<td>' . (!empty($mime->imap_id) ? $mime->imap_id : '&nbsp;') . '</td>';
            
        // name/text part column
        $row .= '<td>';

        if (isset($mime->viewer->conf['no_view']) && $mime->viewer->conf['no_view']) {
            $row .= htmlspecialchars(MIME::decode($mime->description));
        } else {
            $row .= Horde::link('', sprintf(_("View %s"), $mime->description), null, null, "view('" . $mode . "', '" . $mime->imap_id . "', '" . SessionCache::putObject($mime) . "'); return false;") . htmlspecialchars(MIME::decode($mime->description)) . '</a>';
            $row .= '</td>';
        }

        if ($mode === VIEW_ATTACH) {
            $row .= '<td>' . $mime->TYPE . '/' . $mime->subtype . '</td>';
            $row .= '<td>' . $mime->size . ' KB</td>';
        } else {
            $row .= '<td>&nbsp;</td>';
            $row .= '<td>' . _("Save as...") . '</td>';
        }
            
        // download column
        if (!empty($mime->viewer->conf['no_download'])) {
            $row .= '<td>&nbsp;</td>';
        } else {
            global $imp;
            $row .= '<td>';
            $row .= Horde::link(Horde::url('view.php?mailbox=' . urlencode($imp['mailbox'])) . '&amp;index=' . Horde::getFormData('index') . '&amp;array_index=' . Horde::getFormData('array_index') . "&amp;id=$mime->imap_id&amp;actionID=" . ($mode == VIEW_ATTACH ? DOWNLOAD_ATTACH : SAVE_MESSAGE) . '&amp;mime=' . SessionCache::putObject($mime) . '&amp;f=/' . urlencode(MIME::decode($mime->name)), sprintf(_("Download %s"), $mime->description));
            $row .= Horde::img('download.gif', 'alt="' . _("Download") . '"');
            $row .= '</a></td>';
        }
        $row .= "</tr>\n";
        return $row;
    }
    
    /**
     * If there is information available to tell us about a prefix in
     * front of mailbox names that shouldn't be displayed to the user,
     * then use it to strip that prefix out.
     *
     * @param string $folder The folder name to display.
     *
     * @return string The folder, with any prefix gone.
     */
    function displayFolder($folder) {
        $prefix = IMP::preambleString();
        if (substr($folder, 0, strlen($prefix)) == $prefix) {
            $folder = substr($folder, strlen($prefix));
        }
        
        return $folder;
    }
    
    /**
     * Add any site-specific headers defined in config/header.txt to
     * an array of headers.
     *
     * @param array &$headers The array to add headers to.
     */
    function addSiteHeaders(&$headers)
    {
        global $conf;
        
        // Tack on any site-specific headers.
        if ($conf['msg']['prepend_header'] && @is_readable(IMP_BASE . '/config/header.txt')) {
            include_once HORDE_BASE . '/lib/Text.php';
            
            $lines = @file(IMP_BASE . '/config/header.txt');
            foreach ($lines as $line) {
                $line = Text::expandEnvironment($line);
                if (!empty($line)) {
                    list($key, $val) = explode(':', $line);
                    $headers[$key] = trim($val);
                }
            }
        }
        
        return $headers;
    }
    
}
?>
