/*

Copyright (C) 2003 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <libnd.h>
#include "libnd_traceset.h"

typedef struct lnd_tracesource LND_TraceSource;

struct lnd_tracesource
{
  char              *file_name;
  LND_Trace         *trace;
};

struct lnd_traceset
{
  GList             *traces; /* GList<LND_TraceSource*> */
  int                num_traces;

  LND_TraceSetMode   mode;
};


static LND_TraceSource *
traceset_ts_new_trace(LND_Trace *trace)
{
  LND_TraceSource *src;

  if (!trace)
    return NULL;

  if (! (src = g_new0(LND_TraceSource, 1)))
    {
      D(("Out of memory.\n"));
      return NULL;
    }

  src->trace = trace;
  return src;
}


static LND_TraceSource *
traceset_ts_new_file(const char *file_name)
{
  LND_TraceSource *src;

  if (!file_name)
    return NULL;

  if (! (src = g_new0(LND_TraceSource, 1)))
    {
      D(("Out of memory.\n"));
      return NULL;
    }

  src->file_name = g_strdup(file_name);
  return src;
}


static LND_TraceSource *
traceset_ts_copy(LND_TraceSource *src)
{
  LND_TraceSource *src_copy;

  if (! src)
    return NULL;

  if (src->trace)
    src_copy = traceset_ts_new_trace(src->trace);
  else
    src_copy = traceset_ts_new_file(src->file_name);

  return src_copy;
}


static void             
traceset_ts_free(LND_TraceSource *src)
{
  if (!src)
    return;

  if (src->file_name)
    g_free(src->file_name);

  g_free(src);
}


LND_TraceSet *  
libnd_traceset_new(void)
{
  LND_TraceSet *set;

  if (! (set = g_new0(LND_TraceSet, 1)))
    {
      D(("Out of memory.\n"));
      return NULL;
    }

  set->mode = LND_TRACESET_ERROR_ABORT;

  return set;
}


LND_TraceSet    *
libnd_traceset_copy(const LND_TraceSet *set)
{
  GList *l;
  LND_TraceSet *set_copy;
  LND_TraceSource *src;

  if (! (set_copy = libnd_traceset_new()))
    return NULL;

  for (l = set->traces; l; l = g_list_next(l))
    {
      if (! (src = traceset_ts_copy((LND_TraceSource *) l->data)))
	{
	  D(("Out of memory\n"));
	  libnd_traceset_free(set_copy);
	  return NULL;
	}

      set_copy->traces = g_list_append(set_copy->traces, src);
      set_copy->num_traces++;
    }
  
  return set_copy;
}


void             
libnd_traceset_free(LND_TraceSet *set)
{
  GList *l;

  if (!set)
    return;

  for (l = set->traces; l; l = g_list_next(l))
    traceset_ts_free((LND_TraceSource *) l->data);

  g_list_free(set->traces);
  g_free(set);
}


gboolean
libnd_traceset_add_trace(LND_TraceSet *set, LND_Trace *trace)
{
  LND_TraceSource *src;

  if (!set)
    return FALSE;

  if (! (src = traceset_ts_new_trace(trace)))
    {
      D(("Out of memory.\n"));
      return FALSE;
    }
  
  set->traces = g_list_append(set->traces, src);
  set->num_traces++;

  return TRUE;
}


gboolean
libnd_traceset_add_trace_name(LND_TraceSet *set, const char *file_name)
{
  LND_TraceSource *src;

  if (!set || !file_name)
    return FALSE;

  if (! libnd_misc_is_tcpdump_file(file_name))
    {
      D(("File %s is not a tcpdump trace.\n", file_name));
      return FALSE;
    }
      
  if (! (src = traceset_ts_new_file(file_name)))
    {
      D(("Out of memory.\n"));
      return FALSE;
    }
  
  set->traces = g_list_append(set->traces, src);
  set->num_traces++;

  return TRUE;
}


static const char*
traceset_get_absolute_file_name(const char *file, const char *dir)
{
  static char result[MAXPATHLEN];

  if (file[0] == '/')
    return file;

  /* File name read from text file is not absolute.
   * Prefix it with the path that the text file resides in, itself.
   */
  g_snprintf(result, MAXPATHLEN, "%s/%s", dir, file);

  return result;
}


int
libnd_traceset_add_trace_name_list(LND_TraceSet *set, const char *file_name)
{
  FILE *f;
  char  file[MAXPATHLEN];
  char *dir = NULL;
  int count = 0;

  if (!set || !file_name)
    {
      D(("Input error.\n"));
      return FALSE;
    }

  if ( (f = fopen(file_name, "r")) == NULL)
    {
      D(("Could not open file %s\n", file_name));
      return FALSE;
    }
  
  if (file_name[0] == G_DIR_SEPARATOR)
    dir = g_dirname(file_name);
  else
    dir = g_get_current_dir();
  
  while (fscanf(f, "%s", file) != EOF)
    {
      const char *abs_file = traceset_get_absolute_file_name(file, dir);
      
      if (! libnd_traceset_add_trace_name(set, abs_file))
	{
	  D(("File addition failed.\n"));
	  fclose(f);
	  g_free(dir);
	  return FALSE;
	}
      
      count++;
    }
  
  fclose(f);
  g_free(dir);
  return count;
}


void             
libnd_traceset_set_mode(LND_TraceSet *set,
			LND_TraceSetMode mode)
{
  if (set)
    set->mode = mode;
}


LND_TraceSetMode 
libnd_traceset_get_mode(LND_TraceSet *set)
{
  if (set)
    return set->mode;

  return LND_TRACESET_ERROR;
}


int              
libnd_traceset_get_size(LND_TraceSet *set)
{
  if (!set)
    return 0;

  return set->num_traces;
}


void             
libnd_traceset_foreach(LND_TraceSet *set,
		       LND_TraceSetCB callback,
		       void *user_data)
{
  GList *l;
  LND_TraceSource *src;
  LND_Trace *trace;
  gboolean free_trace, do_continue;

  /* Iterate over all registered trace sources */
  for (l = set->traces; l; l = g_list_next(l))
    {
      src = l->data;
      
      /* If it's a filename, open the trace. */
      if (src->file_name)
	{
	  free_trace = TRUE;
	  
	  if (! (trace = libnd_trace_new(src->file_name)))
	    {
	      D(("Opening trace '%s' failed.\n", src->file_name));

	      /* What we do now depends on the configured behavior: */
	      switch (set->mode)
		{
		case LND_TRACESET_ERROR_SKIP:
		  continue;
		  
		case LND_TRACESET_ERROR_ABORT:
		default:
		  return;
		}
	    }
	}
      else
	{
	  free_trace = FALSE;
	  trace = src->trace;
	}
      
      do_continue = callback(set, trace, user_data);
      
      if (free_trace)
	libnd_trace_free(trace);

      if (! do_continue)
	return;
    }
}


const char *
name(void)
{
  return "Trace-Set";
}

const char *
author(void)
{
  return "Christian Kreibich, <christian@whoop.org>";
}

const char *
version(void)
{
  return VERSION;
}

/* No init(), no run() -- this one is just providing an API ... */
