#include "exec.h"
#include "fs.h"
#include "msg.h"

#include <dirent.h>
#include <errno.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>


char *ent_from_path(const char *path)
	{
	char *slash;

	slash = strrchr(path, '/');
	return slash ? slash + 1 : (char *)path;
	}

enum type file_type(bool follow, const char *path)
	{
	struct stat st;

	if ((follow ? stat : lstat)(path, &st) == 0)
		return S_ISDIR(st.st_mode) ? Dir : Other;
	return errno == ENOENT ? None : (err(MSG_STAT, true, path), Unknown);
	}

bool is_dot(const char *ent)
	{
	return ent[0] == '.' && (!ent[1] || (ent[1] == '.' && !ent[2]));
	}

DIR *open_dir(const char *path)
	{
	DIR *dir;

	dir = opendir(path);
	if (!dir)
		err(MSG_OPENDIR, true, path);
	return dir;
	}

void pretty_print(const char *path, enum type typ)
	{
	WRITE_STR(1, path);
	if (typ == Dir)
		WRITE_STR(1, "/");
	WRITE_STR(1, "\n");
	}

bool rm_rf(const char *path)
	{
	const char *args[] = { "rm", "-rf", "--", path, NULL };

	return spawn(exec, args, MSG_DEL, path);
	}

int safe_rename(const char *src, enum type typ, const char *dst)
	{
	/* caller ensures src and dst don't refer to the same path */

	if ((typ == Dir ? mkdir(dst, S_IRWXU | S_IRWXG | S_IRWXO): link(src, dst)) != 0)
		return errno == EEXIST ? 0 : (err(typ == Dir ? MSG_MKDIR : MSG_LINK, true, dst), -1);
	if ((typ == Dir ? rename(src, dst) : unlink(src)) == 0)
		return 1;
	err(typ == Dir ? MSG_RENAME : MSG_DEL, true, src);
	if ((typ == Dir ? rmdir : unlink)(dst) != 0)
		err(MSG_DEL, true, dst);
	return 0;
	}
