/* DeScumm - Scumm Script Disassembler (version 6-8 scripts)
 * Copyright (C) 2001  Ludvig Strigeus
 * Copyright (C) 2002-2005  The ScummVM Team
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * $Header: /cvsroot/scummvm/tools/descumm6.cpp,v 1.190.2.1 2005/10/18 03:04:04 sev Exp $
 *
 */

#include "descumm.h"

/*
switch/case statements have a pattern that look as follows (they were probably
generated by some Scumm script compiler):

push SOMETHING		- the value we are switching on
dup					- duplicate it
push caseA			- push the first case value
eq					- compare
jump if false 		- if not equal, jump to next case
  kill 				- we entered this case - kill the switch value from the stack
  ...
  jump AFTER		- leave this switch
dup
push caseB
eq
jump if false
  kill
  ...
  jump AFTER
[optional: default case here ?!?]
AFTER:


Unfortunatly, in reality it is somewhat more complicated. E.g. in script 55,
in some cases the "jump AFTER" instead jumps into the middle of some other case.
Maybe they did have some sort of code optimized after all?

Anyway, the fixed key pattern here seems to be for each case:
  dup - push - eq - jumpFalse - kill
And of course a push before the first of these. Note that there doesn't have
to be a jump before each case: after all, "fall through" is possible with C(++)
switch/case statements, too, so it's possible they used that in Scumm, too.
  
*/


struct StackEnt {
	byte type;
	long data;
	StackEnt *left, *right;
	char *str;
	StackEnt **list;
};

enum {
	seInt = 1,
	seVar = 2,
	seArray = 3,
	seBinary = 4,
	seUnary = 5,
	seComplex = 6,
	seStackList = 7,
	seDup = 8,
	seNeg = 9
};

enum {
	isZero,
	isEqual,
	isNotEqual,
	isGreater,
	isLess,
	isLessEqual,
	isGreaterEqual,
	operAdd,
	operSub,
	operMul,
	operDiv,
	operLand,
	operLor,
	operBand,
	operBor,
	operMod
};

static const char *oper_list[] = {
	"!",
	"==",
	"!=",
	">",
	"<",
	"<=",
	">=",
	"+",
	"-",
	"*",
	"/",
	"&&",
	"||",
	"&",
	"|",
	"%"
};

#define MAX_STACK_SIZE	256
static StackEnt *stack[MAX_STACK_SIZE];
static int num_stack = 0;
bool HumongousFlag = false;

const char *var_names72[] = {
	/* 0 */
	"VAR_KEYPRESS",
	"VAR_DEBUGMODE",
	"VAR_TIMER_NEXT",
	"VAR_OVERRIDE",
	/* 4 */
	"VAR_WALKTO_OBJ",
	"VAR_RANDOM_NR",
	NULL,
	NULL,
	/* 8 */
	"VAR_GAME_LOADED",
	"VAR_EGO",
	"VAR_NUM_ACTOR",
	NULL,
	/* 12 */
	NULL,
	"VAR_VIRT_MOUSE_X",
	"VAR_VIRT_MOUSE_Y",
	"VAR_MOUSE_X",
	/* 16 */
	"VAR_MOUSE_Y",
	NULL,
	NULL,
	"VAR_CURSORSTATE",
	/* 20 */
	"VAR_USERPUT",
	"VAR_ROOM",
	"VAR_ROOM_WIDTH",
	"VAR_ROOM_HEIGHT",
	/* 24 */
	"VAR_CAMERA_POS_X",
	"VAR_CAMERA_MIN_X",
	"VAR_CAMERA_MAX_X",
	"VAR_ROOM_RESOURCE",
	/* 28 */
	"VAR_SCROLL_SCRIPT",
	"VAR_ENTRY_SCRIPT",
	"VAR_ENTRY_SCRIPT2",
	"VAR_EXIT_SCRIPT",
	/* 32 */
	"VAR_EXIT_SCRIPT2",
	"VAR_VERB_SCRIPT",
	"VAR_SENTENCE_SCRIPT",
	"VAR_INVENTORY_SCRIPT",
	/* 36 */
	"VAR_CUTSCENE_START_SCRIPT",
	"VAR_CUTSCENE_END_SCRIPT",
	NULL,
	NULL,
	/* 40 */
	NULL,
	NULL,
	"VAR_RESTART_KEY",
	"VAR_PAUSE_KEY",
	/* 44 */
	"VAR_CUTSCENEEXIT_KEY",
	"VAR_TALKSTOP_KEY",
	"VAR_HAVE_MSG",
	"VAR_NOSUBTITLES",
	/* 48 */
	"VAR_CHARINC",
	"VAR_TALK_ACTOR",
	"VAR_LAST_SOUND",
	"VAR_SOUND_CHANNEL",
	/* 52 */
	"VAR_MUSIC_CHANNEL",
	NULL,
	NULL,
	NULL,
	/* 56 */
	"VAR_NUM_SOUND_CHANNELS",
	"VAR_MEMORY_PERFORMANCE",
	"VAR_VIDEO_PERFORMANCE",
	"VAR_NEW_ROOM",
	/* 60 */
	"VAR_TMR_1",
	"VAR_TMR_2",
	"VAR_TMR_3",
	"VAR_TIMEDATE_HOUR",
	/* 64 */
	"VAR_TIMEDATE_MINUTE",
	"VAR_TIMEDATE_DAY",
	"VAR_TIMEDATE_MONTH",
	"VAR_TIMEDATE_YEAR",
	/* 68 */
	"VAR_NUM_ROOMS",
	"VAR_NUM_SCRIPTS",
	"VAR_NUM_SOUNDS",
	"VAR_NUM_COSTUMES",
	/* 72 */
	"VAR_NUM_IMAGES",
	"VAR_NUM_CHARSETS",
	"VAR_NUM_GLOBAL_OBJS",
	NULL,
	/* 76 */
	"VAR_POLYGONS_ONLY",
	NULL,
	NULL,
	"VAR_WINDOWS_VERSION",
	/* 80 */
	NULL,
	"VAR_NO_SUBTITLES",
	"VAR_NO_SPEECH",
	"VAR_MOUSE_OVERRIDE",
	/* 84 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 88 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 92 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 96 */
	NULL,
	NULL,
	NULL,
	NULL,
};

const char *var_names6[] = {
	/* 0 */
	NULL,
	"VAR_EGO",
	"VAR_CAMERA_POS_X",
	"VAR_HAVE_MSG",
	/* 4 */
	"VAR_ROOM",
	"VAR_OVERRIDE",
	"VAR_MACHINE_SPEED",
	NULL,
	/* 8 */
	"VAR_NUM_ACTOR",
	"VAR_V6_SOUNDMODE",
	"VAR_CURRENTDRIVE",
	"VAR_TMR_1",
	/* 12 */
	"VAR_TMR_2",
	"VAR_TMR_3",
	NULL,
	NULL,
	/* 16 */
	NULL,
	"VAR_CAMERA_MIN_X",
	"VAR_CAMERA_MAX_X",
	"VAR_TIMER_NEXT",
	/* 20 */
	"VAR_VIRT_MOUSE_X",
	"VAR_VIRT_MOUSE_Y",
	"VAR_ROOM_RESOURCE",
	"VAR_LAST_SOUND",
	/* 24 */
	"VAR_CUTSCENEEXIT_KEY",
	"VAR_TALK_ACTOR",
	"VAR_CAMERA_FAST_X",
	"VAR_SCROLL_SCRIPT",
	/* 28 */
	"VAR_ENTRY_SCRIPT",
	"VAR_ENTRY_SCRIPT2",
	"VAR_EXIT_SCRIPT",
	"VAR_EXIT_SCRIPT2",
	/* 32 */
	"VAR_VERB_SCRIPT",
	"VAR_SENTENCE_SCRIPT",
	"VAR_INVENTORY_SCRIPT",
	"VAR_CUTSCENE_START_SCRIPT",
	/* 36 */
	"VAR_CUTSCENE_END_SCRIPT",
	"VAR_CHARINC",
	"VAR_WALKTO_OBJ",
	"VAR_DEBUGMODE",
	/* 40 */
	"VAR_HEAPSPACE",
	"VAR_ROOM_WIDTH",
	"VAR_RESTART_KEY",
	"VAR_PAUSE_KEY",
	/* 44 */
	"VAR_MOUSE_X",
	"VAR_MOUSE_Y",
	"VAR_TIMER",
	"VAR_TMR_4",
	/* 48 */
	NULL,
	"VAR_VIDEOMODE",
	"VAR_MAINMENU_KEY",
	"VAR_FIXEDDISK",
	/* 52 */
	"VAR_CURSORSTATE",
	"VAR_USERPUT",
	"VAR_ROOM_HEIGHT",
	NULL,
	/* 56 */
	"VAR_SOUNDRESULT",
	"VAR_TALKSTOP_KEY",
	NULL,
	"VAR_FADE_DELAY",
	/* 60 */
	"VAR_NOSUBTITLES",
	"VAR_SAVELOAD_SCRIPT",
	"VAR_SAVELOAD_SCRIPT2",
	NULL,
	/* 64 */
	"VAR_SOUNDPARAM",
	"VAR_SOUNDPARAM2",
	"VAR_SOUNDPARAM3",
	"VAR_MOUSEPRESENT",
	/* 68 */
	"VAR_MEMORY_PERFORMANCE",
	"VAR_VIDEO_PERFORMANCE",
	"VAR_ROOM_FLAG",
	"VAR_GAME_LOADED",
	/* 72 */
	"VAR_NEW_ROOM",
	NULL,
	"VAR_LEFTBTN_DOWN",
	"VAR_RIGHTBTN_DOWN",
	/* 76 */
	"VAR_V6_EMSSPACE",
	NULL,
	NULL,
	NULL,
	/* 80 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 84 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 88 */
	NULL,
	NULL,
	"VAR_GAME_DISK_MSG",
	"VAR_OPEN_FAILED_MSG",
	/* 92 */
	"VAR_READ_ERROR_MSG",
	"VAR_PAUSE_MSG",
	"VAR_RESTART_MSG",
	"VAR_QUIT_MSG",
	/* 96 */
	"VAR_SAVE_BTN",
	"VAR_LOAD_BTN",
	"VAR_PLAY_BTN",
	"VAR_CANCEL_BTN",
	/* 100 */
	"VAR_QUIT_BTN",
	"VAR_OK_BTN",
	"VAR_SAVE_DISK_MSG",
	"VAR_ENTER_NAME_MSG",
	/* 104 */
	"VAR_NOT_SAVED_MSG",
	"VAR_NOT_LOADED_MSG",
	"VAR_SAVE_MSG",
	"VAR_LOAD_MSG",
	/* 108 */
	"VAR_SAVE_MENU_TITLE",
	"VAR_LOAD_MENU_TITLE",
	"VAR_GUI_COLORS",
	"VAR_DEBUG_PASSWORD",
	/* 112 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 116 */
	NULL,
	"VAR_MAIN_MENU_TITLE",
	"VAR_RANDOM_NR",
	"VAR_TIMEDATE_YEAR",
	/* 120 */
	NULL,
	NULL,
	"VAR_GAME_VERSION",
	"VAR_CHARSET_MASK",
	/* 124 */
	NULL,
	"VAR_TIMEDATE_HOUR",
	"VAR_TIMEDATE_MINUTE",
	NULL,
	/* 128 */
	"VAR_TIMEDATE_DAY",
	"VAR_TIMEDATE_MONTH",
	NULL,
	NULL,
};

const char *var_names7[] = {
	/* 0 */
	NULL,
	"VAR_MOUSE_X",
	"VAR_MOUSE_Y",
	"VAR_VIRT_MOUSE_X",
	/* 4 */
	"VAR_VIRT_MOUSE_Y",
	"VAR_ROOM_WIDTH",
	"VAR_ROOM_HEIGHT",
	"VAR_CAMERA_POS_X",
	/* 8 */
	"VAR_CAMERA_POS_Y",
	"VAR_OVERRIDE",
	"VAR_ROOM",
	"VAR_ROOM_RESOURCE",
	/* 12 */
	"VAR_TALK_ACTOR",
	"VAR_HAVE_MSG",
	"VAR_TIMER",
	"VAR_TMR_4",
	/* 16 */
	"VAR_TIMEDATE_YEAR",
	"VAR_TIMEDATE_MONTH",
	"VAR_TIMEDATE_DAY",
	"VAR_TIMEDATE_HOUR",
	/* 20 */
	"VAR_TIMEDATE_MINUTE",
	"VAR_TIMEDATE_SECOND",
	"VAR_LEFTBTN_DOWN",
	"VAR_RIGHTBTN_DOWN",
	/* 24 */
	"VAR_LEFTBTN_HOLD",
	"VAR_RIGHTBTN_HOLD",
	"VAR_MEMORY_PERFORMANCE",
	"VAR_VIDEO_PERFORMANCE",
	/* 28 */
	NULL,
	"VAR_GAME_LOADED",
	NULL,
	NULL,
	/* 32 */
	"VAR_V6_EMSSPACE",
	"VAR_VOICE_MODE",
	"VAR_RANDOM_NR",
	"VAR_NEW_ROOM",
	/* 36 */
	"VAR_WALKTO_OBJ",
	"VAR_NUM_GLOBAL_OBJS",
	"VAR_CAMERA_DEST_X",
	"VAR_CAMERA_DEST_>",
	/* 40 */
	"VAR_CAMERA_FOLLOWED_ACTOR",
	NULL,
	NULL,
	NULL,
	/* 44 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 48 */
	NULL,
	NULL,
	"VAR_SCROLL_SCRIPT",
	"VAR_ENTRY_SCRIPT",
	/* 52 */
	"VAR_ENTRY_SCRIPT2",
	"VAR_EXIT_SCRIPT",
	"VAR_EXIT_SCRIPT2",
	"VAR_VERB_SCRIPT",
	/* 56 */
	"VAR_SENTENCE_SCRIPT",
	"VAR_INVENTORY_SCRIPT",
	"VAR_CUTSCENE_START_SCRIPT",
	"VAR_CUTSCENE_END_SCRIPT",
	/* 60 */
	"VAR_SAVELOAD_SCRIPT",
	"VAR_SAVELOAD_SCRIPT2",
	"VAR_CUTSCENEEXIT_KEY",
	"VAR_RESTART_KEY",
	/* 64 */
	"VAR_PAUSE_KEY",
	"VAR_MAINMENU_KEY",
	"VAR_VERSION_KEY",
	"VAR_TALKSTOP_KEY",
	/* 68 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 72 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 76 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 80 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 84 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 88 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 92 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 96 */
	NULL,
	"VAR_TIMER_NEXT",
	"VAR_TMR_1",
	"VAR_TMR_2",
	/* 100 */
	"VAR_TMR_3",
	"VAR_CAMERA_MIN_X",
	"VAR_CAMERA_MAX_X",
	"VAR_CAMERA_MIN_Y",
	/* 104 */
	"VAR_CAMERA_MAX_Y",
	"VAR_CAMERA_THRESHOLD_X",
	"VAR_CAMERA_THRESHOLD_Y",
	"VAR_CAMERA_SPEED_X",
	/* 108 */
	"VAR_CAMERA_SPEED_Y",
	"VAR_CAMERA_ACCEL_X",
	"VAR_CAMERA_ACCEL_Y",
	"VAR_EGO",
	/* 112 */
	"VAR_CURSORSTATE",
	"VAR_USERPUT",
	"VAR_DEFAULT_TALK_DELAY",
	"VAR_CHARINC",
	/* 116 */
	"VAR_DEBUGMODE",
	"VAR_FADE_DELAY",
	NULL,
	"VAR_CHARSET_MASK",
	/* 120 */
	NULL,
	NULL,
	NULL,
	"VAR_VIDEONAME",
	/* 124 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 128 */
	NULL,
	NULL,
	"VAR_STRING2DRAW",
	"VAR_CUSTOMSCALETABLE",
	/* 132 */
	NULL,
	"VAR_BLAST_ABOVE_TEXT",
	NULL,
	"VAR_MUSIC_BUNDLE_LOADED",
	/* 136 */
	"VAR_VOICE_BUNDLE_LOADED",
	NULL,
};

const char *var_names8[] = {
	/* 0 */
	NULL,
	"VAR_ROOM_WIDTH",
	"VAR_ROOM_HEIGHT",
	"VAR_MOUSE_X",
	/* 4 */
	"VAR_MOUSE_Y",
	"VAR_VIRT_MOUSE_X",
	"VAR_VIRT_MOUSE_Y",
	"VAR_CURSORSTATE",
	/* 8 */
	"VAR_USERPUT",
	"VAR_CAMERA_POS_X",
	"VAR_CAMERA_POS_Y",
	"VAR_CAMERA_DEST_X",
	/* 12 */
	"VAR_CAMERA_DEST_Y",
	"VAR_CAMERA_FOLLOWED_ACTOR",
	"VAR_TALK_ACTOR",
	"VAR_HAVE_MSG",
	/* 16 */
	"VAR_MOUSE_BUTTONS",
	"VAR_RIGHTBTN_HOLD",
	"VAR_MOUSE_HOLD",
	NULL,
	/* 20 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 24 */
	"VAR_TIMEDATE_YEAR",
	"VAR_TIMEDATE_MONTH",
	"VAR_TIMEDATE_DAY",
	"VAR_TIMEDATE_HOUR",
	/* 28 */
	"VAR_TIMEDATE_MINUTE",
	"VAR_TIMEDATE_SECOND",
	"VAR_OVERRIDE",
	"VAR_ROOM",
	/* 32 */
	"VAR_NEW_ROOM",
	"VAR_WALKTO_OBJ",
	"VAR_TIMER",
	NULL,
	/* 36 */
	NULL,
	NULL,
	NULL,
	"VAR_VOICE_MODE",
	/* 40 */
	"VAR_GAME_LOADED",
	"VAR_LANGUAGE",
	"VAR_CURRENTDISK",
	NULL,
	/* 44 */
	NULL,
	"VAR_MUSIC_BUNDLE_LOADED",
	"VAR_VOICE_BUNDLE_LOADED",
	NULL,
	/* 48 */
	NULL,
	NULL,
	"VAR_SCROLL_SCRIPT",
	"VAR_ENTRY_SCRIPT",
	/* 52 */
	"VAR_ENTRY_SCRIPT2",
	"VAR_EXIT_SCRIPT",
	"VAR_EXIT_SCRIPT2",
	"VAR_VERB_SCRIPT",
	/* 56 */
	"VAR_SENTENCE_SCRIPT",
	"VAR_INVENTORY_SCRIPT",
	"VAR_CUTSCENE_START_SCRIPT",
	"VAR_CUTSCENE_END_SCRIPT",
	/* 60 */
	NULL,
	NULL,
	"VAR_CUTSCENEEXIT_KEY",
	"VAR_RESTART_KEY",
	/* 64 */
	"VAR_PAUSE_KEY",
	"VAR_MAINMENU_KEY",
	"VAR_VERSION_KEY",
	"VAR_TALKSTOP_KEY",
	/* 68 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 72 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 76 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 80 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 84 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 88 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 92 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 96 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 100 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 104 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 108 */
	NULL,
	NULL,
	NULL,
	"VAR_CUSTOMSCALETABLE",
	/* 112 */
	"VAR_TIMER_NEXT",
	"VAR_TMR_1",
	"VAR_TMR_2",
	"VAR_TMR_3",
	/* 116 */
	"VAR_CAMERA_MIN_X",
	"VAR_CAMERA_MAX_X",
	"VAR_CAMERA_MIN_Y",
	"VAR_CAMERA_MAX_Y",
	/* 120 */
	"VAR_CAMERA_SPEED_X",
	"VAR_CAMERA_SPEED_Y",
	"VAR_CAMERA_ACCEL_X",
	"VAR_CAMERA_ACCEL_Y",
	/* 124 */
	"VAR_CAMERA_THRESHOLD_X",
	"VAR_CAMERA_THRESHOLD_Y",
	"VAR_EGO",
	NULL,
	/* 128 */
	"VAR_DEFAULT_TALK_DELAY",
	"VAR_CHARINC",
	"VAR_DEBUGMODE",
	NULL,
	/* 132 */
	"VAR_KEYPRESS",
	"VAR_BLAST_ABOVE_TEXT",
	"VAR_SYNC",
	NULL,
};

const char *getVarName(uint var)
{
	if (heVersion == 72) {
		if (var >= sizeof(var_names72) / sizeof(var_names72[0]))
			return NULL;
		return var_names72[var];
	} else if (scriptVersion == 8) {
		if (var >= sizeof(var_names8) / sizeof(var_names8[0]))
			return NULL;
		return var_names8[var];
	} else if (scriptVersion == 7) {
		if (var >= sizeof(var_names7) / sizeof(var_names7[0]))
			return NULL;
		return var_names7[var];
	} else {
		if (var >= sizeof(var_names6) / sizeof(var_names6[0]))
			return NULL;
		return var_names6[var];
	}
}

void push(StackEnt * se)
{
	assert(se);
	assert(num_stack < MAX_STACK_SIZE);
	stack[num_stack++] = se;
}

void invalidop(const char *cmd, int op)
{
	if (cmd)
		error("Unknown opcode %s:0x%x (stack count %d)", cmd, op, num_stack);
	else
		error("Unknown opcode 0x%x (stack count %d)", op, num_stack);
}

StackEnt *se_new(int type)
{
	StackEnt *se = (StackEnt *) malloc(sizeof(StackEnt));
	se->type = type;
	return se;
}

void se_free(StackEnt * se)
{
	free(se);
}

StackEnt *se_neg(StackEnt * se)
{
	StackEnt *s = se_new(seNeg);
	s->left = se;
	return s;
}

StackEnt *se_int(int i)
{
	StackEnt *se = se_new(seInt);
	se->data = i;
	return se;
}

StackEnt *se_var(int i)
{
	StackEnt *se = se_new(seVar);
	se->data = i;
	return se;
}

StackEnt *se_array(int i, StackEnt * dim2, StackEnt * dim1)
{
	StackEnt *se = se_new(seArray);
	se->left = dim2;
	se->right = dim1;
	se->data = i;
	return se;
}

StackEnt *se_oper(StackEnt * a, int op)
{
	StackEnt *se = se_new(seUnary);
	se->data = op;
	se->left = a;
	return se;
}

StackEnt *se_oper(StackEnt * a, int op, StackEnt * b)
{
	StackEnt *se = se_new(seBinary);
	se->data = op;
	se->left = a;
	se->right = b;
	return se;
}

StackEnt *se_complex(const char *s)
{
	StackEnt *se = se_new(seComplex);
	se->str = strdup(s);
	return se;
}

char *se_astext(StackEnt * se, char *where, bool wantparens = true)
{
	int i;
	int var;
	const char *s;

	switch (se->type) {
	case seInt:
		where += sprintf(where, "%ld", se->data);
		break;
	case seVar:
		if (scriptVersion == 8) {
			if (!(se->data & 0xF0000000)) {
				var = se->data & 0xFFFFFFF;
				if ((s = getVarName(var)) != NULL)
					where = strecpy(where, s);
				else
					where += sprintf(where, "var%ld", se->data & 0xFFFFFFF);
			} else if (se->data & 0x80000000) {
				where += sprintf(where, "bitvar%ld", se->data & 0x7FFFFFFF);
			} else if (se->data & 0x40000000) {
				where += sprintf(where, "localvar%ld", se->data & 0xFFFFFFF);
			} else {
				where += sprintf(where, "?var?%ld", se->data);
			}
		} else {
			if (!(se->data & 0xF000)) {
				var = se->data & 0xFFF;
				if ((s = getVarName(var)) != NULL)
					where = strecpy(where, s);
				else
					where += sprintf(where, "var%ld", se->data & 0xFFF);
			} else if (se->data & 0x8000) {
				where += sprintf(where, "bitvar%ld", se->data & 0x7FFF);
			} else if (se->data & 0x4000) {
				where += sprintf(where, "localvar%ld", se->data & 0xFFF);
			} else {
				where += sprintf(where, "?var?%ld", se->data);
			}
		}
		break;
	case seArray:
		if (se->left) {
			if(scriptVersion == 8 && !(se->data & 0xF0000000) &&
			   (s = getVarName(se->data & 0xFFFFFFF)) != NULL)
				where += sprintf(where, "%s[",s);
			else if(scriptVersion < 8 && !(se->data & 0xF000) &&
				(s = getVarName(se->data & 0xFFF)) != NULL)
				where += sprintf(where, "%s[",s);
			else
				where += sprintf(where, "array%ld[", se->data);
			where = se_astext(se->left, where);
			where = strecpy(where, "][");
			where = se_astext(se->right, where);
			where = strecpy(where, "]");
		} else {
			if(scriptVersion == 8 && !(se->data & 0xF0000000) &&
			   (s = getVarName(se->data & 0xFFFFFFF)) != NULL)
				where += sprintf(where, "%s[",s);
			else if(scriptVersion < 8 && !(se->data & 0xF000) &&
				(s = getVarName(se->data & 0xFFF)) != NULL)
				where += sprintf(where, "%s[",s);
			else
				where += sprintf(where, "array%ld[", se->data);
			where = se_astext(se->right, where);
			where = strecpy(where, "]");
		}
		break;
	case seUnary:
		where += sprintf(where, "%s", oper_list[se->data]);
		where = se_astext(se->left, where);
		break;
	case seBinary:
		if (wantparens)
			*where++ = '(';
		where = se_astext(se->left, where);
		where += sprintf(where, " %s ", oper_list[se->data]);
		where = se_astext(se->right, where);
		if (wantparens)
			*where++ = ')';
		*where = 0;
		break;
	case seComplex:
		where = strecpy(where, se->str);
		break;
	case seStackList:
		*where++ = '[';
		for (i = se->data; --i >= 0;) {
			where = se_astext(se->list[i], where);
			if (i)
				*where++ = ',';
		}
		*where++ = ']';
		*where = 0;
		break;
	case seDup:
		where += sprintf(where, "dup[%ld]", se->data);
		break;
	case seNeg:
		*where++ = '!';
		where = se_astext(se->left, where);
		break;
	}
	return where;
}

StackEnt *pop()
{
	if (num_stack == 0) {
		printf("ERROR: No items on stack to pop!\n");

		if (!haltOnError)
			return se_complex("**** INVALID DATA ****");
 		exit(1);
	}
	return stack[--num_stack];
}


void kill(char *output, StackEnt * se)
{
	if (se->type != seDup) {
		char *e = strecpy(output, "pop(");
		e = se_astext(se, e);
		strcpy(e, ")");
		se_free(se);
	} else {
		push(se);
	}
}

void doAssign(char *output, StackEnt * dst, StackEnt * src)
{
	if (src->type == seDup && dst->type == seDup) {
		dst->data = src->data;
		return;
	}
	char *e = se_astext(dst, output);
	e = strecpy(e, " = ");
	se_astext(src, e);
}

void doAdd(char *output, StackEnt * se, int val)
{
	char *e = se_astext(se, output);
	if (val == 1) {
		sprintf(e, "++");
	} else if (val == -1) {
		sprintf(e, "--");
	} else {
		/* SCUMM doesn't support this */
		sprintf(e, " += %d", val);
	}
}

StackEnt *dup(char *output, StackEnt * se)
{
	static int dupindex = 0;

	if (se->type == seInt)
		return se;

	StackEnt *dse = se_new(seDup);
	dse->data = ++dupindex;
	doAssign(output, dse, se);
	return dse;
}

void writeArray(char *output, int i, StackEnt * dim2, StackEnt * dim1, StackEnt * value)
{
	StackEnt *array = se_array(i, dim2, dim1);
	doAssign(output, array, value);
	se_free(array);
}

void writeVar(char *output, int i, StackEnt * value)
{
	StackEnt *se = se_var(i);
	doAssign(output, se, value);
	se_free(se);
}

void addArray(char *output, int i, StackEnt * dim1, int val)
{
	StackEnt *array = se_array(i, NULL, dim1);
	doAdd(output, array, val);
	se_free(array);
}

void addVar(char *output, int i, int val)
{
	StackEnt *se = se_var(i);
	doAdd(output, se, val);
	se_free(se);
}

StackEnt *se_get_string()
{
	byte cmd;
	char buf[1024];
	char *e = buf;
	bool in = false;
	int i;

	while ((cmd = get_byte()) != 0) {
		if (cmd == 0xFF || cmd == 0xFE) {
			if (in) {
				*e++ = '"';
				in = false;
			}
			i = get_byte();
			switch (i) {
			case 1:
				e += sprintf(e, ":newline:");
				break;
			case 2:
				e += sprintf(e, ":keeptext:");
				break;
			case 3:
				e += sprintf(e, ":wait:");
				break;
			case 4:		// addIntToStack
			case 5:		// addVerbToStack
			case 6:		// addNameToStack
			case 7:		// addStringToStack
				{
				StackEnt foo;
				foo.type = seVar;
				foo.data = get_word();
				e += sprintf(e, ":");
				e = se_astext(&foo, e);
				e += sprintf(e, ":");
				}
				break;
			case 9:
				e += sprintf(e, ":startanim=%d:", get_word());
				break;
			case 10:
				e += sprintf(e, ":sound:");
				cur_pos += 14;
				break;
			case 14:
				e += sprintf(e, ":setfont=%d:", get_word());
				break;
			case 12:
				e += sprintf(e, ":setcolor=%d:", get_word());
				break;
			case 13:
				e += sprintf(e, ":unk2=%d:", get_word());
				break;
			default:
				e += sprintf(e, ":unk%d=%d:", i, get_word());
			}
		} else {
			if (!in) {
				*e++ = '"';
				in = true;
			}
			*e++ = cmd;
		}
	}
	if (in)
		*e++ = '"';
	*e = 0;
	return se_complex(buf);
}

StackEnt *se_get_list()
{
	StackEnt *se = se_new(seStackList);
	StackEnt *senum = pop();
	int num, i;

	if (senum->type != seInt) {
		error("stackList with variable number of arguments, cannot disassemble");
	}
	se->data = num = senum->data;
	se->list = (StackEnt **) calloc(num, sizeof(StackEnt *));

	for(i = 0; i < num; i++) {
		se->list[i] = pop();
	}
	return se;
}

void ext(char *output, const char *fmt)
{
	bool wantresult;
	byte cmd, extcmd;
	const char *extstr = NULL;
	const char *prep = NULL;
	StackEnt *args[10];
	int numArgs = 0;
	char *e = (char *)output;

	/* return the result? */
	wantresult = false;
	if (*fmt == 'r') {
		wantresult = true;
		fmt++;
	}

	while ((cmd = *fmt++) != '|') {
		if (cmd == 'x' && !extstr) {
			/* Sub-op: next byte specifies which one */
			extstr = fmt;
			while (*fmt++)
				;
			e += sprintf(e, "%s.", extstr);

			/* extended thing */
			extcmd = get_byte();

			/* locate our extended item */
			while ((cmd = *fmt++) != extcmd) {
				/* scan until we find , or \0 */
				while ((cmd = *fmt++) != ',') {
					if (cmd == 0) {
						invalidop(extstr, extcmd);
					}
				}
			}
			/* found a command, continue at the beginning */
			continue;
		}
		if (cmd == 'y' && !extstr) {
			/* Sub-op: parameters are in a list, first element of the list specified the command */
			StackEnt *se;
			extstr = fmt;
			while (*fmt++)
				;
			e += sprintf(e, "%s.", extstr);
			
			args[numArgs++] = se_get_list();
			
			/* extended thing */
			se = args[numArgs - 1];
			se->data--;
			extcmd = (byte) se->list[se->data]->data;

			/* locate our extended item */
			while ((cmd = *fmt++) != extcmd) {
				/* scan until we find , or \0 */
				while ((cmd = *fmt++) != ',') {
					if (cmd == 0) {
						se->data++;
						fmt = "Unknown";
						goto output_command;
					}
				}
			}

			/* found a command, continue at the beginning */
			continue;
		}

		if (cmd == 'p') {
			args[numArgs++] = pop();
		} else if (cmd == 'z') {	// = popRoomAndObj()
			args[numArgs++] = pop();
			if (scriptVersion < 7 && !heVersion)
				args[numArgs++] = pop();
		} else if (cmd == 'h') {
			//TODO Add proper support for HE72+ string system
			if (heVersion == 72)
				args[numArgs++] = pop();
		} else if (cmd == 's') {
			args[numArgs++] = se_get_string();
		} else if (cmd == 'w') {
			args[numArgs++] = se_int(get_word());
		} else if (cmd == 'f') {
			args[numArgs++] = se_int(get_byte());
		} else if (cmd == 'l') {
			args[numArgs++] = se_get_list();
		} else if (cmd == 'j') {
			args[numArgs++] = se_int(get_word());
		} else if (cmd == 'v') {
			args[numArgs++] = se_var(get_word());
		} else {
			error("Character '%c' unknown in argument string '%s', \n", fmt, cmd);
		}
	}

output_command:

	/* create a string from the arguments */
	if (prep)
		e = strecpy(e, prep);
	while (*fmt != 0 && *fmt != ',')
		*e++ = *fmt++;
	*e++ = '(';
	while (--numArgs >= 0) {
		e = se_astext(args[numArgs], e);
		if (numArgs)
			*e++ = ',';
	}
	*e++ = ')';
	*e = 0;

	if (wantresult) {
		push(se_complex((char *)output));
		output[0] = 0;
		return;
	}
}

void jump(char *output)
{
	int offset = get_word();
	int cur = get_curoffs();
	int to = cur + offset;

	if (offset == 1) {
		// Sometimes, jumps with offset 1 occur. I used to suppress those.
		// But it turns out that's not quite correct in some cases. With this
		// code, it can sometimes happens that you get an empty 'else' branch;
		// but in many other cases, an otherwise hidden instruction is revealed,
		// or an instruction is placed into an else branch instead of being
		// (incorrectly) placed inside the body of the 'if' itself.
		sprintf(output, "/* jump %x; */", to);
	} else if (!dontOutputElse && maybeAddElse(cur, to)) {
		pendingElse = true;
		pendingElseTo = to;
		pendingElseOffs = cur;
		pendingElseOpcode = g_jump_opcode;
		pendingElseIndent = num_block_stack;
	} else {
		if (num_block_stack && !dontOutputWhile) {
			BlockStack *p = &block_stack[num_block_stack - 1];
			if (p->isWhile && cur == p->to)
				return;		// A 'while' ends here.
			if (!dontOutputBreaks && maybeAddBreak(cur, to)) {
				sprintf(output, "break");
				return;
		  }
		}
		sprintf(output, "jump %x", to);
	}
}

void jumpif(char *output, StackEnt * se, bool negate)
{
	int offset = get_word();
	int cur = get_curoffs();
	int to = cur + offset;
	char *e = output;

	if (!dontOutputElseif && pendingElse) {
		if (maybeAddElseIf(cur, pendingElseTo, to)) {
			pendingElse = false;
			haveElse = true;
			e = strecpy(e, "} else if (");
			e = se_astext(se, e, false);
			sprintf(e, alwaysShowOffs ? ") /*%.4X*/ {" : ") {", to);
			return;
		}
	}

	if (!dontOutputIfs && maybeAddIf(cur, to)) {
		if (!dontOutputWhile && block_stack[num_block_stack - 1].isWhile)
			e = strecpy(e, negate ? "until (" : "while (");
		else
			e = strecpy(e, negate ? "unless (" : "if (");
		e = se_astext(se, e, false);
		sprintf(e, alwaysShowOffs ? ") /*%.4X*/ {" : ") {", to);
		return;
	}

	e = strecpy(e, negate ? "if (" : "unless (");
	e = se_astext(se, e);
	sprintf(e, ") jump %x", to);
}


#define PRINT_V7HE(name)           \
	do {                          \
		ext(output, "x" name "\0"         \
				"\x41pp|XY,"      \
				"\x42p|color,"    \
				"\x43p|right,"    \
				"\x45|center,"    \
				"\x47|left,"      \
				"\x48|overhead,"  \
				"\x4A|mumble,"    \
				"\x4Bs|msg,"      \
				"\xF9l|colors,"	  \
				"\xFE|begin,"     \
				"\xFF|end"        \
				);                \
	} while(0)

void next_line_HE_V72(char *output)
{
	byte code = get_byte();
	StackEnt *se_a, *se_b;

	switch (code) {
	case 0x0:
		push(se_int(get_byte()));
		break;
	case 0x1:
		push(se_int(get_word()));
		break;
	case 0x2:
		push(se_var(get_byte()));
		break;
	case 0x3:
		push(se_var(get_word()));
		break;
	case 0x4:
		ext(output, "s|addMessageToStack");
		break;
	case 0x7:
		push(se_array(get_word(), NULL, pop()));
		break;
	case 0xB:
		se_a = pop();
		push(se_array(get_word(), pop(), se_a));
		break;
	case 0xC:
		se_a = dup(output, pop());
		push(se_a);
		push(se_a);
		break;
	case 0xD:
		push(se_oper(pop(), isZero));
		break;
	case 0xE:
	case 0xF:
	case 0x10:
	case 0x11:
	case 0x12:
	case 0x13:
	case 0x14:
	case 0x15:
	case 0x16:
	case 0x17:
	case 0x18:
	case 0x19:
		se_a = pop();
		se_b = pop();
		push(se_oper(se_b, (code - 0xE) + isEqual, se_a));
		break;
	case 0x1A:
	case 0xA7:
		kill(output, pop());
		break;
	case 0x43:
		writeVar(output, get_word(), pop());
		break;
	case 0x47:
		writeArray(output, get_word(), NULL, pop(), pop());
		break;
	case 0x4B:
		writeArray(output, get_word(), pop(), pop(), pop());
		break;
	case 0x4F:
		addVar(output, get_word(), 1);
		break;
	case 0x53:
		addArray(output, get_word(), pop(), 1);
		break;
	case 0x54:
		ext(output, "rp|objectX");
		break;
	case 0x55:
		ext(output, "rp|objectY");
		break;
	case 0x57:
		addVar(output, get_word(), -1);
		break;
	case 0x5B:
		addArray(output, get_word(), pop(), -1);
		break;
	case 0x5C:
		jumpif(output, pop(), true);
		break;
	case 0x5D:
		jumpif(output, pop(), false);
		break;
	case 0x5E:
		ext(output, "lpf|startScript");
		break;
	case 0x5F:
		ext(output, "lp|startScriptQuick");
		break;
	case 0x60:
		ext(output, "lppf|startObject");
		break;
	case 0x61:
		ext(output, "x" "drawObject\0"
				"\x3Epppp|unk1,"
				"\x3Fpp|unk2,"
				"\x41ppp|unk3,");
		break;
	case 0x62:
		ext(output, "ppp|drawObjectAt");
		break;
	case 0x65:
		ext(output, "|stopObjectCodeA");
		break;
	case 0x66:
		ext(output, "|stopObjectCodeB");
		break;
	case 0x67:
		ext(output, "|endCutscene");
		break;
	case 0x68:
		ext(output, "l|beginCutscene");
		break;
	case 0x69:
		ext(output, "|stopMusic");
		break;
	case 0x6A:
		ext(output, "p|freezeUnfreeze");
		break;
	case 0x6B:
		ext(output, "x" "cursorCommand\0"
				"\x13z|setCursorImg,"
				"\x14z|setCursorImg,"
				"\x90|cursorOn,"
				"\x91|cursorOff,"
				"\x92|userPutOn,"
				"\x93|userPutOff,"
				"\x94|softCursorOn,"
				"\x95|softCursorOff,"
				"\x96|softUserputOn,"
				"\x97|softUserputOff,"
				"\x99z|setCursorImg,"
				"\x9App|setCursorHotspot,"
				"\x9Cp|initCharset,"
				"\x9Dl|charsetColors,"
				"\xD6p|makeCursorColorTransparent");
		break;
	case 0x6C:
		ext(output, "|breakHere");
		break;
	case 0x6D:
		ext(output, "rlp|ifClassOfIs");
		break;
	case 0x6E:
		ext(output, "lp|setClass");
		break;
	case 0x6F:
		ext(output, "rp|getState");
		break;
	case 0x70:
		ext(output, "pp|setState");
		break;
	case 0x71:
		ext(output, "pp|setOwner");
		break;
	case 0x72:
		ext(output, "rp|getOwner");
		break;
	case 0x73:
		jump(output);
		break;
	case 0x74:
		ext(output, "x" "startSound\0"
				"\xE0|soundFrequency,"
				"\xE6|soundTimer,"
				"\xE7|soundOffset,"
				"\xE8|soundID,"
				"\xF5|soundLoopToggle,"
				"\xFF|start");
		break;
	case 0x75:
		ext(output, "p|stopSound");
		break;
	case 0x76:
		ext(output, "p|startMusic");
		break;
	case 0x77:
		ext(output, "p|stopObjectScript");
		break;
	case 0x78:
		ext(output, "p|panCameraTo");
		break;
	case 0x79:
		ext(output, "p|actorFollowCamera");
		break;
	case 0x7A:
		ext(output, "p|setCameraAt");
		break;
	case 0x7B:
		ext(output, "p|loadRoom");
		break;
	case 0x7C:
		ext(output, "p|stopScript");
		break;
	case 0x7D:
		ext(output, "ppp|walkActorToObj");
		break;
	case 0x7E:
		ext(output, "ppp|walkActorTo");
		break;
	case 0x7F:
		ext(output, "pppp|putActorInXY");
		break;
	case 0x80:
		ext(output, "zp|putActorAtObject");
		break;
	case 0x81:
		ext(output, "pp|faceActor");
		break;
	case 0x82:
		ext(output, "pp|animateActor");
		break;
	case 0x83:
		ext(output, "pppp|doSentence");
		break;
	case 0x84:
		ext(output, "z|pickupObject");
		break;
	case 0x85:
		ext(output, "ppz|loadRoomWithEgo");
		break;
	case 0x87:
		ext(output, "rp|getRandomNumber");
		break;
	case 0x88:
		ext(output, "rpp|getRandomNumberRange");
		break;
	case 0x8A:
		ext(output, "rp|getActorMoving");
		break;
	case 0x8B:
		ext(output, "rp|isScriptRunning");
		break;
	case 0x8C:
		ext(output, "rp|getActorRoom");
		break;
	case 0x8D:
		ext(output, "rp|getObjectX");
		break;
	case 0x8E:
		ext(output, "rp|getObjectY");
		break;
	case 0x8F:
		ext(output, "rp|getObjectDir");
		break;
	case 0x90:
		ext(output, "rp|getActorWalkBox");
		break;
	case 0x91:
		ext(output, "rp|getActorCostume");
		break;
	case 0x92:
		ext(output, "rpp|findInventory");
		break;
	case 0x93:
		ext(output, "rp|getInventoryCount");
		break;
	case 0x94:
		ext(output, "rpp|getVerbFromXY");
		break;
	case 0x95:
		ext(output, "|beginOverride");
		break;
	case 0x96:
		ext(output, "|endOverride");
		break;
	case 0x97:
		ext(output, "ps|setObjectName");
		break;
	case 0x98:
		ext(output, "rp|isSoundRunning");
		break;
	case 0x99:
		ext(output, "pl|setBoxFlags");
		break;
	case 0x9B:
		ext(output, "x" "resourceRoutines\0"
				"\x64p|loadScript,"
				"\x65p|loadSound,"
				"\x66p|loadCostume," 
				"\x67p|loadRoom,"
				"\x68p|nukeScript," 
				"\x69p|nukeSound," 
				"\x6Ap|nukeCostume,"
				"\x6Bp|nukeRoom," 
				"\x6Cp|lockScript," 
				"\x6Dp|lockSound," 
				"\x6Ep|lockCostume,"
				"\x6Fp|lockRoom," 
				"\x70p|unlockScript," 
				"\x71p|unlockSound," 
				"\x72p|unlockCostume,"
				"\x73p|unlockRoom," 
				"\x75p|loadCharset," 
				"\x76p|nukeCharset," 
				"\x77z|loadFlObject,"
				"\x78p|queueloadScript,"
				"\x79p|queueloadSound,"
				"\x7Ap|queueloadCostume,"
				"\x7Bp|queueloadRoomImage,"
				"\x9fp|unlockImage,"
				"\xc0p|nukeImage,"
				"\xc9p|loadImage,"
				"\xcap|lockImage,"
				"\xcbp|queueloadImage,"
				"\xe9p|lockFlObject,"
				"\xebp|unlockFlObject");
		break;
	case 0x9C:
		ext(output, "x" "roomOps\0"
				"\xACpp|roomScroll,"
				"\xAEpp|setScreen,"
				"\xAFpppp|setPalColor,"
				"\xB0|shakeOn,"
				"\xB1|shakeOff,"
				"\xB3ppp|darkenPalette,"
				"\xB4pp|saveLoadThing,"
				"\xB5p|screenEffect,"
				"\xB6ppppp|darkenPalette,"
				"\xB7ppppp|setupShadowPalette,"
				"\xBApppp|palManipulate,"
				"\xBBpp|colorCycleDelay,"
				"\xD5p|setPalette,"
				"\xDCpp|roomFunc220,"
				"\xDDsp|saveLoadThing2");
		break;
	case 0x9D:
		ext(output, "x" "actorOps\0"
				"\xC5p|setCurActor,"
				"\x4Cp|setCostume,"
				"\x4Dpp|setWalkSpeed,"
				"\x4El|setSound,"
				"\x4Fp|setWalkFrame,"
				"\x50pp|setTalkFrame,"
				"\x51p|setStandFrame,"
				"\x52ppp|actorSet:82:??,"
				"\x53|init,"
				"\x54p|setElevation,"
				"\x55|setDefAnim,"
				"\x56pp|setPalette,"
				"\x57p|setTalkColor,"
				"\x58s|setName,"
				"\x59p|setInitFrame,"
				"\x5Bp|setWidth,"
				"\x5Cp|setScale,"
				"\x5D|setNeverZClip,"
				"\x5Ep|setAlwayZClip?,"
				"\xE1ps|setTalkieSlot,"
				"\x5F|setIgnoreBoxes,"
				"\x60|setFollowBoxes,"
				"\x61p|setAnimSpeed,"
				"\x62p|setShadowMode,"
				"\x63pp|setTalkPos,"
				"\xC6pp|setAnimVar,"
				"\xD7|setIgnoreTurnsOn,"
				"\xD8|setIgnoreTurnsOff,"
				"\xD9|initLittle,"
				"\xDA|drawVirScr,"
				"\xE3p|setLayer,"
				"\xE4p|setWalkScript,"
				"\xE5|setStanding,"
				"\xE6p|setDirection,"
				"\xE7p|turnToDirection,"
				"\xE9|freeze,"
				"\xEA|unfreeze,"
				"\xEBp|setTalkScript");
		break;
	case 0x9E:
		ext(output, "x" "verbOps\0"
				"\xC4p|setCurVerb,"
				"\x7Cp|loadImg,"
				"\x7Ds|loadString,"
				"\x7Ep|setColor,"
				"\x7Fp|setHiColor,"
				"\x80pp|setXY,"
				"\x81|setOn,"
				"\x82|setOff,"
				"\x83p|kill,"
				"\x84|init,"
				"\x85p|setDimColor,"
				"\x86|setDimmed,"
				"\x87p|setKey,"
				"\x88|setCenter,"
				"\x89p|setToString,"
				"\x8Bpp|setToObject,"
				"\x8Cp|setBkColor,"
				"\xFF|redraw");
		break;
	case 0x9F:
		ext(output, "rpp|getActorFromXY");
		break;
	case 0xA0:
		ext(output, "rpp|findObject");
		break;
	case 0xA1:
		ext(output, "lp|pseudoRoom");
		break;
	case 0xA2:
		ext(output, "rp|getActorElevation");
		break;
	case 0xA3:
		ext(output, "rpp|getVerbEntrypoint");
		break;
	case 0xA4:
		switch (get_byte()) {
		case 5:{
			int array = get_word();
			writeArray(output, array, NULL, pop(), se_get_string());
			}
			break;
		case 208:
			se_a = pop();
			se_b = se_get_list();
			writeArray(output, get_word(), NULL, se_a, se_b);
			break;
		case 212:
			se_a = pop();
			se_b = se_get_list();
			writeArray(output, get_word(), pop(), se_a, se_b);
			break;
		}
		break;
	case 0xA5:
		ext(output, "x" "saveRestoreVerbs\0"
				"\x8Dppp|saveVerbs,"
				"\x8Eppp|restoreVerbs,"
				"\x8Fppp|deleteVerbs");
		break;
	case 0xA6:
		ext(output, "ppppp|drawBox");
		break;
	case 0xA8:
		ext(output, "rp|getActorWidth");
		break;
	case 0xA9:
		ext(output, "x" "wait\0"
				"\xA8pj|waitForActor,"
				"\xA9|waitForMessage,"
				"\xAA|waitForCamera,"
				"\xAB|waitForSentence,"
				"\xE2pj|waitUntilActorDrawn,"
				"\xE8pj|waitUntilActorTurned,"
				);
		break;
	case 0xAA:
		ext(output, "rp|getActorScaleX");
		break;
	case 0xAB:
		ext(output, "rp|getActorAnimCounter1");
		break;
	case 0xAD:
		ext(output, "rlp|isAnyOf");
		break;
	case 0xAE:
		ext(output, "x" "systemOps\0"
				 "\x9E|pauseGame,"
				 "\xA0|shutDown");
		break;
	case 0xAF:
		ext(output, "rp|isActorInBox");
		break;
	case 0xB0:
		ext(output, "p|delay");
		break;
	case 0xB1:
		ext(output, "p|delaySeconds");
		break;
	case 0xB2:
		ext(output, "p|delayMinutes");
		break;
	case 0xB3:
		ext(output, "|stopSentence");
		break;
	case 0xB4:
		PRINT_V7HE("printLine");
		break;
	case 0xB5:
		PRINT_V7HE("printCursor");
		break;
	case 0xB6:
		PRINT_V7HE("printDebug");
		break;
	case 0xB7:
		PRINT_V7HE("printSystem");
		break;
	case 0xB8:
		// This is *almost* identical to the other print opcodes, only the 'begine' subop differs
		ext(output, "x" "printActor\0"
				"\x41pp|XY,"
				"\x42p|color,"
				"\x43p|right,"
				"\x45|center,"
				"\x47|left,"
				"\x48|overhead,"
				"\x4A|mumble,"
				"\x4Bs|msg,"
				"\xFEp|begin,"
				"\xFF|end");
		break;
	case 0xB9:
		PRINT_V7HE("printEgo");
		break;
	case 0xBA:
		ext(output, "ps|talkActor");
		break;
	case 0xBB:
		ext(output, "s|talkEgo");
		break;
	case 0xBC:
		ext(output, "x" "dimArray\0"
				"\x5pw|int,"
				"\x2pw|bit,"
				"\x3pw|nibble,"
				"\x4pw|byte,"
				"\x7pw|string,"
				"\xCCw|nukeArray");
		break;
	case 0xBD:
		ext(output, "|stopObjectCode");
		break;
	case 0xBE:
		// TODO: this loads another script which does something like
		// stack altering and then finishes (usually with opcode 0xBD).
		// When stack is changed, further disassembly is wrong.
		// This is widely used in HE games.
		// As there are cases when called script does not alter the
		// stack, it's not correct to use "rlpp|..." here
		ext(output, "lpp|startObjectQuick");
		break;
	case 0xBF:
		ext(output, "lp|startScriptQuick2");
		break;
	case 0xC0:
		ext(output, "x" "dim2dimArray\0"
				"\x2ppw|bit,"
				"\x3ppw|nibble,"
				"\x4ppw|byte,"
				"\x5ppw|int,"
				"\x6ppw|dword,"
				"\x7ppw|string");
		break;
	case 0xC4:
		ext(output, "rp|abs");
		break;
	case 0xC5:
		ext(output, "rpp|getDistObjObj");
		break;
	case 0xC6:
		ext(output, "rppp|getDistObjPt");
		break;
	case 0xC7:
		ext(output, "rpppp|getDistPtPt");
		break;
	case 0xC8:
		ext(output, "rl|kernelGetFunctions");
		break;
	case 0xC9:
		ext(output, "y" "kernelSetFunctions\0"
				"\x1|virtScreenLoad,"
				"\x14|eraseAuxActor,"
				"\x15|pauseDrawObjects,"
				"\x16|resumeDrawObjects,"
				"\x17|clearCharsetMask,"
				"\x18|pauseActors,"
				"\x19|resumActors,"
				"\x1E|actorBottomClipOverride,"
				"\x2A|drawWizUnk1,"
				"\x2B|drawWizUnk2,"
				);
		break;
	case 0xCA:
		ext(output, "p|breakXTimes");
		break;
	case 0xCB:
		ext(output, "rlp|pickOneOf");
		break;
	case 0xCC:
		ext(output, "rplp|pickOneOfDefault");
		break;
	case 0xCD:
		ext(output, "pppp|stampObject");
		break;
	case 0xD0:
		ext(output, "|getDateTime");
		break;
	case 0xD1:
		ext(output, "|stopTalking");
		break;
	case 0xD2:
		ext(output, "rpp|getAnimateVariable");
		break;
	case 0xD4:
		ext(output, "wpp|shuffle");
		break;
	case 0xD5:
		ext(output, "lpf|jumpToScript");
		break;
	case 0xD6:
		se_a = pop();
		se_b = pop();
		push(se_oper(se_b, operBand, se_a));
		break;
	case 0xD7:
		se_a = pop();
		se_b = pop();
		push(se_oper(se_b, operBor, se_a));
		break;
	case 0xD8:
		ext(output, "rp|isRoomScriptRunning");
		break;
	case 0xD9:
		ext(output, "p|closeFile");
		break;
	case 0xDA:
		ext(output, "rsp|openFile");
		break;
	case 0xDB:
		ext(output, "rpp|readFile");
		break;
	case 0xDC:
		ext(output, "ppp|writeFile");
		break;

	case 0xDD:
		ext(output, "rp|findAllObjects");
		break;
	case 0xDE:
		ext(output, "h|deleteFile");
		break;
	case 0xDF:
		ext(output, "hh|renameFile");
		break;
	case 0xE0:
		ext(output, "x" "soundOps\0" 
				"\xE0p|setSoundFrequency");
		break;
	case 0xE1:
		ext(output, "rpp|getPixel");
		break;
	case 0xE2:
		ext(output, "p|localizeArray");
		break;
	case 0xE3:
		ext(output, "rl|pickVarRandom");
		break;
	case 0xE4:
		ext(output, "p|setBotSet");
		break;
	case 0xE9:
		ext(output, "ppp|seekFilePos");
		break;
	case 0xEA:
		ext(output, "x" "redimArray\0"
				"\xC7ppw|int,"
				"\xCAppw|byte");
		break;
	case 0xEB:
		ext(output, "rp|readFilePos");
		break;
	case 0xEE:
		ext(output, "rp|stringLen");
		break;
	case 0xF3:
		ext(output, "rx" "readINI\0"
				"\x06h|number,"
				"\x07h|string");
		break;
	case 0xF4:
		ext(output, "x" "writeINI\0"
				"\x06ph|number,"
				"\x07hh|string");
		break;
	case 0xF9:
		ext(output, "h|setFilePath");
		break;
	case 0xFA:
		ext(output, "hf|setWindowCaption");
		break;
	case 0xFB:
		ext(output, "x" "polygonOps\0"
				"\xF6ppppppppp|polygonStore,"
				"\xF7pp|polygonErase,"
				"\xF8ppppppppp|polygonStore,"
				);
		break;
	case 0xFC:
			ext(output, "rpp|polygonHit");
		break;

	default:
		invalidop(NULL, code);
		break;
	}
}

#define PRINT_V8(name)            \
	do {                          \
		ext(output,               \
				"x" name "\0"     \
				"\xC8|baseop,"    \
				"\xC9|end,"       \
				"\xCApp|XY,"      \
				"\xCBp|color,"    \
				"\xCC|center,"    \
				"\xCDp|charset,"  \
				"\xCE|left,"      \
				"\xCF|overhead,"  \
				"\xD0|mumble,"    \
				"\xD1s|msg,"      \
				"\xD2|wrap"       \
				);                \
	} while(0)


void next_line_V8(char *output)
{
	byte code = get_byte();
	StackEnt *se_a, *se_b;

	switch (code) {
	case 0x1:
		push(se_int(get_word()));
		break;
	case 0x2:
		push(se_var(get_word()));
		break;
	case 0x3:
		push(se_array(get_word(), NULL, pop()));
		break;
	case 0x4:
		se_a = pop();
		push(se_array(get_word(), pop(), se_a));
		break;
	case 0x5:
		se_a = dup(output, pop());
		push(se_a);
		push(se_a);
		break;
	case 0x6:
		kill(output, pop());
		break;
	case 0x7:
		push(se_oper(pop(), isZero));
		break;
	case 0x8:
	case 0x9:
	case 0xA:
	case 0xB:
	case 0xC:
	case 0xD:
	case 0xE:
	case 0xF:
	case 0x10:
	case 0x11:
	case 0x12:
	case 0x13:
	case 0x14:
	case 0x15:
	case 0x16:
		se_a = pop();
		se_b = pop();
		push(se_oper(se_b, (code - 0x8) + isEqual, se_a));
		break;
		
	case 0x64:
		jumpif(output, pop(), true);
		break;
	case 0x65:
		jumpif(output, pop(), false);
		break;
	case 0x66:
		jump(output);
		break;
	case 0x67:
		ext(output, "|breakHere");
		break;
	case 0x68:
		ext(output, "p|breakHereVar");
		break;
	case 0x69:
		ext(output, "x" "wait\0"
				"\x1Epj|waitForActor,"
				"\x1F|waitForMessage,"
				"\x20|waitForCamera,"
				"\x21|waitForSentence,"
				"\x22pj|waitUntilActorDrawn,"
				"\x23pj|waitUntilActorTurned,"
				);
		break;
	case 0x6A:
		ext(output, "p|delay");
		break;
	case 0x6B:
		ext(output, "p|delaySeconds");
		break;
	case 0x6C:
		ext(output, "p|delayMinutes");
		break;
	case 0x6D:
		writeVar(output, get_word(), pop());
		break;
	case 0x6E:
		addVar(output, get_word(), +1);
		break;
	case 0x6F:
		addVar(output, get_word(), -1);
		break;
	case 0x70:
		// FIXME - is this correct?!? Also, make the display nicer...
		ext(output, "x" "dimArray\0"
				"\x0Apw|dim-scummvar,"
				"\x0Bpw|dim-string,"
				"\xCAw|undim"
				);
		break;
	case 0x71:
		se_a = pop();
		writeArray(output, get_word(), NULL, pop(), se_a);
		break;

	case 0x74:
		// FIXME - is this correct?!? Also, make the display nicer...
		ext(output, "x" "dim2dimArray\0"
				"\x0Appw|dim2-scummvar,"
				"\x0Bppw|dim2-string,"
				"\xCAw|undim2"
				);
		break;
	case 0x75:
		se_a = pop();
		se_b = pop();
		writeArray(output, get_word(), pop(), se_b, se_a);
		break;
	case 0x76:
		switch (get_byte()) {
		case 0x14:{
			int array = get_word();
			writeArray(output, array, NULL, pop(), se_get_string());
			}
			break;
		case 0x15:
			se_a = pop();
			se_b = se_get_list();
			writeArray(output, get_word(), NULL, se_a, se_b);
			break;
		case 0x16:
			se_a = pop();
			se_b = se_get_list();
			writeArray(output, get_word(), pop(), se_a, se_b);
			break;
		}
		break;

	case 0x79:
		ext(output, "lpp|startScript");
		break;
	case 0x7A:
		ext(output, "lp|startScriptQuick");
		break;
	case 0x7B:
		ext(output, "|stopObjectCode");
		break;
	case 0x7C:
		ext(output, "p|stopScript");
		break;
	case 0x7D:
		ext(output, "lpp|jumpToScript");
		break;
	case 0x7E:
		ext(output, "p|return");
		break;
	case 0x7F:
		ext(output, "lppp|startObject");
		break;

	case 0x81:
		ext(output, "l|beginCutscene");
		break;
	case 0x82:
		ext(output, "|endCutscene");
		break;
	case 0x83:
		ext(output, "p|freezeUnfreeze");
		break;
	case 0x84:
		ext(output, "|beginOverride");
		break;
	case 0x85:
		ext(output, "|endOverride");
		break;
	case 0x86:
		ext(output, "|stopSentence");
		break;
	case 0x87:
		ext(output, "p|debug");
		break;
	case 0x89:
		ext(output, "lp|setClassOf?");
		break;
	case 0x8A:
		ext(output, "pp|setState");
		break;
	case 0x8B:
		ext(output, "pp|setOwner");
		break;

	case 0x8C:
		ext(output, "pp|panCameraTo");
		break;
	case 0x8D:
		ext(output, "p|actorFollowCamera");
		break;
	case 0x8E:
		ext(output, "pp|setCameraAt");
		break;
	case 0x8F:
		ext(output,
				"x" "printActor\0"
				"\xC8p|baseop,"
				"\xC9|end,"
				"\xCApp|XY,"
				"\xCBp|color,"
				"\xCC|center,"
				"\xCDp|charset,"
				"\xCE|left,"
				"\xCF|overhead,"
				"\xD0|mumble,"
				"\xD1s|msg,"
				"\xD2|wrap"
				);
		break;
	case 0x90:
		PRINT_V8("printEgo");
		break;
	case 0x91:
		ext(output, "ps|talkActor");
		break;
	case 0x92:
		ext(output, "s|talkEgo");
		break;
	case 0x93:
		PRINT_V8("printLine");
		break;
	case 0x94:
		PRINT_V8("printCursor");
		break;
	case 0x95:
		PRINT_V8("printDebug");
		break;
	case 0x96:
		PRINT_V8("printSystem");
		break;
	case 0x97:
		PRINT_V8("blastText");
		break;
	case 0x98:
		ext(output, "pppp|drawObject");
		break;

	case 0x9C:
		ext(output, "x" "cursorCommand\0"
				"\xDC|cursorOn,"
				"\xDD|cursorOff,"
				"\xDE|userPutOn,"
				"\xDF|userPutOff,"
				"\xE0|softCursorOn,"
				"\xE1|softCursorOff,"
				"\xE2|softUserputOn,"
				"\xE3|softUserputOff,"
				"\xE4pp|setCursorImg,"
				"\xE5pp|setCursorHotspot,"
				"\xE6p|makeCursorColorTransparent,"
				"\xE7p|initCharset,"
				"\xE8l|charsetColors,"
		    		"\xE9pp|setCursorPosition");
		break;
	case 0x9D:
		ext(output, "p|loadRoom");
		break;
	case 0x9E:
		ext(output, "ppz|loadRoomWithEgo");
		break;
	case 0x9F:
		ext(output, "ppp|walkActorToObj");
		break;
	case 0xA0:
		ext(output, "ppp|walkActorTo");
		break;
	case 0xA1:
		ext(output, "pppp|putActorAtXY");
		break;
	case 0xA2:
		ext(output, "zp|putActorAtObject");
		break;
	case 0xA3:
		ext(output, "pp|faceActor");
		break;
	case 0xA4:
		ext(output, "pp|animateActor");
		break;
	case 0xA5:
		ext(output, "ppp|doSentence");
		break;
	case 0xA6:
		ext(output, "z|pickupObject");
		break;
	case 0xA7:
		ext(output, "pl|setBoxFlags");
		break;
	case 0xA8:
		ext(output, "|createBoxMatrix");
		break;

	case 0xAA:
		ext(output, "x" "resourceRoutines\0"
				"\x3Cp|loadCharset,"
				"\x3Dp|loadCostume,"
				"\x3Ep|loadObject,"
				"\x3Fp|loadRoom,"
				"\x40p|loadScript,"
				"\x41p|loadSound,"
				"\x42p|lockCostume,"
				"\x43p|lockRoom,"
				"\x44p|lockScript,"
				"\x45p|lockSound,"
				"\x46p|unlockCostume,"
				"\x47p|unlockRoom,"
				"\x48p|unlockScript,"
				"\x49p|unlockSound,"
				"\x4Ap|nukeCostume,"
				"\x4Bp|nukeRoom,"
				"\x4Cp|nukeScript,"
				"\x4Dp|nukeSound"
				);
		break;
	case 0xAB:
		ext(output, "x" "roomOps\0"
				"\x52pppp|setRoomPalette,"
				"\x55ppp|setRoomIntensity,"
				"\x57p|fade,"
				"\x58ppppp|setRoomRBGIntensity,"
				"\x59pppp|transformRoom,"
				"\x5App|colorCycleDelay,"
				"\x5Bpp|copyPalette,"
				"\x5Cp|newPalette,"
				"\x5D|saveGame,"
				"\x5E|LoadGame,"
				"\x5Fppppp|setRoomSaturation"
				);
		break;
	case 0xAC:
		// Note: these are guesses and may partially be wrong
		ext(output, "x" "actorOps\0"
				"\x64p|setActorCostume,"
				"\x65pp|setActorWalkSpeed,"
				"\x67|setActorDefAnim,"
				"\x68p|setActorInitFrame,"
				"\x69pp|setActorTalkFrame,"
				"\x6Ap|setActorWalkFrame,"
				"\x6Bp|setActorStandFrame,"
				"\x6C|setActorAnimSpeed,"
				"\x6D|setActorDefault,"	// = initActorLittle ?
				"\x6Ep|setActorElevation,"
				"\x6Fpp|setActorPalette,"
				"\x70p|setActorTalkColor,"
				"\x71s|setActorName,"
				"\x72p|setActorWidth,"
				"\x73p|setActorScale,"
				"\x74|setActorNeverZClip,"
				"\x75p|setActorAlwayZClip?,"
				"\x76|setActorIgnoreBoxes,"
				"\x77|setActorFollowBoxes,"
				"\x78p|setShadowMode,"
				"\x79pp|setActorTalkPos,"
				"\x7Ap|setCurActor,"
				"\x7Bpp|setActorAnimVar,"
				"\x7C|setActorIgnoreTurnsOn,"
				"\x7D|setActorIgnoreTurnsOff,"
				"\x7E|newActor,"
				"\x7Fp|setActorLayer,"
				"\x80|setActorStanding,"
				"\x81p|setActorDirection,"
				"\x82p|actorTurnToDirection,"
				"\x83p|setActorWalkScript,"
				"\x84p|setTalkScript,"
				"\x85|freezeActor,"
				"\x86|unfreezeActor,"
				"\x87p|setActorVolume,"
				"\x88p|setActorFrequency,"
				"\x89p|setActorPan"
				);
		break;
	case 0xAD:
		ext(output, "x" "cameraOps\0"
				"\x32|freezeCamera,"
				"\x33|unfreezeCamera"
				);
		break;
	case 0xAE:
		ext(output, "x" "verbOps\0"
				"\x96p|verbInit,"
				"\x97|verbNew,"
				"\x98|verbDelete,"
				"\x99s|verbLoadString,"
				"\x9App|verbSetXY,"
				"\x9B|verbOn,"
				"\x9C|verbOff,"
				"\x9Dp|verbSetColor,"
				"\x9Ep|verbSetHiColor,"

				"\xA0p|verbSetDimColor,"
				"\xA1|verbSetDim,"
				"\xA2p|verbSetKey,"
				"\xA3p|verbLoadImg,"
				"\xA4p|verbSetToString,"
				"\xA5|verbSetCenter,"
				"\xA6p|verbSetCharset,"
				"\xA7p|verbSetLineSpacing"
				);
		break;
	case 0xAF:
		ext(output, "p|startSound");
		break;

	case 0xB1:
		ext(output, "p|stopSound");
		break;
	case 0xB2:
		ext(output, "l|soundKludge");
		break;
	case 0xB3:
		ext(output, "x" "systemOps\0"
				"\x28|restart,"
				"\x29|quit");
		break;
	case 0xB4:
		ext(output, "x" "saveRestoreVerbs\0"
				"\xB4ppp|saveVerbs,"
				"\xB5ppp|restoreVerbs,"
				"\xB6ppp|deleteVerbs");
		break;
	case 0xB5:
		ext(output, "ps|setObjectName");
		break;
	case 0xB6:
		ext(output, "|getDateTime");
		break;
	case 0xB7:
		ext(output, "ppppp|drawBox");
		break;

	case 0xB9:
		ext(output, "s|startVideo");
		break;
	case 0xBA:
		ext(output, "y" "kernelSetFunctions\0"
				"\xB|lockObject,"
				"\xC|unlockObject,"
				"\xD|remapCostume,"
				"\xE|remapCostumeInsert,"
				"\xF|setVideoFrameRate,"

				"\x14|setBoxScale,"
				"\x15|setScaleSlot,"
				"\x16|setBannerColors,"
				"\x17|setActorChoreLimbFrame,"
				"\x18|clearTextQueue,"
				"\x19|saveGameWrite,"
				"\x1A|saveGameRead,"
				"\x1B|saveGameReadName,"
				"\x1C|saveGameStampScreenshot,"
				"\x1D|setKeyScript,"
				"\x1E|killAllScriptsButMe,"
				"\x1F|stopAllVideo,"
				"\x20|writeRegistryValue,"
				"\x21|paletteSetIntensity,"
				"\x22|queryQuit,"
				
				"\x6C|buildPaletteShadow,"
				"\x6D|setPaletteShadow,"

				"\x76|blastShadowObject,"
				"\x77|superBlastObject"
				);
		break;

	case 0xC8:
		ext(output, "rlp|startScriptQuick2");
		break;
	case 0xC9:
		ext(output, "lppp|startObjectQuick");
		break;
	case 0xCA:
		ext(output, "rlp|pickOneOf");
		break;
	case 0xCB:
		ext(output, "rplp|pickOneOfDefault");
		break;

	case 0xCD:
		ext(output, "rlp|isAnyOf");
		break;
	case 0xCE:
		ext(output, "rp|getRandomNumber");
		break;
	case 0xCF:
		ext(output, "rpp|getRandomNumberRange");
		break;
	case 0xD0:
		ext(output, "rlp|ifClassOfIs");
		break;
	case 0xD1:
		ext(output, "rp|getState");
		break;
	case 0xD2:
		ext(output, "rp|getOwner");
		break;
	case 0xD3:
		ext(output, "rp|isScriptRunning");
		break;

	case 0xD5:
		ext(output, "rp|isSoundRunning");
		break;
	case 0xD6:
		ext(output, "rp|abs");
		break;

	case 0xD8:
		ext(output, "ry" "kernelGetFunctions\0"
				"\x73|getWalkBoxAt,"
				"\x74|isPointInBox,"
				"\xCE|getRGBSlot,"
				"\xD3|getKeyState,"
				"\xD7|getBox,"
				"\xD8|findBlastObject,"
				"\xD9|actorHit,"
				"\xDA|lipSyncWidth,"
				"\xDB|lipSyncHeight,"
				"\xDC|actorTalkAnimation,"
				"\xDD|getMasterSFXVol,"
				"\xDE|getMasterVoiceVol,"
				"\xDF|getMasterMusicVol,"
				"\xE0|readRegistryValue,"
				"\xE1|imGetMusicPosition,"
				"\xE2|musicLipSyncWidth,"
				"\xE3|musicLipSyncHeight"
				);
		break;
	case 0xD9:
		ext(output, "rpp|isActorInBox");
		break;
	case 0xDA:
		ext(output, "rpp|getVerbEntrypoint");
		break;
	case 0xDB:
		ext(output, "rpp|getActorFromXY");
		break;
	case 0xDC:
		ext(output, "rpp|findObject");
		break;
	case 0xDD:
		ext(output, "rpp|getVerbFromXY");
		break;

	case 0xDF:
		ext(output, "rpp|findInventory");
		break;
	case 0xE0:
		ext(output, "rp|getInventoryCount");
		break;
	case 0xE1:
		ext(output, "rpp|getAnimateVariable");
		break;
	case 0xE2:
		ext(output, "rp|getActorRoom");
		break;
	case 0xE3:
		ext(output, "rp|getActorWalkBox");
		break;
	case 0xE4:
		ext(output, "rp|getActorMoving");
		break;
	case 0xE5:
		ext(output, "rp|getActorCostume");
		break;
	case 0xE6:
		ext(output, "rp|getActorScaleX");
		break;
	case 0xE7:
		ext(output, "rp|getActorLayer");
		break;
	case 0xE8:
		ext(output, "rp|getActorElevation");
		break;
	case 0xE9:
		ext(output, "rp|getActorWidth");
		break;
	case 0xEA:
		ext(output, "rp|getObjectDir");
		break;
	case 0xEB:
		ext(output, "rp|getObjectX");
		break;
	case 0xEC:
		ext(output, "rp|getObjectY");
		break;
	case 0xED:
		ext(output, "rp|getActorChore");
		break;
	case 0xEE:
		ext(output, "rpp|getDistObjObj");
		break;
	case 0xEF:
		ext(output, "rpppp|getDistPtPt");
		break;
	case 0xF0:
		ext(output, "rp|getObjectImageX");
		break;
	case 0xF1:
		ext(output, "rp|getObjectImageY");
		break;
	case 0xF2:
		ext(output, "rp|getObjectImageWidth");
		break;
	case 0xF3:
		ext(output, "rp|getObjectImageHeight");
		break;
	case 0xF4:
		ext(output, "rp|getVerbX");
		break;
	case 0xF5:
		ext(output, "rp|getVerbY");
		break;
	case 0xF6:
		ext(output, "rps|stringWidth");
		break;
	case 0xF7:
		ext(output, "rp|getActorZPlane");
		break;

	default:
		invalidop(NULL, code);
		break;
	}
}

#define PRINT_V67(name)           \
	do {                          \
		ext(output, "x" name "\0"         \
				"\x41pp|XY,"      \
				"\x42p|color,"    \
				"\x43p|right,"    \
				"\x45|center,"    \
				"\x47|left,"      \
				"\x48|overhead,"  \
				"\x4A|mumble,"    \
				"\x4Bs|msg,"      \
				"\xFE|begin,"     \
				"\xFF|end"        \
				);                \
	} while(0)

void next_line_V67(char *output)
{
	byte code = get_byte();
	StackEnt *se_a, *se_b;

	switch (code) {
	case 0x0:
		push(se_int(get_byte()));
		break;
	case 0x1:
		push(se_int(get_word()));
		break;
	case 0x2:
		push(se_var(get_byte()));
		break;
	case 0x3:
		push(se_var(get_word()));
		break;
	case 0x6:
		push(se_array(get_byte(), NULL, pop()));
		break;
	case 0x7:
		push(se_array(get_word(), NULL, pop()));
		break;
	case 0xA:
		se_a = pop();
		push(se_array(get_byte(), pop(), se_a));
		break;
	case 0xB:
		se_a = pop();
		push(se_array(get_word(), pop(), se_a));
		break;
	case 0xC:
		se_a = dup(output, pop());
		push(se_a);
		push(se_a);
		break;
	case 0xD:
		push(se_oper(pop(), isZero));
		break;
	case 0xE:
	case 0xF:
	case 0x10:
	case 0x11:
	case 0x12:
	case 0x13:
	case 0x14:
	case 0x15:
	case 0x16:
	case 0x17:
	case 0x18:
	case 0x19:
		se_a = pop();
		se_b = pop();
		push(se_oper(se_b, (code - 0xE) + isEqual, se_a));
		break;
	case 0x1A:
	case 0xA7:
		kill(output, pop());
		break;
	case 0x42:
		writeVar(output, get_byte(), pop());
		break;
	case 0x43:
		writeVar(output, get_word(), pop());
		break;
	case 0x46:
		writeArray(output, get_byte(), NULL, pop(), pop());
		break;
	case 0x47:
		writeArray(output, get_word(), NULL, pop(), pop());
		break;
	case 0x4A:
		writeArray(output, get_byte(), pop(), pop(), pop());
		break;
	case 0x4B:
		writeArray(output, get_word(), pop(), pop(), pop());
		break;
	case 0x4E:
		addVar(output, get_byte(), 1);
		break;
	case 0x4F:
		addVar(output, get_word(), 1);
		break;
	case 0x52:
		addArray(output, get_byte(), pop(), 1);
		break;
	case 0x53:
		addArray(output, get_word(), pop(), 1);
		break;
	case 0x56:
		addVar(output, get_byte(), -1);
		break;
	case 0x57:
		addVar(output, get_word(), -1);
		break;
	case 0x5A:
		addArray(output, get_byte(), pop(), -1);
		break;
	case 0x5B:
		addArray(output, get_word(), pop(), -1);
		break;
	case 0x5C:
		jumpif(output, pop(), true);
		break;
	case 0x5D:
		jumpif(output, pop(), false);
		break;
	case 0x5E:
		ext(output, "lpp|startScript");
		break;
	case 0x5F:
		ext(output, "lp|startScriptQuick");
		break;
	case 0x60:
		ext(output, "lppp|startObject");
		break;
	case 0x61:
		ext(output, "pp|drawObject");
		break;
	case 0x62:
		ext(output, "ppp|drawObjectAt");
		break;
	case 0x63:
		if (HumongousFlag)
			invalidop(NULL, code);
		else
			ext(output, "ppppp|drawBlastObject");
		break;
	case 0x64:
		if (HumongousFlag)
			invalidop(NULL, code);
		else
			ext(output, "pppp|setBlastObjectWindow");
		break;
	case 0x65:
		ext(output, "|stopObjectCodeA");
		break;
	case 0x66:
		ext(output, "|stopObjectCodeB");
		break;
	case 0x67:
		ext(output, "|endCutscene");
		break;
	case 0x68:
		ext(output, "l|beginCutscene");
		break;
	case 0x69:
		ext(output, "|stopMusic");
		break;
	case 0x6A:
		ext(output, "p|freezeUnfreeze");
		break;
	case 0x6B:
		ext(output, "x" "cursorCommand\0"
				"\x90|cursorOn,"
				"\x91|cursorOff,"
				"\x92|userPutOn,"
				"\x93|userPutOff,"
				"\x94|softCursorOn,"
				"\x95|softCursorOff,"
				"\x96|softUserputOn,"
				"\x97|softUserputOff,"
				"\x99z|setCursorImg,"
				"\x9App|setCursorHotspot,"
				"\x9Cp|initCharset,"
				"\x9Dl|charsetColors,"
				"\xD6p|makeCursorColorTransparent");
		break;
	case 0x6C:
		ext(output, "|breakHere");
		break;
	case 0x6D:
		ext(output, "rlp|ifClassOfIs");
		break;
	case 0x6E:
		ext(output, "lp|setClass");
		break;
	case 0x6F:
		ext(output, "rp|getState");
		break;
	case 0x70:
		ext(output, "pp|setState");
		break;
	case 0x71:
		ext(output, "pp|setOwner");
		break;
	case 0x72:
		ext(output, "rp|getOwner");
		break;
	case 0x73:
		jump(output);
		break;
	case 0x74:
		if (HumongousFlag)
			ext(output, "pp|startSound");
		else
			ext(output, "p|startSound");
		break;
	case 0x75:
		ext(output, "p|stopSound");
		break;
	case 0x76:
		ext(output, "p|startMusic");
		break;
	case 0x77:
		ext(output, "p|stopObjectScript");
		break;
	case 0x78:
		if (scriptVersion < 7)
			ext(output, "p|panCameraTo");
		else
			ext(output, "pp|panCameraTo");
		break;
	case 0x79:
		ext(output, "p|actorFollowCamera");
		break;
	case 0x7A:
		if (scriptVersion < 7)
			ext(output, "p|setCameraAt");
		else
			ext(output, "pp|setCameraAt");
		break;
	case 0x7B:
		ext(output, "p|loadRoom");
		break;
	case 0x7C:
		ext(output, "p|stopScript");
		break;
	case 0x7D:
		ext(output, "ppp|walkActorToObj");
		break;
	case 0x7E:
		ext(output, "ppp|walkActorTo");
		break;
	case 0x7F:
		ext(output, "pppp|putActorInXY");
		break;
	case 0x80:
		ext(output, "zp|putActorAtObject");
		break;
	case 0x81:
		ext(output, "pp|faceActor");
		break;
	case 0x82:
		ext(output, "pp|animateActor");
		break;
	case 0x83:
		ext(output, "pppp|doSentence");
		break;
	case 0x84:
		ext(output, "z|pickupObject");
		break;
	case 0x85:
		ext(output, "ppz|loadRoomWithEgo");
		break;
	case 0x87:
		ext(output, "rp|getRandomNumber");
		break;
	case 0x88:
		ext(output, "rpp|getRandomNumberRange");
		break;
	case 0x8A:
		ext(output, "rp|getActorMoving");
		break;
	case 0x8B:
		ext(output, "rp|isScriptRunning");
		break;
	case 0x8C:
		ext(output, "rp|getActorRoom");
		break;
	case 0x8D:
		ext(output, "rp|getObjectX");
		break;
	case 0x8E:
		ext(output, "rp|getObjectY");
		break;
	case 0x8F:
		ext(output, "rp|getObjectDir");
		break;
	case 0x90:
		ext(output, "rp|getActorWalkBox");
		break;
	case 0x91:
		ext(output, "rp|getActorCostume");
		break;
	case 0x92:
		ext(output, "rpp|findInventory");
		break;
	case 0x93:
		ext(output, "rp|getInventoryCount");
		break;
	case 0x94:
		ext(output, "rpp|getVerbFromXY");
		break;
	case 0x95:
		ext(output, "|beginOverride");
		break;
	case 0x96:
		ext(output, "|endOverride");
		break;
	case 0x97:
		ext(output, "ps|setObjectName");
		break;
	case 0x98:
		ext(output, "rp|isSoundRunning");
		break;
	case 0x99:
		ext(output, "pl|setBoxFlags");
		break;
	case 0x9A:
		if (HumongousFlag)
			invalidop(NULL, code);
		else
			ext(output, "|createBoxMatrix");
		break;
	case 0x9B:
		if (HumongousFlag)
			ext(output, "x" "resourceRoutines\0"
					"\x64p|loadScript,"
					"\x65p|loadSound,"
					"\x66p|loadCostume," 
					"\x67p|loadRoom,"
					"\x68p|nukeScript," 
					"\x69p|nukeSound," 
					"\x6Ap|nukeCostume,"
					"\x6Bp|nukeRoom," 
					"\x6Cp|lockScript," 
					"\x6Dp|lockSound," 
					"\x6Ep|lockCostume,"
					"\x6Fp|lockRoom," 
					"\x70p|unlockScript," 
					"\x71p|unlockSound," 
					"\x72p|unlockCostume,"
					"\x73p|unlockRoom," 
					"\x75p|loadCharset," 
					"\x76p|nukeCharset," 
					"\x77z|loadFlObject,"
					"\x78p|queueloadScript,"
					"\x79p|queueloadSound,"
					"\x7Ap|queueloadCostume,"
					"\x7Bp|queueloadRoomImage,"
					"\x9fp|unlockImage,"
					"\xc0p|nukeImage,"
					"\xc9p|loadImage,"
					"\xcap|lockImage,"
					"\xcbp|queueloadImage,"
					"\xe9p|lockFlObject,"
					"\xebp|unlockFlObject");
		else
			ext(output, "x" "resourceRoutines\0"
					"\x64p|loadScript,"
					"\x65p|loadSound,"
					"\x66p|loadCostume," 
					"\x67p|loadRoom,"
					"\x68p|nukeScript," 
					"\x69p|nukeSound," 
					"\x6Ap|nukeCostume,"
					"\x6Bp|nukeRoom," 
					"\x6Cp|lockScript," 
					"\x6Dp|lockSound," 
					"\x6Ep|lockCostume,"
					"\x6Fp|lockRoom," 
					"\x70p|unlockScript," 
					"\x71p|unlockSound," 
					"\x72p|unlockCostume,"
					"\x73p|unlockRoom," 
					"\x75p|loadCharset," 
					"\x76p|nukeCharset," 
					"\x77z|loadFlObject");
		break;
	case 0x9C:
		if (HumongousFlag)
			ext(output, "x" "roomOps\0"
					"\xACpp|roomScroll,"
					"\xAEpp|setScreen,"
					"\xAFpppp|setPalColor,"
					"\xB0|shakeOn,"
					"\xB1|shakeOff,"
					"\xB3ppp|darkenPalette,"
					"\xB4pp|saveLoadThing,"
					"\xB5p|screenEffect,"
					"\xB6ppppp|darkenPalette,"
					"\xB7ppppp|setupShadowPalette,"
					"\xBApppp|palManipulate,"
					"\xBBpp|colorCycleDelay,"
					"\xD5p|setPalette,"
					"\xDCpp|copyPalColor,"
					"\xDDsp|saveLoadThing2");
		else
			ext(output, "x" "roomOps\0"
					"\xACpp|roomScroll,"
					"\xAEpp|setScreen,"
					"\xAFpppp|setPalColor,"
					"\xB0|shakeOn,"
					"\xB1|shakeOff,"
					"\xB3ppp|darkenPalette,"
					"\xB4pp|saveLoadThing,"
					"\xB5p|screenEffect,"
					"\xB6ppppp|darkenPalette,"
					"\xB7ppppp|setupShadowPalette,"
					"\xBApppp|palManipulate,"
					"\xBBpp|colorCycleDelay,"
					"\xD5p|setPalette,"
					"\xDCpp|copyPalColor");
		break;
	case 0x9D:
		ext(output, "x" "actorOps\0"
				"\xC5p|setCurActor,"
				"\x4Cp|setCostume,"
				"\x4Dpp|setWalkSpeed,"
				"\x4El|setSound,"
				"\x4Fp|setWalkFrame,"
				"\x50pp|setTalkFrame,"
				"\x51p|setStandFrame,"
				"\x52ppp|actorSet:82:??,"
				"\x53|init,"
				"\x54p|setElevation,"
				"\x55|setDefAnim,"
				"\x56pp|setPalette,"
				"\x57p|setTalkColor,"
				"\x58s|setName,"
				"\x59p|setInitFrame,"
				"\x5Bp|setWidth,"
				"\x5Cp|setScale,"
				"\x5D|setNeverZClip,"
				"\x5Ep|setAlwayZClip?,"
				"\xE1p|setAlwayZClip?,"
				"\x5F|setIgnoreBoxes,"
				"\x60|setFollowBoxes,"
				"\x61p|setAnimSpeed,"
				"\x62p|setShadowMode,"
				"\x63pp|setTalkPos,"
				"\xC6pp|setAnimVar,"
				"\xD7|setIgnoreTurnsOn,"
				"\xD8|setIgnoreTurnsOff,"
				"\xD9|initLittle,"
				"\xDA|drawVirScr,"
				"\xE3p|setLayer,"
				"\xE4p|setWalkScript,"
				"\xE5|setStanding,"
				"\xE6p|setDirection,"
				"\xE7p|turnToDirection,"
				"\xE9|freeze,"
				"\xEA|unfreeze,"
				"\xEBp|setTalkScript");
		break;
	case 0x9E:
		if (HumongousFlag)
			ext(output, "x" "verbOps\0"
					"\xC4p|setCurVerb,"
					"\x7Cp|loadImg,"
					"\x7Ds|loadString,"
					"\x7Ep|setColor,"
					"\x7Fp|setHiColor,"
					"\x80pp|setXY,"
					"\x81|setOn,"
					"\x82|setOff,"
					"\x83p|kill,"
					"\x84|init,"
					"\x85p|setDimColor,"
					"\x86|setDimmed,"
					"\x87p|setKey,"
					"\x88|setCenter,"
					"\x89p|setToString,"
					"\x8Bpp|setToObject,"
					"\x8Cp|setBkColor,"
					"\xFF|redraw");
		else
			ext(output, "x" "verbOps\0"
					"\xC4p|setCurVerb,"
					"\x7Cp|loadImg,"
					"\x7Ds|loadString,"
					"\x7Ep|setColor,"
					"\x7Fp|setHiColor,"
					"\x80pp|setXY,"
					"\x81|setOn,"
					"\x82|setOff,"
					"\x83|kill,"
					"\x84|init,"
					"\x85p|setDimColor,"
					"\x86|setDimmed,"
					"\x87p|setKey,"
					"\x88|setCenter,"
					"\x89p|setToString,"
					"\x8Bpp|setToObject,"
					"\x8Cp|setBkColor,"
					"\xFF|redraw");
		break;
	case 0x9F:
		ext(output, "rpp|getActorFromXY");
		break;
	case 0xA0:
		ext(output, "rpp|findObject");
		break;
	case 0xA1:
		ext(output, "lp|pseudoRoom");
		break;
	case 0xA2:
		ext(output, "rp|getActorElevation");
		break;
	case 0xA3:
		ext(output, "rpp|getVerbEntrypoint");
		break;
	case 0xA4:
		switch (get_byte()) {
		case 205:{
			int array = get_word();
			writeArray(output, array, NULL, pop(), se_get_string());
			}
			break;
		case 208:
			se_a = pop();
			se_b = se_get_list();
			writeArray(output, get_word(), NULL, se_a, se_b);
			break;
		case 212:
			se_a = pop();
			se_b = se_get_list();
			writeArray(output, get_word(), pop(), se_a, se_b);
			break;
		}
		break;
	case 0xA5:
		ext(output, "x" "saveRestoreVerbs\0"
				"\x8Dppp|saveVerbs,"
				"\x8Eppp|restoreVerbs,"
				"\x8Fppp|deleteVerbs");
		break;
	case 0xA6:
		ext(output, "ppppp|drawBox");
		break;
	case 0xA8:
		ext(output, "rp|getActorWidth");
		break;
	case 0xA9:
		ext(output, "x" "wait\0"
				"\xA8pj|waitForActor,"
				"\xA9|waitForMessage,"
				"\xAA|waitForCamera,"
				"\xAB|waitForSentence,"
				"\xE2pj|waitUntilActorDrawn,"
				"\xE8pj|waitUntilActorTurned,"
				);
		break;
	case 0xAA:
		ext(output, "rp|getActorScaleX");
		break;
	case 0xAB:
		ext(output, "rp|getActorAnimCounter1");
		break;
	case 0xAC:
		ext(output, "l|soundKludge");
		break;
	case 0xAD:
		ext(output, "rlp|isAnyOf");
		break;
	case 0xAE:
		ext(output, "x" "systemOps\0"
				"\x9E|restartGame,"
				"\x9F|pauseGame,"
				"\xA0|shutDown");
		break;
	case 0xAF:
		ext(output, "rp|isActorInBox");
		break;
	case 0xB0:
		ext(output, "p|delay");
		break;
	case 0xB1:
		ext(output, "p|delaySeconds");
		break;
	case 0xB2:
		ext(output, "p|delayMinutes");
		break;
	case 0xB3:
		ext(output, "|stopSentence");
		break;
	case 0xB4:
		if (HumongousFlag)
			PRINT_V7HE("printLine");
		else
			PRINT_V67("printLine");
		break;
	case 0xB5:
		if (HumongousFlag)
			PRINT_V7HE("printCursor");
		else
			PRINT_V67("printCursor");
		break;
	case 0xB6:
		if (HumongousFlag)
			PRINT_V7HE("printDebug");
		else
			PRINT_V67("printDebug");
		break;
	case 0xB7:
		if (HumongousFlag)
			PRINT_V7HE("printSystem");
		else
			PRINT_V67("printSystem");
		break;
	case 0xB8:
		// This is *almost* identical to the other print opcodes, only the 'begine' subop differs
		ext(output, "x" "printActor\0"
				"\x41pp|XY,"
				"\x42p|color,"
				"\x43p|right,"
				"\x45|center,"
				"\x47|left,"
				"\x48|overhead,"
				"\x4A|mumble,"
				"\x4Bs|msg,"
				"\xFEp|begin,"
				"\xFF|end");
		break;
	case 0xB9:
		if (HumongousFlag)
			PRINT_V7HE("printEgo");
		else
			PRINT_V67("printEgo");
		break;
	case 0xBA:
		ext(output, "ps|talkActor");
		break;
	case 0xBB:
		ext(output, "s|talkEgo");
		break;
	case 0xBC:
		ext(output, "x" "dimArray\0"
				"\xC7pv|int,"
				"\xC8pv|bit,"
				"\xC9pv|nibble,"
				"\xCApv|byte,"
				"\xCBpv|string,"
				"\xCCv|nukeArray");
		break;
	case 0xBD:
		if (HumongousFlag)
			ext(output, "|stopObjectCode");
		else
			invalidop(NULL, code);
		break;
	case 0xBE:
		// TODO: this loads another script which does something like
		// stack altering and then finishes (usually with opcode 0xBD).
		// When stack is changed, further disassembly is wrong.
		// This is widely used in HE games.
		// As there are cases when called script does not alter the
		// stack, it's not correct to use "rlpp|..." here
		ext(output, "lpp|startObjectQuick");
		break;
	case 0xBF:
		ext(output, "lp|startScriptQuick2");
		break;
	case 0xC0:
		ext(output, "x" "dim2dimArray\0"
				"\xC7ppv|int,"
				"\xC8ppv|bit,"
				"\xC9ppv|nibble,"
				"\xCAppv|byte,"
				"\xCBppv|string");
		break;
	case 0xC1:
		ext(output, "ps|unknownOp");
		break;
	case 0xC4:
		ext(output, "rp|abs");
		break;
	case 0xC5:
		ext(output, "rpp|getDistObjObj");
		break;
	case 0xC6:
		ext(output, "rppp|getDistObjPt");
		break;
	case 0xC7:
		ext(output, "rpppp|getDistPtPt");
		break;
	case 0xC8:
		if (HumongousFlag)
			ext(output, "rl|kernelGetFunctions");
		else {
			if (scriptVersion == 7) 
				ext(output, "ry" "kernelGetFunctions\0"
					"\x73|getWalkBoxAt,"
					"\x74|isPointInBox,"
					"\xCE|getRGBSlot,"
					"\xCF|getObjectXPos,"
					"\xD0|getObjectYPos,"
					"\xD1|getObjectWidth,"
					"\xD2|getObjectHeight,"
					"\xD3|getKeyState,"
					"\xD4|getActorFrame,"
					"\xD5|getVerbXPos,"
					"\xD6|getVerbYPos,"
					"\xD7|getBoxFlags"
					);
			else 
				ext(output, "ry" "kernelGetFunctions\0"
					"\x71|getPixel"
					);
		}
		break;
	case 0xC9:
		if (scriptVersion == 7) 
			ext(output, "y" "kernelSetFunctions\0"
					"\x4|grabCursor,"
					"\x6|startVideo,"
					"\xC|setCursorImg,"
					"\xD|remapCostume,"
					"\xE|remapCostumeInsert,"
					"\xF|setVideoFrameRate,"

					"\x10|enqueueTextCentered,"
					"\x11|enqueueTextNormal,"
					"\x12|setMouseXY,"
					"\x14|setRadioChatter,"

					"\x6B|setActorScale,"
					"\x6C|buildPaletteShadow,"
					"\x6D|setPaletteShadow,"

					"\x72|unk114,"
					"\x75|freezeScripts,"
					"\x76|blastShadowObject,"
					"\x77|superBlastObject,"
					"\x7C|setSaveSound,"
					"\xD7|setSubtitles,"
					);
		else 
			ext(output, "y" "kernelSetFunctions\0"
					"\x3|dummy,"
					"\x4|grabCursor,"
					"\x5|fadeOut,"
					"\x6|redrawScreen,"
					"\x8|startManiac,"
					"\x9|killAllScriptsExceptCurrent,"
					"\x68|nukeFlObjects,"
					"\x6B|setActorScale,"
					"\x6C|setupShadowPalette,"
					"\x6D|setupShadowPalette,"
					"\x6E|clearCharsetMask,"
					"\x6F|setActorShadowMode,"
					"\x70|shiftShadowPalette,"
					"\x72|noirMode,"
					"\x75|freezeScripts,"
					"\x77|superBlastObject,"
					"\x78|swapPalColors,"
					"\x7A|setSoundResult,"
					"\x7B|copyPalColor,"
					"\x7C|setSaveSound,"
					);
		break;
	case 0xCA:
		ext(output, "p|breakXTimes");
		break;
	case 0xCB:
		ext(output, "rlp|pickOneOf");
		break;
	case 0xCC:
		ext(output, "rplp|pickOneOfDefault");
		break;
	case 0xCD:
		ext(output, "pppp|stampObject");
		break;
	case 0xD0:
		ext(output, "|getDateTime");
		break;
	case 0xD1:
		ext(output, "|stopTalking");
		break;
	case 0xD2:
		ext(output, "rpp|getAnimateVariable");
		break;
	case 0xD4:
		ext(output, "vpp|shuffle");
		break;
	case 0xD5:
		ext(output, "lpp|jumpToScript");
		break;
	case 0xD6:
		se_a = pop();
		se_b = pop();
		push(se_oper(se_b, operBand, se_a));
		break;
	case 0xD7:
		se_a = pop();
		se_b = pop();
		push(se_oper(se_b, operBor, se_a));
		break;
	case 0xD8:
		ext(output, "rp|isRoomScriptRunning");
		break;
	case 0xD9:
		if (HumongousFlag)
			ext(output, "p|closeFile");
		else
			invalidop(NULL, code);
		break;
	case 0xDA:
		if (HumongousFlag)
			ext(output, "rsp|openFile");
		else
			invalidop(NULL, code);
		break;
	case 0xDB:
		if (HumongousFlag)
			ext(output, "rpp|readFile");
		else
			invalidop(NULL, code);
		break;
	case 0xDC:
		if (HumongousFlag)
			ext(output, "ppp|writeFile");
		else
			invalidop(NULL, code);
		break;

	case 0xDD:
		ext(output, "rp|findAllObjects");
		break;
	case 0xDE:
		if (HumongousFlag)
			ext(output, "s|deleteFile");
		else
			invalidop(NULL, code);
		break;
	case 0xDF:
		if (HumongousFlag)
			ext(output, "ss|renameFile");
		else
			invalidop(NULL, code);
		break;
	case 0xE0:
		if (HumongousFlag)
			ext(output, "x" "soundOps\0" 
				"\xDEp|setMusicVolume,"
				"\xE0p|setSoundFrequency");
		else
			invalidop(NULL, code);
		break;
	case 0xE1:
		ext(output, "rpp|getPixel");
		break;
	case 0xE2:
		if (HumongousFlag)
			ext(output, "p|localizeArrayToScript");
		else
			invalidop(NULL, code);
		break;
	case 0xE3:
		//TODO add proper decoding
		ext(output, "rl|pickVarRandom");
		break;
	case 0xE4:
		ext(output, "p|setBotSet");
		break;
	case 0xE9:
		if (HumongousFlag)
			ext(output, "ppp|seekFilePos");
		else
			invalidop(NULL, code);
		break;
	case 0xEA:
		if (HumongousFlag)
			ext(output, "x" "redimArray\0"
					"\xC7ppw|int,"
					"\xCAppw|byte");
		else
			invalidop(NULL, code);
		break;
	case 0xEB:
		if (HumongousFlag)
			ext(output, "rp|readFilePos");
		else
			invalidop(NULL, code);
		break;
	case 0xEC:
		if (HumongousFlag)
			invalidop(NULL, code);
		else
			ext(output, "rp|getActorLayer");
		break;
	case 0xED:
		if (HumongousFlag)
			invalidop(NULL, code);
		else
			ext(output, "rp|getObjectNewDir");
		break;
	case 0xEE:
		if (HumongousFlag)
			ext(output, "rp|stringLen");
		else
			invalidop(NULL, code);
		break;
	case 0xF3:
		if (HumongousFlag)
			ext(output, "rsp|readINI");
		else
			invalidop(NULL, code);
		break;
	case 0xF4:
		if (HumongousFlag)
			ext(output, "rsp|writeINI");
		else
			invalidop(NULL, code);
		break;
	case 0xF9:
		if (HumongousFlag)
			ext(output, "s|setFilePath");
		else
			invalidop(NULL, code);
		break;
	case 0xFA:
		if (HumongousFlag) {
			ext(output, "fs|setWindowCaption");
		} else
			invalidop(NULL, code);
		break;
	case 0xFB:
		if (HumongousFlag)
			ext(output, "x" "polygonOps\0"
					"\xF6ppppppppp|polygonStore,"
					"\xF7pp|polygonErase,"
					"\xF8ppppppppp|polygonStore,"
					);
		else
			invalidop(NULL, code);
		break;
	case 0xFC:
		if (HumongousFlag)
			ext(output, "rpp|polygonHit");
		else
			invalidop(NULL, code);
		break;
	default:
		invalidop(NULL, code);
		break;
	}
}
