/*  This file is part of the KDE project
    Copyright (C) 2000 Simon Hausmann <hausmann@kde.org>
    Copyright (C) 2003 Luciano Montanaro <mikelima@cirulla.net>
    Copyright (C) 2001 Dirk Mueller <mueller@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#include "dcopclient.h"
#include "dcopobject.h"
#include "kdebug.h"

#include "kapplication.h"

#include <kio/connection.h>

#include <assert.h>
#include <ctype.h>

static inline bool isIdentChar( char x )
{                                               // Avoid bug in isalnum
    return x == '_' || (x >= '0' && x <= '9') ||
         (x >= 'a' && x <= 'z') || (x >= 'A' && x <= 'Z');
}

QCString normalizeFunctionSignature( const QCString& fun ) {
    if ( fun.isEmpty() )                                // nothing to do
        return fun.copy();
    QCString result( fun.size() );
    char *from  = fun.data();
    char *to    = result.data();
    char *first = to;
    char last = 0;
    while ( true ) {
        while ( *from && isspace(*from) )
            from++;
        if ( last && isIdentChar( last ) && isIdentChar( *from ) )
            *to++ = 0x20;
        while ( *from && !isspace(*from) ) {
            last = *from++;
            *to++ = last;
        }
        if ( !*from )
            break;
    }
    if ( to > first && *(to-1) == 0x20 )
        to--;
    *to = '\0';
    result.resize( (int)((long)to - (long)result.data()) + 1 );
    return result;
}

KIO::Connection *DCOPClient::s_connection = 0;

DCOPClient::DCOPClient()
{
}

DCOPClient::~DCOPClient()
{
}

bool DCOPClient::send( const QCString &remApp, 
	const QCString &remObj, const QCString &remFun, const QByteArray &data )
{
    kdDebug() << "dcopclient::send (" << 
	remApp << ", " << remObj << ", " << remFun << ")" << endl;
    QCString replyType;
    QByteArray replyData;

    if ( !globalConnection() )
        return DCOPDispatcher::self()->process( remObj, 
		normalizeFunctionSignature( remFun ),
		data, replyType, replyData );

    QByteArray realData;
#ifndef QT_NO_DATASTREAM
    QDataStream stream( realData, IO_WriteOnly );
    stream << remApp << remObj << normalizeFunctionSignature( remFun ) << data;

    globalConnection()->send( static_cast<int>( DCOPClient::Send ), realData );
#endif
    return true;
}

bool DCOPClient::call( const QCString &remApp, const QCString &remObj, 
	const QCString &remFun, const QByteArray &data, 
	QCString &replyType, QByteArray &replyData )
{
    kdDebug() << "dcopclient::call(" << 
	remApp << ", " << remObj << ", " << remFun << ")" << endl;

    if ( !globalConnection() )
        return DCOPDispatcher::self()->process( remObj, 
		normalizeFunctionSignature( remFun ), data, 
		replyType, replyData );

    QByteArray realData;
#ifndef QT_NO_DATASTREAM
    QDataStream stream( realData, IO_WriteOnly );
    stream << remApp << remObj << normalizeFunctionSignature( remFun ) << data;

    globalConnection()->send( static_cast<int>( DCOPClient::Call ), realData );
#endif
    QByteArray reply;
    int cmd = 0;
    int res = globalConnection()->read( &cmd, reply );

    assert( res != -1 );

#ifndef QT_NO_DATASTREAM
    QDataStream replyStream( reply, IO_ReadOnly );
    replyStream >> replyType >> replyData;
#endif
    return true; // ###
}

void DCOPClient::setDaemonMode( bool daemonMode )
{
    QByteArray data;
    QDataStream ds(data, IO_WriteOnly);
    ds << static_cast<Q_INT8>( daemonMode );

    QCString replyType;
    QByteArray reply;
    if (!call("DCOPServer", "", "setDaemonMode(bool)", data, replyType, reply))
        qWarning("I couldn't enable daemon mode at the dcopserver!");
}

DCOPClient *
DCOPClient::mainClient()
{
    return KApplication::kApplication()->dcopClient();
}
