/* logjam - a GTK client for LiveJournal.
 * Copyright (C) 2000-2002 Evan Martin <evan@livejournal.com>
 *
 * vim: tabstop=4 shiftwidth=4 noexpandtab :
 * $Id: settings.c,v 1.26 2002/11/01 10:00:13 martine Exp $
 */

#include "config.h"

#include <gtk/gtk.h>
#ifdef HAVE_GTKSPELL
#include <gtkspell/gtkspell.h>
#endif /* HAVE_GTKSPELL */
#include <stdlib.h> /* atoi */
#include <string.h>

#include "conf.h"
#include "security.h"
#include "util.h"
#include "spawn.h"
#include "lj.h"
#include "checkfriends.h"
#include "settings.h"

/* what's this?  all of these funny structures in the settings box?
 * well, instead of creating and tearing down all of these widgets, i
 * try to pull out shared functionality.
 *
 * it's loosely inspired by george lebl's pong 
 *   http://www.5z.com/jirka/pong-documentation/
 * but i've never actaully read how that works.
 *
 *
 * we have a collection of SettingsWidgets that track their name and
 * configuration option, and then it's all nicely automated through functions
 * of the form sw_*(). 
 */

typedef enum {
	SW_TOGGLE,
	SW_TEXT,
	SW_INTEGER,
	SW_COMBO,
	SW_CUSTOM
} SettingsWidgetType;

typedef struct _SettingsWidget SettingsWidget;
struct _SettingsWidget {
	char *name;
	void *conf;
	SettingsWidgetType type;
	char *caption;
	void (*custom)(SettingsWidget *sw);
	GtkWidget *widget;
	gpointer data; /* extra pointer, to be used as needed. */
	gpointer data2; /* extra pointer for wimps */
};

static void sec_save_cb(SettingsWidget *sw);
static void font_save_cb(SettingsWidget *sw);
static void cfmask_save_cb(SettingsWidget *sw);

static void run_fontsel_settings_dlg(SettingsWidget *sw);
static void run_cfmask_settings_dlg(SettingsWidget *sw);

static SettingsWidget settingswidgets[] = {
	{ "ui_revertusejournal", &conf.options.revertusejournal, 
		SW_TOGGLE, "_Revert to primary journal after posting on a community", NULL },
	{ "ui_keepmetadata", &conf.options.keepmetadata, 
		SW_TOGGLE, "Keep _metadata after posting", NULL },
	{ "ui_defaultsecurity", &conf.defaultsecurity, 
		SW_CUSTOM, "Default security on posts:", &sec_save_cb },
	{ "ui_autosave", &conf.options.autosave,
		SW_TOGGLE, "Automatically save drafts (for crash recovery)", NULL },
	{ "ui_noautologin", &conf.options.noautologin,
		SW_TOGGLE, "Don't automatically login", NULL },

#ifdef HAVE_GTKSPELL
	{ "ui_spellcheck", &conf.options.usespellcheck, 
		SW_TOGGLE, "_Use spell check" },
#endif
	{ "ui_font", &conf.uifont,
		SW_CUSTOM, "Entry display font:", &font_save_cb },

	{ "web_spawn", &conf.spawn_command, 
		SW_COMBO, NULL },

	{ "net_useproxy", &conf.options.useproxy, 
		SW_TOGGLE, "Use _proxy server" },
	{ "net_proxy", &conf.proxy, 
		SW_TEXT, "UR_L:" },

	{ "net_useproxyauth", &conf.options.useproxyauth, 
		SW_TOGGLE, "Use proxy _authentication" },
	{ "net_proxyuser", &conf.proxyuser, 
		SW_TEXT, "_User:" },
	{ "net_proxypass", &conf.proxypass, 
		SW_TEXT, "_Password:" },

	{ "debug_netdump", &conf.options.netdump, 
		SW_TOGGLE, "Dump _network data on stderr" },
#ifndef G_OS_WIN32
	{ "debug_nofork", &conf.options.nofork, 
		SW_TOGGLE, "Don't _fork on network requests" },
#else
	{ "debug_nofork", &conf.options.nofork, 
		SW_TOGGLE, "Don't run network requests in a separate _thread" },
#endif


	{ "cf_autostart", &conf.options.cfautostart,
		SW_TOGGLE, "_Monitor friends list for new entries upon login" },
	{ "cf_usemask", &conf.options.cfusemask,
		SW_TOGGLE, "_Limit monitoring to groups of friends" },
	{ "cf_mask", &conf.cfmask,
		SW_CUSTOM, "", &cfmask_save_cb },
	{ "cf_threshold", &conf.cfthreshold,
		SW_INTEGER, "Threshold for _notification (number of new entries): " },
	{ "cf_userinterval", &conf.cfuserinterval,
		SW_INTEGER, "_Check friends list interval (seconds): " },
	{ "cf_autofloat", &conf.options.cfautofloat,
		SW_TOGGLE, "_Show floating indicator on application start" },
	{ "cf_autofloatraise", &conf.options.cfautofloatraise,
		SW_TOGGLE, "_Raise floating indicator when new entries detected" },
	{ "cf_float_decorate", &conf.options.cffloat_decorate,
		SW_TOGGLE, "Show _titlebar on floating indicator" },
	
	{ NULL }
};

static SettingsWidget*
sw_lookup(char *name) {
	SettingsWidget *sw;
	for (sw = settingswidgets; sw->name; sw++) 
		if (strcmp(name, sw->name) == 0) return sw;
	g_error("sw_lookup failed for %s", name);
	return NULL;
}

static void
toggle_enable_cb(GtkWidget *toggle, GtkWidget *target) {
	gtk_widget_set_sensitive(target, 
			gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(toggle)));
}
static void
toggle_tie(GtkWidget *toggle, GtkWidget *target) {
	gtk_widget_set_sensitive(target, 
			gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(toggle)));
	g_signal_connect(G_OBJECT(toggle), "toggled",
			G_CALLBACK(toggle_enable_cb), target);
}

static GtkWidget*
sw_make(char *name) {
	SettingsWidget *sw = sw_lookup(name);
	GtkWidget *hbox, *label = NULL;
	switch (sw->type) {
		case SW_TOGGLE:
			sw->widget = gtk_check_button_new_with_mnemonic(_(sw->caption));
			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(sw->widget),
					*(gboolean*)sw->conf);
			return sw->widget;
			break;
		case SW_TEXT:
			hbox = gtk_hbox_new(FALSE, 5);
			label = gtk_label_new_with_mnemonic(_(sw->caption));
			gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
			sw->widget = gtk_entry_new();
			gtk_box_pack_start(GTK_BOX(hbox), sw->widget, TRUE, TRUE, 0);
			if (*(char**)sw->conf)
				gtk_entry_set_text(GTK_ENTRY(sw->widget), *(char**)sw->conf);
			gtk_label_set_mnemonic_widget(GTK_LABEL(label), sw->widget);
			return hbox;
		case SW_INTEGER:
			hbox = gtk_hbox_new(FALSE, 5);
			label = gtk_label_new_with_mnemonic(_(sw->caption));
			gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
			sw->widget = gtk_entry_new();
			gtk_entry_set_width_chars(GTK_ENTRY(sw->widget), 4);
			gtk_box_pack_start(GTK_BOX(hbox), sw->widget, TRUE, TRUE, 0);
			{
				char buf[30];
				sprintf(buf, "%d", *(int*)sw->conf);
				gtk_entry_set_text(GTK_ENTRY(sw->widget), buf);
			}
			gtk_label_set_mnemonic_widget(GTK_LABEL(label), sw->widget);
			return hbox;
		case SW_COMBO:
			hbox = gtk_hbox_new(FALSE, 5);
			if (sw->caption) {
				label = gtk_label_new_with_mnemonic(_(sw->caption));
				gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
			}
			sw->widget = gtk_combo_new();
			gtk_box_pack_start(GTK_BOX(hbox), sw->widget, TRUE, TRUE, 0);
			if (*(char**)sw->conf)
				gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(sw->widget)->entry), 
						*(char**)sw->conf);
			if (label)
				gtk_label_set_mnemonic_widget(GTK_LABEL(label), sw->widget);
			return hbox;
		case SW_CUSTOM:
			if (sw->caption) {
				hbox = gtk_hbox_new(FALSE, 5);
				gtk_box_pack_start(GTK_BOX(hbox), 
						gtk_label_new(_(sw->caption)), FALSE, FALSE, 0);
				return hbox;
			}
			break;
		default:
			break;
	}
	return NULL;
}

static void
sw_save(SettingsWidget *sw) {
	switch (sw->type) {
		case SW_TOGGLE:
			*(gboolean*)sw->conf = 
				gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(sw->widget));
			break;
		case SW_TEXT:
			string_replace((char**)sw->conf, 
					gtk_editable_get_chars(GTK_EDITABLE(sw->widget), 0, -1));
			break;
		case SW_INTEGER:
			{
				char *text = 
					gtk_editable_get_chars(GTK_EDITABLE(sw->widget), 0, -1);
				*(int*)sw->conf = atoi(text);
				g_free(text);
			}
			break;
		case SW_COMBO:
			string_replace((char**)sw->conf, 
					gtk_editable_get_chars(
						GTK_EDITABLE(GTK_COMBO(sw->widget)->entry), 0, -1));
			break;
		default:
			if (sw->custom)
				sw->custom(sw);
			else
				g_error("custom cb undefined for %s\n", sw->name);
	}
}

static GtkWidget*
uisettings(LJWin *ljw) {
	SettingsWidget *sw;
	GtkWidget *vbox, *hbox, *button;
	char *fontname = NULL;

	vbox = gtk_vbox_new(FALSE, 5);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 10);

	gtk_box_pack_start(GTK_BOX(vbox), 
			sw_make("ui_revertusejournal"), FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), 
			sw_make("ui_keepmetadata"), FALSE, FALSE, 0);

	sw = sw_lookup("ui_defaultsecurity");
	sw->widget = secmgr_new(FALSE);
	secmgr_security_set(SECMGR(sw->widget), (Security*)sw->conf);
	hbox = sw_make("ui_defaultsecurity");
	gtk_box_pack_start(GTK_BOX(hbox), sw->widget, TRUE, TRUE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	gtk_box_pack_start(GTK_BOX(vbox), 
			sw_make("ui_autosave"), FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), 
			sw_make("ui_noautologin"), FALSE, FALSE, 0);
#ifdef HAVE_GTKSPELL
	gtk_box_pack_start(GTK_BOX(vbox), 
			sw_make("ui_spellcheck"), FALSE, FALSE, 0);
#endif
	
	sw = sw_lookup("ui_font");
	if (conf.uifont == NULL) {
		fontname = pango_font_description_to_string(
				pango_context_get_font_description(
					gtk_widget_get_pango_context(GTK_WIDGET(ljw->eentry))));
	}
	sw->widget = gtk_label_new(conf.uifont ? conf.uifont : (fontname ? fontname : _("[gtk default]")));
	if (fontname)
		g_free(fontname);
	button = gtk_button_new_from_stock(GTK_STOCK_SELECT_FONT);
	sw->data = ljw->eentry;
	hbox = sw_make("ui_font");
	gtk_box_pack_start(GTK_BOX(hbox), sw->widget, TRUE, TRUE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), GTK_WIDGET(button), FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
	g_signal_connect_swapped(G_OBJECT(button), "clicked",
			G_CALLBACK(run_fontsel_settings_dlg), sw);

	return vbox;
}

static void
sec_save_cb(SettingsWidget *sw) {
	secmgr_security_get(SECMGR(sw->widget), (Security*)sw->conf);
}

/* FIXME: hack around limitation of the settings system... */
static GtkWidget*
proxyuserpass(void) {
	GtkWidget *table;
	SettingsWidget *sw;

	table = lj_table_new(2, 2);

	sw = sw_lookup("net_proxyuser");
	sw->widget = gtk_entry_new();
	if (*(char**)sw->conf)
		gtk_entry_set_text(GTK_ENTRY(sw->widget), *(char**)sw->conf);
	lj_table_label_content(GTK_TABLE(table), 0, _(sw->caption), sw->widget);

	sw = sw_lookup("net_proxypass");
	sw->widget = gtk_entry_new();
	gtk_entry_set_visibility(GTK_ENTRY(sw->widget), FALSE);
	if (*(char**)sw->conf)
		gtk_entry_set_text(GTK_ENTRY(sw->widget), *(char**)sw->conf);
	lj_table_label_content(GTK_TABLE(table), 1, _(sw->caption), sw->widget);

	return table;

}
static GtkWidget*
proxysettings(void) {
	GtkWidget *vbox, *wbox, *hbox, *cb, *table;

	vbox = gtk_vbox_new(FALSE, 5);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 10);

	cb = sw_make("net_useproxy");
	gtk_box_pack_start(GTK_BOX(vbox), cb, FALSE, FALSE, 0);

	wbox = gtk_vbox_new(FALSE, 5);
	toggle_tie(cb, wbox);
	gtk_box_pack_start(GTK_BOX(vbox), wbox, FALSE, FALSE, 0);

	hbox = sw_make("net_proxy");
	gtk_box_pack_start(GTK_BOX(wbox), hbox, FALSE, FALSE, 0);

	cb = sw_make("net_useproxyauth");
	gtk_box_pack_start(GTK_BOX(wbox), cb, FALSE, FALSE, 0);
	table = proxyuserpass();
	gtk_box_pack_start(GTK_BOX(wbox), table, FALSE, FALSE, 0);

	toggle_tie(cb, table);

	return vbox;
}

static GtkWidget*
websettings(void) {
	GtkWidget *vbox;
	GtkWidget *combo, *label;
	GList *list = NULL;
	int i;

	vbox = gtk_vbox_new(FALSE, 5);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 10);

	label = gtk_label_new(
			_("This command is evaluated by /bin/sh.\n"
			"Use %s in the command line in the place of the URL.\n"
			"You can use %s more than once."));
	gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);

	for (i = 0; spawn_commands[i] != NULL; i++) {
		list = g_list_append(list, spawn_commands[i]);
	}

	gtk_box_pack_start(GTK_BOX(vbox), sw_make("web_spawn"), FALSE, FALSE, 0);

	combo = sw_lookup("web_spawn")->widget;
	gtk_combo_set_popdown_strings(GTK_COMBO(combo), list);
	if (conf.spawn_command) 
		gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(combo)->entry), 
				conf.spawn_command);

	return vbox;
}

static GtkWidget*
debugsettings(void) {
	GtkWidget *vbox;

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 10);
	gtk_box_pack_start(GTK_BOX(vbox), 
			sw_make("debug_netdump"), FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), 
			sw_make("debug_nofork"), FALSE, FALSE, 0);

	return vbox;
}

static GtkWidget*
cfriendssettings(GtkWidget *parent) {
	GtkWidget *vbox;
	GtkWidget *frame, *framebox;
	GtkWidget *maskhbox;
	SettingsWidget *sw;

	vbox = gtk_vbox_new(FALSE, 5);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 10);
	frame = gtk_frame_new(_("Floating indicator"));
	framebox = gtk_vbox_new(FALSE, 5);
	gtk_container_set_border_width(GTK_CONTAINER(framebox), 5);

	gtk_box_pack_start(GTK_BOX(vbox), 
			sw_make("cf_autostart"), FALSE, FALSE, 0);

	maskhbox = gtk_hbox_new(FALSE, 5);
	gtk_box_pack_start(GTK_BOX(maskhbox), 
			sw_make("cf_usemask"), FALSE, FALSE, 0);

	sw = sw_lookup("cf_mask");
	sw->widget = gtk_button_new_with_label(_("Choose filter"));
	sw->data2 = parent;
	sw->data = GUINT_TO_POINTER(atoi(conf.cfmask));
	gtk_box_pack_start(GTK_BOX(maskhbox), GTK_WIDGET(sw->widget), FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), maskhbox, FALSE, FALSE, 0);
	g_signal_connect_swapped(G_OBJECT(sw->widget), "clicked",
			G_CALLBACK(run_cfmask_settings_dlg), sw);

	gtk_box_pack_start(GTK_BOX(vbox), 
			sw_make("cf_threshold"), FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), 
			sw_make("cf_userinterval"), FALSE, FALSE, 0);

	gtk_box_pack_start(GTK_BOX(framebox), 
			sw_make("cf_autofloat"), FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(framebox), 
			sw_make("cf_autofloatraise"), FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(framebox), 
			sw_make("cf_float_decorate"), FALSE, FALSE, 0);
	gtk_container_add(GTK_CONTAINER(frame), framebox);
	gtk_box_pack_start(GTK_BOX(vbox),
			frame, FALSE, FALSE, 0);

	return vbox;
}

static gboolean
run_settings_dialog(LJWin *ljw, SettingsPage page) {
	GtkWidget *dlg, *nb;
	SettingsWidget *sw;

	dlg = lj_dialog_new(GTK_WIDGET(ljw), _("Preferences"), -1, -1);

	/* the order of notebook pages created here should match the
	 * SettingsPage enum in settings.h */
	nb = gtk_notebook_new();
	gtk_notebook_append_page(GTK_NOTEBOOK(nb), 
			uisettings(ljw), gtk_label_new_with_mnemonic(_("User _Interface")));
	gtk_notebook_append_page(GTK_NOTEBOOK(nb), 
			websettings(), gtk_label_new_with_mnemonic(_("_Web Browser")));
	gtk_notebook_append_page(GTK_NOTEBOOK(nb), 
			proxysettings(), gtk_label_new_with_mnemonic(_("_Proxy")));
	gtk_notebook_append_page(GTK_NOTEBOOK(nb), 
			cfriendssettings(dlg), gtk_label_new_with_mnemonic(_("Check _Friends")));
	gtk_notebook_append_page(GTK_NOTEBOOK(nb), 
			debugsettings(), gtk_label_new_with_mnemonic(_("_Debug")));

	lj_dialog_set_contents(dlg, nb);
	gtk_notebook_set_current_page(GTK_NOTEBOOK(nb), page);
	gtk_widget_show_all(GTK_DIALOG(dlg)->vbox);

	lj_dialog_add_okcancel(dlg, NULL);

	if (lj_dialog_run(dlg)) {
		for (sw = settingswidgets; sw->name; sw++) 
			sw_save(sw);
		gtk_widget_destroy(dlg);
		return TRUE;
	}
	return FALSE;
}

void
settings_run(LJWin *ljw, SettingsPage initialpage) {
#ifdef HAVE_GTKSPELL
	gboolean hadspell            = conf.options.usespellcheck;
#endif
	gboolean hadcffloat_decorate = conf.options.cffloat_decorate;
	gchar   *oldfont             = g_strdup(conf.uifont);
	gboolean hadautosave         = conf.options.autosave;

	if (!run_settings_dialog(ljw, initialpage)) {
		g_free(oldfont);
		return;
	}
	
	g_slist_foreach(app.secmgr_list, (GFunc)secmgr_security_set_force,
			&conf.defaultsecurity);

	g_slist_foreach(app.cfmgr_list, (GFunc)cfmgr_mask_set,
			conf.options.cfusemask ? conf.cfmask : "0");

	if (conf.uifont != oldfont) {
		lj_font_set(ljw->eentry, conf.uifont);
	}
	g_free(oldfont);
	
	if (conf.options.cffloat_decorate != hadcffloat_decorate)
		cf_float_decorate_refresh();

	cf_threshold_normalize(&conf.cfthreshold);

	if (!hadautosave && conf.options.autosave)
		lj_autosave_init(ljw);
	if (hadautosave && !conf.options.autosave)
		lj_autosave_stop(ljw);

#ifdef HAVE_GTKSPELL
	if (conf.options.usespellcheck != hadspell) {
		GtkSpell *spell;
		if (conf.options.usespellcheck) {
			GError *err = NULL;
			if (gtkspell_new_attach(GTK_TEXT_VIEW(ljw->eentry), "en", &err) == NULL) {
				lj_warning(GTK_WIDGET(ljw), _("GtkSpell error: %s"), err->message);
				conf.options.usespellcheck = FALSE;
				g_error_free(err);
			}
		} else {
			spell = gtkspell_get_from_text_view(GTK_TEXT_VIEW(ljw->eentry));
			if (spell)
				gtkspell_detach(spell);
		}
	}
#endif
}

static void
run_fontsel_settings_dlg(SettingsWidget *sw) {
	GtkWidget    *dlg = gtk_font_selection_dialog_new(_("Select font"));

	gtk_font_selection_dialog_set_font_name(GTK_FONT_SELECTION_DIALOG(dlg),
			gtk_label_get_text(GTK_LABEL(sw->widget))); 
	
	if (gtk_dialog_run(GTK_DIALOG(dlg)) == GTK_RESPONSE_OK) {
		gtk_label_set_text(GTK_LABEL(sw->widget), 
				gtk_font_selection_dialog_get_font_name(
						GTK_FONT_SELECTION_DIALOG(dlg)));
	}
	
	gtk_widget_destroy(dlg); 
}

static void
font_save_cb(SettingsWidget *sw) {
	const gchar *newfont;
	gchar *oldfont;

	newfont = gtk_label_get_text(GTK_LABEL(sw->widget));
	oldfont = pango_font_description_to_string(
			pango_context_get_font_description(
				gtk_widget_get_pango_context(GTK_WIDGET(sw->data))));

	if (newfont && g_ascii_strcasecmp(oldfont, newfont) != 0)
		string_replace(sw->conf, newfont ? g_strdup(newfont) : newfont);
	g_free(oldfont);
}

static void
run_cfmask_settings_dlg(SettingsWidget *sw) {
	sw->data = GINT_TO_POINTER(custom_security_dlg_run(          /* ick!  */
			GTK_WIDGET(sw->data2), GPOINTER_TO_UINT(sw->data))); /* ick!! */
}

static void
cfmask_save_cb(SettingsWidget *sw) {
	string_replace(&conf.cfmask,
			g_strdup_printf("%u", GPOINTER_TO_UINT(sw->data)));
}

