/* logjam - a GTK client for LiveJournal.
 * Copyright (C) 2000-2002 Evan Martin <evan@livejournal.com>
 *
 * vim: tabstop=4 shiftwidth=4 noexpandtab :
 * $Id: protocol.c,v 1.11 2002/10/03 01:11:43 martine Exp $
 */

#include <glib.h>

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <ctype.h>

#include "util.h"
#include "protocol.h"

NetResult* 
lj_protocol_parse_response(const char *res) {
	char *key;
	char *val;
	GHashTable *hash = lj_protocol_hash_new();
	int read_keys = 0;

	while (*res != 0) {
		key = util_getline(res);
		res = util_skipline(res);
		if (*res) {
			val = util_getline(res);
			res = util_skipline(res);
			g_hash_table_insert(hash, key, val);
			read_keys++;
		}
	}
	if (read_keys == 0) { /* error. */
		lj_protocol_hash_free(hash);
		return NULL;
	}
	return (NetResult*)hash;
}

int 
lj_protocol_request_succeeded(NetResult *result) {
	char *error;

	if (result == NULL) return FALSE;

	error = net_result_get(result, "success");
	if (error == NULL) return FALSE;

	return (g_ascii_strcasecmp(error, "OK") == 0);
}

NetRequest*
lj_protocol_request_new(const char *mode, const char *username, const char *password, const char *usejournal) {
	unsigned char buf[33];

	NetRequest *request = (NetRequest*)lj_protocol_hash_new();

	net_request_copys(request, "mode", mode);
	net_request_copys(request, "user", username);
	if (usejournal)
		net_request_copys(request, "usejournal", usejournal);
	net_request_seti(request, "ver", 1);

	md5_hash(password, buf);
	net_request_copys(request, "hpassword", buf);

	return request;
}

GHashTable*
lj_protocol_hash_new(void) {
	return g_hash_table_new(g_str_hash, g_str_equal);
}

static void
hash_append_str_encoded(gpointer key, gpointer value, gpointer data) {
	GString *string = data;
	gchar *en_key, *en_value;

	if (key == NULL) return;
	en_key = urlencode(key);
	en_value = urlencode(value);
	g_string_append_printf(string, "%s=%s&", en_key, en_value);

	g_free(en_key);
	g_free(en_value);
}

GString*
lj_protocol_request_to_string(GHashTable* request) {
	GString *str = g_string_sized_new(2048);
	g_hash_table_foreach(request, hash_append_str_encoded, str);
	return str;
}

static gint free_all_hash_cb(gpointer key, gpointer val, gpointer user_data) {
	g_free(key); g_free(val);
	return TRUE;
}

void
lj_protocol_hash_free(GHashTable *hash) {
	g_hash_table_foreach_remove(hash, free_all_hash_cb, NULL);
	g_hash_table_destroy(hash);
}

