/* logjam - a GTK client for LiveJournal.
 * Copyright (C) 2000-2002 Evan Martin <evan@livejournal.com>
 *
 * vim: tabstop=4 shiftwidth=4 noexpandtab :
 * $Id: menu.c,v 1.35 2002/11/26 05:41:29 martine Exp $
 */

#include "config.h"

#include <gtk/gtk.h>
#include <glib.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "conf.h"
#include "network.h"
#include "menu.h"
#include "lj.h"
#include "friends.h"
#include "history.h"
#include "util.h"
#include "spawn.h"
#include "security.h"
#include "icons.h"
#include "meta.h"
#include "settings.h"
#include "pollcreator.h"
#include "tools.h"

void about_dlg(GtkWidget *mainwin);
void link_dialog_run(LJWin *ljw);
void console_dialog_run(GtkWidget *parent);

static void
new_cb(LJWin *ljw) {
	if (!lj_confirm_lose_entry(ljw)) return;
	lj_clear_entry(ljw);
}
static void
load_last_entry(LJWin *ljw) { 
	Entry *entry;

	if (!lj_confirm_lose_entry(ljw)) return;

	if ((entry = history_load_latest(GTK_WIDGET(ljw))) == NULL)
		return;

	lj_load_entry(ljw, entry);
	entry_free(entry);
}
static void
run_history_recent_dlg(LJWin *ljw) { 
	Entry *entry;

	if (!lj_confirm_lose_entry(ljw)) return;

	if ((entry = history_recent_dialog_run(GTK_WIDGET(ljw))) == NULL)
		return;

	lj_load_entry(ljw, entry);
	entry_free(entry);
}

static void
loginout_cb(LJWin *ljw) {
	lj_do_loginout(ljw, FALSE);
}

static GtkWidget*
menu_item_from_action(GtkItemFactory *item_factory, int action) {
	GtkWidget *menu, *item;
	menu = gtk_item_factory_get_widget_by_action(item_factory, action);
	item = gtk_menu_get_attach_widget(GTK_MENU(menu));
	return item;
}

enum {
	ACTION_NONE=0,
	ACTION_JOURNAL,
	ACTION_WEB_LINKS,
	ACTION_ENTRY_SUBMIT_SEP,
	ACTION_ENTRY_SUBMIT_NEW,
	ACTION_ENTRY_SAVE_SERVER,

	ACTION_JOURNAL_POSTAS,
	ACTION_JOURNAL_POSTAS_SEP,
	ACTION_LOGIN,

	ACTION_TOOLS_CONSOLE
};

GtkWidget* 
menu_make_bar(LJWin *ljw) {
	GtkWidget *bar;
	GtkAccelGroup *accelgroup = NULL;
	GtkItemFactory *item_factory = NULL;

/* a note on accelerators:
 *  shift-ctl-[x]  is used to type in UTF-8 chars; 
 *  this means we can't use any shift-ctl accels. */
static GtkItemFactoryEntry menu_items[] = {
	/* path accel cb cb_action type extra */
	{ N_("/_LogJam"),                 NULL,     NULL, 0, "<Branch>" },
	{ N_("/LogJam/_About LogJam..."), NULL,     about_dlg, 0, "<StockItem>", "logjam-goat" },
	{ N_("/LogJam/sep0"),             NULL,     NULL, 0, "<Separator>" },
	{ N_("/LogJam/_Preferences..."),  NULL,     settings_run, SETTINGS_PAGE_UNSPECIFIED, "<StockItem>", GTK_STOCK_PREFERENCES },
	{ N_("/LogJam/sep2"),             NULL,     NULL, 0, "<Separator>" },
	{ N_("/LogJam/_Login"),           NULL,     loginout_cb, ACTION_LOGIN, NULL },
	{ N_("/LogJam/_Quit"),            "<ctl>Q", lj_quit, 0, "<StockItem>", GTK_STOCK_QUIT },
	
	{ N_("/_Entry"),                        NULL, NULL, 0, "<Branch>" },
	{ N_("/Entry/_New"),                    NULL, new_cb, 0, "<StockItem>", GTK_STOCK_NEW },
	{ N_("/Entry/_Open..."),                NULL, lj_open_entry, 0, "<StockItem>", GTK_STOCK_OPEN },
	{ N_("/Entry/Save _As..."),             NULL, lj_save_entry, 0, "<StockItem>", GTK_STOCK_SAVE_AS },
	{ N_("/Entry/Submit sep"),              NULL, NULL, ACTION_ENTRY_SUBMIT_SEP, "<Separator>" },
	{ N_("/Entry/Submit as New _Entry"),    NULL, lj_submit_entry, ACTION_ENTRY_SUBMIT_NEW, NULL },
	{ N_("/Entry/_Save Changes to Server"), NULL, lj_save_entry_server, ACTION_ENTRY_SAVE_SERVER, NULL },

	{ N_("/_Journal"),                      NULL,          NULL, ACTION_JOURNAL, "<Branch>" },
	{ N_("/Journal/_Friends..."),           "<ctl><alt>F", friends_manager_show, 0, NULL },
	{ N_("/Journal/_Load Last Entry"),      NULL,          load_last_entry, 0, NULL },
	{ N_("/Journal/_Load Recent Entry..."), NULL,          run_history_recent_dlg, 0, NULL },
	{ N_("/Journal/Post As sep"),           NULL,          NULL, ACTION_JOURNAL_POSTAS_SEP, "<Separator>" },
	{ N_("/Journal/_Post As"),              NULL,         NULL, ACTION_JOURNAL_POSTAS, "<Branch>" },
	
	{ N_("/_Tools"),                        NULL,          NULL, 0, "<Branch>" },
	{ N_("/Tools/_Insert File..."),         NULL,          tools_insert_file, 0, NULL },
	{ N_("/Tools/_Make Link..."),           "<ctl><alt>L", link_dialog_run, 0, NULL },
	{ N_("/Tools/_HTML Escape"),            "<ctl><alt>H", tools_html_escape_selected, 0, NULL },
	{ N_("/Tools/_Remove Linebreaks"),      "<ctl><alt>R", tools_remove_linebreak_selected, 0, NULL },
	{ N_("/Tools/_LiveJournal Console..."), NULL,          console_dialog_run, ACTION_TOOLS_CONSOLE, NULL },
	{ N_("/Tools/_Poll Creator"),           "<ctl><alt>P", run_poll_creator_dlg, 0, NULL },
	
	{ N_("/_Web Links"),                    NULL,         NULL, ACTION_WEB_LINKS, "<Branch>" },
};
	int itemcount = sizeof(menu_items) / sizeof(menu_items[0]);

	accelgroup = gtk_accel_group_new();
	item_factory = gtk_item_factory_new(GTK_TYPE_MENU_BAR,
			"<main>", accelgroup);
	gtk_item_factory_create_items(item_factory, itemcount, menu_items, ljw);
	gtk_window_add_accel_group(GTK_WINDOW(ljw), accelgroup);

	bar = gtk_item_factory_get_widget(item_factory, "<main>");

	ljw->mjournal = menu_item_from_action(item_factory, ACTION_JOURNAL);
	gtk_widget_hide(ljw->mjournal);
	ljw->mweb = menu_item_from_action(item_factory, ACTION_WEB_LINKS);
	gtk_widget_hide(ljw->mweb);
	
	ljw->msubmitsep = gtk_item_factory_get_widget_by_action(item_factory,
			ACTION_ENTRY_SUBMIT_SEP);
	ljw->msubmit = gtk_item_factory_get_widget_by_action(item_factory,
			ACTION_ENTRY_SUBMIT_NEW);
	ljw->msaveserver = gtk_item_factory_get_widget_by_action(item_factory,
			ACTION_ENTRY_SAVE_SERVER);
	gtk_widget_hide(ljw->msaveserver);

	ljw->musejournal = menu_item_from_action(item_factory,
			ACTION_JOURNAL_POSTAS);
	ljw->musejournalsep = gtk_item_factory_get_widget_by_action(item_factory,
			ACTION_JOURNAL_POSTAS_SEP);
	ljw->mloginout = gtk_item_factory_get_widget_by_action(item_factory,
			ACTION_LOGIN);

	ljw->mconsole = gtk_item_factory_get_widget_by_action(item_factory,
			ACTION_TOOLS_CONSOLE);
	gtk_widget_hide(ljw->mconsole);

	return bar;
}

static void
usejournal_activate(GtkWidget *w, LJWin *ljw) {
	const gchar *user;

	user = gtk_label_get_text(GTK_LABEL(GTK_BIN(w)->child));

	if (g_ascii_strcasecmp(user, conf_cur_user()->username) == 0) {
		string_replace(&conf.usejournal, NULL); /* post as normal user. */
	} else {
		string_replace(&conf.usejournal, g_strdup(user));
	}
	lj_update_usejournal(ljw);
}

/* "having a shared prefix" is a pretty vague concept.
 * we try a few heuristics. */

static int
sharedprefix(char *a, char *b) {
	const int minmatch = 4;
	int i = 0;
	char *ulpos;

	/* does this journal have an underscore? go from there. */
	ulpos = strchr(b, '_');
	if (ulpos) {
		if (strncmp(a, b, ulpos-b) == 0)
			return ulpos - b;
	}

	/* or if the shared prefix is at least minmatch chars... */
	while (*a++ == *b++)
		i++;
	return (i > minmatch) ? i : 0;
}

static GtkWidget*
build_usejournal_menu(LJWin *ljw) {
	GtkWidget *menu, *curmenu, *item, *label;
	GSList *group = NULL;
	GList *l;
	char *journal;
	char *curmenuprefix = NULL;
	char prefix[30];

	curmenu = menu = gtk_menu_new(); 

	item = gtk_radio_menu_item_new_with_label(group, conf_cur_user()->username);
	ljw->musejournal_defaultjournal = item;
	group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(item));
	g_signal_connect(G_OBJECT(item), "activate",
			G_CALLBACK(usejournal_activate), ljw);

	/* if they haven't specified a usejournal, default to their username. */
	if (conf.usejournal == NULL) 
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), TRUE);

	gtk_widget_show(item);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);

	for (l = conf_cur_user()->usejournals; l != NULL; l = l->next) {
		journal = (char*)l->data;

		if (curmenuprefix) {
			/* try to match this item to the prefix. */
			if (sharedprefix(curmenuprefix, journal)) {
				/* match. */
			} else {
				curmenu = menu;
				curmenuprefix = NULL;
			}
		} 
		if (!curmenuprefix && l->next) {
			/* try to see if this begins a new prefix. */
			char *nextjournal = (char*)l->next->data;
			int ofs;
			ofs = sharedprefix(journal, nextjournal);
			if (ofs) {
				/* make a new submenu for these shared-prefix journals. */
				char *markup;
				memcpy(prefix, journal, ofs);
				prefix[ofs] = 0;
				markup = g_strdup_printf("<i>%s</i>", prefix);

				item = gtk_menu_item_new();
				label = gtk_label_new(NULL);
				gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
				gtk_label_set_markup(GTK_LABEL(label), markup);
				gtk_container_add(GTK_CONTAINER(item), label);
				gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
				curmenu = gtk_menu_new();
				curmenuprefix = prefix;
				gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), curmenu);
				gtk_widget_show_all(item);
			}
		}
		item = gtk_radio_menu_item_new_with_label(group, journal);
		if (conf.usejournal && strcmp(conf.usejournal, journal) == 0)
			gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), TRUE);
		g_signal_connect(G_OBJECT(item), "activate",
				G_CALLBACK(usejournal_activate), ljw);
		gtk_widget_show(item);
		gtk_menu_shell_append(GTK_MENU_SHELL(curmenu), item);

		group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(item));
	}

	return menu;
}

static void 
menu_update_usejournal(LJWin *ljw, NetResult *result) {
	if (!(conf_cur_user() && conf_cur_user()->usejournals)) {
		gtk_widget_hide(ljw->musejournal);
		gtk_widget_hide(ljw->musejournalsep);
		ljw->musejournal_defaultjournal = NULL;
		return;
	} 

	gtk_menu_item_remove_submenu(GTK_MENU_ITEM(ljw->musejournal));
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(ljw->musejournal), build_usejournal_menu(ljw));

	gtk_widget_show(ljw->musejournal);
	gtk_widget_show(ljw->musejournalsep);
}

static void
menu_spawn_url(GtkWidget *w, GtkWidget *ljw) {
	char *url;
	url = g_object_get_data(G_OBJECT(w), "url");
	if (url)
		spawn_url(ljw, url);
}

static GtkWidget*
web_recurse(LJWin *ljw, NetResult *hash, int base) {
	GtkWidget *menu, *item;
	char *text, *str, *url, *sub, *dupstr;
	int i, count;

	str = net_result_getf(hash, "menu_%d_count", base);
	if (str == NULL) {
		g_warning("menu with no elements?");
		return NULL;
	}
	count = atoi(str);

	menu = gtk_menu_new();

	for (i = 1; i < count+1; i++) {
		text = net_result_getf(hash, "menu_%d_%d_text", base, i);
		if (text == NULL) {
			g_warning("menu item has no text?");
			continue;
		}

		url = net_result_getf(hash, "menu_%d_%d_url", base, i);
		sub = net_result_getf(hash, "menu_%d_%d_sub", base, i);

		if (text[0] == '-' && text[1] == 0) {
			/* separator */
			item = gtk_menu_item_new();
			gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
		} else if (url != NULL) {
			/* url menu item */
			item = gtk_image_menu_item_new_with_label(text);
			gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(item),
					gtk_image_new_from_stock(GTK_STOCK_JUMP_TO, GTK_ICON_SIZE_MENU));

			dupstr = g_strdup(url);
			/* free the dup'd string after the menu item is destroyed! */
			g_object_set_data_full(G_OBJECT(item), "url",
					dupstr, g_free);
			g_signal_connect(G_OBJECT(item), "activate",
					G_CALLBACK(menu_spawn_url), ljw);
			gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
		} else if (sub != NULL) {
			/* submenu ! */
			GtkWidget *submenu;
			int subbase;

			subbase = atoi(sub);
			item = gtk_menu_item_new_with_mnemonic(text);
			submenu = web_recurse(ljw, hash, subbase);
			gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), submenu);
			gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
		} else {
			g_warning("unknown menu item type...");
		}
	}
	return menu;
}

static void 
menu_update_web(LJWin *ljw, NetResult *hash) {
	gtk_menu_item_remove_submenu(GTK_MENU_ITEM(ljw->mweb));
	if (hash) {
		gtk_menu_item_set_submenu(GTK_MENU_ITEM(ljw->mweb),
				web_recurse(ljw, hash, 0));
		gtk_widget_show_all(ljw->mweb);
	} else {
		lj_widget_set_visible(ljw->mweb, FALSE);
	}
}

void
menu_update(LJWin *ljw, NetResult *result) {
	menu_update_usejournal(ljw, result);
	menu_update_web(ljw, result);
	gtk_label_set_text(GTK_LABEL(GTK_BIN(ljw->mloginout)->child), conf.loginok ? _("Logout") : _("Login"));
	gtk_label_set_pattern(GTK_LABEL(GTK_BIN(ljw->mloginout)->child), "_");
	lj_widget_set_visible(ljw->mjournal, conf.loginok);
	lj_widget_set_visible(ljw->mweb, conf.loginok);
	lj_widget_set_visible(ljw->msubmitsep, conf.loginok);
	lj_widget_set_visible(ljw->msubmit, conf.loginok);
	lj_widget_set_visible(ljw->msaveserver, (conf.loginok && ljw->itemid > 0));
	lj_widget_set_visible(ljw->mconsole, conf.loginok);
}

