/* logjam - a GTK client for LiveJournal.
 * Copyright (C) 2000-2002 Evan Martin <evan@livejournal.com>
 *
 * vim: tabstop=4 shiftwidth=4 noexpandtab :
 * $Id: lj.c,v 1.35 2002/12/04 04:50:59 martine Exp $
 */

#include <config.h>
#include <gtk/gtk.h>
#ifdef HAVE_GTKSPELL
#include <gtkspell/gtkspell.h>
#endif /* HAVE_GTKSPELL */

#include <stdio.h>
#include <time.h>
#include <stdlib.h>
#ifndef G_OS_WIN32
#include <unistd.h> /* unlink */
#endif /* G_OS_WIN32 */

#include "network.h"
#include "conf.h"
#include "util.h"
#include "ljtypes.h"
#include "login.h"
#include "lj.h"
#include "menu.h"
#include "security.h"
#include "icons.h"
#include "meta.h"
#include "checkfriends.h"

static void lj_draft_delete();
static const int LJ_AUTOSAVE_INTERVAL = 5*1000;

/* set the font (by name) of a widget. used on ljw->eentry by settings. */
void
lj_font_set(GtkWidget *w, gchar *font_name) {
	PangoFontDescription *font_desc;
	
	font_desc = pango_font_description_from_string(font_name);
	gtk_widget_modify_font(w, font_desc);
	pango_font_description_free(font_desc);
}

void 
lj_update_usejournal(LJWin *ljw) {
	char *title;
	if (conf.loginok) {
		char *status = NULL;
		User *u = conf_cur_user();
		char *fullname = u->fullname;
		if (fullname == NULL)
			fullname = "(?)";

		if (conf.usejournal) {
			title = g_strdup_printf(_("%s's LiveJournal (using %s)"), fullname, conf.usejournal);
			status = g_strdup_printf(_("Using '%s':"), conf.usejournal);
		} else {
			title = g_strdup_printf(_("%s's LiveJournal"), fullname);
		}
		gtk_label_set_text(GTK_LABEL(ljw->lstatus), status ? status : "");
		g_free(status);
	} else {
		title = g_strdup_printf("LogJam %s", VERSION);
	}
	gtk_window_set_title(GTK_WINDOW(ljw), title);
	g_free(title);
}

gboolean
lj_confirm_lose_entry(LJWin *ljw) {
	if (!ljw->isdirty)
		return TRUE;

	return lj_confirm(GTK_WIDGET(ljw), _("Entry Changed"), _("Lose changes to current entry?"));
}

void
lj_clear_entry(LJWin *ljw) {
	Entry *entry;
	
	entry = entry_new();
	lj_load_entry(ljw, entry);
	entry_free(entry);
}

static void
lj_hideshow_buttons(LJWin *ljw) {
	if (conf.loginok) {
		gtk_widget_show(ljw->baction);
		gtk_widget_show(ljw->lstatus);
		if (ljw->itemid)
			gtk_widget_show(ljw->bdelete);
		else
			gtk_widget_hide(ljw->bdelete);
	} else {
		gtk_widget_hide(ljw->baction);
		gtk_widget_hide(ljw->bdelete);
		gtk_widget_hide(ljw->lstatus);
	}
}

Entry*
lj_read_entry(LJWin *ljw) {
	Entry *entry;
	GtkTextBuffer *buffer;
	GtkTextIter start, end;

	entry = entry_new();

	entry->itemid = ljw->itemid;

	buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(ljw->eentry));
	gtk_text_buffer_get_bounds(buffer, &start, &end);

	entry->event = gtk_text_buffer_get_text(buffer, &start, &end, TRUE);

	metamgr_to_entry(METAMGR(ljw->metamgr), entry);

	return entry;
}

void
lj_load_entry(LJWin *ljw, Entry *entry) {
	GtkTextIter start, end;
	GtkTextBuffer *buffer;

	ljw->itemid = entry->itemid;
	if (ljw->itemid > 0) {
		gtk_label_set_text(GTK_LABEL(ljw->laction), _("  Save Changes  "));
		if (conf.loginok)
			gtk_widget_show(ljw->msaveserver);
	} else {
		gtk_label_set_text(GTK_LABEL(ljw->laction), _("  Submit  "));
		gtk_widget_hide(ljw->msaveserver);
	}

	buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(ljw->eentry));
	gtk_text_buffer_get_bounds(buffer, &start, &end);
	gtk_text_buffer_delete(buffer, &start, &end);
	if (entry->event)
		gtk_text_buffer_insert(buffer, &start, entry->event, -1);

	metamgr_from_entry(METAMGR(ljw->metamgr), entry);
	ljw->isdirty = FALSE;

	lj_hideshow_buttons(ljw);
}

void
lj_insert_file(LJWin *ljw, const char *filename) {
	GtkTextBuffer *buffer;
	char *text;

	if (!g_file_get_contents(filename, &text, NULL, NULL))
		return;
	buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(ljw->eentry));
	gtk_text_buffer_insert_at_cursor(buffer, text, -1);
	g_free(text);
}

static gboolean
do_action(LJWin *ljw, const char *type, const char *action, gboolean use_itemid, gboolean delete) {
	NetRequest *request;
	NetResult *result;
	Entry *entry;
	gboolean success;

	request = net_request_new(type);

	entry = lj_read_entry(ljw);
	if (!use_itemid)
		entry->itemid = 0;
	if (delete) {
		g_free(entry->event);
		entry->event = g_strdup("");
	}
	entry_set_request_fields(entry, request);
	entry_free(entry);

	result = net_request_run(GTK_WIDGET(ljw), action, request);
	net_request_free(request);

	success = net_result_succeeded(result);
	net_result_free(result);
	return success;
}
		
void
lj_save_entry_server(LJWin *ljw) {
	if (do_action(ljw, "editevent", _("Saving changes..."), TRUE, FALSE)) {
		lj_clear_entry(ljw);
		lj_draft_delete();
	}
}

void
lj_submit_entry(LJWin *ljw) {
	if (do_action(ljw, "postevent", _("Submitting entry..."), FALSE, FALSE)) {
		lj_clear_entry(ljw);
		if (conf.options.revertusejournal) {
			if (ljw->musejournal_defaultjournal) {
				gtk_check_menu_item_set_active(
						GTK_CHECK_MENU_ITEM(ljw->musejournal_defaultjournal),
						TRUE);
			}
			lj_update_usejournal(ljw);
		}
		lj_draft_delete();
	}
}

static void 
action_cb(GtkWidget *w, LJWin *ljw) {
	/* note that a negative itemid will eventually be a draft.
	 * should the default option on a draft be save changes or submit? */
	if (ljw->itemid > 0) {
		lj_save_entry_server(ljw);
	} else {
		lj_submit_entry(ljw);
	}
}

static void 
delete_cb(GtkWidget *w, LJWin *ljw) {
	if (!lj_confirm(GTK_WIDGET(ljw), _("Delete"), _("Delete this entry from server?")))
		return;

	if (do_action(ljw, "editevent", _("Deleting entry..."), TRUE, TRUE))
		lj_clear_entry(ljw);
}

static void
lj_draft_delete(void) {
	char path[1024];
	conf_make_path("draft", path, 1024);
	if (unlink(path) < 0) {
		/* FIXME handle error. */
	}
}

static void 
do_logout(LJWin *ljw) {
	string_replace(&conf.usejournal, NULL);
	lj_update_usejournal(ljw);
	cfmgr_set_state(ljw->cfmgr, CF_DISABLED);

	conf.loginok = FALSE;
}

void 
lj_do_loginout(LJWin *ljw, int autolog) {
	if (conf.loginok) {
		do_logout(ljw);
		menu_update(ljw, NULL);
		lj_update_usejournal(ljw);
	}
	login_dlg_run(ljw, autolog);

	lj_update_usejournal(ljw);
	if (conf.loginok) {
		gtk_widget_show_all(cfindicator_box_get(ljw->cfi_main));

		if (conf.options.cfautostart)
			cfmgr_set_state(ljw->cfmgr, CF_ON);
	} else {
		gtk_widget_hide_all(cfindicator_box_get(ljw->cfi_main));
	}
	lj_hideshow_buttons(ljw);
	metamgr_login_changed(METAMGR(ljw->metamgr));
	gtk_widget_grab_focus(ljw->eentry);
}

void 
lj_save_entry(LJWin *ljw) {
	GtkWidget *filesel;
	
	filesel = gtk_file_selection_new(_("Save Entry"));
	/* this should really be a systemwide setting, but the gtk file selector
	 * sucks. */
	gtk_file_selection_hide_fileop_buttons(GTK_FILE_SELECTION(filesel));
	lj_dialog_init(filesel, GTK_WIDGET(ljw), NULL, -1, -1);

	if (gtk_dialog_run(GTK_DIALOG(filesel)) == GTK_RESPONSE_OK) {
		const gchar *filename = NULL;
		Entry *entry;

		filename = gtk_file_selection_get_filename(
				GTK_FILE_SELECTION(filesel));
		entry = lj_read_entry(ljw);
		if (entry_to_xml_file(entry, filename) < 0) {
			/* FIXME handle error. */
		} else {
			ljw->isdirty = FALSE;
		}
		entry_free(entry);
	}
	gtk_widget_destroy(filesel);
}

void
lj_save_draft(LJWin *ljw) {
	Entry *entry;
	char path[1024];
	
	entry = lj_read_entry(ljw);
	conf_make_path("draft", path, 1024);
	if (entry_to_xml_file(entry, path) < 0) {
		/* FIXME handle error. */
	}
	entry_free(entry);
}
	
gboolean
lj_save_draft_cb(LJWin *ljw) {
	if (!ljw->isdirty)
		return TRUE;

	lj_save_draft(ljw);

	return TRUE; /* perpetuate timeout */
}

void 
lj_open_entry(LJWin *ljw) {
	GtkWidget *filesel;
	GtkWidget *hbox, *filetype;
	GtkWidget *menu, *item;
	static struct {
		char *label;
		EntryFileType type;
	} filetypes[] = {
		{ N_("Autodetect"),   ENTRY_FILE_AUTODETECT },
		{ N_("XML"),          ENTRY_FILE_XML        },
		{ N_("RFC822-Style"), ENTRY_FILE_RFC822     },
		{ N_("Plain Text"),   ENTRY_FILE_PLAIN      },
		{ NULL                                  },
	}, *ft;

	if (!lj_confirm_lose_entry(ljw)) return;
	
	filesel = gtk_file_selection_new(_("Open Entry"));
	/* this should really be a systemwide setting, but the gtk file selector
	 * sucks. */
	gtk_file_selection_hide_fileop_buttons(GTK_FILE_SELECTION(filesel));

	filetype = gtk_option_menu_new();
	menu = gtk_menu_new();
	for (ft = filetypes; ft->label; ft++) {
		item = gtk_menu_item_new_with_label(ft->label);
		gtk_widget_show(item);
		gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
	}
	gtk_option_menu_set_menu(GTK_OPTION_MENU(filetype), menu);

	hbox = gtk_hbox_new(FALSE, 5);
	gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new(_("File Type:")),
			FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), filetype, TRUE, TRUE, 0);
	gtk_widget_show_all(hbox);

	gtk_box_pack_start(GTK_BOX(GTK_FILE_SELECTION(filesel)->main_vbox),
			hbox, FALSE, FALSE, 0);
	
	lj_dialog_init(filesel, GTK_WIDGET(ljw), NULL, -1, -1);

	while (gtk_dialog_run(GTK_DIALOG(filesel)) == GTK_RESPONSE_OK) {
		const gchar *filename;
		Entry *entry;
		GError *err = NULL;
		int op;

		filename = gtk_file_selection_get_filename(
				GTK_FILE_SELECTION(filesel));
		op = gtk_option_menu_get_history(GTK_OPTION_MENU(filetype));

		entry = entry_new_from_filename(filename, filetypes[op].type, &err);
		if (entry != NULL) {
			lj_load_entry(ljw, entry);
			entry_free(entry);
			break;
		} else {
			lj_warning(filesel, err->message);
			g_error_free(err);
		}
	}
	gtk_widget_destroy(filesel);
}

static void
buffer_changed(GtkWidget *w, LJWin *ljw) {
	ljw->isdirty = TRUE;
}

static GtkWidget*
make_main_entry(LJWin *ljw) {
	ljw->eentry = gtk_text_view_new();
	gtk_text_view_set_editable(GTK_TEXT_VIEW(ljw->eentry), TRUE);
	gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(ljw->eentry), GTK_WRAP_WORD);
	g_signal_connect(G_OBJECT(
				gtk_text_view_get_buffer(GTK_TEXT_VIEW(ljw->eentry))),
			"modified-changed",
			G_CALLBACK(buffer_changed), ljw);
#ifdef HAVE_GTKSPELL
	if (conf.options.usespellcheck) {
		GError *err = NULL;
		if (gtkspell_new_attach(GTK_TEXT_VIEW(ljw->eentry), "en", &err) == NULL) {
			lj_warning(GTK_WIDGET(ljw), _("GtkSpell error: %s"), err->message);
			g_error_free(err);
		}
	}
#endif /* HAVE_GTKSPELL */

	/* fixme gtk2; this would submit the entry when you hit ctl-enter
	g_signal_connect(G_OBJECT(ljw->eentry), "activate",
		G_CALLBACK(submit_entry), ljw);*/
	if (conf.uifont)
		lj_font_set(ljw->eentry, conf.uifont);

	return scroll_wrap(ljw->eentry);
}

static GtkWidget*
make_action_bar(LJWin *ljw) {
	GtkWidget *box;
	
	box = gtk_hbox_new(FALSE, 5); 

	ljw->cfi_main = cfindicator_new(ljw->cfmgr, CF_MAIN, GTK_WIDGET(ljw));
	cfmgr_set_state(ljw->cfmgr, CF_DISABLED);
	gtk_box_pack_start(GTK_BOX(box), cfindicator_box_get(ljw->cfi_main),
			FALSE, FALSE, 0);

	ljw->lstatus = gtk_label_new(NULL);
	gtk_misc_set_alignment(GTK_MISC(ljw->lstatus), 1.0, 0.5);
	gtk_box_pack_start(GTK_BOX(box), ljw->lstatus, TRUE, TRUE, 10);

	ljw->bdelete = gtk_button_new_from_stock(GTK_STOCK_DELETE);
	g_signal_connect(G_OBJECT(ljw->bdelete), "clicked",
			G_CALLBACK(delete_cb), ljw);
	gtk_box_pack_start(GTK_BOX(box), ljw->bdelete, FALSE, FALSE, 0);

	ljw->baction = gtk_button_new();
	ljw->laction = gtk_label_new(NULL);
	g_signal_connect(G_OBJECT(ljw->baction), "clicked",
			G_CALLBACK(action_cb), ljw);
	gtk_container_add(GTK_CONTAINER(ljw->baction), ljw->laction);
	gtk_box_pack_start(GTK_BOX(box), ljw->baction, FALSE, FALSE, 0);

	return box;
}

static GtkWidget*
make_app_contents(LJWin *ljw) {
	GtkWidget *vbox;

	vbox = gtk_vbox_new(FALSE, 5);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 5);

	/* the cfmgr will very soon keep track of the
	 * default cf indicator in the submit bar. */
	ljw->cfmgr = cfmgr_new();
	cfmgr_parent_set(ljw->cfmgr, GTK_WIDGET(ljw));
	/* (see comment in checkfriends.c about this circular reference.) */

	ljw->metamgr = metamgr_new();
	gtk_box_pack_start(GTK_BOX(vbox), ljw->metamgr, FALSE, FALSE, 0);

	gtk_box_pack_start(GTK_BOX(vbox), 
			make_main_entry(ljw), TRUE, TRUE, 0);
	gtk_box_pack_end(GTK_BOX(vbox), 
			make_action_bar(ljw), FALSE, FALSE, 0);
	
	return vbox;
}

void
lj_autosave_init(LJWin* ljw) {
	if (app.autosave || !conf.options.autosave || !app.autosave_ok)
		return;

	app.autosave = g_timeout_add(LJ_AUTOSAVE_INTERVAL,
			(GSourceFunc)lj_save_draft_cb, ljw);
}

void
lj_autosave_stop(LJWin* ljw) {
	if (!app.autosave)
		return;
	lj_draft_delete();
	g_source_remove(app.autosave);
	app.autosave = 0;
}

/* load a saved draft if there is one and it's appropriate to do so */
static Entry*
lj_load_draft(LJWin* ljw) {
	if (conf.options.autosave && app.autoloaddraft_ok) {
		GError *err = NULL;
		char path[1024];
		Entry *entry;

		conf_make_path("draft", path, 1024);

		if (!g_file_test(path, G_FILE_TEST_EXISTS))
			return FALSE;

		entry = entry_new_from_filename(path, ENTRY_FILE_XML, &err);
		if (entry == NULL) {
			lj_warning(GTK_WIDGET(ljw), _("Error loading draft: %s."),
					err->message);
			g_error_free(err);
		}
		return entry;
	}
	return NULL; /* didn't read draft */
}

void
lj_quit(LJWin *ljw) {
	if (lj_confirm_lose_entry(ljw))
		gtk_main_quit();
}

static gboolean
delete_event_cb(LJWin *ljw) {
	lj_quit(ljw);
	return TRUE; /* don't ever let this delete the window; quit will do it. */
}

/* gtk stuff */
static GType
ljwin_get_type(void) {
	static GType ljw_type = 0;
	if (!ljw_type) {
		const GTypeInfo ljw_info = {
			sizeof (GtkWindowClass),
			NULL,
			NULL,
			NULL,
			NULL,
			NULL,
			sizeof (LJWin),
			0,
			//(GInstanceInitFunc) ljwin_init, /* no init func needed since */
			NULL,     /* GTK_WINDOW_TOPLEVEL is the default GtkWindow:type */
		};
		ljw_type = g_type_register_static(GTK_TYPE_WINDOW,
				"LJWin", &ljw_info, 0);
	}
	return ljw_type;
}

#define LJWIN(obj) (G_TYPE_CHECK_INSTANCE_CAST((obj), ljwin_get_type(), LJWin))

static GtkWidget*
ljwin_new(void) {
	LJWin *ljw = LJWIN(g_object_new(ljwin_get_type(), NULL));
	return GTK_WIDGET(ljw);
}

void 
lj_run(Entry *entry) {
	GtkWidget *vbox;
	Entry *draftentry;

	LJWin *ljw = LJWIN(ljwin_new());
	gtk_window_set_default_size(GTK_WINDOW(ljw), 300, 200);
	geometry_tie(GTK_WIDGET(ljw), GEOM_MAIN);

	lj_update_usejournal(ljw);
	lj_win_set_icon(GTK_WIDGET(ljw));
	app.tooltips = gtk_tooltips_new();
	gtk_tooltips_enable(app.tooltips);
	
	g_signal_connect(G_OBJECT(ljw), "delete-event",
		G_CALLBACK(delete_event_cb), NULL);

	vbox = gtk_vbox_new(FALSE, 0); 

	gtk_box_pack_start(GTK_BOX(vbox), 
			menu_make_bar(ljw), FALSE, FALSE, 0);

	gtk_box_pack_start(GTK_BOX(vbox), make_app_contents(ljw), 
			TRUE, TRUE, 0);

	gtk_container_add(GTK_CONTAINER(ljw), vbox);

	lj_autosave_init(ljw);

	gtk_widget_show_all(vbox);
	lj_hideshow_buttons(ljw);
	gtk_widget_hide_all(cfindicator_box_get(ljw->cfi_main));

	menu_update(ljw, NULL);

	gtk_widget_show(GTK_WIDGET(ljw));

	/* suck a bunch of events in. */
	while (gtk_events_pending())
		gtk_main_iteration();

	draftentry = lj_load_draft(ljw);
	if (draftentry) {
		if (lj_confirm(GTK_WIDGET(ljw), _("Autosaved Draft Found"),
					_("An autosaved draft was found, possibly from a previous run of LogJam that crashed.  "
					  "Would you like to recover it?"))) {
			entry_free(entry);
			entry = draftentry;
		}
	}

	lj_load_entry(ljw, entry);
	if (draftentry)
		ljw->isdirty = TRUE;
	entry_free(entry);

	/* to prevent flicker, make this GtkWindow immediately after
	 * pending events have been handled, and immediately handle its
	 * own event afterwards.
	 *
	 * XXX: Should make_cf_float() have its own event-sucking loop? */
	if (conf.options.cfautofloat) {
		app.cfi_float = make_cf_float(ljw->cfmgr, GTK_WIDGET(ljw));
		while (gtk_events_pending())
			gtk_main_iteration();
	}

	/* autolog if username and password known. */
	lj_do_loginout(ljw, conf_cur_user() != NULL);
	gtk_main();

	lj_draft_delete();
}
