/*
 * kfortunewindow.cpp
 *
 * Copyright (c) 2001 Frerich Raabe <raabe@kde.org>
 * Copyright (c) 2001 Ian Reinhart Geiser <geiseri@kde.org>
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. For licensing and distribution details, check the
 * accompanying file 'COPYING'.
 */
#include <dcopclient.h>
#include <dcopobject.h>
#include <kaction.h>
#include <kapplication.h>
#include <kfiledialog.h>
#include <klocale.h>
#include <kmainwindow.h>
#include <knotifyclient.h>
#include <kprocess.h>
#include <kstdaction.h>

#include <qclipboard.h>
#include <qcstring.h>
#include <qdatastream.h>
#include <qfile.h>
#include <qtextstream.h>

#include "kfortunewindow.h"
#include "kfortuneview.h"

KFortuneWindow::KFortuneWindow() : KMainWindow(), DCOPObject("KFortuneWindow")
{
	setupActions();
	
	setCaption(i18n("KDE Fortune"));

	m_view = new KFortuneView(this);
	m_prefsView = new prefs(this);
	connect(m_view, SIGNAL(newFortuneClicked()), this, SLOT(newFortune()));
	connect(m_view, SIGNAL(closeClicked()), this, SLOT(close()));
	connect(m_prefsView, SIGNAL(okay->released()), this, SLOT(slotSavePrefs()));

	setCentralWidget(m_view);

	newFortune();
}

KFortuneWindow::~KFortuneWindow()
{
	delete m_view;
}

void KFortuneWindow::nextFortune()
{
	newFortune();
}

QString KFortuneWindow::currentFortune() const
{
	return m_view->fortune();
}

void KFortuneWindow::setupActions()
{
	KStdAction::openNew(this, SLOT(newFortune()), actionCollection());
	KStdAction::save(this, SLOT(saveFortune()), actionCollection());
	KStdAction::quit(this, SLOT(close()), actionCollection());
	KStdAction::copy(this, SLOT(copyFortune()), actionCollection());
	KStdAction::preferences(this, SLOT(preferences()), actionCollection());
        
	createGUI();
}

void KFortuneWindow::newFortune()
{
	m_fortuneText = QString::null;

	KProcess *process = new KProcess;
	connect(process, SIGNAL(receivedStdout(KProcess *, char *, int)),
	        this, SLOT(receivedStdout(KProcess *, char *, int)));
	connect(process, SIGNAL(processExited(KProcess *)),
	        this, SLOT(processExited(KProcess *)));
	*process << "fortune";

	if (!process->start(KProcess::NotifyOnExit, KProcess::Stdout)) {
		KNotifyClient::event(QString::fromLatin1("ExecFailed"),
			i18n("Could not run the 'fortune' program."
			" Please make sure that it is in your $PATH."));
		delete process;
	}
}

void KFortuneWindow::saveFortune()
{
	QString fileName = KFileDialog::getSaveFileName(QString::null, QString::null, NULL, i18n("Save Fortune..."));
	if (!fileName.isEmpty()) {
		QFile file(fileName);
		if (file.open(IO_WriteOnly)) {
			QTextStream stream(&file);

			stream << m_view->fortune();

			file.close();
		} else
			KNotifyClient::event(QString::fromLatin1("WriteFailed"),
				i18n("Could not create file %1 for writing!").arg(fileName));
	}	
}

void KFortuneWindow::copyFortune()
{
	kapp->clipboard()->setText(m_view->fortune());
}

void KFortuneWindow::setFortune(const QString &fortune)
{
	m_view->setFortune(fortune);
	adjustSize();

	QByteArray data;
	QDataStream stream(data, IO_WriteOnly);
	stream << fortune;
	kapp->dcopClient()->emitDCOPSignal("newFortune(QString)", data);
}

void KFortuneWindow::preferences()
{
	// Set the data in m_presView
	// Somewhere here we need to do a fortune -l to get
	// all of the databases.

	// Show
	m_prefsView->show();
}

void KFortuneWindow::slotSavePrefs()
{
	// Grab all info from m_prefsView

	// Hide
	m_prefsView->hide();
}

void KFortuneWindow::receivedStdout(KProcess *, char *data, int len)
{
	m_fortuneText += QString::fromLocal8Bit(data, len);
}

void KFortuneWindow::processExited(KProcess *process)
{
	// Get rid of the trailing \n
	m_fortuneText.truncate(m_fortuneText.length() - 1);

	setFortune(m_fortuneText);

	delete process;
}
#include "kfortunewindow.moc"

// vim:ts=4:sw=4:noet
