/*
Copyright (c) 2007 Andre Noll

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:
1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ANDRE NOLL
``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/select.h>
#include <signal.h>
#include <errno.h>
#include <sys/time.h>
#include <time.h>
#include <sys/types.h>
#include <sys/wait.h>


static int signal_pipe[2];

static int tv_diff(const struct timeval *b, const struct timeval *a, struct timeval *diff)
{
	int ret = 1;

	if ((b->tv_sec < a->tv_sec) ||
			((b->tv_sec == a->tv_sec) && (b->tv_usec < a->tv_usec))) {
		const struct timeval *tmp = a;
		a = b;
		b = tmp;
		ret = -1;
	}
	if (!diff)
		return ret;
	diff->tv_sec = b->tv_sec - a->tv_sec;
	if (b->tv_usec < a->tv_usec) {
		diff->tv_sec--;
		diff->tv_usec = 1000 * 1000 - a->tv_usec + b->tv_usec;
	} else
		diff->tv_usec = b->tv_usec - a->tv_usec;
	return ret;
}

static void signal_handler(int s)
{
	write(signal_pipe[1], &s, sizeof(int));
}


void signal_init_or_die(void)
{
	if (pipe(signal_pipe))
		exit(EXIT_FAILURE);
	if (signal(SIGCHLD, &signal_handler) == SIG_ERR)
		exit(EXIT_FAILURE);
}

static int reap_child(pid_t child_pid)
{
	int status;
	pid_t pid;

	do
		pid = waitpid(child_pid, &status, 0);
	while (pid == -1 && errno == EINTR);
	if (pid <= 0) /* oops, that should never happen */
		return EXIT_FAILURE;
	if (WIFEXITED(status))
		printf("child %i exited. Exit status: %i\n", (int)pid,
			WEXITSTATUS(status));
	else if (WIFSIGNALED(status))
		printf("child %i was killed by signal %i\n", (int)pid,
			WTERMSIG(status));
	else
		printf("child %i terminated abormally\n", (int)pid);
	return pid;
}


int main(int argc, char **argv)
{
	int ret, err, expired = 0;
	pid_t pid;
	struct timeval start, timeout = {.tv_sec = 0}, tv;

	if (argc < 3)
		exit(EXIT_FAILURE);
	timeout.tv_sec = atoi(argv[1]);
	argv[1] = strdup(argv[2]);
	pid = fork();
	if (pid < 0)
		exit(EXIT_FAILURE);
	signal_init_or_die();
	if (!pid) {
		close(signal_pipe[0]);
		close(signal_pipe[1]);
		execvp(argv[1], argv + 2);
		_exit(EXIT_FAILURE);
	}
	free(argv[1]);
	gettimeofday(&start, NULL);
	tv = timeout;
	do {
		fd_set rfds;
		struct timeval diff, now;

		FD_ZERO(&rfds);
		FD_SET(signal_pipe[0], &rfds);
		ret = select(signal_pipe[0] + 1, &rfds, NULL, NULL, &tv);
		err = errno;
		gettimeofday(&now, NULL);
		tv_diff(&start, &now, &diff);
		if (tv_diff(&diff, &timeout, &tv) > 0)
			expired = 1;
	} while ((!ret && !expired) || (ret < 0 && err == EINTR));
	if (ret <= 0)
		kill(pid, SIGINT);
	ret = reap_child(pid);
	close(signal_pipe[0]);
	close(signal_pipe[1]);
	return ret;
}
