/*
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2 of the License, or
 *      (at your option) any later version.
 *
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include "position.h"
#include "def.h"

/*
 * alloc_pixmap : Allocate a new GdkPixmap
 * which is like an offscreen Pixmap to 
 * draw different part of the game into.
 * 
 * We need to do that when first starting 
 * the game, and after that when resizing 
 * the window a new pixmap should be 
 * allocated with the size of the resized 
 * window.
 * 
 * struct _prog *prog_data: the main 
 * struct which is what makes possible 
 * to pass pointer to every usefull 
 * part of the memory.
 * 
 * Here we need :
 * 
 * prog_data->target - the GtkPixmap
 * 
 * prog_data->area - the widget where 
 * the pixmap will be copied to be 
 * visible to the user.
 * 
 * prog_data->output - the main stream 
 * for printing message. Default is 
 * stdout.
 */

void alloc_pixmap(struct _prog *prog_data)
{
	if(prog_data->target != NULL)
	{
		g_object_unref(prog_data->target);
		prog_data->target = NULL;
	}
	
	prog_data->target = gdk_pixmap_new( 
		prog_data->area->window, 
		prog_data->area->allocation.width, 
		prog_data->area->allocation.height, 
		-1) ;
	
	if(prog_data->target == NULL)
	{
		if(prog_data->output != NULL)
			fprintf(prog_data->output, 
				_("Unable to allocated target pixmap\n") );
	}
}

/* alloc_targetzone : a target zone is a 
 * rectangle in the game area where you 
 * can drop a card during the game.
 * 
 * Example: the top left area represent 
 * the pile. The bottom of the screen is 
 * the user hand. And the middle is the 
 * table where you drop one of your card 
 * to play it.
 * 
 * This function allocate a new target zone 
 * and prepend it to a GList.
 * 
 * struct _prog *prog_data: see file def.h
 * 
 * Here we need : 
 * 
 * prog_data->dropping : a GList in which we put 
 * every dropping area in a data struct _target 
 * which contains a dim (x,y, w,h) and a flag 
 * (active or not).
 * 
 * return : struct _target *
 * pointer to the target zone allocated. 
 * NULL in case of an error.
 */

struct _target * alloc_targetzone(struct _prog *prog_data)
{
	struct _target *zone = (struct _target *) 
		g_malloc(sizeof(struct _target) );
	
	if(zone)
	{
		zone->active = FALSE;
		zone->dim.x = 0;
		zone->dim.y = 0;
		zone->dim.w = 0;
		zone->dim.h = 0;
		
		prog_data->dropping = g_list_prepend(
			prog_data->dropping, zone);
	}
	
	return( zone );
}

/* alloc_prog: takes care of allocation memory 
 * for every important part of the program. 
 * 
 * Each element is considered as allocated during 
 * the game. Which is why this alloc_prog function 
 * is needed.
 * 
 * struct _prog *prog_data: see file def.h
 * 
 * Here we need: 
 * 
 * prog_data->icons : an array composed by four 
 * GdkPixbuf containing icons of each family 
 * of the card game (diamond, spade, heart, club)
 * This is used to reflect the main family chosen 
 * by the master player in the beginning of each 
 * round.
 * 
 * prog_data->players : array composed by four 
 * GList, each corresponding to a player hand.
 * Each element in all lists contains a struct 
 * _card which is composed of informations 
 * to know which card it is, like the family, 
 * the points, the position on the game area, 
 * etc. See file def.h for more details about 
 * the _card structure.
 * 
 * prog_data->teams : array composed by two
 * GList, each containing cards winned during 
 * all turns played by teams. This is where 
 * cards disappears. 
 * 
 * All GList (players and teams) are freed 
 * when a round is finished. It's more like 
 * a real list of cards you put next to you 
 * when playing a real card game. 
 * 
 * 
 */
void alloc_prog(struct _prog *prog_data)
{
	int i, maxicon = 5, teams = 2, players = 4;
	
	prog_data->icons = (GdkPixbuf**) g_malloc(sizeof(GdkPixbuf*) *maxicon );
	if(prog_data->icons)
	{
		for(i = 0; i < maxicon; i ++)
			prog_data->icons[i] = NULL;
	}
	
	
	
	alloc_pixmap(prog_data);
		
	
	prog_data->teams = (GList **) g_malloc( sizeof(GList*) *teams );
	
	if(prog_data->teams != NULL)
	{
		for ( i=0; i< teams; i++)
			prog_data->teams[i] = NULL;
	}
	
	
	prog_data->players = (GList **) g_malloc( sizeof(GList*) *players );
	
	if(prog_data->players != NULL)
	{
		for ( i=0; i< players; i++)
			prog_data->players[i] = NULL;
	}
	
	alloc_targetzone(prog_data);
	

}

gchar** alloc_default_names()
{
	gchar **list = (gchar **) g_malloc( sizeof(gchar *) * 5 );
	
	if(list)
	{
		list[0] = g_strdup ( _("Elisa") );
		list[1] = g_strdup ( _("Ronald") );
		list[2] = g_strdup ( _("You") );
		list[3] = g_strdup ( _("Antonio") );
		list[4] = NULL;
	}
	
	return list;
}
