/* AbiSource
 * 
 * Copyright (C) 2005 Daniel d'Andrada T. de Carvalho
 * <daniel.carvalho@indt.org.br>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
 * 02111-1307, USA.
 */


#ifndef _OD_STYLE_STYLE_H_
#define _OD_STYLE_STYLE_H_

// Internal includes
#include "OD_ListenerState.h"

// AbiWord includes
#include <ut_string_class.h>

// AbiWord classes
class PD_Document;


/**
 * An OpenDocument regular style (<style:style>).
 */
class OD_Style_Style : public OD_ListenerState {

public:

    OD_Style_Style(OD_ElementStack& rElementStack);
    virtual ~OD_Style_Style() {}

    void startElement(const XML_Char* pName, const XML_Char** ppAtts,
                      OD_ListenerStateAction& rAction);
                      
    void endElement(const XML_Char* pName, OD_ListenerStateAction& rAction);
    
    void charData (const XML_Char* pBuffer, int length) {}
    
    
    const UT_UTF8String& getDisplayName() const {return m_displayName;}
    void setDisplayName(UT_UTF8String& rDisplayName) {
        m_displayName = rDisplayName;
    }
    
    /**
     * Defines an AbiWord style that is equivalent to this
     * OpenDocument style.
     * 
     * Called by text and paragraph styles.
     * 
     * @param pDocument The AbiWord document on which the style will be defined.
     */
    void defineAbiStyle(PD_Document* pDocument);
    
    
    /**
     * Builds the AbiWord "props" attribute value that describes this
     * Style.
     */
    void buildAbiPropsAttrString();
    
    /**
     * @param rProps The string that will have appended to it the properties of this
     *               style.
     */
    void OD_Style_Style::getAbiPropsAttrString(UT_UTF8String& rProps) const;
    
    
    const UT_UTF8String& getParentStyleName() const {
        return m_parentStyleName;
    }
    
    void setParentStylePointer(const OD_Style_Style* pParentStyle) {
        m_pParentStyle = pParentStyle;        
    }
    
    void setNextStylePointer(const OD_Style_Style* pNextStyle) {
        m_pNextStyle = pNextStyle;
    }
    
    bool getPageBreakBefore() const {return m_bPageBreakBefore;}
    bool getColBreakBefore() const {return m_bColBreakBefore;}
    
    const UT_UTF8String& getName() const {return m_name;}
    void setName(UT_UTF8String& rName) {
        m_name = rName;
    }
    
    const UT_UTF8String& getParentName() const {return m_parentStyleName;}
    void setParentName(const char* pName) {m_parentStyleName.assign(pName);}
    void setParentName(const UT_UTF8String& rName) {m_parentStyleName = rName;}

    inline const UT_UTF8String& getNextStyleName() const {return m_nextStyleName;}
    inline void setNextStyleName(const char* pName) {m_nextStyleName.assign(pName);}
    void setNextStyleName(const UT_UTF8String& rName) {m_nextStyleName = rName;}
 
    bool hasProperties() const {
        return m_bColBreakBefore ||
               m_bPageBreakBefore ||
               !m_lineHeight.empty() ||
               !m_align.empty() ||
               !m_widows.empty() ||    
               !m_orphans.empty() || 
               !m_marginLeft.empty() ||
               !m_marginRight.empty() ||
               !m_marginTop.empty() ||
               !m_marginBottom.empty() ||       
               !m_bgcolor.empty() ||
               !m_keepWithNext.empty() ||
            
               !m_color.empty() ||
               !m_textDecoration.empty() ||
               !m_textPos.empty() ||
               !m_fontName.empty() ||
               !m_fontSize.empty() ||
               !m_lang.empty() ||
               !m_fontStyle.empty() ||
               !m_fontWeight.empty() ||
            
               !m_columns.empty() ||
               !m_columnGap.empty() ||
               
               !m_wrap.empty() ||
               
               !m_backgroundColor.empty() ||
    
               !m_columnWidth.empty() ||
               
               !m_minRowHeight.empty() ||
               !m_rowHeight.empty();
    }
    
    bool isAutomatic() const {return m_bAutomatic;}
    const UT_UTF8String* getMarginLeft() const {return &m_marginLeft;}
    const UT_UTF8String* getFamily() const {return &m_family;}
    const UT_UTF8String* getFontName() const {return &m_fontName;}
    
    /**
     * @param local If "true", It returns the plain value of the corresponding
     *              variable. Otherwise, it considers the final value of this
     *              property, taking into account its value on the parent styles.
     */
    const UT_UTF8String* getWrap(bool local) const;
    
    const UT_UTF8String* getBackgroundColor() const {return &m_backgroundColor;}
    
    const UT_UTF8String* getColumnWidth() const {return &m_columnWidth;}
    
    const UT_UTF8String* getMinRowHeight() const {return &m_minRowHeight;}
    const UT_UTF8String* getRowHeight() const {return &m_rowHeight;}
 

    const UT_UTF8String* getBorderTop_thickness() const {return &m_borderTop_thickness;}
    const UT_UTF8String* getBorderTop_color() const {return &m_borderTop_color;}
    const UT_UTF8String* getBorderBottom_thickness() const {return &m_borderBottom_thickness;}
    const UT_UTF8String* getBorderBottom_color() const {return &m_borderBottom_color;}
    const UT_UTF8String* getBorderLeft_thickness() const {return &m_borderLeft_thickness;}
    const UT_UTF8String* getBorderLeft_color() const {return &m_borderLeft_color;}
    const UT_UTF8String* getBorderRight_thickness() const {return &m_borderRight_thickness;}
    const UT_UTF8String* getBorderRight_color() const {return &m_borderRight_color;}
 
private:

    // <style:style />
    void _parse_style_style(const XML_Char** ppAtts);
    
    // <style:paragraph-properties />
    void _parse_style_paragraphProperties(const XML_Char** ppProps);
    
    // <style:text-properties />
    void _parse_style_textProperties(const XML_Char** ppProps);
    
    // <style:section-properties />
    void _parse_style_sectionProperties(const XML_Char** ppProps);
    
    // <style:graphic-properties />
    void _parse_style_graphicProperties(const XML_Char** ppProps);
    
    // <style:table-properties />
    void _parse_style_tableProperties(const XML_Char** ppProps);
    
    // <style:table-column-properties />
    void _parse_style_tableColumnProperties(const XML_Char** ppProps);
    
    // <style:table-row-properties />
    void _parse_style_tableRowProperties(const XML_Char** ppProps);
    
    // <style:table-cell-properties />
    void _parse_style_tableCellProperties(const XML_Char** ppProps);
    
    /**
     * If pString is "0.0556in solid #0000ff", rColor will receive "#0000ff" and
     * rLength "0.0556in".
     */
    void _stripColorLength(UT_UTF8String& rColor, UT_UTF8String& rLength,
                          const XML_Char* pString) const;
                          
    /**
     * This function shouldn't exist. The code should use
     * UT_isValidDimensionString instead. The problem with the UT function is
     * that it doesn't check the dimension specifier and only accepts NULL
     * terminated strings.
     * 
     * @param length 0 for NULL terminated strings.
     */
    bool _isValidDimensionString(const XML_Char* pString, UT_uint32 length=0) const;
    
    // true if it is an OpenDocument automatic style.
    // ie., it's defined inside a <office:automatic-styles> element.
    bool m_bAutomatic;
    
    const OD_Style_Style* m_pParentStyle;
    const OD_Style_Style* m_pNextStyle;
    
    UT_UTF8String m_abiPropsAttr;
    

    // <attribute name="style:name">
    UT_UTF8String m_name;

    // <attribute name="style:display-name"> (optional)
    // If this attribute is not present, the display name equals the style name.
    // In AbiWord, maps to the "name" attribute.
    UT_UTF8String m_displayName;
    
    // Maps to the "type" attribute.
    // OpenDocument | AbiWord
    // "character"  - "C"
    // "paragraph"  - "P"
    // "section"    - none (AbiWord don't have section styles)
    //
    // An exception is "graphic" styles. AbiWord don't have then.
    UT_UTF8String m_family;
    
    // <attribute name="style:parent-style-name"> (optional)
    // If a parent style is not specified, a default parent style defined by
    // the application is used.
    //
    // In AbiWord, maps to the "basedon" attribute.
    UT_UTF8String m_parentStyleName;
    
    // <attribute name="style:next-style-name">
    // By default, the current style is used as the next style.
    // In AbiWord, maps to the "followedby" attribute.
    UT_UTF8String m_nextStyleName;
    
    // <attribute name="style:list-style-name"> (optional)
    //
    // Is only applied to headings and to paragraphs that are contained in a
    // list, where the list does not specify a list style itself, and the list
    // has no list style specification for any of its parents.
    //
    // Maps to AbiWord, but not directly.
    UT_UTF8String m_listStyleName;
    
    // <attribute name="style:master-page-name"> (optional)
    //
    // If this attribute is associated with a style, a page break is inserted
    // when the style is applied and the specified master page is applied to the
    // preceding page.
    // This attribute is ignored if it is associated with a paragraph style that
    // is applied to a paragraph within a table.
    //
    // Maps to AbiWord, but not directly.
    UT_UTF8String m_MasterPageName;
    
    
    ////
    // <style:paragraph-properties> attributes
    // These goes inside the Abi "props" attribute
    bool m_bColBreakBefore;
    bool m_bPageBreakBefore;    
    UT_UTF8String m_lineHeight;
    UT_UTF8String m_align;
    UT_UTF8String m_widows;    
    UT_UTF8String m_orphans; 
    UT_UTF8String m_marginLeft;
    UT_UTF8String m_marginRight;
    UT_UTF8String m_marginTop;
    UT_UTF8String m_marginBottom;       
    UT_UTF8String m_bgcolor;
    UT_UTF8String m_keepWithNext;
    
    ////
    // <style:text-properties />
    // These goes inside the Abi "props" attribute
    UT_UTF8String m_color;
    UT_UTF8String m_textDecoration;
    UT_UTF8String m_textPos;
    UT_UTF8String m_fontName;
    UT_UTF8String m_fontSize;
    UT_UTF8String m_lang;
    UT_UTF8String m_fontStyle;
    UT_UTF8String m_fontWeight;


    // fo:background-color
    // For <style:table-properties> and <style:table-cell-properties>
    UT_UTF8String m_backgroundColor;

    
    ////
    // <style:section-properties>
    // These goes inside the Abi "props" attribute
    UT_UTF8String m_columns;
    UT_UTF8String m_columnGap;
    
    ////
    // <style:graphic-properties>
    UT_UTF8String m_wrap; // style:wrap

    
    ////
    // <style:table-column-properties>
    UT_UTF8String m_columnWidth; // style:column-width
    
    ////
    // <style:table-row-properties>
    UT_UTF8String m_minRowHeight; // style:min-row-height
    UT_UTF8String m_rowHeight; // style:row-height
    
    
    ////
    // <style:table-cell-properties>
    
    // fo:border-top
    UT_UTF8String m_borderTop_thickness;
    UT_UTF8String m_borderTop_color;
    
    // fo:border-bottom
    UT_UTF8String m_borderBottom_thickness;
    UT_UTF8String m_borderBottom_color;
    
    // fo:border-left
    UT_UTF8String m_borderLeft_thickness;
    UT_UTF8String m_borderLeft_color;
    
    // fo:border-right
    UT_UTF8String m_borderRight_thickness;
    UT_UTF8String m_borderRight_color;
    
    // OBS: If "fo:border" is defined, its value will fill all "fo:border-*"
};

#endif //_OD_STYLE_STYLE_H_
