/*
 * Copyright (C) 2002 Daniel Heck
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * This file contains the code for initializing the video hardware and
 * for various visual effects in the widest sense.  This includes
 *
 * a) managing and rendering the mouse cursor,
 * b) changing between windowed and fullscreen mode,
 * c) fading between different screen using one of a number of
 *    predefined effects.
 *
 * The code in this file is independent of the game.  For game-specific
 * display code, refer to file display.cc
 */

#include "enigma.hh"
#include "video.hh"
#include "lua.hh"
#include "options.hh"
#include "px/px.hh"
#include "SDL.h"
#include <cassert>
#include <cstdio>
#include <fstream>

using namespace std;
using namespace px;

namespace
{
    class Video_SDL {
        SDL_Surface*    sdlScreen;
        bool            fullScreen;
        string          caption;
        px::Screen*     screen;
        bool            initialized;
    public:
        Video_SDL();
        ~Video_SDL();

        void init(int w, int h);
        void toggle_inputgrab();
        void toggle_fullscreen();
        void set_fullscreen(bool on_off);
        bool is_fullscreen() const { return fullScreen; }
        void set_caption(const char *str);
        px::Screen *get_screen() { return screen; }
    };

    class MouseCursor {
    public:
        MouseCursor (px::Screen *scr);
        ~MouseCursor();

        void set_image (px::Surface *s, int hotx_, int hoty_);
        void move (int newx, int newy);
        void redraw ();         // Redraw if position/image changed
        void draw();            // Draw cursor if visible
        void show ();
        void hide ();
        Rect get_rect() const;
        Rect get_oldrect() const;

	bool has_changed() { return changedp; }
        int get_x() const { return x; }
        int get_y() const { return y; }

    private:
        void grab_bg ();

        Screen  *screen;
        Surface *background;    // Copy of screen contents behind cursor
        Surface *cursor;        // Pixmap of the cursor
        int      x, y;
        int      oldx, oldy;
	int      hotx, hoty;
	int      visible;
	bool     changedp;
    };
}

//----------------------------------------
// Video engine impl
//----------------------------------------
Video_SDL::Video_SDL()
: sdlScreen(0)
, fullScreen(false)
, screen(0)
, initialized(false)
{}

Video_SDL::~Video_SDL()
{
    SDL_WM_GrabInput(SDL_GRAB_OFF);
    if (sdlScreen != 0 && fullScreen)
        SDL_WM_ToggleFullScreen(sdlScreen);
    delete screen;
}

void Video_SDL::set_caption(const char *str)
{
    if (initialized)
        SDL_WM_SetCaption(str, 0);
}

void Video_SDL::init(int w, int h)
{
    SDL_WM_SetCaption(caption.c_str(), 0);

    int bpp = options::BitsPerPixel;
    assert(bpp==16 || bpp==8 || bpp==32);

    const SDL_VideoInfo *vidinfo = SDL_GetVideoInfo();
    if (vidinfo->vfmt->BitsPerPixel==8)
        bpp = 8;

    Uint32 flags = SDL_SWSURFACE;
    if (options::FullScreen)
        flags |= SDL_FULLSCREEN;

    if (bpp == 8)
        flags |= SDL_HWPALETTE;

    sdlScreen = SDL_SetVideoMode(w, h, bpp, flags);
    if (sdlScreen == 0)
    {
	fprintf(stderr, "Couldn't open screen: %s\n", SDL_GetError());
        exit(1);
    }
    if (sdlScreen->flags & SDL_FULLSCREEN)
    {
        fullScreen = true;
        options::FullScreen = true;
    }
    screen = new Screen(sdlScreen);

    SDL_ShowCursor(0);
    SDL_EnableKeyRepeat(SDL_DEFAULT_REPEAT_DELAY / 2,
                        SDL_DEFAULT_REPEAT_INTERVAL / 2);

    initialized = true;
}

void Video_SDL::toggle_inputgrab()
{
    if (SDL_WM_GrabInput(SDL_GRAB_QUERY) == SDL_GRAB_ON)
        SDL_WM_GrabInput(SDL_GRAB_OFF);
    else
        SDL_WM_GrabInput(SDL_GRAB_ON);
}

void Video_SDL::set_fullscreen(bool on_off)
{
    if (on_off != fullScreen)
    {
        toggle_fullscreen();
        fullScreen = on_off;
    }
}

void Video_SDL::toggle_fullscreen()
{
    SDL_WM_ToggleFullScreen(sdlScreen);
    fullScreen = !fullScreen;
}

//----------------------------------------
// MouseCursor impl
//----------------------------------------

MouseCursor::MouseCursor (Screen *screen_)
: screen(screen_)
, background(0)
, cursor(0)
{
    oldx = oldy = 0;
    hotx = hoty = 0;
    visible = 0;
    changedp = true;
}

MouseCursor::~MouseCursor()
{
    delete background;
    delete cursor;
}

void
MouseCursor::set_image (px::Surface *s, int hx, int hy)
{
    delete cursor;
    cursor = s;
    hotx   = hx;
    hoty   = hy;
}

void
MouseCursor::draw ()
{
    if (visible > 0) {
        grab_bg();

        GC gc(screen->get_surface());
        blit (gc, x-hotx, y-hoty, cursor);
        screen->update_rect (get_rect());

        changedp = false;
    }
}

void
MouseCursor::redraw ()
{
    if (visible > 0 && changedp)
    {
        if (background) {
            GC gc(screen->get_surface());
            blit (gc, oldx-hotx, oldy-hoty, background);
            screen->update_rect (get_oldrect());
        }
        draw();
    }
}

void
MouseCursor::move(int newx, int newy)
{
    x        = newx;
    y        = newy;
    changedp = true;
}

void
MouseCursor::show ()
{
    if (++visible == 1)
        changedp=true;
}

void
MouseCursor::hide ()
{
    if (--visible == 0) {
        changedp = true;
        if (background) {
            GC gc(screen->get_surface());
            blit (gc, oldx-hotx, oldy-hoty, background);
        }
        delete background;
        background=0;
    }
}

Rect
MouseCursor::get_rect() const
{
    int scrx=x-hotx;
    int scry=y-hoty;
    return Rect(scrx,scry,cursor->width(), cursor->height());
}

Rect
MouseCursor::get_oldrect() const
{
    int scrx=oldx-hotx;
    int scry=oldy-hoty;
    return Rect(scrx,scry,cursor->width(), cursor->height());
}

void
MouseCursor::grab_bg ()
{
    if (background == 0)
        background = Grab(screen->get_surface(), get_rect());
    else {
        GC gc(background);
        blit (gc, 0,0, screen->get_surface(), get_rect());
    }
    oldx=x;
    oldy=y;
}



//----------------------------------------------------------------------
// FUNCTIONS.
//----------------------------------------------------------------------
namespace
{
    Video_SDL   *video_engine = 0;
    MouseCursor *cursor;
    Surface     *back_buffer  = 0;
}

/* This function is installed as an event filter by video::Init.  It
   intercepts two kinds of events: Mouse motions, which are used to
   update the position of the mouse cursor (but passed on to the event
   queue), and pressing of Alt-Ret, which switches between fullscreen
   and windowed mode (also passed on to the event queue). */
static int
event_filter(const SDL_Event *e)
{
    if (e->type == SDL_MOUSEMOTION)
    {
        cursor->move(e->motion.x, e->motion.y);
        cursor->redraw();
    }
    else if (e->type == SDL_KEYDOWN)
    {
        if ((e->key.keysym.sym==SDLK_RETURN) &&
            (e->key.keysym.mod & KMOD_ALT))
        {
            video::ToggleFullscreen();
        }
    }
    return 1;
}

void
video::SetMouseCursor(px::Surface *s, int hotx, int hoty)
{
    cursor->set_image(s, hotx, hoty);
    cursor->redraw();
}

void
video::HideMouse()
{
    cursor->hide();
    cursor->redraw();
}

void
video::ShowMouse()
{
    cursor->show();
    cursor->redraw();
}

int
video::Mousex()
{
    return cursor->get_x();
}

int
video::Mousey()
{
    return cursor->get_y();
}



void
video::SetPalette(const char *palname)
{
    if (GetColorDepth() != 8)
        return;

    SDL_Surface *s = GetScreen()->get_surface()->get_surface();
    string filename = enigma::FindDataFile("gfx", palname);
    ifstream f(filename.c_str());

    string restofline;

    getline(f, restofline);    // skip first two lines
    getline(f, restofline);

    SDL_Color palette[256];
    int n=0;
    while (f && n<256) {
        int r=0, g=0, b=0;
        f >> r >> g >> b;
        palette[n].r = r;
        palette[n].g = g;
        palette[n].b = b;
        getline(f, restofline); // skip rest of line
        ++n;
    }
//     palette[255].r = 0xff;
//     palette[255].g = 0;
//     palette[255].b = 0xff;
    SDL_SetColors (s, palette,0,n);
}

int
video::GetColorDepth()
{
    return GetScreen()->get_surface()->bipp();
}

Surface*
video::BackBuffer()
{
    if (back_buffer==0) {
        back_buffer= Duplicate(GetScreen()->get_surface());
    }
    return back_buffer;
}

static video::VMInfo video_modes[] = {
    { 640, 480, "640x480" },
    { 640, 512, "640x512" }
};

#define NUMENTRIES(array) (sizeof(array)/sizeof(*array))

const video::VMInfo *video::GetInfo (VideoModes vm)
{
    return &video_modes[vm];
}


void video::Init()
{
    assert (NUMENTRIES(video_modes) == VM_COUNT);

    int vidmode = Clamp (options::VideoMode, 0, VM_COUNT-1);
    VMInfo &vminfo = video_modes[vidmode];

    video_engine = new Video_SDL();
    video_engine->init (vminfo.width, vminfo.height);

    int x, y;
    SDL_GetMouseState(&x, &y);
    cursor = new MouseCursor (GetScreen());
    cursor->move(x,y);

    SDL_SetEventFilter(event_filter);

    options::FullScreen = video_engine->is_fullscreen();
}

void
video::Shutdown()
{
    delete video_engine;
    delete cursor;
    delete back_buffer;
}


px::Screen *
video::GetScreen()
{
    return video_engine->get_screen();
}


bool
video::ToggleFullscreen()
{
    video_engine->toggle_fullscreen();
    return options::FullScreen = video_engine->is_fullscreen();
}

void
video::ToggleInputgrab() {
    ; //video_engine->toggle_inputgrab();
}

void
video::SetCaption(const char *str)
{
    video_engine->set_caption(str);
}


//======================================================================
// SPECIAL EFFECTS
//======================================================================
namespace video
{
    void FX_Push (Surface *newscr, int originx, int originy);
}

namespace video {
void fade_pal(FadeMode mode)
{
//     const double fadesec = 0.6;
//     double v = 1/fadesec;
//     double a = mode==FADEIN ? 0 : 1;

//     SDL_Surface *screen = GetScreen()->get_surface();

//     SDL_Palette *pal = screen->format->palette;
//     assert(pal);

//     while (true) {
//         Uint32 otime = SDL_GetTicks();


//         for (int i=0; i<pal->ncolors; ++i) {
//             colors[i].r = int (a*pal.colors[i].r)

//         dt = (SDL_GetTicks()-otime)/1000.0;
//         if (mode==FADEIN && (a+=v*dt) > 1)
//             break;
//         else if (mode==FADEOUT && (a-=v*dt) < 0)
//             break;
//     }
}

void fade_alpha(FadeMode mode)
{
//     const double fadesec = 0.6;
//     double v = 255/fadesec;

//     px::Screen *screen = GetScreen();
//     px::Surface *buffer = Duplicate(screen);
//     double dt;

//     double a = mode==FADEIN ? 0 : 255;
//     Drawable *d = screen->get_drawable();
//     while (true) {
//         Uint32 otime = SDL_GetTicks();

//         d->box(0,0,640,480, 0);
//         buffer->set_alpha(int(a));
//         d->blit(0,0,buffer);
//         screen->update_all();
//         screen->flush_updates();

//         dt = (SDL_GetTicks()-otime)/1000.0;
//         if (mode==FADEIN && (a+=v*dt) > 255)
//             break;
//         else if (mode==FADEOUT && (a-=v*dt) < 0)
//             break;
//     }

//     if (mode==FADEIN) {
//         buffer->set_alpha(255);
//         d->blit(0,0,buffer);
//     } else
//         d->box(0,0,640,480, 0);
//     screen->update_all();
//     screen->flush_updates();
//     delete buffer;
}
}
void
video::FX_Fade(FadeMode mode)
{
    SDL_Surface *s = GetScreen()->get_surface()->get_surface();
    if (s->format->palette)
        fade_pal(mode);
    else
        fade_alpha(mode);
}

void
video::FX_Fly (Surface *newscr, int originx, int originy)
{
    double rest_time = 0.5;

    double velx = -originx / rest_time;
    double vely = -originy / rest_time;

    double origx = originx;
    double origy = originy;

    Screen *scr = GetScreen();
    GC scrgc(scr->get_surface());
        
    while (rest_time > 0)
    {
        Uint32 otime = SDL_GetTicks();

        Rect r(static_cast<int>(origx),
               static_cast<int>(origy),
               scr->width(), scr->height());
        blit (scrgc, r.x, r.y, newscr);

        scr->update_rect(r);
        scr->flush_updates();

        double dt = (SDL_GetTicks()-otime)/1000.0;
        if (dt > rest_time)
            dt = rest_time;
        rest_time -= dt;
        origx += velx * dt;
        origy += vely * dt;
    }
}

void
video::FX_Push (Surface *newscr, int originx, int originy)
{
    double rest_time = 0.7;

    double velx = -2 * originx / rest_time;
    double vely = -2 * originy / rest_time;

    double accx = -0.5*velx/rest_time;
    double accy = -0.5*vely/rest_time;

    double x = originx;
    double y = originy;

    Screen *scr = GetScreen();
    GC scrgc(scr->get_surface());

    Surface *oldscr = Duplicate(scr->get_surface());

    Uint32 otime = SDL_GetTicks();
    double t=0;
    while (rest_time > 0)
    {
        double dt = (SDL_GetTicks()-otime)/1000.0;
        otime = SDL_GetTicks();
        if (dt > rest_time)
            dt = rest_time;
        rest_time -= dt;
        t+=dt;

        x = (accx*t + velx)*t + originx;
        y = (accy*t + vely)*t + originy;

        blit (scrgc, (int)x-originx, (int)y, oldscr);
        blit (scrgc, (int)x, (int)y-originy, oldscr);
        blit (scrgc, (int)x-originx, (int)y-originy, oldscr);

        blit (scrgc, (int)x, (int) y, newscr);

        scr->update_all();
        scr->flush_updates();
    }
    delete oldscr;
    blit(scrgc, 0,0, newscr);
    scr->update_all();
    scr->flush_updates();
}

void
video::ShowScreen (TransitionModes tm, Surface *newscr)
{
    int scrw = GetScreen()->width();
    int scrh = GetScreen()->height();

    switch (tm) {
    case TM_RANDOM:
	break;
    case TM_FADEOUTIN:
	break;
    case TM_SQUARES:
	break;
    case TM_FLY_N: FX_Fly (newscr, 0, -scrh); break;
    case TM_FLY_S: FX_Fly (newscr, 0, +scrh); break;
    case TM_FLY_E: FX_Fly (newscr, +scrw, 0); break;
    case TM_FLY_W: FX_Fly (newscr, -scrw, 0); break;

    case TM_PUSH_RANDOM:
        {
            int xo=0, yo=0;
            while (xo==0 && yo==0) {
                xo = enigma::IntegerRand(-1,1)*scrw;
                yo = enigma::IntegerRand(-1,1)*scrh;
            }
            FX_Push (newscr, xo, yo);
        }
        break;
    case TM_PUSH_N: FX_Push (newscr, 0, -scrh); break;
    case TM_PUSH_S: FX_Push (newscr, 0, +scrh); break;
    case TM_PUSH_E: FX_Push (newscr, +scrw, 0); break;
    case TM_PUSH_W: FX_Push (newscr, -scrw, 0); break;
    default:
        break;
    }
}

void
video::Screenshot(const char *fname)
{
    SDL_Surface *s = GetScreen()->get_surface()->get_surface();
    SDL_SaveBMP(s, fname);
    enigma::Log << "screenshot: "<<s->w << "x"<<s->h<<endl;
}
