/*
 * Copyright (C) 2002,2003 Daniel Heck
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: d_models.hh,v 1.11 2003/07/03 21:46:32 dheck Exp $
 */
#ifndef D_MODELS_HH
#define D_MODELS_HH

#include <vector>
#include <string>
#include <cassert>

//----------------------------------------
// Image
//----------------------------------------
namespace display
{
    using std::string;
    using std::vector;

    class Image {
    public:
        // Constructors.
	Image(px::Surface *sfc)
        : surface(sfc), rect(surface->size()), refcount(1)
	{}

	Image(px::Surface *sfc, const px::Rect &r)
        : surface(sfc), rect(r), refcount(1)
        {}

        // Destructor.
        ~Image() {
	    // `surface' need not be deleted; this is handled in the
	    // global image cache.
	}

        // Functions.
        void incref() { ++refcount; }
        void decref() {if (--refcount == 0) delete this;}

        void draw(px::GC &gc, int x, int y) {
            blit(gc, x, y, surface, rect);
        }

	// Variables.
	px::Surface *surface;
	px::Rect rect;          // location of image inside surface
	int refcount;
    };

//----------------------------------------
// ImageModel
//----------------------------------------

    class ImageModel : public Model {
	Image* image;
	int xoff, yoff;         // relative origin of the image
    public:
	ImageModel(Image *i, int xo, int yo)
        : image(i), xoff(xo), yoff(yo) {
            image->incref();
        }
	ImageModel(Surface *s, int xo, int yo)
        : image(new Image(s)), xoff(xo), yoff(yo)
        {}
	ImageModel(Surface *s, const Rect &r, int xo, int yo)
        : image(new Image(s, r)), xoff(xo), yoff(yo)
        {}
	~ImageModel() { image->decref(); }
	void draw(px::GC &gc, int x, int y)
        {
	    assert(image);
            image->draw(gc, x+xoff, y+yoff);
	}

        Image *get_image() { return image; }

	Model *clone() { return new ImageModel(image, xoff, yoff); }
    };

//----------------------------------------
// ShadowModel
//----------------------------------------
    class ShadowModel : public Model {
    public:
        ShadowModel(Model *m, Model *sh) {model=m; shade=sh;}
        ~ShadowModel() { delete model; delete shade; }

        // Model interface

        virtual void expose (ModelLayer *ml, const px::V2 &pos) {
            model->expose(ml, pos);
            shade->expose(ml, pos);
        }
        virtual void remove (ModelLayer *ml) {
            shade->remove(ml);
            model->remove(ml);
        }

	void set_callback(ModelCallback *cb) { model->set_callback(cb); }
	void reverse() { model->reverse(); shade->reverse(); }
        void draw(px::GC &gc, int x, int y) {
            model->draw(gc,x,y);
        }
        void draw_shadow(px::GC &gc, int x, int y) {
            shade->draw(gc,x,y);
        }
        Model *get_shadow() const { return shade; }
        Model *clone() {
            return new ShadowModel(model->clone(), shade->clone());
        }

    private:
        Model *model, *shade;
    };

//----------------------------------------------------------------------
// CompositeModel
//
// This kind of model consists of two layers: one background layer and
// some foreground layer that is drawn over the background.  Both
// layers may be animated.  This model make it easy to compose complex
// animations from simple ones; see models-2d.lua for a few examples.
//----------------------------------------------------------------------
    class CompositeModel : public Model {
        Model *bg, *fg;
    public:
        CompositeModel(Model *b, Model *f) : bg(b), fg(f) {}
        ~CompositeModel() {
            delete bg; delete fg;
        }

        // Animation interface
        void set_callback(ModelCallback *cb) {
            bg->set_callback(cb);
            fg->set_callback(cb);
        }
        void reverse() {
            bg->reverse();
            fg->reverse();
        }

        // Model interface
	Model *get_shadow() const { return bg->get_shadow(); }
        virtual void expose (ModelLayer *ml, const px::V2 &pos) {
            fg->expose (ml, pos);
            bg->expose (ml, pos);
        }
        virtual void remove (ModelLayer *ml) {
            fg->remove (ml);
            bg->remove (ml);
        }
        void draw(px::GC &gc, int x, int y) {
            bg->draw(gc,x,y);
            fg->draw(gc,x,y);
        }
        void draw_shadow(px::GC &gc, int x, int y) {
            bg->draw_shadow(gc,x,y);
        }
        Model *clone() {
            return new CompositeModel(bg->clone(), fg->clone());
        }
    };

//----------------------------------------
// RandomModel
//----------------------------------------
    class RandomModel : public Model {
        std::vector<std::string> modelnames;
    public:
        void add_model(const std::string &name) {modelnames.push_back(name);}

        // Model interface
        Model *clone();
    };

//----------------------------------------
// Alias
//----------------------------------------
    class AliasModel : public Model {
        string name;
    public:
        AliasModel(const string &modelname) : name(modelname) {}
        Model *clone();
    };

//----------------------------------------
// Animation
//----------------------------------------
    class AnimFrame : public px::Nocopy {
    public:
        AnimFrame(Model *m, double dur)
        : model(m), duration(dur)
        {}
        ~AnimFrame() { delete model; }
        Model *model;
        double  duration;
    };

    class AnimRep {
    public:
        AnimRep(bool l) : loop(l), refcount(1) {}
        ~AnimRep() { delete_sequence(frames.begin(), frames.end()); }
        vector<AnimFrame*>  frames;
        bool                loop;
        int                 refcount;
    };

    class Anim2d : public Model, public px::Nocopy {
    public:
        Anim2d(bool loop) : rep(new AnimRep(loop)) {}
        ~Anim2d();
        void set_callback(ModelCallback *cb) { callback = cb; }

        void add_frame(Model *m, double duration);

        /*
        ** Model interface
        */
        void draw(px::GC &gc, int x, int y);
        void draw_shadow(px::GC &gc, int x, int y);
        Model *clone() { return new Anim2d(rep); }
        void reverse() { reversep = !reversep; }

        void expose (ModelLayer *ml, const px::V2 &pos);
        void update (ModelLayer *ml, double dtime) {
            tick (dtime);
            Rect r;
            if (has_changed(r)) {
                ml->mark_redraw_area (r);
            }
        }
        void remove (ModelLayer *ml);

        void tick(double dtime);
        bool has_changed(Rect &changed_region);
        bool is_garbage() const { return finishedp; }

        void move (const px::V2 &newpos) {
            posx = newpos[0];
            posy = newpos[1];
        }
    private:
        Anim2d(AnimRep *r);

        /*
        ** Variables
        */
        AnimRep *rep;
        unsigned curframe;      // Current frame number.
        double  frametime;      // Elapsed time since frame was activated
        bool    finishedp;      // Animation has finished
        bool    changedp;       // Model state has changed since last redraw
        bool    reversep;       // Play the animation in reverse direction

        double posx, posy;      // World coordinates of the sprite
        bool is_sprite;         // False if item/stone animation
        ModelCallback *callback;
    };
}

#endif
