%  Copyright (C) 2002-2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

\begin{code}
module RepoPrefs ( add_to_preflist, get_preflist, set_preflist,
                   readFileQuick,
                   lastrepo, set_lastrepo,
                   get_prefval, set_prefval, change_prefval,
                   def_prefval,
                   write_default_prefs,
                   boring_file_filter, darcsdir_filter,
                   FileType(..), filetype_function,
                 ) where

import IO
import Directory
import Monad
import RegexString
\end{code}

\section{prefs}

The \verb!_darcs! directory will also typically contain a \verb!prefs!
directory.  This directory exists simply to hold user configuration
settings specific to this repository.  This directory may be freely
deleted, and its contents are intended to be modifiable by the user,
although a mistake in such a modification may cause darcs to behave
strangely or cause your cat to cough up a hairball.

\input{ArgumentDefaults.lhs}

\begin{code}
write_default_prefs = sequence_ [default_boring, default_binaries]
\end{code}

\subsection{repos}
The \verb!_darcs/prefs/repos! file contains a list of repositories you have
pulled from or pushed to, and is used for autocompletion of pull and push
commands in bash.  Feel free to delete any lines from this list that might
get in there, or to delete the file as a whole.

\subsection{boring}
The \verb!_darcs/prefs/boring! file may contain a list of regular
expressions describing files, such as object files, that you do not expect
to add to your project.  As an example, the boring file that I use with
my darcs repository is:
\begin{verbatim}
\.hi$
\.o$
^\.[^/]
^_
~$
(^|/)CVS($|/)
\end{verbatim}
The last line is actually irrelevant because I don't use CVS, but if I did
it would be handy.  You may want to have the boring file under version
control.  To do this you can use darcs setpref to set the value
``boringfile'' to the name of your desired boring file (e.g. ``darcs
setpref boringfile ./.boring'', where the .boring is a file that has been
darcs added to your repository).  You could also use this to configure your
repo to work with a global boring file, something like
``/etc/darcs/boring''.

\begin{code}
default_boring = set_preflist "boring" ["\\.hi$","\\.o$","(^|/)CVS($|/)","~$",
                                        "^\\.[^/]", "^_"]

fixregex ('^':r) = '^':r
fixregex r = ".*"++r
darcsdir_filter :: [FilePath] -> [FilePath]
darcsdir_filter = filter (not.is_darcsdir)
is_darcsdir ('.':'/':f) = is_darcsdir f
is_darcsdir "." = True
is_darcsdir "" = True
is_darcsdir ".." = True
is_darcsdir "../" = True
is_darcsdir "_darcs" = True
is_darcsdir ('_':'d':'a':'r':'c':'s':'/':_) = True
is_darcsdir _ = False
boring_file_filter :: IO ([FilePath] -> [FilePath])
boring_file_filter = do
    borefile <- def_prefval "boringfile" "_darcs/prefs/boring"
    bores <- (liftM lines $ readFile borefile) `catch`
             (\_-> return [])
    return $ actual_boring_file_filter $ map (mkRegex.fixregex) bores
actual_boring_file_filter :: [Regex] -> [FilePath] -> [FilePath]
actual_boring_file_filter rs fs =
    filter (abf (not.is_darcsdir) rs) fs
abf fi (r:rs) = abf (\f -> fi f && matchRegex r f == Nothing) rs
abf fi [] = fi
\end{code}

\subsection{binary}
The \verb!_darcs/prefs/binaries! file may contain a list of regular
expressions describing files that should be treated as binary files rather
than text files.  You probably will want to have the binaries file under
version control.  To do this you can use darcs setpref to set the value
``binariesfile'' to the name of your desired binaries file (e.g. ``darcs
setpref binariesfile ./.binaries'', where .binaries is a file that has been
darcs added to your repository).

\begin{code}
data FileType = BinaryFile | TextFile
                deriving (Eq)
default_binaries =
    set_preflist "binaries" ["\\.png$","\\.gz$","\\.pdf$","\\.jpg$",
                             "\\.gif$", "\\.tar$","\\.bz2"]

filetype_function :: IO (FilePath -> FileType)
filetype_function = do
    binsfile <- def_prefval "binariesfile" "_darcs/prefs/binaries"
    bins <- (liftM lines $ readFile binsfile) `catch`
             (\e-> if isDoesNotExistError e then return [] else ioError e)
    let isbin f = or $ map (\r-> matchRegex (mkRegex $ fixregex r) f /= Nothing) bins
        ftf f = if isbin f then BinaryFile else TextFile
        in
        return ftf
\end{code}

\begin{code}
add_to_preflist :: String -> String -> IO ()
get_preflist :: String -> IO [String]
set_preflist :: String -> [String] -> IO ()

lastrepo :: [String] -> IO [String]
set_lastrepo :: String -> IO ()
\end{code}

\begin{code}
add_to_preflist p s = do
  hasprefs <- doesDirectoryExist "_darcs/prefs/"
  unless hasprefs $ createDirectory "_darcs/prefs/"
  pl <- get_preflist p
  writeFile ("_darcs/prefs/"++p) $ unlines $ add_to_list s pl
get_preflist p = do
  hasprefs <- doesFileExist ("_darcs/prefs/"++p)
  if hasprefs
    then liftM lines $ readFileQuick ("_darcs/prefs/"++p)
    else return []
set_preflist p ls = do
  haspref <- doesDirectoryExist "_darcs/prefs/"
  if haspref
     then writeFile ("_darcs/prefs/"++p) (unlines ls)
     else return ()

add_to_list s [] = [s]
add_to_list s (s':ss) = if s == s' then (s:ss) else s': add_to_list s ss
\end{code}

\begin{code}
set_prefval :: String -> String -> IO ()
get_prefval :: String -> IO (Maybe String)
def_prefval :: String -> String -> IO String
def_prefval p d = do
  pv <- get_prefval p
  case pv of Nothing -> return d
             Just v -> return v
get_prefval p =
    do pl <- get_preflist "prefs"
       case map snd $ filter ((==p).fst) $ map (break (==' ')) pl of
           [val] -> return $ Just $ tail val
           _ -> return Nothing
set_prefval p v = do pl <- get_preflist "prefs"
                     set_preflist "prefs" $
                       filter ((/=p).fst.(break (==' '))) pl ++ [p++" "++v]
change_prefval p f t =
    do pl <- get_preflist "prefs"
       ov <- get_prefval p
       newval <- case ov of
                 Nothing -> return t
                 Just old -> if old == f then return t else return old
       set_preflist "prefs" $
                    filter ((/=p).fst.(break(==' '))) pl ++ [p++" "++newval]
\end{code}

\begin{code}
lastrepo [] = do last <- get_preflist "lastrepo"
                 case last of [_] -> return last
                              _ -> return []
lastrepo r = return r
set_lastrepo r = do set_preflist "lastrepo" [r]
                    add_to_preflist "repos" r
\end{code}

\begin{code}
readFileQuick :: FilePath -> IO String
readFileQuick f = do
  h <- openFile f ReadMode
  out <- readHandleQuick h
  hClose h
  return out
readHandleQuick :: Handle -> IO String
readHandleQuick h = do
  done <- hIsEOF h
  if done
     then return ""
     else do
          c <- hGetChar h
          rest <- readHandleQuick h
          return (c:rest)
\end{code}
