
module Posix where


import Foreign
import Foreign.Ptr
import Foreign.C.Types
import Foreign.C.String

import Monad

foreign import ccall "_stat" c_stat :: CString -> Ptr () -> IO Int


type FileOffset = CInt
 
data FileStatus = FileStatus {
    accessTime :: EpochTime,
    modificationTime :: EpochTime,
    fileSize :: FileOffset
  }

peek_int p o = do
    i <- peek (castPtr (plusPtr p o)) :: IO CInt
    return (fromIntegral i)

getFileStatus     :: FilePath -> IO FileStatus
getFileStatus path = do
    p <- mallocBytes 36
    r <- path `withCString` (`c_stat` p)
    when (r /= 0) $ print ("stat error: ", r)
    sz <- peek_int p 20
    at <- peek_int p 24
    mt <- peek_int p 28
    free p
    return (FileStatus (EpochTime (fromIntegral at)) (EpochTime (fromIntegral mt)) sz)

foreign import ccall "sys/utime.h _utime" c_utime :: CString -> Ptr a -> IO CInt

setFileTimes :: FilePath -> EpochTime -> EpochTime -> IO ()
setFileTimes path (EpochTime atime) (EpochTime mtime) = path `withCString` \s -> do
    p <- mallocBytes 8
    poke (castPtr p :: Ptr Int32) (fromIntegral atime)
    poke (castPtr (plusPtr p 4) :: Ptr Int32) (fromIntegral mtime)
    c_utime s p
    free p
    return ()



foreign import ccall "time" c_ctime :: CInt -> IO CInt

epochTime :: IO EpochTime
epochTime = do
	t <- c_ctime 0
	return (EpochTime (fromIntegral t))

newtype EpochTime = EpochTime Int32
 deriving (Eq, Ord, Num, Real, Integral, Enum, Show)


createLink p1 p2 = do
	s <- readFile p1
	writeFile p2 s
	return ()

foreign import ccall "stdlib.h _sleep" c_sleep :: CULong -> IO ()

sleep n = c_sleep (1000 * fromIntegral n)


-- Dummy implementation of signals

data Handler = Default | Ignore | Catch (IO ())
type Signal = Int
type SignalSet = ()

installHandler :: Signal -> Handler -> Maybe SignalSet -> IO Handler
installHandler _ _ _ = return Ignore

sigINT :: Signal
sigINT = 0

