%  Copyright (C) 2002-2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\begin{code}
module PatchInfo ( PatchInfo, patchinfo, invert_name,
                   make_filename, readPatchInfo, hGetPatchInfo,
                   readPatchInfoPS,
                   just_name, repopatchinfo, RepoPatchInfo,
                   human_friendly,
                 ) where
import Char ( isSpace )
import Time
import Text.Html
import IO
import Monad ( when )
import FastPackedString

data RepoPatchInfo = RPI String PatchInfo
repopatchinfo r pi = RPI r pi

data PatchInfo = PatchInfo String String String [String] Bool
                 deriving (Eq,Ord)

patchinfo :: String -> String -> String -> [String] -> PatchInfo
patchinfo date name author log = PatchInfo date name author log False
\end{code}

\section{Patch info formatting}


\begin{code}
invert_name (PatchInfo d n a l inv) = PatchInfo d n a l (not inv)
\end{code}

\begin{code}
just_name :: PatchInfo -> String
just_name (PatchInfo d n a l inv) = n

human_friendly :: PatchInfo -> String
human_friendly (PatchInfo d n a [] inv) =
    d ++ "  " ++ a ++ "\n" ++ hfn n ++ "\n"
human_friendly (PatchInfo d n a l inv) =
    d ++ "  " ++ a ++ "\n" ++ hfn n ++ "\n" ++ unlines (map ("  "++) l)
hfn n = if take 4 n == "TAG " then "  tagged "++drop 4 n else "  * "++n
\end{code}

\begin{code}
make_filename :: PatchInfo -> String
make_filename (PatchInfo d n a l inv) =
    fix_up_fname (midtrunc n++"-"++a++"-"++d)

midtrunc :: String -> String
midtrunc s
    | length s < 73 = s
    | otherwise = (take 40 s)++"..."++(reverse $ take 30 $ reverse s)
fix_up_fname :: String -> String
fix_up_fname = map munge_char

munge_char :: Char -> Char
munge_char '*' = '+'
munge_char '?' = '2'
munge_char '>' = '7'
munge_char '<' = '2'
munge_char ' ' = '_'
munge_char '"' = '~'
munge_char '`' = '.'
munge_char '\'' = '.'
munge_char '/' = '1'
munge_char '\\' = '1'
munge_char '!' = '1'
munge_char ':' = '.'
munge_char ';' = ','
munge_char '{' = '~'
munge_char '}' = '~'
munge_char '(' = '~'
munge_char ')' = '~'
munge_char '[' = '~'
munge_char ']' = '~'
munge_char '=' = '+'
munge_char '#' = '+'
munge_char '%' = '8'
munge_char '&' = '6'
munge_char '@' = '9'
munge_char '|' = '1'
munge_char  c  =  c
\end{code}


\begin{code}
instance  HTML RepoPatchInfo  where
    toHtml = htmlPatchInfo
instance  Show PatchInfo  where
    show = showPatchInfo
instance  Read PatchInfo  where
    readsPrec _           = readPatchInfo
\end{code}

\paragraph{Patch info}
Patch is stored between square brackets.
\begin{verbatim}
[ <patch name>
<patch author>*<patch date>
 <patch log (may be empty)> (indented one)
 <can have multiple lines in patch log,>
 <as long as they're preceded by a space>
 <and don't end with a square bracket.>
]
\end{verbatim}
\begin{code}
myunlines [] = ""
myunlines (s:ss) = "\n"++s++ myunlines ss
-- note that below I assume the name has no newline in it.
showPatchInfo (PatchInfo ct name author log False) =
  "[" ++ name ++"\n"++
  author ++ "**" ++ ct ++
  myunlines (map (' ':) log) ++
  "] "
showPatchInfo (PatchInfo ct name author log True) =
    "[" ++ name ++"\n"++
    author ++ "*-" ++ ct ++
    myunlines (map (' ':) log) ++
    "] "
readPatchInfo :: ReadS PatchInfo
readPatchInfo s =
  [(PatchInfo ct name author log False, zz) |
       (name, w) <- mygetline $ skip_char '[' s,
       (author, x) <- get_line_or_tochar '*' w,
       (ct, y) <- get_line_or_tochar ']' $ skip_char '*' $ skip_char '*' x,
       (log, z) <- lines_starting_with_ending_with ' ' ']' y,
       ("]", zz) <- mylex z]
  ++
  [(PatchInfo ct name author log True, zz) |
       (name, w) <- mygetline $ skip_char '[' s,
       (author, x) <- get_line_or_tochar '*' w,
       (ct, y) <- get_line_or_tochar ']' $ skip_char '-' $ skip_char '*' x,
       (log, z) <- lines_starting_with_ending_with ' ' ']' y,
       ("]", zz) <- mylex z]
readPatchInfoPS :: PackedString -> Maybe (PatchInfo,PackedString)
readPatchInfoPS s =
    if headPS (dropWhilePS Char.isSpace s) /= '[' -- ]
    then Nothing
    else case breakPS (=='\n') $ tailPS $ dropWhilePS Char.isSpace s of
         (name,s') ->
             case breakPS (=='*') $ tailPS s' of
             (author,s2) ->
                 case breakPS (\c->c==']'||c=='\n') $ dropPS 2 s2 of
                 (ct,s''') ->
                     case lines_starting_with_ending_withPS ' ' ']'$dnPS s''' of
                     Just (log, s4) ->
                         if indexPS s2 1 == '*'
                         then Just (PatchInfo (unpackPS ct) (unpackPS name)
                                  (unpackPS author) (map unpackPS log) False, s4)
                         else Just (PatchInfo (unpackPS ct) (unpackPS name)
                                  (unpackPS author) (map unpackPS log) True, s4)
dnPS s = if nullPS s then s
         else if headPS s == '\n' then tailPS s else s
hGetPatchInfo h = do hSkipWhite h
                     nex <- hLookAhead h
                     when (nex /= '[') $ ioError $ userError "bad start to a PatchInfo" -- ]
                     hGetChar h
                     name <- hGetLine h
                     author <- hGetLineOrToChar h '*'
                     hSkipChar h '*'
                     sep <- hGetChar h
                     ct <- hGetLineOrToChar h ']'
                     log <- hGetLinesStartingWithEndingWith h ' ' ']'
                     hSkipWhite h
                     nc <- hLookAhead h
                     hSkipChar h ']'
                     case sep of
                        '*' -> return $ PatchInfo ct  name author log False
                        '-' -> return $ PatchInfo ct  name author log True
hGetLinesStartingWithEndingWith h c e = do
    nex <- hLookAhead h
    if nex /= c then return []
       else do hGetChar h
               l <- hGetLine h
               if (e `elem` l)
                  then do hSeek h RelativeSeek $
                                fromIntegral ((-1) - (length $ dropWhile (/=e) l))
                          return [takeWhile (/=e) l]
                  else do rest <- hGetLinesStartingWithEndingWith h c e
                          return $ l : rest
hSkipWhite h = do nex <- hLookAhead h
                  if Char.isSpace nex then do hGetChar h
                                              hSkipWhite h
                                      else return ()
hSkipChar :: Handle -> Char -> IO ()
hSkipChar h c = do nex <- hLookAhead h
                   if nex == c then do hGetChar h
                                       return ()
                               else ioError $ userError "bad char"
hGetLineOrToChar h c = do nex <- hLookAhead h
                          if nex == c || nex == '\n'
                             then if nex == '\n'
                                  then do hGetChar h
                                          return ""
                                  else return ""
                             else do ch <- hGetChar h
                                     rest <- hGetLineOrToChar h c
                                     return $ ch : rest
\end{code}

\begin{code}
skip_char :: Char -> String -> String
skip_char tc [] = []
skip_char tc (c:cs)
    | tc == c = cs
    | Char.isSpace c = skip_char tc cs
    | otherwise = []
get_line_or_tochar :: Char -> ReadS String
get_line_or_tochar _ [] = []
get_line_or_tochar tc s =
    [(w, rest)]
    where (w, s') = break (\c -> c == '\n' || c == tc) s
          rest = if s' == "" then ""
                 else if head s' == tc then s'
                      else tail s'
\end{code}

\begin{code}
isntSpace c = not (isSpace c)
mylex              :: ReadS String
mylex s          =  case dropWhile Char.isSpace s of
                      "" -> [("","")]
                      s' -> [(w, s'')]
                            where (w, s'') = break Char.isSpace s'

mygetline          :: ReadS String
mygetline [] = []
mygetline s = [(w, if s' == "" then "" else tail s')]
    where (w, s') = break (\c -> c == '\n') s

lines_starting_with_ending_withPS :: Char -> Char -> PackedString
                                  -> Maybe ([PackedString],PackedString)
lines_starting_with_ending_withPS st en s =
    if headPS s /= st then Just ([],tailPS s)
    else case breakPS (\c->c==en||c=='\n') $ tailPS s of
         (l,r) ->
           if headPS r == en then Just ([l],tailPS r)
           else
             case lines_starting_with_ending_withPS st en $ tailPS r of
             Just (ls,rest) -> Just (l:ls,rest)

lines_starting_with_ending_with :: Char -> Char -> ReadS [String]
lines_starting_with_ending_with = read_more_lines_starting_with_ending_with []
read_more_lines_starting_with_ending_with ::
    [String] -> Char -> Char -> ReadS [String]
read_more_lines_starting_with_ending_with lns tc ec (c:cs) =
    if c /= tc then [(lns, c:cs)]
    else read_more_lines_starting_with_ending_with (lns ++ [line]) tc ec s'
         where [(line, s')] = get_line_or_tochar ec cs

lines_starting_with           :: Char -> ReadS [String]
lines_starting_with = read_more_lines_starting_with []

read_more_lines_starting_with :: [String] -> Char -> ReadS [String]
read_more_lines_starting_with lns tc "" = [(lns, "")]
read_more_lines_starting_with lns tc (c:cs) =
    if c /= tc then [(lns, c:cs)]
    else read_more_lines_starting_with (lns ++ [line]) tc s'
         where [(line, s')] = mygetline cs
\end{code}

\begin{code}
htmlPatchInfo :: RepoPatchInfo -> Html
htmlPatchInfo (RPI r (PatchInfo ct name author log _)) =
    toHtml $ (td << patch_link r (PatchInfo ct name author log True)) `above`
               ((td ! [align "right"] << mail_link author) `beside` (td << ct))

patch_link :: String -> PatchInfo -> Html
patch_link r (PatchInfo ct name author log _) =
    toHtml $ hotlink
               ("darcs?"++r++"**"++make_filename (PatchInfo ct name author log True))
               [toHtml name]
mail_link :: String -> Html
mail_link email = toHtml $ hotlink ("mailto:"++email) [toHtml email]
\end{code}
