%  Copyright (C) 2002-2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\chapter{Theory of patches}
\label{Patch}

\newtheorem{thm}{Theorem}
\newtheorem{dfn}{Definition}

\section{Background}

I think a little background on the author is in order.  I am a physicist,
and think like a physicist.  The proofs and theorems given here are what I
would call ``physicist'' proofs and theorems, which is to say that while
the proofs may not be rigorous, they are practical, and the theorems are
intended to give physical insight.  It would be great to have a
mathematician work on this, but I am not a mathematician, and don't care
for math.

From the beginning of this theory, which originated as the result of a
series of email discussions with Tom Lord, I have looked at patches as
being analagous to the operators of quantum mechanics.  I include in this
appendix footnotes explaining the theory of patches in terms of the theory
of quantum mechanics.  I know that for most people this won't help at all,
but many of my friends (and as I write this all three of darcs' users) are
physicists, and this will be helpful to them.  To nonphysicists, perhaps it
will provide some insight into how at least this physicist things.

\section{Introduction}

\begin{code}
module Patch ( Patch, rmfile, addfile, rmdir, adddir, move,
               hunk, tokreplace, join_patches, namepatch,
               binary,
               infopatch, changepref,
               is_similar, is_addfile,
               hPutPatch, writePatch, hGetPatch,
               readPatch, read_one_patch, invert, commute, merge,
               readPatchPS, prop_readPS_show,
               canonize, reorder, submerge_in_dir, flatten,
               apply_to_slurpy, patchname, unjoin_patches,
               kludge_to_slurpy,
               LineMark(AddedLine, RemovedLine, AddedRemovedLine, None),
               MarkedUpFile, markup_file,
               MarkedUpRepo, markup_repo, clean_markedup_repo,
               prop_inverse_composition, prop_commute_twice,
               prop_inverse_valid, prop_other_inverse_valid,
               prop_commute_equivalency, prop_commute_either_order,
               prop_commute_either_way, prop_merge_is_commutable_and_correct,
               prop_merge_is_swapable, prop_merge_valid,
               prop_glump_order_independent,
               prop_glump_seq_merge, prop_glump_seq_merge_valid,
               prop_glump_three_merge, prop_glump_three_merge_valid,
               prop_unravel_three_merge, prop_unravel_seq_merge,
               prop_unravel_order_independent,
               prop_read_show, quickmerge, check_patch, check_a_patch,
               test_patch, adddeps, getdeps,
               list_conflicted_files, list_touched_files,
               resolve_conflicts,
               -- for Population
               DirMark(..), patchChanges, applyToPop,
             ) where
import Char ( isSpace, isDigit )
import IO
import Lcs
import SlurpDirectory
import PatchInfo
import PatchCheck
import Debug.QuickCheck
import Monad
import List ( sort, nub )
import IOExts
import RegexString( mkRegex, matchRegex )
import FastPackedString
import Data.Char ( ord, chr )
import Maybe ( catMaybes, fromJust )
import RepoPrefs ( change_prefval )
import PopulationData ( Population(..), Info(..), PopTree(..), DirMark(..),
                        setState )

data Patch = NamedP PatchInfo [PatchInfo] Patch
           | Move String String
           | DP String DirPatchType
           | FP String FilePatchType
           | Split [Patch]
           | ComP [Patch]
           | Merger Bool String Patch Patch
           | ChangePref String String String
             deriving (Eq,Ord)
instance Arbitrary Patch where
    arbitrary = sized arbpatch
data FilePatch = FilePatch String FilePatchType
                 deriving (Eq,Ord)

data FilePatchType = RmFile | AddFile
                   | Hunk Int [PackedString] [PackedString]
                   | TokReplace String String String
                   | Binary PackedString PackedString
                     deriving (Eq,Ord)

data DirPatch = DirPatch String DirPatchType
                deriving (Eq,Ord)
data DirPatchType = RmDir | AddDir
                    deriving (Eq,Ord)
\end{code}

\begin{code}
addfile :: String -> Patch
rmfile :: String -> Patch
adddir :: String -> Patch
rmdir :: String -> Patch
hunk :: String -> Int -> [PackedString] -> [PackedString] -> Patch
join_patches :: [Patch] -> Patch
unjoin_patches :: Patch -> Maybe [Patch]
namepatch :: String -> String -> String -> [String] -> Patch -> Patch
infopatch :: PatchInfo -> Patch -> Patch
adddeps :: Patch -> [PatchInfo] -> Patch
getdeps :: Patch -> [PatchInfo]

addfile f = FP (n_fn f) AddFile
rmfile f = FP (n_fn f) RmFile
adddir d = DP (n_fn d) AddDir
rmdir d = DP (n_fn d) RmDir
move f f' = Move (n_fn f) (n_fn f')
changepref p f t = ChangePref p f t
hunk f line old new = FP (n_fn f) (Hunk line old new)
tokreplace f tokchars old new = FP (n_fn f) (TokReplace tokchars old new)
binary f old new = FP (n_fn f) $ Binary old new
join_patches ps = ComP ps
unjoin_patches (ComP ps) = Just ps
unjoin_patches _ = Nothing
namepatch date name author log p =
    NamedP (patchinfo date name author log) [] p
infopatch pi p = NamedP pi [] p
adddeps (NamedP pi ds p) ds' = NamedP pi (ds++ds') p
getdeps (NamedP _ ds _) = ds

patchname :: Patch -> Maybe String
patchname (NamedP i _ _) = Just $ make_filename i
patchname _ = Nothing
\end{code}

\begin{code}
hunkgen = do
  i <- frequency [(1,choose (0,5)),(1,choose (0,35)),
                  (2,return 0),(3,return 1),(2,return 2),(1,return 3)]
  j <- frequency [(1,choose (0,5)),(1,choose (0,35)),
                  (2,return 0),(3,return 1),(2,return 2),(1,return 3)]
  if i == 0 && j == 0 then hunkgen
    else liftM4 hunk filepathgen linenumgen
                (sequence [filelinegen|i<-[1..i::Int]])
                (sequence [filelinegen|i<-[1..j::Int]])
tokreplacegen = do
  f <- filepathgen
  o <- tokengen
  n <- tokengen
  if o == n
     then return $ tokreplace f "A-Za-z" "old" "new"
     else return $ tokreplace f "A-Za-z_" o n
twofilegen p = do
  n1 <- filepathgen
  n2 <- filepathgen
  if n1 /= n2 && (check_a_patch $ p n1 n2)
     then return $ p n1 n2
     else twofilegen p
chprefgen = do
  f <- oneof [return "color", return "movie"]
  o <- tokengen
  n <- tokengen
  if o == n then return $ changepref f "old" "new"
            else return $ changepref f o n
simplepatchgen = frequency [(1,liftM addfile filepathgen),
                            (1,liftM adddir filepathgen),
                            (1,liftM3 binary filepathgen arbitrary arbitrary),
                            (1,twofilegen move),
                            (1,tokreplacegen),
                            (1,chprefgen),
                            (7,hunkgen)
                           ]
onepatchgen = oneof [simplepatchgen, liftM invert simplepatchgen]
norecursgen 0 = onepatchgen
norecursgen n = oneof [onepatchgen,flatcompgen n]
arbpatch 0 = onepatchgen
arbpatch n = frequency [(2,onepatchgen),
                       -- (1,compgen n),
                        (3,flatcompgen n),
                        (1,mergegen n),
                        (1,namedgen n),
                        (1,depgen n),
                        (1,onepatchgen)
                       ]
unempty :: Arbitrary a => Gen [a]
unempty = do
  as <- arbitrary
  case as of
    [] -> unempty
    _ -> return as
mergegen n = do
  p1 <- norecursgen len
  p2 <- norecursgen len
  if (check_a_patch $ join_patches [invert p1,p2]) &&
         (check_a_patch $ join_patches [invert p2,p1])
     then case merge (p2,p1) of
          Just (p2',p1') ->
              if check_a_patch $ join_patches [p1',p2']
              then return $ join_patches [p1',p2']
              else return $ join_patches [addfile "Error_in_mergegen",
                                          addfile "Error_in_mergegen",
                                          p1,p2,p1',p2']
     else mergegen n
  where len = if n < 15 then n`div`3 else 3
namedgen n =
    liftM5 namepatch unempty unempty unempty arbitrary $ arbpatch (n-1)
arbpi = liftM4 patchinfo unempty unempty unempty unempty
instance Arbitrary PatchInfo where
    arbitrary = arbpi
instance Arbitrary PackedString where
    arbitrary = liftM packString arbitrary
depgen n =
    liftM3 NamedP arbitrary arbitrary $ arbpatch (n-1)
plistgen s n
    | n <= 0 = return []
    | otherwise = do
                  next <- arbpatch s
                  rest <- plistgen s (n-1)
                  return $ next : rest
compgen n = do
    size <- choose (0,n)
    myp <- liftM join_patches $ plistgen size ((n+1) `div` (size+1))
-- here I assume we only want to consider valid patches...
    if check_a_patch myp
       then return myp
       else compgen n
flatlistgen n = sequence [ onepatchgen | i <- [1..n] ]
flatcompgen n = do
  myp <- liftM (join_patches . regularize_patches) $ flatlistgen n
  if check_a_patch myp
     then return myp
     else flatcompgen n
linenumgen = frequency [(1,return 1), (1,return 2), (1,return 3),
                    (3,liftM (\n->1+abs n) arbitrary) ]
tokengen = oneof [return "hello", return "world", return "this",
                  return "is", return "a", return "silly",
                  return "token", return "test"]
toklinegen = liftM unwords $ sequence [tokengen|i<-[1..5]]
filelinegen = liftM packString $
              frequency [(1,arbitrary),(5,toklinegen),
                         (1,return ""), (1,return "{"), (1,return "}") ]
filepathgen = liftM fixpath badfpgen
fixpath "" = "test"
fixpath p = fpth p
fpth ('/':'/':cs) = fpth ('/':cs)
fpth (c:cs) = c : fpth cs
fpth [] = []
badfpgen =  frequency [(1,return "test"), (1,return "hello"), (1,return "world"),
                       (1,arbitrary),
                       (1,liftM2 (\a b-> a++"/"++b) filepathgen filepathgen) ]
instance Arbitrary Char where
    arbitrary = oneof $ map return
                (['a'..'z']++['A'..'Z']++['1'..'9']++['0','~','.',',','-','/'])
\end{code}

\begin{code}
n_fn :: String -> String
n_fn ('.':'/':f) = n_fn f
n_fn f = '.':'/': (n_fn_helper $ reverse f)
n_fn_helper ('/':rf) = n_fn_helper rf
n_fn_helper rf = reverse rf
\end{code}

A patch describes a change to the tree.  It could be either a primitive
patch (such as a file add/remove, a directory rename, or a hunk replacement
within a file), or a compostive patch describing many such changes.  Every
patch type must satisfy the conditions described in this appendix.  The
theory of patches is independent of the data which the patches manipulate,
which is what makes it both powerful and useful, as it provides a framework
upon which one can build a revision control system in a sane manner.

Although in a sense, the defining property of any patch is that it can be
applied to a certain tree, and thus make a certain change, this change does
not wholly define the patch.  A patch is defined by a
\emph{representation}, together with a set of rules for how it behaves
(which it has in common with its patch type).  The \emph{representation} of
a patch defines what change that particular patch makes, and must be
defined in the context of a specific tree.  The theory of patches is a
theory of the many ways one can change the representation of a patch to
place it in the context of a different tree.  The patch itself is not
changed, since it describes a single change, which must be the same
regardless of its representation\footnote{For those comfortable with
quantum mechanics, think of a patch as a quantum mechanical operator, and
the representation as the basis set.  The analogy breaks down pretty
quickly, however, since an operator could be described in any complete
basis set, while a patch modifying the file {\tt foo} can only be described
in the rather small set of contexts which have a file {\tt foo} to be
modified.}.

So how does one define a tree, or the context of a patch? The simplest way
to define a tree is as the result of a series of patches applied to the
empty tree\footnote{This is very similar to the second-quantized picture,
in which any state is seen as the result of a number of creation operators
acting on the vacuum, and provides a similar set of simplifications---in
particular, the exclusion principle is very elegantly enforced by the
properties of the anti-hermitian fermion creation operators.}.  Thus, the
context of a patch consists of the set of patches that precede it.

\begin{code}
apply_to_slurpy :: Patch -> Slurpy -> Maybe Slurpy

apply_to_slurpy (NamedP n d p) s = apply_to_slurpy p s
apply_to_slurpy (Merger True g p1 p2) s =
    apply_to_slurpy (merger_equivalent $ Merger True g p1 p2) s
apply_to_slurpy (Merger False g p1 p2) s =
    apply_to_slurpy (merger_equivalent $ Merger False g p1 p2) s
apply_to_slurpy (ComP []) s = Just s
apply_to_slurpy (ComP (p:ps)) s =
  apply_to_slurpy p s >>= apply_to_slurpy (ComP ps)
apply_to_slurpy (Split []) s = Just s
apply_to_slurpy (Split (p:ps)) s =
  apply_to_slurpy p s >>= apply_to_slurpy (Split ps)

apply_to_slurpy (FP f RmFile) s = slurp_removefile f s
apply_to_slurpy (FP f AddFile) s = slurp_addfile f s
apply_to_slurpy (FP f (Hunk line old new)) s =
    slurp_modfile f (applyHunkLines line old new) s
apply_to_slurpy (FP f (TokReplace tcs old new)) s =
    slurp_modfile f (applyTokReplace tcs old new) s
apply_to_slurpy (FP f (Binary o n)) s =
    slurp_modfile f (applyBinary o n) s

apply_to_slurpy (DP d AddDir) s = slurp_adddir d s
apply_to_slurpy (DP d RmDir) s = slurp_removedir d s

apply_to_slurpy (Move f f') s = slurp_move f f' s
apply_to_slurpy (ChangePref p f t) s =
    slurp_runfunc (change_prefval p f t) s
--apply_to_slurpy _ _ = Nothing
\end{code}

\begin{code}
kludge_to_slurpy :: Patch -> Slurpy -> Maybe Patch
kludge_to_slurpy p s
    | apply_to_slurpy p s /= Nothing = Nothing
kludge_to_slurpy (ComP (p:ps)) s =
    case kludge_to_slurpy p s of
    Just p' ->
      case apply_to_slurpy p' s of
      Nothing -> error "Bad kludge in kludge_to_slurpy of composite."
      Just s' ->
        case kludge_to_slurpy (ComP ps) s' of
        Nothing -> Just $ ComP (p':ps)
        Just (ComP ps') -> Just $ ComP (p':ps')
    Nothing ->
      case apply_to_slurpy p s of
      Nothing -> error "A bad kludge in kludge_to_slurpy of composite."
      Just s' ->
        case kludge_to_slurpy (ComP ps) s' of
        Nothing -> Nothing
        Just (ComP ps') -> Just $ ComP (p:ps')
kludge_to_slurpy (Split ps) s =
    case kludge_to_slurpy (ComP ps) s of
    Nothing -> Nothing
    Just (ComP ps') -> Just (Split ps')
kludge_to_slurpy m@(Merger b g p1 p2) s =
    kludge_to_slurpy (merger_equivalent m) s
kludge_to_slurpy p s = error $ "Couldn't figure out how to kludge patch "++show p
\end{code}

\begin{code}
check_patch :: Patch -> PatchCheck Bool
check_a_patch :: Patch -> Bool
check_a_patch p = (do_check $ check_patch p) && (do_check $ check_patch $ invert p)
verbose_check_a_patch :: Patch -> Bool
verbose_check_a_patch p =
    (do_verbose_check $ check_patch p) && (do_check $ check_patch $ invert p)

check_patch (NamedP n d p) = check_patch p
check_patch (Merger True g p1 p2) = do
  --check_patch $ join_patches [invert p1,p2,invert p2,p1]
  check_patch $ merger_equivalent $ Merger True g p1 p2
check_patch (Merger False g p1 p2) = do
  check_patch $ merger_equivalent $ Merger False g p1 p2
  --check_patch $ join_patches [invert p1,p2,invert p2,p1]
check_patch (ComP []) = is_valid
check_patch (ComP (p:ps)) =
  check_patch p >> check_patch (ComP ps)
check_patch (Split []) = is_valid
check_patch (Split (p:ps)) =
  check_patch p >> check_patch (Split ps)

check_patch (FP f RmFile) = remove_file f
check_patch (FP f AddFile) =  create_file f
check_patch (FP f (Hunk line old new)) = do
    file_exists f
    sequence $ map (delete_line f line) old
    sequence $ map (insert_line f line) (reverse new)
    is_valid
check_patch (FP f (TokReplace t old new)) =
    modify_file f (try_tok_possibly t old new)
-- note that the above isn't really a sure check, as it leaves PSomethings
-- and PNothings which may have contained new...
check_patch (FP f (Binary o n)) = do
    file_exists f
    sequence $ map (delete_line f 1) (linesPS o)
    file_empty f
    sequence $ map (insert_line f 1) (reverse $ linesPS n)
    is_valid

check_patch (DP d AddDir) = create_dir d
check_patch (DP d RmDir) = remove_dir d

check_patch (Move f f') = check_move f f'
check_patch (ChangePref _ _ _) = return True

regularize_patches :: [Patch] -> [Patch]
regularize_patches ps = rpint [] ps
rpint ps [] = ps
rpint ok (p:ps) =
  if check_a_patch (join_patches $ p:ok)
  then rpint (p:ok) ps
  else rpint ok ps
\end{code}

The simplest relationship between two patches is that of ``sequential''
patches, which means that the context of the second patch (the one on the
left) consists of the first patch (on the right) plus the context of the
first patch.  The composition of two patches (which is also a patch) refers
to the patch which is formed by first applying one and then the other.  The
composition of two patches, $P_1$ and $P_2$ is represented as $P_2P_1$,
where $P_1$ is to be applied first, then $P_2$\footnote{This notation is
inspired by the notation of matrix multiplication or the application of
operators upon a Hilbert space.  In the algebra of patches, there is
multiplication (i.e. composition), which is associative but not
commutative, but no addition or subtraction.}

There is one other very useful relationship that two patches can have,
which is to be parallel patches, which means that the two patches have an
identical context (i.e. their representation applies to identical trees).
This is represented by $P_1\parallel P_2$.  Of course, two patches may also
have no simple relationship to one another.  In that case, if you want to
do something with them, you'll have to manipulate them with respect to
other patches until they are either in sequence or in parallel.

The most fundamental and simple property of patches is that they must be
invertible.  The inverse of a patch is decribed by: $P^{ -1}$.  In the
darcs implementation, the inverse is required to be computable from
knowledge of the patch only, without knowledge of its context, but that
(although convenient) is not required by the theory of patches.
\begin{dfn}
The inverse of patch $P$ is $P^{ -1}$, which is the ``simplest'' patch for
which the composition \( P^{ -1} P \) makes no changes to the tree.
\end{dfn}
Using this definition, it is trivial to prove the following theorem
relating to the inverse of a composition of two patches.
\begin{thm} The inverse of the composition of two patches is
\[ (P_2 P_1)^{ -1} = P_1^{ -1} P_2^{ -1}. \]
\end{thm}
Moreover, it is possible to show that the right inverse of a patch is equal
to its left inverse.  In this respect, patches continue to be analagous to
square matrices, and indeed the proofs relating to these properties of the
inverse are entirely analagous to the proofs in the case of matrix
multiplication.  The compositions proofs can also readily be extended to
the composition of more than two patches.
\begin{code}
prop_inverse_composition :: Patch -> Patch -> Bool
prop_inverse_composition p1 p2 =
    invert (join_patches [p1,p2]) == join_patches [invert p2, invert p1]
prop_inverse_valid p1 = check_a_patch $ join_patches [invert p1,p1]
prop_other_inverse_valid p1 = check_a_patch $ join_patches [p1,invert p1]
\end{code}
\begin{code}
invert :: Patch -> Patch
invert (NamedP n d p)  = NamedP (invert_name n) (map invert_name d) (invert p)
invert (Merger b g p1 p2)  = Merger (not b) g p1 p2
invert (FP f RmFile)  = FP f AddFile
invert (FP f AddFile)  = FP f RmFile
invert (FP f (Hunk line old new))  = FP f $ Hunk line new old
invert (FP f (TokReplace t o n)) = FP f $ TokReplace t n o
invert (FP f (Binary o n)) = FP f $ Binary n o
invert (DP d RmDir) = DP d AddDir
invert (DP d AddDir) = DP d RmDir
invert (Move f f') = Move f' f
invert (ChangePref p f t) = ChangePref p t f
-- I need to see if there is a combined map-reverse, which I think would
-- be more efficient.
invert (ComP ps)  = ComP (map invert (reverse ps))
invert (Split ps) = Split (map invert (reverse ps))
\end{code}

\newcommand{\commute}{\longleftrightarrow}
\newcommand{\commutes}{\longleftrightarrow}

The first way (of only two) to change the context of a patch is by
commutation, which is the process of changing the order of two sequential
patches.
\begin{dfn}
The commutation of patches $P_1$ and $P_2$ is represented by
\[ P_2 P_1 \commutes {P_1}' {P_2}'. \]
Here $P_1'$ is intended to describe the same change as $P_1$, with the
only difference being that $P_1'$ is applied after $P_2'$ rather than
before $P_2$.
\end{dfn}
The above definition is obviously rather vague, the reason being that what
is the ``same change'' has not been defined, and we simply assume (and
hope) that the code's view of what is the ``same change'' will match those
of its human users.  The `$\commutes$' operator should be read as something
like the $==$ operator in C, indicating that the right hand side performs
identical changes to the left hand side, but the two patches are in
reversed order.  When read in this manner, it is clear that commutation
must be a reversible process, and indeed this means that commutation
\emph{can} fail, and must fail in certain cases.  For example, the creation
and deletion of the same file cannot be commuted.  When two patches fail to
commute, it is said that the second patch depends on the first, meaning
that it must have the first patch in its context (remembering that the
context of a patch is a set of patches, which is how we represent a tree).
\footnote{The fact that commutation can fail makes a huge difference in the
whole patch formalism.  It may be possible to create a formalism in which
commutation always succeeds, with the result of what would otherwise be a
commutation that fails being something like a virtual particle (which can
violate conservation of energy), and it may be that such a formalism would
allow strict mathematical proofs (whereas those used in the current
formalism are mostly only hand waving ``physicist'' proofs).  However, I'm
not sure how you'd deal with a request to delete a file that has not yet
been created, for example.  Obviously you'd need to create some kind of
antifile, which would annihilate with the file when that file finally got
created, but I'm not entirely sure how I'd go about doing this.
$\ddot\frown$ So I'm sticking with my hand waving formalism.}
\begin{code}
prop_commute_twice :: Patch -> Patch -> Property
prop_commute_twice p1 p2 =
    (does_commute p1 p2) ==> (Just (p2,p1) == (commute (p2,p1) >>= commute))
does_commute :: Patch -> Patch -> Bool
does_commute p1 p2 =
    commute (p2,p1) /= Nothing && (check_a_patch $ join_patches [p1,p2])
prop_commute_equivalency :: Patch -> Patch -> Property
prop_commute_equivalency p1 p2 =
    (does_commute p1 p2) ==>
    case commute (p2,p1) of
    Just (p1',p2') -> check_a_patch $ join_patches [p1,p2,invert p1',invert p2']
\end{code}

%I should add that one using the inversion relationship of sequential
%patches, one can avoid having to provide redundant definitions of
%commutation.
\begin{code}
prop_commute_either_way :: Patch -> Patch -> Property
prop_commute_either_way p1 p2 =
    does_commute p1 p2 ==> does_commute (invert p2) (invert p1)
\end{code}

% There is another interesting property which is that a commute's results
% can't be affected by commuting another thingamabopper.

\begin{code}
prop_commute_either_order :: Patch -> Patch -> Patch -> Property
prop_commute_either_order p1 p2 p3 =
    check_a_patch (join_patches [p1,p2,p3]) &&
    does_commute p1 (join_patches [p2,p3]) &&
    does_commute p2 p3 ==>
    case commute (p2,p1) of
    Nothing -> False
    Just (p1',p2') ->
        case commute (p3,p1') of
        Nothing -> False
        Just (p1'',p3') ->
            case commute (p3',p2') of
            Nothing -> False
            Just (p2'', p3'') ->
                case commute (p3,p2) of
                Nothing -> False
                Just (p2''a,p3'a) ->
                    case commute (p3'a,p1) of
                    Just (p1'a,p3''a) -> p3''a == p3''
                    Nothing -> False
\end{code}
\begin{code}
is_in_directory :: FilePath -> FilePath -> Bool
is_in_directory (cd:d) (cf:f)
    | cd /= cf = False
    | otherwise = is_in_directory d f
is_in_directory [] ('/':_) = True
is_in_directory [] [] = True -- Count directory itself as being in directory...
is_in_directory _ _ = False

clever_commute :: ((Patch, Patch) -> Maybe (Patch, Patch)) ->
                (Patch, Patch) -> Maybe (Patch, Patch)
clever_commute c (p1,p2) = c (p1,p2) `or_maybe`
    (case c (invert p2,invert p1) of
     Just (p1', p2') -> Just (invert p2', invert p1')
     Nothing -> Nothing)

commute :: (Patch,Patch) -> Maybe (Patch,Patch)
commute (NamedP n1 d1 p1, NamedP n2 d2 p2) =
    if n2 `elem` d1 || n1 `elem` d2
    then Nothing
    else case commute (p1,p2) of
         Just (p2',p1') -> Just (NamedP n2 d2 p2', NamedP n1 d1 p1')
         Nothing -> Nothing
commute (p2, p1) =
    clever_commute commute_nameconflict (p2, p1) `or_maybe`
    clever_commute commute_filedir (p2, p1) `or_maybe`
    clever_commute commute_merger (p2, p1) `or_maybe`
    clever_commute commute_split (p2, p1) `or_maybe`
    clever_commute commute_prefs (p2,p1) `or_maybe`
    clever_commute commute_composite (p2, p1) `or_maybe`
    clever_commute commute_named (p2, p1) `or_maybe`
    clever_commute commute_recursive_merger (p2, p1) `or_maybe`
    clever_commute other_commute_recursive_merger (p2, p1) `or_maybe`
    Nothing

commute_no_merger :: (Patch,Patch) -> Maybe (Patch,Patch)
commute_no_merger (NamedP n1 d1 p1, NamedP n2 d2 p2) =
    if n2 `elem` d1 || n1 `elem` d2
    then Nothing
    else case commute_no_merger (p1,p2) of
         Just (p2',p1') -> Just (NamedP n2 d2 p2', NamedP n1 d1 p1')
         Nothing -> Nothing
commute_no_merger (p2, p1) =
    clever_commute commute_nameconflict (p2, p1) `or_maybe`
    clever_commute commute_filedir (p2, p1) `or_maybe`
    clever_commute commute_split (p2, p1) `or_maybe`
    clever_commute commute_prefs (p2,p1) `or_maybe`
    clever_commute commute_composite (p2, p1) `or_maybe`
    clever_commute commute_named (p2, p1) `or_maybe`
    clever_commute commute_recursive_merger (p2, p1) `or_maybe`
    clever_commute other_commute_recursive_merger (p2, p1) `or_maybe`
    Nothing

commute_prefs (ChangePref p f t,p1) = Just (p1,ChangePref p f t)
commute_prefs _ = Nothing
commute_merger (Merger True g p1 p2, pA) =
  if pA == p1
  then Just (Merger True g p2 p1, p2)
  else Nothing
commute_merger _ = Nothing
\end{code}

\begin{code}
commute_recursive_merger (Merger True g p1 p2, pA) =
  case commute (merger_undo p, pA) of -- This should be same as the above.
  Nothing -> Nothing
  Just (pA',_) ->
    case commute (invert $ merger_undo p, pA') of
    Nothing -> Nothing
    Just (pAa,_) -> if pAa /= pA then Nothing else
      case sequence [commute (p1, pA'),commute (p2, pA'),commute (glump g p1 p2, pA')] of
      Nothing -> Nothing
      Just [(_,p1'),(_,p2'),(pA'',gl')] ->
        if gl' /= glump g p1' p2'
        then Nothing
        else
          case sequence [commute (pA',invert p1'), commute (pA',invert p2')] of
          Nothing -> Nothing
          Just _ -> Just (pA'', Merger True g p1' p2')
  where p = Merger True g p1 p2
commute_recursive_merger _ = Nothing
other_commute_recursive_merger (pA'', Merger True g p1' p2') =
  case commute (pA'',glump g p1' p2') of
  Nothing -> Nothing
  Just (gl,pA') ->
      case sequence [liftM (invert.fst) $ commute (pA',invert p1'),
                     liftM (invert.fst) $ commute (pA',invert p2')] of
      Nothing -> Nothing
      Just [p1,p2] ->
        if gl /= glump g p1 p2
        then Nothing
        else
          case sequence [commute (p1,pA'), commute (p2,pA')] of
          Nothing -> Nothing
          Just _ ->
            case commute (invert $ merger_undo p, pA') of
            Nothing -> Nothing
            Just (pA,_) ->
              if pA == p1 then Nothing
              else case commute (merger_undo p,pA) of
                Nothing -> Nothing
                Just (pA'a,_) ->
                  if pA'a /= pA' then Nothing
                  else Just (p, pA)
          where p = Merger True g p1 p2
other_commute_recursive_merger _ = Nothing

commute_filedir :: (Patch,Patch) -> Maybe (Patch,Patch)

movedirfilename :: String -> String -> String -> String
movedirfilename d d' f =
    if length f > length d && take (length d+1) f == d ++ "/"
    then d'++drop (length d) f
    else if f == d
         then d'
         else f

is_superdir d1 d2 =
    length d2 >= length d1 + 1 && take (length d1 + 1) d2 == d1 ++ "/"
make_conflicted (FP f AddFile) = FP (f++"-conflict") AddFile
make_conflicted (DP f AddDir ) = DP (f++"-conflict") AddDir
make_conflicted (Move a f) = Move a (f++"-conflict")

create_conflict_merge (Move d d', FP f AddFile)
    | d' == f = Just (Move d $ f++"-conflict", FP f AddFile)
create_conflict_merge (Move d d', DP f AddDir)
    | d' == f = Just (Move d $ f++"-conflict", DP f AddDir)
create_conflict_merge (FP d AddFile, DP f AddDir)
    | d == f = Just (FP (d++"-conflict") AddFile, DP f AddDir)
create_conflict_merge (Move d d', Move f f')
    | d' == f' && d > f = Just (Move (movedirfilename f f' d) $ f'++"-conflict",
                                Move f f')
create_conflict_merge (p, Split [Move a b, p2])
    | b == a++"-conflict" =
        case create_conflict_merge (p, make_conflicted p2) of
        Nothing -> Nothing
        Just (p',_) -> Just (p', Split [Move a b, p2])
create_conflict_merge _ = Nothing

commute_nameconflict (Move d d', FP f2 AddFile)
    | d == f2 && d' == f2++"-conflict" = Just (FP d' AddFile, ComP [])
    | d' == f2++"-conflict" = Just (Split [Move f2 d', FP f2 AddFile],
                                    Move d f2)
commute_nameconflict (Move d d', DP f2 AddDir)
    | d == f2 && d' == f2++"-conflict" = Just (DP d' AddDir, ComP [])
    | d' == f2++"-conflict" = Just (Split [Move f2 d', DP f2 AddDir],
                                    Move d f2)
commute_nameconflict (Move d d', Move f f')
    | d' == d++"-conflict" && d == f'
        = Just (Move f d', ComP [])
    | d' == f'++"-conflict" && (movedirfilename f' f d) > f =
        Just (Split [Move f' d', Move (movedirfilename d d' f) f'],
              Move (movedirfilename f' f d) f')
commute_nameconflict (FP f AddFile, DP d AddDir)
    | f == d++"-conflict" = Just (Split [Move d f, DP d AddDir],
                                  FP d AddFile)
commute_nameconflict (DP f AddDir, Split [Move a b, p2])
    | b == a++"-conflict" && f == b++"-conflict" =
        Just (Split [Move b f, Split [Move a b, p2]], DP b AddDir)
commute_nameconflict (FP f AddFile, Split [Move a b, p2])
    | b == a++"-conflict" && f == b++"-conflict" =
        Just (Split [Move b f, Split [Move a b, p2]], FP b AddFile)
commute_nameconflict (Move old f, Split [Move a b, p2])
    | b == a++"-conflict" && f == b++"-conflict" =
        Just (Split [Move b f, Split [Move a b, p2]], Move old b)
--commute_nameconflict (Split [Move a b, p2], DP f AddDir)
--    | b == a++"-conflict" && f == a = Just (DP b AddDir, p2)
--commute_nameconflict (Split [Move a b, p2], FP f AddFile)
--    | b == a++"-conflict" && f == a = Just (FP b AddFile, p2)
--commute_nameconflict (Split [Move a b, p2], Move old f)
--    | b == a++"-conflict" && f == a = Just (Move old b, p2)
commute_nameconflict _ = Nothing

commute_filedir (FP f1 p1, FP f2 p2) =
  if f1 /= f2 then Just ( FP f2 p2, FP f1 p1 )
  else commuteFP (FilePatch f1 p1, FilePatch f2 p2)
commute_filedir (DP d1 p1, DP d2 p2) =
  if (not $ is_in_directory d1 d2) && (not $ is_in_directory d2 d1) &&
     d1 /= d2
  then Just ( DP d2 p2, DP d1 p1 )
  else Nothing
commute_filedir (DP d dp, FP f fp) =
    if not $ is_in_directory d f then Just (FP f fp, DP d dp)
    else Nothing

commute_filedir (Move d d', FP f2 p2)
    | f2 == d' = Nothing
    | otherwise = Just (FP (movedirfilename d d' f2) p2, Move d d')
commute_filedir (Move d d', DP d2 p2)
    | is_superdir d2 d' || is_superdir d2 d = Nothing
    | d2 == d' = Nothing
    | otherwise = Just (DP (movedirfilename d d' d2) p2, Move d d')
commute_filedir (Move d d', Move f f')
    | f == d' || f' == d = Nothing
    | f == d || f' == d' = Nothing
    | d `is_superdir` f && f' `is_superdir` d' = Nothing
    | otherwise =
        Just (Move (movedirfilename d d' f) (movedirfilename d d' f'),
              Move (movedirfilename f' f d) (movedirfilename f' f d'))

commute_filedir _ = Nothing

commute_named (NamedP n d p, p1) =
    case commute (p,p1) of
    Just (p1',p') -> Just (p1', NamedP n d p')
    Nothing -> Nothing
commute_named _ = Nothing
\end{code}

\paragraph{Merge}
\newcommand{\merge}{\Longrightarrow}
The second way one can change the context of a patch is by a {\bf merge}
operation.  A merge is an operation that takes two parallel patches and
gives a pair of sequenctial patches.  The merge operation is represented by
the arrow ``\( \merge \)''.
\begin{dfn}\label{merge_dfn}
The result of a merge of two patches, $P_1$ and $P_2$ is one of two patches,
$P_1'$ and $P_2'$, which satisfy the relationship:
\[  P_2 \parallel P_1 \merge {P_2}' P_1 \commute {P_1}' P_2. \]
\end{dfn}
Note that the sequential patches resulting from a merge are \emph{required}
to commute.  This is an important consideration, as without it most of the
manipulations we would like to perform would not be possible.  The other
important fact is that a merge \emph{cannot fail}.  Naively, those two
requirements seem contradictory.  In reality, what it means is that the
result of a merge may be a patch which is much more complex than any we
have yet considered\footnote{Alas, I don't know how to prove that the two
constraints even \emph{can} be satisfied.  The best I have been able to do
is to believe that they can be satisfied, and to be unable to find an case
in which my implementation fails to satisfy them.  These two requirements
are the foundation of the entire theory of patches (have you been counting
how many foundations it has?).}.

\begin{code}
merge :: (Patch, Patch) -> Maybe (Patch, Patch)
quickmerge :: (Patch, Patch) -> Patch
quickmerge (p2,p1) = case merge (p2,p1) of
                     Just (p1',p2') -> p1'
\end{code}

\begin{code}
prop_merge_is_commutable_and_correct :: Patch -> Patch -> Property
prop_merge_is_commutable_and_correct p1 p2 =
    (check_a_patch $ ComP [invert p1,p2]) ==>
    case merge (p2,p1) of
    Nothing -> False
    Just (p2',p1') ->
        case commute (p2',p1') of
        Nothing -> False
        Just (p1'',p2'') -> p2'' == p2 && p1' == p1
prop_merge_is_swapable :: Patch -> Patch -> Property
prop_merge_is_swapable p1 p2 =
    (check_a_patch $ ComP [invert p1,p2]) ==>
    case merge (p2,p1) of
    Nothing -> False
    Just (p2',p1') ->
        case commute (p2',p1') of
        Nothing -> False
        Just (p1'',p2'') ->
           case merge (p1,p2) of
           Nothing -> False
           Just (p1''', p2''') -> p1'' == p1''' && p2'' == p2'''

prop_merge_valid :: Patch -> Patch -> Property
prop_merge_valid p1 p2 =
    (check_a_patch $ ComP [invert p1,p2]) ==>
    case merge (p2,p1) of
    Nothing -> False
    Just (p2',p1') ->
        check_a_patch $ join_patches [invert p1,p2,invert p2,p1',p2']


does_merge :: Patch -> Patch -> Bool
does_merge p1 p2 = merge (p2,p1) /= Nothing
\end{code}

\section{How merges are actually performed}

The constraint that any two compatible patches (patches which can
successfully be applied to the same tree) can be merged is actually quite
difficult to apply.  The above merge constraints also imply that the result
of a series of merges must be independent of the order of the merges.  So
I'm putting a whole section here for the interested to see what algorithms
I use to actually perform the merges (as this is pretty close to being the
most difficult part of the code).

The first case is that in which the two merges don't actually conflict, but
don't trivially merge either (e.g. hunk patches on the same file, where the
line number has to be shifted as they are merged).  This kind of merge can
actually be very elegantly dealt with using only commutation and inversion.

There is a handy little theorem which is immensely useful when trying to
merge two patches.
\begin{thm}\label{merge_thm}
$ P_2' P_1 \commute P_1' P_2 $ if and only if $ P_1'^{ -1}
P_2' \commute P_2 P_1^{ -1} $, provided both commutations succeed.  If
either commute fails, this theorem does not apply.
\end{thm}
This can easily be proven by multiplying both sides of the first
commutation by $P_1'^{ -1}$ on the left, and by $P_1^{ -1}$ on the right.
Besides being used in merging, this theorem is also useful in the recursive
commutations of mergers.  From Theorem~\ref{merge_thm}, we see that the
merge of $P_1$ and $P_2'$ is simply the commutation of $P_2$ with $P_1^{
-1}$ (making sure to do the commutation the right way.  Of course, if this
commutation fails, the patches conflict.  Moreover, one must check that the
merged result actually commutes with $P_1$, as the theorem applies only
when \emph{both} commutations are successful.

\begin{code}
smart_merge :: (Patch, Patch) -> Maybe (Patch, Patch)
smart_merge (p1,p2) =
  case simple_smart_merge (p1,p2) of
  Nothing -> Nothing
  Just (p1'a,p2a) ->
      case simple_smart_merge (p2,p1) >>= commute of
      Nothing -> Nothing
      Just (p1'b, p2b) ->
          if p1'a == p1'b && p2a == p2b && p2a == p2
          then Just (p1'a, p2)
          else Nothing
simple_smart_merge (p1, p2) =
  case commute (p1, invert p2) of
  Just (p2i',p1') ->
      case commute (p1', p2) of
      Just (p2', p1o) ->
          if p1o == p1 then Just (p1', p2)
          else Nothing
      Nothing -> Nothing
  Nothing -> Nothing
\end{code}

Of couse, there are patches that actually conflict, meaning a merge where
the two patches truly cannot both be applied (e.g. trying to create a file
and a directory with the same name).  We deal with this case by creating a
special kind of patch to support the merge, which we will call a
``merger''.  Basically, a merger is a patch that contains the two patches
that conflicted, and instructs darcs basically to resolve the conflict.  By
construction a merger will satisfy the commutation property (see
Definition~\ref{merge_dfn}) that characterizes all merges.  Moreover the
merger's properties are what makes the order of merges unimportant (which
is a rather critical property for darcs as a whole).

The job of a merger is basically to undo the two conflicting patches, and
then apply some sort of a ``resolution'' of the two instead.  In the case
of two conflicting hunks, this will look much like what CVS does, where it
inserts both versions into the file.  In general, of course, the two
conflicting patches may both be mergers themselves, in which case the
situation is considerably more complicated.

\begin{code}
list_conflicted_files :: Patch -> [FilePath]
list_conflicted_files p =
    nubsort $ concat $ map ltf $ concat $ resolve_conflicts p
list_touched_files :: Patch -> [FilePath]
list_touched_files p = ltf p
ltf (NamedP _ _ p) = ltf p
ltf (Split ps) = nubsort $ concatMap ltf ps
ltf (ComP ps) = nubsort $ concatMap ltf ps
ltf (FP f _) = [f]
ltf (DP d _) = [d]
ltf (Merger g b p1 p2) = nubsort $ ltf p1 ++ ltf p2
ltf p = []
nubsort = nubsorted . sort
nubsorted (a:b:l) | a == b = nubsorted (a:l)
                  | otherwise = a: nubsorted (b:l)
nubsorted l = l
\end{code}

\begin{code}
merge (p1,p2) =
                smart_merge (p1,p2) `or_maybe`
                clever_merge create_conflict_merge (p1,p2) `or_maybe`
                clever_merge named_merge (p1,p2) `or_maybe`
                clever_merge mergeComP (p1,p2) `or_maybe`
                panic_merge(p1,p2)

or_maybe (Just e) _ = Just e
or_maybe Nothing f = f

clever_merge :: ((Patch, Patch) -> Maybe (Patch, Patch)) ->
                (Patch, Patch) -> Maybe (Patch, Patch)
clever_merge m (p1,p2) = m (p1,p2) `or_maybe` (m (p2,p1) >>= commute)

named_merge :: (Patch, Patch) -> Maybe (Patch, Patch)
named_merge (p, NamedP n d ps) =
    case merge (p, ps) of
    Just (p',ps') -> Just (p', NamedP n d ps')
    Nothing -> Nothing
named_merge _ = Nothing

merger_merge :: (Patch, Patch) -> Maybe (Patch, Patch)
merger_merge (Merger True g p1 p2, p) =
  case commute (p, p1) of
  Nothing -> Just (Merger True g p (Merger True g p1 p2), p)
  Just (_,p') ->
    case [merge (glump g p1 p2, p'), merge (p1, p'), merge (p2, p')] of
    [Just (gl',_),Just (p1',_),Just (p2',_)] ->
      if gl' /= glump g p1' p2'
      then Nothing
      else Just (Merger True g p1' p2', p)
    _ -> Nothing
merger_merge (Merger False g p1 p2, p) =
    case [merge (glump g p1 p2, p), merge (p1, p), merge (p2, p)] of
    [Just _,Just (p1',_),Just (p2',_)] ->
        Just (Merger False g p1' p2', p)
    _ -> Nothing
merger_merge _ = Nothing

panic_merge :: (Patch, Patch) -> Maybe (Patch, Patch)
panic_merge (p2,p1) = Just (Merger True "0.0" p1 p2, p1)
\end{code}

Much of the merger code depends on a routine which recreates from a single
merger the entire sequence of patches which led up to that merger (this is,
of course, assuming that this is the complicated general case of a merger
of mergers of mergers).  This ``unwind'' procedure is rather complicated,
but absolutely critical to the merger code, as without it we wouldn't even
be able to undo the effects of the patches involved in the merger, since we
wouldn't know what patches were all involved in it.

Basically, unwind takes a merger such as
\begin{verbatim}
M( M(A,B), M(A,M(C,D)))
\end{verbatim}
From which it recreates a merge history:
\begin{verbatim}
C
A
M(A,B)
M( M(A,B), M(A,M(C,D)))
\end{verbatim}
(For the curious, yes I can easily unwind this merger in my head [and on
paper can unwind insanely more complex mergers]---that's what comes of
working for a few months on an algorithm.)  Let's start with a simple
unwinding.  The merger \verb!M(A,B)! simply means that two patches
(\verb!A! and \verb!B!) conflicted, and of the two of them \verb!A! is
first in the history.  The last two patches in the unwinding of any merger
are always just this easy.  So this unwinds to:
\begin{verbatim}
A
M(A,B)
\end{verbatim}
What about a merger of mergers? How about \verb!M(A,M(C,D))!.  In this case
we know the two most recent patches are:
\begin{verbatim}
A
M(A,M(C,D))
\end{verbatim}
But obviously the unwinding isn't complete, since we don't yet see where
\verb!C! and \verb!D! came from.  In this case we take the unwinding of
\verb!M(C,D)! and drop its latest patch (which is \verb!M(C,D)! itself) and
place that at the beginning of our patch train:
\begin{verbatim}
C
A
M(A,M(C,D))
\end{verbatim}
As we look at \verb!M( M(A,B), M(A,M(C,D)))!, we consider the unwindings of
each of its subpatches:
\begin{verbatim}
          C
A         A
M(A,B)    M(A,M(C,D))
\end{verbatim}
As we did with \verb!M(A,M(C,D))!, we'll drop the first patch on the
right and insert the first patch on the left.  That moves us up to the two
\verb!A!'s.  Since these agree, we can use just one of them (they
``should'' agree).  That leaves us with the \verb!C! which goes first.

The catch is that things don't always turn out this easily.  There is no
guarantee that the two \verb!A!'s would come out at the same time, and if
they didn't, we'd have to rearrange things until they did.  Or if there was
no way to rearrange things so that they would agree, we have to go on to
plan B, which I will explain now.

Consider the case of \verb!M( M(A,B), M(C,D))!.  We can easily unwind the
two subpatches
\begin{verbatim}
A         C
M(A,B)    M(C,D)
\end{verbatim}
Now we need to reconcile the \verb!A! and \verb!C!.  How do we do this?
Well, as usual, the solution is to use the most wonderful
Theorem~\ref{merge_thm}.  In this case we have to use it in the reverse of
how we used it when merging, since we know that \verb!A! and \verb!C! could
either one be the \emph{last} patch applied before \verb!M(A,B)! or
\verb!M(C,D)!.  So we can find \verb!C'! using
\[
A^{ -1} C \commute C' A'^{ -1}
\]
Giving an unwinding of
\begin{verbatim}
C'
A
M(A,B)
M( M(A,B), M(C,D) )
\end{verbatim}
There is a bit more complexity to the unwinding process (mostly having to
do with cases where you have deeper nesting), but I think the general
principles that are followed are pretty much included in the above
discussion.

\begin{code}
unwind :: Patch -> [Patch] -- Recreates a patch history in reverse.
unwind p@(Merger b g p1 p2) =
    case (unwind p1, unwind p2) of
    (_:p1s,_:p2s) -> (Merger b g p1 p2) : p1 : reconcile_unwindings p p1s p2s
unwind p = [p];

reconcile_unwindings _ [] p2s = p2s
reconcile_unwindings _ p1s [] = p1s
reconcile_unwindings p (p1:p1s) p2s =
    case [(p1s', p2s')|
          p1s' <- all_head_permutations (p1:p1s),
          p2s' <- all_head_permutations p2s,
          head p1s' == head p2s'] of
    ((p1':p1s', p2':p2s'):_) -> p1' : reconcile_unwindings p p1s' p2s'
    [] -> case liftM reverse $ put_before p1 $ reverse p2s of
          Just p2s' -> p1 : reconcile_unwindings p p1s p2s'
          Nothing ->
              case liftM reverse $ put_before (head p2s) $ reverse (p1:p1s) of
              Just p1s' -> (head p2s) : reconcile_unwindings p p1s' (tail p2s)
              Nothing -> error $ "r_u commute bug, contact droundy@ag.o!\n"
                         ++ "Original patch:\n" ++ show p

put_before p1 (p2:p2s) =
    case commute (invert p1,p2) of
    Nothing -> Nothing
    Just (p2',p1') -> case commute (p1,p2') of
                      Nothing -> Nothing
                      Just _ -> liftM (p2' :) $ put_before p1' p2s
put_before _ [] = Just []

-- NOTE: all_head_permutations accepts a list of patches IN REVERSE
-- ORDER!!!

all_head_permutations :: [Patch] -> [[Patch]]
all_head_permutations [] = []
all_head_permutations [p] = [[p]]
all_head_permutations ps =
  reverse $ map reverse $ nub $ tail_permutations_normal_order $ reverse ps

tail_permutations_normal_order :: [Patch] -> [[Patch]]
tail_permutations_normal_order [] = []
tail_permutations_normal_order (p1:ps) =
    case swap_to_back_n_o (p1:ps) of
    Just ps' -> ps' : map (p1:) (tail_permutations_normal_order ps)
    Nothing -> map (p1:) (tail_permutations_normal_order ps)

swap_to_back_n_o [] = Just []
swap_to_back_n_o [p] = Just [p]
swap_to_back_n_o (p1:p2:ps) =
    case commute (p2,p1) of
    Just (p1',p2') ->
        case swap_to_back_n_o (p1':ps) of
        Just ps' -> Just $ p2': ps'
        Nothing -> Nothing
    Nothing -> Nothing
\end{code}

There are a couple of simple constraints on the routine which determines
how to resolve two conflicting patches (which is called `glump').  These
must be satisfied in order that the result of a series of merges is always
independent of their order.  Firstly, the output of glump cannot change
when the order of the two conflicting patches is switched.  If it did, then
commuting the merger could change the resulting patch, which would be bad.
\begin{code}
prop_glump_order_independent :: String -> Patch -> Patch -> Property
prop_glump_order_independent g p1 p2 =
    (check_a_patch $ ComP [invert p1,p2]) ==>
    glump g p1 p2 == glump g p2 p1
\end{code}
Secondly, the result of the merge of three (or more) conflicting patches
cannot depend on the order in which the merges are performed.

\begin{code}
prop_glump_seq_merge :: String -> Patch -> Patch -> Patch -> Property
prop_glump_seq_merge g p1 p2 p3 =
    (check_a_patch $ ComP [invert p1,p2, p3]) ==>
    glump g p3 (Merger True g p2 p1) == glump g (Merger True g p2 p1) p3
prop_glump_seq_merge_valid :: String -> Patch -> Patch -> Patch -> Property
prop_glump_seq_merge_valid g p1 p2 p3 =
    (check_a_patch $ ComP [invert p1,p2, p3]) ==>
    (check_a_patch $ join_patches [invert p1,p2,p3,invert p3,invert p2])
test_patch = test_str ++ test_note
tp1 = read "
move ./test/test ./hello
"
tp2 = read "
move ./test ./hello
"
tp2' = quickmerge (tp2,tp1)
tp1' = quickmerge (tp1,tp2)
test_note = (if commute (tp2',tp1) == Just (tp1', tp2)
              then "At least they commute right.\n"
              else "Argh! they don't even commute right.\n")
         ++(if check_a_patch $ tp2
              then "tp2 itself is valid!\n"
              else "Oh my! tp2 isn't even valid!\n")
         ++(if check_a_patch $ tp2'
              then "tp2' itself is valid!\n"
              else "Aaack! tp2' itself is invalid!\n")
         ++(if check_a_patch $ join_patches [tp1, tp2']
              then "Valid merge tp2'!\n"
              else "Bad merge tp2'!\n")
         ++ (if check_a_patch $ join_patches [tp2, tp1']
              then "Valid merge tp1'!\n"
              else "Bad merge tp1'!\n")
         ++ (if check_a_patch $ join_patches [tp2,tp1',invert tp2',invert tp1]
              then "Both agree!\n"
              else "The two merges don't agree!\n")
         ++ (if check_a_patch $ join_patches [invert tp2, tp1]
              then "They should be mergable!\n"
              else "Wait a minute, these guys can't be merged!\n")
tp = tp1'

test_str = "Patches are:\n"++(show tp)
           ++(if check_a_patch tp
              then "At least the patch itself is valid.\n"
              else "The patch itself is bad!\n")
           ++"commute of tp1' and tp2 is "++show (commute (tp1',tp2))++"\n"
           ++"commute of tp2' and tp1 is "++show (commute (tp2',tp1))++"\n"
           {-++ "\nSimply flattened, it is:\n"
                  ++ (show $ map (join_patches.flatten.merger_equivalent) $ flatten tp)
           ++ "\n\nUnravelled, it gives:\n" ++ (show $ map unravel $ flatten tp)
           ++ "\n\nUnwound, it gives:\n" ++ (show $ map unwind $ flatten tp)
           ++(if check_a_patch (join_patches$ reverse $ unwind tp)
              then "Unwinding is valid.\n"
              else "Bad unwinding!\n")
           ++(if check_a_patch $ join_patches [tp,invert tp]
              then "Inverse is valid.\n"
              else "Bad inverse!\n")
           ++(if check_a_patch $ join_patches [invert tp, tp]
              then "Other inverse is valid.\n"
              else "Bad other inverse!\n")-}
\end{code}

\begin{code}
prop_glump_three_merge :: String -> Patch -> Patch -> Patch -> Property
prop_glump_three_merge g p1 p2 p3 =
    (check_a_patch $ ComP [invert p1,p2,invert p2, p3]) ==>
    glump g (Merger True g p2 p1) (Merger True g p2 p3) ==
          glump g (Merger True g p1 p2) (Merger True g p1 p3)
              &&
    glump g (Merger True g p2 p1) (Merger True g p2 p3) ==
          glump g (Merger True g p1 p3) (Merger True g p1 p2)
prop_glump_three_merge_valid :: String -> Patch -> Patch -> Patch -> Property
prop_glump_three_merge_valid g p1 p2 p3 =
    (check_a_patch $ ComP [invert p1,p2,invert p2, p3]) ==>
    (check_a_patch $
     join_patches [invert p1,p2,invert p2,p3,invert p3,
                   glump g (Merger True g p2 p1) (Merger True g p2 p3)])
\end{code}

The conflict resolution code (glump) begins by ``unravelling'' the merger
into a set of sequences of patches.  Each sequence of patches corresponds
to one non-conflicted patch that got merged together with the others.  The
result of the unravelling of a series of merges must obviously be
independent of the order in which those merges are performed.  This
unravelling code (which uses the unwind code mentioned above) uses probably
the second most complicated algorithm.  Fortunately, if we can successfully
unravel the merger, almost any function of the unravelled merger satisfies
the two constraints mentioned above that the conflict resolution code must
satisfy.

\begin{code}
unravel :: Patch -> [[Patch]]
prop_unravel_three_merge :: Patch -> Patch -> Patch -> Property
prop_unravel_three_merge p1 p2 p3 =
    (check_a_patch $ ComP [invert p1,p2,invert p2,p3]) ==>
    (unravel $ Merger True "a" (Merger True "a" p2 p3) (Merger True "a" p2 p1)) ==
    (unravel $ Merger True "a" (Merger True "a" p1 p3) (Merger True "a" p1 p2))
\end{code}
\begin{code}
prop_unravel_seq_merge :: Patch -> Patch -> Patch -> Property
prop_unravel_seq_merge p1 p2 p3 =
    (check_a_patch $ ComP [invert p1,p2,p3]) ==>
    (unravel $ Merger True "a" p3 $ Merger True "a" p2 p1) ==
    (unravel $ Merger True "a" (Merger True "a" p2 p1) p3)
\end{code}
\begin{code}
prop_unravel_order_independent :: Patch -> Patch -> Property
prop_unravel_order_independent p1 p2 =
    (check_a_patch $ ComP [invert p1,p2]) ==>
    (unravel $ Merger True "a" p2 p1) == (unravel $ Merger True "a" p1 p2)
\end{code}

\begin{code}
resolve_conflicts :: Patch -> [[Patch]]
resolve_conflicts p = rcs [] $ reverse $ flatten p
rcs _ [] = []
rcs passedby (p@(Merger True "0.0" _ _):ps) =
    case commute (p,join_patches passedby) of
    Just (_,p'@(Merger True "0.0" p1 p2)) ->
        (nub $ glump "0.9" p1 p2 : map join_patches (unravel p))
        : rcs (p : passedby) ps
    Nothing -> rcs (p : passedby) ps
rcs passedby (NamedP n d p:ps) = rcs passedby $ (reverse $ flatten p)++ps
rcs passedby (p:ps) = rcs (p : passedby) ps
\end{code}

\begin{code}
--dbs e s = s
dbs e s = unsafePerformIO $
          do putStr $ "Dbg string: "++e++"... "++show s++"\n"
             return s
dbp :: String -> Patch -> Patch
--dbp s p = p
dbp s p = unsafePerformIO $
          do putStr $ "Dbg: "++s++"...\n"++show p++"\n"
             return p
dbcp s p = p
dbcp s p = if check_a_patch p then p
           else dbp s p
quickcanon p = case canonize p of
    Just p' -> p'
    Nothing -> dbp "a bad quickcanon" p
unravel p = sort $ nub $
            map (sort_coalesce_composite) $
            map (concat . (map (flatten.merger_equivalent))) $
            get_supers $ map reverse $ new_ur p $ unwind p

get_supers :: [[Patch]] -> [[Patch]]
get_supers (x:xs) =
    case filter (not.(x `is_superpatch_of`)) xs of
    xs' -> if or $ map (`is_superpatch_of` x) xs'
           then get_supers xs'
           else x : get_supers xs'
get_supers [] = []
is_superpatch_of :: [Patch] -> [Patch] -> Bool
_ `is_superpatch_of` [] = True
[] `is_superpatch_of` _ = False
a `is_superpatch_of` b | a == b = True
                       | length b > length a = False
a `is_superpatch_of` (b:bs) =
    case filter ((==b).head) $ head_permutations_normal_order a of
    ((_:as):_) -> as `is_superpatch_of` bs
    [] -> False

-- Following function unused but may come in handy later...
simplify :: Patch -> Patch
simplify (ComP [p]) = p
simplify (ComP ps) =
    case sort_coalesce_composite $ sort_coalesce_composite $
         map simplify $ concat $ map flatten ps of
    [p] -> p
    ps' -> ComP ps'
simplify p@(Merger _ _ _ _) = simplify $ merger_equivalent p
simplify (Split ps) = simplify (ComP ps)
simplify p = p

head_permutations_normal_order [] = []
head_permutations_normal_order (p:ps) =
    (p:ps) : catMaybes (map (swapfirst.(p:)) $
                        head_permutations_normal_order ps)
swapfirst (p1:p2:ps) = case commute (p2,p1) of
                       Just (p1',p2') -> Just $ p2':p1':ps
                       Nothing -> Nothing
swapfirst _ = Nothing

new_ur :: Patch -> [Patch] -> [[Patch]]
new_ur p (Merger b g p1 p2 : ps) =
   case filter (\pp-> head pp == p1) $ all_head_permutations ps of
   ((p1':ps'):_) -> new_ur p (p1:ps') ++ new_ur p (p2:ps')
   _ -> error $ "Bug in new_ur - contact droundy@abridgegame.org!\n"
              ++ "Original patch:\n" ++ show p
              ++ "Unwound:\n" ++ unlines (map show $ unwind p)

new_ur op ps =
    case filter (is_merger.head) $ all_head_permutations ps of
    [] -> [ps]
    (ps':_) -> new_ur op ps'
is_merger (Merger _ _ _ _) = True
is_merger _ = False

merger_undo :: Patch -> Patch
merger_undo (Merger b g p1 p2) =
    case (is_merger p1, is_merger p2) of
    (True ,True ) -> join_patches $ map invert $ tail $ unwind p
                     where p = Merger b g p1 p2
    (False,False) -> invert p1
    (True ,False) -> unglump p1
    (False,True ) -> join_patches $ [invert p1, merger_undo p2]
unglump (Merger True g p1 p2) = invert $ glump g p1 p2

merger_equivalent :: Patch -> Patch
merger_equivalent (Merger True g p1 p2) =
    join_patches $ sort_coalesce_composite
                     ((flatten $ merger_equivalent $ merger_undo p)++
                      (flatten $ merger_equivalent $ glump g p1 p2))
    where p = Merger True g p1 p2
merger_equivalent (Merger False g p1 p2) =
    invert $ merger_equivalent $ Merger True g p1 p2
merger_equivalent (Split ps) = Split $ map merger_equivalent ps
merger_equivalent (ComP ps) = ComP $ map merger_equivalent ps
merger_equivalent (NamedP n d p) = NamedP n d $ merger_equivalent p
merger_equivalent p = p
\end{code}

\begin{code}
glump "0.1" p1 p2 = case unravel $ Merger True "0.1" p1 p2 of
                    (ps:_) -> join_patches ps
glump "a" p1 p2 = glump "0.9" p1 p2
glump "0.0" _ _ = ComP []

glump "0.9" p1 p2 = case map (flatten.(dbcp $ "\n\nhello "++
                                       show (length$show$unwind$Merger True "0.9" p1 p2)
                                       ++" "++
                                       show (length $show$ Merger True "0.9" p1 p2)
                                       ++"\n"++
                                       show (Merger True "0.9" p1 p2)).join_patches) $
                         unravel $ Merger True "0.9" p1 p2 of
                    [ps] -> join_patches ps
                    pss -> if only_hunks pss
                           then mangle_unravelled_hunks pss
                           else join_patches $ head pss
\end{code}
\begin{code}
only_hunks [] = False
only_hunks pss = case get_a_filename pss of
                 "" -> False
                 f  -> ohs f pss
ohs f (ps:pss) = if oh f ps then ohs f pss else False
ohs f [] = True
oh f (FP f' (Hunk _ _ _):ps) = if f == f' then oh f ps else False
oh f (p:ps) = False
oh f [] = True

nothings = Nothing : nothings
rls 1 o n mls = map Just n ++ drop (length o) mls
rls l o n (ml:mls) = ml : rls (l-1) o n mls

apply_hunks :: [Maybe PackedString] -> [Patch] -> [Maybe PackedString]
apply_hunks ms (FP _ (Hunk l o n):ps) = apply_hunks (rls l o n ms) ps
apply_hunks ms [] = ms

get_hunks_old mls ps = apply_hunks (apply_hunks mls ps) (map invert $ reverse ps)
get_old mls (ps:pss) = get_old (get_hunks_old mls ps) pss
get_old mls [] = mls
get_hunks_new mls ps = apply_hunks mls ps

get_hunkline = ghl 1
ghl n pps =
    if hasjust $ foldr (:) [] $ map head pps
    then n
    else ghl (n+1) $ foldr (:) [] $ map tail pps
hasjust [] = False
hasjust (Just _:_) = True
hasjust (Nothing:ms) = hasjust ms

get_a_filename ((FP f _:_):_) = f
get_a_filename _ = ""

make_chunk n mls = pull_chunk $ drop (n-1) mls
pull_chunk (Just l:mls) = l : pull_chunk mls
pull_chunk (Nothing: mls) = []

mangle_unravelled_hunks :: [[Patch]] -> Patch
--mangle_unravelled_hunks [[h1],[h2]] = Deal with simple cases handily?
mangle_unravelled_hunks pss =
  case get_old nothings pss of
  oldf ->
    case map (get_hunks_new oldf) pss of
    newfs ->
      case get_hunkline $ oldf : newfs of
      l ->
        case sort $ map (make_chunk l) newfs of
        (nch:nchs) ->
          if 0 == length (make_chunk l oldf)
          then case foldl (++) nch nchs of
               mixedch ->
                 FP (get_a_filename pss) (Hunk l (make_chunk l oldf) mixedch)
          else case foldl (\a b->a++[packString "*************"]++b) nch nchs of
               mixedch ->
                  FP (get_a_filename pss)
                     (Hunk l (make_chunk l oldf) $
                      [packString "v v v v v v v"]++
                      mixedch++
                      [packString "^ ^ ^ ^ ^ ^ ^"])
\end{code}

It can sometimes be handy to have a canonical representation of a given
patch.  We achieve this by defining a canonical form for each patch type,
and a function ``{\tt canonize}'' which takes a patch and puts it into
canonical form.  This routine is used by the diff function to create an
optimal patch (based on an LCS algorithm) from a simple hunk describing the
old and new version of a file.
\begin{code}
canonize :: Patch -> Maybe Patch
canonize (NamedP n d p) =
    case canonize p of
    Just p' -> Just $ NamedP n d p'
    Nothing -> Nothing
canonize (Merger b g p1 p2) =
    liftM2 (Merger b g) (canonize p1) (canonize p2)
canonize (Split ps) = Just $ Split $ sort_coalesce_composite ps
canonize (ComP ps) = canonizeComposite ps
canonize (FP f (Hunk line old new)) = canonizeHunk f line old new
canonize p = Just p
\end{code}
Note that canonization may fail, if the patch is internally inconsistent.

A simpler, faster (and more generally useful) cousin of canonize is the
coalescing function.  This takes two sequential patches, and tries to turn
them into one patch.  This function is used to deal with ``split'' patches,
which are created when the commutation of a primitive patch can only be
represented by a composite patch.  In this case the resulting composite
patch must return to the original primitive patch when the commutation is
reversed, which a split patch accomplishes by trying to coalesce its
contents each time it is commuted.

\begin{code}
coalesce (p2, p1) | p2 == invert p1 = Just $ join_patches []
coalesce (FP f1 p1, FP f2 p2)
    | f1 /= f2 = Nothing
    | otherwise = coalesceFilePatch (FilePatch f1 p1, FilePatch f2 p2)
coalesce (ComP [], p) = Just p
coalesce (p, ComP []) = Just p
coalesce (Split [], p) = Just p
coalesce (p, Split []) = Just p
coalesce (p2,p1) = Nothing
\end{code}

\section{File patches} A file patch is a patch which only modifies a single
file.  There are some rules which can be made about file patches in
general, which makes them a handy class.
\begin{code}
get_filename :: Patch -> Maybe String
set_filename :: String -> Patch -> Patch

is_filepatch (FP _ _) = True
--is_filepatch _ = False

get_filename (FP f _) = Just f
--get_filename _ = Nothing

set_filename f (FP _ fp) = FP f fp
--set_filename f p = p
\end{code}
For example, commutation of two filepatches is trivial if they modify
different files.  There is an exception when one of the files has a name
ending with ``-conflicted'', in which case it may not commute with a file
having the same name, but without the ``-conflicted.''  If they happen to
modify the same file, we'll have to check whether or not they commute.
\begin{code}
commuteFP :: (FilePatch, FilePatch) -> Maybe (Patch, Patch)
commuteFP (FilePatch f1 (Hunk line1 old1 new1),
           FilePatch f2 (Hunk line2 old2 new2)) =
  commuteHunk f1 (Hunk line1 old1 new1, Hunk line2 old2 new2)
commuteFP (FilePatch f1 (TokReplace t o n),
           FilePatch f2 (Hunk line2 old2 new2)) =
    case try_tok_replace t o n old2 of
    Nothing -> Nothing
    Just old2' ->
      case try_tok_replace t o n new2 of
      Nothing -> Nothing
      Just new2' -> Just (FP f2 $ Hunk line2 old2' new2',
                          FP f1 $ TokReplace t o n)
commuteFP (FilePatch f1 (TokReplace t o n),
           FilePatch f2 (TokReplace t2 o2 n2))
    | t /= t2 = Nothing
    | o == o2 = Nothing
    | n == o2 = Nothing
    | o == n2 = Nothing
    | n == n2 = Nothing
    | otherwise = Just (FP f2 $ TokReplace t2 o2 n2,
                        FP f1 $ TokReplace t o n)
commuteFP (_,_) = Nothing
\end{code}

\begin{code}
coalesceFilePatch (FilePatch f1 (Hunk line1 old1 new1),
                   FilePatch f2 (Hunk line2 old2 new2))
    = coalesceHunk f1 line1 old1 new1 line2 old2 new2
coalesceFilePatch (FilePatch f1 AddFile, FilePatch f2 RmFile)
    = Just (ComP [])
coalesceFilePatch (FilePatch f1 (TokReplace t1 o1 n1),
                   FilePatch f2 (TokReplace t2 o2 n2))
    | t1 == t2 && n2 == o1 = Just $ FP f1 $ TokReplace t1 o2 n1
coalesceFilePatch (FilePatch f (Binary m n), FilePatch f' (Binary o m'))
    | m == m' = Just $ FP f $ Binary o n
coalesceFilePatch _ = Nothing
\end{code}

There is another handy function, which primarily affects file patches
(although it can also affect other patches, such as rename patches or dir
add/remove patches), which is the submerge-in-directory function.  This
function changes the patch to act on a patch within a subdirectory rather
than in the current directory, and is useful when performing the recursive
diff.

\begin{code}
submerge_in_dir :: FilePath -> Patch -> Patch
submerge_in_dir dir (Move ('.':'/':f) ('.':'/':f')) =
    Move (n_fn $ dir++"/"++f) (n_fn $ dir++"/"++f')
submerge_in_dir dir (DP ('.':'/':d) dp) = DP (n_fn (dir++"/"++d)) dp
submerge_in_dir dir (DP d dp) = DP (n_fn (dir++"/"++d)) dp
submerge_in_dir dir (FP ('.':'/':f) fp) = FP (n_fn (dir++"/"++f)) fp
submerge_in_dir dir (FP f fp) = FP (n_fn (dir++"/"++f)) fp
submerge_in_dir dir (Split ps) = Split $ map (submerge_in_dir dir) ps
submerge_in_dir dir (ComP ps) = ComP $ map (submerge_in_dir dir) ps
submerge_in_dir dir (NamedP n d p) = NamedP n d (submerge_in_dir dir p)
submerge_in_dir dir (Merger b g p1 p2) = Merger b g (sub p1) (sub p2)
    where sub = submerge_in_dir dir
\end{code}

Hunks are an example of a complex filepatch.  A hunk is a set of lines of a
text file to be replaced by a different set of lines.  Either of these sets
may be empty, which would mean a deletion or insertion of lines.
\begin{code}
applyHunkLines :: Int -> [PackedString] -> [PackedString]
               -> FileContents -> Maybe FileContents
applyHunkLines l o n (c,_) =
    case sequence $ apply_hunk_internal l o n c of
    Just c' -> Just (c',Nothing)
    Nothing -> Nothing
apply_hunk_internal :: Int -> [PackedString] -> [PackedString]
                    -> [PackedString] -> [Maybe PackedString]
apply_hunk_internal l o (n:ns) (c:cs)
    | l == 1 = Just n : apply_hunk_internal l o ns (c:cs)
    | l > 0 = Just c : apply_hunk_internal (l-1) o (n:ns) cs
apply_hunk_internal l [] n []
    | l == 1 = map Just n
    | otherwise = [Nothing]
apply_hunk_internal l (o:os) [] (c:cs)
    | l == 1 && o == c = apply_hunk_internal l os [] cs
    | l == 1 && o /= c = [Nothing]
    | l > 1 = Just c : apply_hunk_internal (l-1) (o:os) [] cs
apply_hunk_internal _ [] [] cs = map Just cs
\end{code}
The hunk is the simplest patch that has a commuting pattern in which the
commuted patches differ from the originals (rather than simple success or
failure).  This makes commuting or merging two hunks a tad tedious.
\begin{code}
commuteHunk f (Hunk line2 old2 new2, Hunk line1 old1 new1)
  | line1 + length new1 < line2 =
      Just (FP f (Hunk line1 old1 new1),
            FP f (Hunk (line2-(length new1)+(length old1)) old2 new2))
  | line2 + length old2 < line1 =
      Just (FP f (Hunk (line1+(length new2)-(length old2)) old1 new1),
            FP f (Hunk line2 old2 new2))
  | line1 + length new1 == line2 &&
      ((length new2 /= 0 && length new1 /= 0) ||
       (length old2 /= 0 && length old1 /= 0)) =
      Just (FP f (Hunk line1 old1 new1),
            FP f (Hunk (line2-(length new1)+(length old1)) old2 new2))
  | line2 + length old2 == line1 &&
      ((length new2 /= 0 && length new1 /= 0) ||
       (length old2 /= 0 && length old1 /= 0)) =
      Just (FP f (Hunk (line1+(length new2)-(length old2)) old1 new1),
            FP f (Hunk line2 old2 new2))
  | otherwise = Nothing
\end{code}
Hunks, of course, can be coalesced if they have any overlap.  Note that
coalesce code doesn't check if the two patches are conflicting.  If you are
coalescing two conflicting hunks, you've already got a bug somewhere.

\begin{code}
coalesceHunk f line1 old1 new1 line2 old2 new2 =
    docoalesceHunk f line1 old1 new1 line2 old2 new2
    --case commute (FP f (Hunk line1 old1 new1),
    --              FP f (Hunk line2 old2 new2)) of
    --Just (p1,p2) -> Nothing -- They don't coalesce
    --Nothing ->
    --    docoalesceHunk f line1 old1 new1 line2 old2 new2
docoalesceHunk f line1 old1 new1 line2 old2 new2
    | line1 == line2 && length old1 < length new2 =
        if take (length old1) new2 /= old1
        then Nothing
        else case drop (length old1) new2 of
        extranew -> Just (FP f (Hunk line1 old2 (new1++extranew)))
    | line1 == line2 && length old1 > length new2 =
        if take (length new2) old1 /= new2
        then Nothing
        else case drop (length new2) old1 of
        extraold -> Just (FP f (Hunk line1 (old2++extraold) new1))
    | line1 == line2 = if new2 == old1 then Just (FP f (Hunk line1 old2 new1))
                       else Nothing
    | line1 < line2 && length old1 >= line2 - line1 =
        case take (line2 - line1) old1 of
        extra->docoalesceHunk f line1 old1 new1 line1 (extra++old2) (extra++new2)
    | line1 > line2 && length new2 >= line1 - line2 =
        case take (line1 - line2) new2 of
        extra->docoalesceHunk f line2 (extra++old1) (extra++new1) line2 old2 new2
    | otherwise = Nothing
\end{code}

One of the most important pieces of code is the canonization of a hunk,
which is where the ``diff'' algorithm is performed.  This algorithm begins
with chopping off the identical beginnings and endings of the old and new
hunks.  This isn't strictly necesary, but is a good idea, since this
process is $O(n)$, while the primary diff algorithm is something
considerably more painful than that... actually the head would be dealt
with all right, but with more space complexity.  I think it's more
efficient to just chop the head and tail off first.

\begin{code}
canonizeHunk f line old []  = Just (FP f (Hunk line old []))
canonizeHunk f line [] new  = Just (FP f (Hunk line [] new))
canonizeHunk f line old new =
    case make_holey f line old new thelcs of
    [p] -> Just p
    [] -> Nothing
    ps -> Just $ join_patches ps
    where thelcs = lcs old new

make_holey :: String -> Int -> [PackedString] -> [PackedString]
           -> [PackedString] -> [Patch]
make_holey f line old new thelcs =
    map (\ (l,o,n) -> FP f (Hunk l o n))
        (make_holey_hunkdata line [] [] old new thelcs)

make_holey_hunkdata :: Int -> [PackedString] -> [PackedString] ->
                       [PackedString] -> [PackedString] -> [PackedString]->
                       [(Int,[PackedString],[PackedString])]
make_holey_hunkdata line ol nl o n []
    | ol++o == [] && nl++n == [] = []
    | otherwise = [(line,ol++o, nl++n)]
make_holey_hunkdata line ol nl (o:os) (n:ns)  (l:ls)
    | o /= l =
        make_holey_hunkdata line (ol++[o]) nl os (n:ns) (l:ls)
    | n /= l =
        make_holey_hunkdata line ol (nl++[n]) (o:os) ns (l:ls)
    | ol == [] && nl == [] =
        make_holey_hunkdata (line+1) [] [] os ns ls
    | otherwise = (line,ol,nl) :
                  make_holey_hunkdata (line+1+length nl) [] [] os ns ls
applyBinary :: PackedString -> PackedString
            -> FileContents -> Maybe FileContents
applyBinary o n (_,Just c) | c == o = Just (linesPS n, Just n)
applyBinary o n (ls,Nothing) | unlinesPS ls == o = Just (linesPS n, Just n)
applyBinary _ _ _ = Nothing
\end{code}

\section{Token replace patches}\label{token_replace}

Although most filepatches will be hunks, darcs is clever enough to support
other types of changes as well.  A ``token replace'' patch replaces all
instances of a given token with some other version.  A token, here, is
defined by a regular expression, which must be of the simple [a-z...] type,
indicating which characters are allowed in a token, with all other
characters acting as delimiters.  For example, a C identifier would be a
token with the flag \verb![A-Za-z_0-9]!.

\begin{code}
applyTokReplace :: String -> String -> String
                -> FileContents -> Maybe FileContents
applyTokReplace t o n (c,_) =
    case sequence $ map (try_tok_internal t o n) c of
    Nothing -> Nothing
    Just c' -> Just (c', Nothing)

try_tok_possibly :: String -> String -> String
                -> [Possibly PackedString] -> Maybe [Possibly PackedString]
try_tok_possibly t o n mss =
    sequence $ map (silly_maybe_possibly $ try_tok_internal t o n) $ take 1000 mss

try_tok_replace :: String -> String -> String
                -> [PackedString] -> Maybe [PackedString]
try_tok_replace t o n mss =
    sequence $ map (try_tok_internal t o n) mss

silly_maybe_possibly :: (PackedString -> Maybe PackedString) ->
                        (Possibly PackedString -> Maybe (Possibly PackedString))
silly_maybe_possibly f =
    \px -> case px of
           PNothing -> Just PNothing
           PSomething -> Just PSomething
           PJust x -> case f x of
                      Nothing -> Nothing
                      Just x' -> Just $ PJust x'

try_tok_internal :: String -> String -> String
                 -> PackedString -> Maybe PackedString
try_tok_internal t o n s =
    case matchRegex (mkRegex $ "([^"++t++"]*)(["++t++"]+)(.*)") $ unpackPS s of
    Just [before,tok,after] ->
        case try_tok_internal t o n $ packString after of
        Nothing -> Nothing
        Just rest ->
            if tok == o
            then Just $ packString $ before ++ n ++ unpackPS rest
            else if tok == n
                 then Nothing
                 else Just $ packString $ before ++ tok ++ unpackPS rest
    Nothing -> Just s
\end{code}

What makes the token replace patch special is the fact that a token replace
can be merged with almost any ordinary hunk, giving exactly what you would
want.  For example, you might want to change the patch type {\tt
TokReplace} to {\tt TokenReplace} (if you decided that saving two
characters of space was stupid).  If you did this using hunks, it would
modify every line where {\tt TokReplace} occurred, and quite likely provoke
a conflict with another patch modifying those lines.  On the other hand, if
you did is using a token replace patch, the only change that it could
conflict with would be if someone else had used the token ``{\tt
TokenReplace}'' in their patch rather than TokReplace---and that actually
would be a real conflict!

\section{Composite patches}
Composite patches are made up of a series of patches intended to be applied
sequentially.  They are represented by a list of patches, with the first
patch in the list being applied first.
\begin{code}
commute_composite (ComP ps, p1) =
    case commuteCompositeRightToLeft ps p1 [] of
    ([],p1',ps') -> Just (p1', ComP ps')
    _ -> Nothing
commute_composite _ = Nothing

commute_split (Split ps, p1) =
    case commuteCompositeRightToLeft ps p1 [] of
    ([],p1',ps') ->
        case sort_coalesce_composite ps' of
        [p'] -> Just (p1', p')
        ps'' -> Just (p1', Split ps'')
    _ -> Nothing
commute_split _ = Nothing

commuteCompositeRightToLeft :: [Patch] -> Patch -> [Patch]
                           -> ([Patch], Patch, [Patch])
commuteCompositeRightToLeft [] p1 ps = ([], p1, reverse ps)
commuteCompositeRightToLeft (p:ps) p1 newp =
  case commute (p, p1) of
  Just (p1', p') -> commuteCompositeRightToLeft ps p1' (p':newp)
  Nothing -> (p:ps, p1, reverse newp)
\end{code}

\begin{code}
reorder :: Patch -> Patch
reorder (NamedP n d p) = NamedP n d $ reorder p
reorder (ComP ps) = ComP $ sortps ps
reorder p = p
sortps [] = []
sortps (p1:ps) =
    case sortps ps of
    [] -> [p1]
    (p2:ps') -> if p1 < p2 then p1 : p2 : ps'
                else case commute (p2,p1) of
                     Nothing -> p1 : p2 : ps'
                     Just (p1',p2') -> p2': sortps (p1':ps')

sort_coalesce_composite :: [Patch] -> [Patch]
sort_coalesce_composite [] = []
sort_coalesce_composite (p:ps) =
    case sort_coalesce_composite ps of
    [] -> [p]
    (np:nps) ->
        case coalesce (np, p) of
        Just pp -> sort_coalesce_composite (pp:nps)
        Nothing ->
            if p < np
            then p:np: nps
            else case commute (np, p) of
                 Just (p',np') -> np': sort_coalesce_composite (p':nps)
                 Nothing -> p:np: nps
simplify_composite :: [Patch] -> Maybe Patch
simplify_composite [p] = canonize p
simplify_composite ps = Just $ ComP ps
subcanonize_composite :: [Patch] -> [Patch]
subcanonize_composite [] = []
subcanonize_composite (p:ps) =
    case canonize p of
    Just p' -> p' : subcanonize_composite ps
    --Nothing -> subcanonize_composite ps
canonizeComposite ps =
    simplify_composite $ sort_coalesce_composite $ subcanonize_composite ps
\end{code}

\begin{code}
mergeComP :: (Patch, Patch) -> Maybe (Patch, Patch)
mergeComP (p, ComP ps) =
    case convenient_comp_merge p ps of
    Just p' -> Just (p', ComP ps)
    Nothing -> Nothing
mergeComP _ = Nothing

convenient_comp_merge :: Patch -> [Patch] -> Maybe Patch
convenient_comp_merge p [] = Just p
convenient_comp_merge p (rhp:ps) =
    case merge (p, rhp) of
    Just (p', rhp') -> convenient_comp_merge p' ps
    Nothing -> Nothing
\end{code}

%Another nice thing to be able to do with composite patches is to `flatten'
%them, that is, turn them into a simple list of patches (appropriately
%ordered, of course), with all nested compositeness unnested.

\begin{code}
{- INLINE flatten -}
flatten :: Patch -> [Patch]

flatten (ComP ps) = concat $ map flatten ps
flatten p = [p]
\end{code}

%\section{Outputting interesting and useful information}

%Just being able to manipulate patches and trees is not enough.  We also
%want to be able to view the patches and files.  This requires another set
%of functions, closely related to the patch application functions, which
%will give us the necesary information to browse the changes we have made.
%It is \emph{not} the Patch module's responsibility to add any sort of
%markup or formatting, but simply to provide the information necesary for an
%external module to do the formatting.

\begin{code}
data LineMark = AddedLine PatchInfo | RemovedLine PatchInfo
              | AddedRemovedLine PatchInfo PatchInfo | None
                deriving (Show)
type MarkedUpFile = [(PackedString, LineMark)]
markup_file :: PatchInfo -> Patch
            -> (FilePath, MarkedUpFile) -> (FilePath, MarkedUpFile)
markup_file n (NamedP n' d' p') (f, mk) = markup_file n p' (f, mk)
markup_file n (Merger b g p1 p2) (f, mk) =
    markup_file n (merger_equivalent $ Merger b g p1 p2) (f, mk)
markup_file _ (ComP []) (f, mk) = (f, mk)
markup_file n (ComP (p:ps)) (f, mk) = markup_file n (ComP ps) $
                                      markup_file n p (f, mk)
markup_file _ (Split []) (f, mk) = (f, mk)
markup_file n (Split (p:ps)) (f, mk) = markup_file n (Split ps) $
                                       markup_file n p (f, mk)
markup_file n (FP f' AddFile) (f, mk) = (f, mk)
markup_file n (FP f' RmFile) (f, mk) = (f, mk)
markup_file n (FP f' (Hunk line old new)) (f, mk)
    | f' /= f = (f, mk)
    | otherwise = (f, markup_hunk n line old new mk)
markup_file name (FP f' (TokReplace t o n)) (f, mk)
    | f' /= f = (f, mk)
    | otherwise = (f, markup_tok name t o n mk)
markup_file n (DP d dp) (f, mk) = (f, mk)
markup_file n (Move d d') (f, mk) = (movedirfilename d d' f, mk)
markup_file n (ChangePref _ _ _) (f,mk) = (f,mk)

markup_hunk :: PatchInfo -> Int -> [PackedString] -> [PackedString]
            -> MarkedUpFile -> MarkedUpFile
markup_hunk n l old new ((sf, RemovedLine pi):mk) =
    (sf, RemovedLine pi) : markup_hunk n l old new mk
markup_hunk n l old new ((sf, AddedRemovedLine po pn):mk) =
    (sf, AddedRemovedLine po pn) : markup_hunk n l old new mk

markup_hunk name 1 old (n:ns) mk =
    (n, AddedLine name) : markup_hunk name 1 old ns mk
markup_hunk n 1 (o:os) [] ((sf, None):mk)
    | o == sf = (sf, RemovedLine n) : markup_hunk n 1 os [] mk
    | otherwise = [(packString "Error in patch application", AddedLine n)]
markup_hunk n 1 (o:os) [] ((sf, AddedLine nold):mk)
    | o == sf = (sf, AddedRemovedLine nold n) : markup_hunk n 1 os [] mk
    | otherwise = [(packString "Error in patch application", AddedLine n)]
markup_hunk n 1 [] [] mk = mk

markup_hunk n l old new ((sf, AddedLine pi):mk)
    | l > 1 = (sf, AddedLine pi) : markup_hunk n (l-1) old new mk
    | l < 1 = (sf, AddedLine pi) : markup_hunk n (l-1) old new mk
markup_hunk n l old new ((sf, None):mk)
    | l > 1 = (sf, None) : markup_hunk n (l-1) old new mk
    | l < 1 = (sf, None) : markup_hunk n (l-1) old new mk

markup_hunk _ _ _ _ [] = []

markup_hunk n l old new mk =
    (packString "Error: ",None) : mk

markup_tok name t o n mk =
    case unzip mk of
    (lines,markup) ->
        case try_tok_replace t o n lines of
        Just l' -> zip l' markup
\end{code}
\begin{code}
type MarkedUpRepo = [(String, PatchInfo, String, DirMark)]
clean_markedup_repo :: MarkedUpRepo -> MarkedUpRepo
clean_markedup_repo [] = []
clean_markedup_repo ((f,n,cr,AddedFile):mk) = (f,n,cr,DullFile) : clean_markedup_repo mk
clean_markedup_repo ((f,n,cr,ModifiedFile):mk) = (f,n,cr,DullFile) : clean_markedup_repo mk
clean_markedup_repo ((f,n,cr,RemovedFile):mk) = clean_markedup_repo mk
clean_markedup_repo ((f,n,cr,MovedFile _):mk) = (f,n,cr,DullFile) : clean_markedup_repo mk
clean_markedup_repo ((f,n,cr,AddedDir):mk) = (f,n,cr,DullDir) : clean_markedup_repo mk
clean_markedup_repo ((f,n,cr,MovedDir _):mk) = (f,n,cr,DullDir) : clean_markedup_repo mk
clean_markedup_repo ((f,n,cr,RemovedDir):mk) = clean_markedup_repo mk
clean_markedup_repo (m:mk) = m : clean_markedup_repo mk
markup_repo :: PatchInfo -> Patch -> MarkedUpRepo -> MarkedUpRepo
markup_repo n (NamedP n' d' p') mk = markup_repo n p' mk
markup_repo n (Merger b g p1 p2) mk =
    markup_repo n (merger_equivalent $ Merger b g p1 p2) mk
markup_repo _ (ComP []) mk = mk
markup_repo n (ComP (p:ps)) mk = markup_repo n (ComP ps) $
                                 markup_repo n p mk
markup_repo _ (Split []) mk = mk
markup_repo n (Split (p:ps)) mk = markup_repo n (Split ps) $
                                  markup_repo n p mk
markup_repo n (FP f AddFile) mk = (f, n, f, AddedFile) : mk
markup_repo n (FP f RmFile) ((f',n',cr',m):mk)
    | f == f' = (f,n',cr',RemovedFile) : mk
    | otherwise = (f',n',cr',m) : markup_repo n (FP f RmFile) mk
markup_repo n (FP f fp) ((f',n',cr',m):mk)
    | f' /= f = (f',n',cr',m) : markup_repo n (FP f fp) mk
    | otherwise = (f, n', cr', ModifiedFile) : mk
markup_repo n (DP d AddDir) mk = (d, n, d, AddedDir) : mk
markup_repo n (DP d RmDir) ((f',n',cr',m):mk)
    | d == f' = (d,n',cr',RemovedDir) : mk
    | otherwise = (f',n',cr',m) : markup_repo n (DP d RmDir) mk
markup_repo n (Move fi fi') ((f',n',cr',m):mk)
    | fi == f' && mur_is_file m = (fi',n',cr',MovedFile fi) : mk
    | fi == f' && mur_is_dir  m = (fi',n',cr',MovedDir fi) : mk
    | otherwise = (f',n',cr',m) : markup_repo n (Move fi fi') mk
markup_repo n (ChangePref _ _ _) mk = mk
markup_repo _ _ [] = []

mur_is_dir RemovedDir = True
mur_is_dir (MovedDir _) = True
mur_is_dir AddedDir = True
mur_is_dir DullDir = True
mur_is_dir _ = False
mur_is_file = not . mur_is_dir
\end{code}

\section{Patch string formatting}

Of course, in order to store our patches in a file, we'll have to save them
as some sort of strings.  The convention is that each patch string will end
with a newline, but on parsing we skip any amount of whitespace between
patches.
\begin{code}
prop_read_show :: Patch -> Bool
prop_read_show p = case readPatch $ show p of
                   [(p',_)] -> p' == p
                   _ -> False
prop_readPS_show :: Patch -> Bool
prop_readPS_show p = case readPatchPS $ packString $ show p of
                     Just (p',_) -> p' == p
                     Nothing -> False
\end{code}
\begin{code}
instance  Show Patch  where
    show (FP f AddFile) = showAddFile f
    show (FP f RmFile)  = showRmFile f
    show (FP f (Hunk line old new))  = showHunk f line old new
    show (FP f (TokReplace t old new))  = showTok f t old new
    show (FP f (Binary old new))  = showBinary f old new
    show (DP d AddDir) = showAddDir d
    show (DP d RmDir)  = showRmDir d
    show (Move f f') = showMove f f'
    show (ChangePref p f t) = showChangePref p f t
    show (ComP ps)  = showComP ps
    show (Split ps)  = showSplit ps
    show (NamedP n d p) = showNamed n d p
    show (Merger b p1 p2 pr) = showMerger b p1 p2 pr
hPutPatch :: Handle -> Patch -> IO ()
writePatch :: FilePath -> Patch -> IO ()
hPutPatch h p@(ComP _) = hPutComP h p
hPutPatch h p@(NamedP _ _ _) = hPutNamed h p
hPutPatch h p@(Merger _ _ _ _) = hPutMerger h p
hPutPatch h p@(Split _) = hPutSplit h p
hPutPatch h p@(FP _ (Hunk _ _ _)) = hPutHunk h p
hPutPatch h p = hPutStr h $ show p
writePatch f p = do h <- openFile f WriteMode
                    hPutPatch h p
                    hClose h

instance  Read Patch  where
    readsPrec _           = readPatch

readPatchPS :: PackedString -> Maybe (Patch,PackedString)
readPatchPS s = case (unpackPS . fst) `liftM` mylexPS s of
                Just "{" -> readComPPS s -- }
                Just "(" -> readSplitPS s -- )
                Just "hunk" -> readHunkPS s
                Just "replace" -> readTokPS s
                Just "binary" -> readBinaryPS s
                Just "addfile" -> readAddFilePS s
                Just "adddir" -> readAddDirPS s
                Just "rmfile" -> readRmFilePS s
                Just "rmdir" -> readRmDirPS s
                Just "move" -> readMovePS s
                Just "changepref" -> readChangePrefPS s
                Just "merger" -> readMergerPS True s
                Just "regrem" -> readMergerPS False s
                Just ('[':_) -> readNamedPS s -- ]
                _ -> Nothing

readPatch                 :: ReadS Patch
readPatch s =
    case mylex s of
    [("{", _)]       -> readComP s -- }
    [("(", _)]       -> readSplit s -- )
    [("hunk", _)]    -> readHunk s
    [("replace", _)] -> readTok s
    [("binary", _)]  -> readBinary s
    [("addfile", _)] -> readAddFile s
    [("adddir", _)]  -> readAddDir s
    [("rmfile", _)]  -> readRmFile s
    [("rmdir", _)]   -> readRmDir s
    [("move", _)]    -> readMove s
    [("changepref", _)]-> readChangePref s
    [("merger", _)]  -> readMerger s
    [("regrem", _)]  -> readMerger s
    [('[':_, _)]     -> readNamed s -- ]
    _                -> []
hGetLinePS :: Handle -> IO PackedString
hGetLinePS h = liftM packString $ hGetLine h
hLex h = do nex <- hLookAhead h
            if Char.isSpace nex then do hGetChar h
                                        hLex h
               else hLexHelper h
hUnLex h s = hSeek h RelativeSeek $ fromIntegral (- length s)
hLexHelper h = do nex <- hLookAhead h
                  if Char.isSpace nex then return ""
                     else do hGetChar h
                             rest <- hLexHelper h
                             return $ nex : rest
hGetPatch :: Handle -> IO Patch
hGetPatch h = do
     tok <- hLex h
     hUnLex h tok
     case tok of
         "{" -> hGetComP h
         "(" -> hGetSplit h
         "hunk" -> hGetHunk h
         "replace" -> hGetReplace h
         "binary" -> hGetBinary h
         "addfile" -> hGetAddFile h
         "adddir" -> hGetAddDir h
         "rmfile" -> hGetRmFile h
         "rmdir" -> hGetRmDir h
         "move" -> hGetMove h
        -- "changepref" -> hGetChangePref h
         "merger" -> hGetMerger h
         "regrem" -> hGetMerger h
         ('[':_) -> hGetNamed h
         _ -> ioError $ userError "no patch here!"
\end{code}

\begin{code}
read_one_patch :: String -> Maybe Patch
read_one_patch s =
    case readPatch s of
    [(p,_)] -> Just p
    _ -> Nothing
\end{code}

\paragraph{Composite patch}
A patch made up of a few other patches.
\begin{verbatim}
{
  <put patches here> (indented two)
}
\end{verbatim}
\begin{code}
showComP ps = foldl (++) "{\n" (map show ps) ++ "}\n"
hPutComP h (ComP ps) = do hPutStr h "{\n"
                          sequence_ $ map (hPutPatch h) ps
                          hPutStr h "}\n"
readComPPS :: PackedString -> Maybe (Patch,PackedString)
readComPPS s =
    case mylexPS s of
    Just (start,t) ->
        case read_patchesPS t of
        Just (ps,w) ->
            case mylexPS w of
            Just (end,x) -> if unpackPS end == "}" && unpackPS start == "{"
                            then Just (ComP ps, dropWhilePS Char.isSpace x)
                            else Nothing
        
readComP :: ReadS Patch
readComP s =
  [(ComP ps, z) |
       ("{", t) <- mylex s,
       (ps, w) <- [read_more_patches [] t],
       ("}", x) <- mylex w,
       z <- [dropWhile Char.isSpace x]]
read_patchesPS :: PackedString -> Maybe ([Patch],PackedString)
read_patchesPS s =
    case readPatchPS s of
    Nothing -> Just ([],s)
    Just (p,s') ->
        case read_patchesPS s' of
        Just (ps,s'') -> Just (p:ps,s'')

read_more_patches :: [Patch] -> String -> ([Patch], String)
read_more_patches ps s =
  case readPatch s of
    [] -> (ps, s)
    [(p,s')] -> read_more_patches (ps++[p]) s'
    _ -> error "Ooops, could be more than one patch!"
hGetComP h = do hLex h -- trust that the first token is "{"
                ps <- h_get_patches h
                end <- hLex h
                case end of
                  "}" -> return $ ComP ps
                  bad -> error $ "Composite patch shouldn't end with "++bad
h_get_patches h = do mp <- (liftM Just (hGetPatch h)) `catch`
                           (\ _ -> return Nothing )
                     case mp of
                             Just p -> do rest <- h_get_patches h
                                          return $ p : rest
                             Nothing -> return []
\end{code}

\paragraph{Split patch}
A split patch is similar to a composite patch (identical in how it's
stored), but rather than being composed of several patches grouped
together, it is created from one patch that has been split apart, typically
through a merge or commutation.
\begin{verbatim}
(
  <put patches here> (indented two)
)
\end{verbatim}
\begin{code}
showSplit ps =
  "(\n" ++
  (foldl (++) "" (map show ps)) ++
  ")\n"
hPutSplit h (Split ps) = do hPutStr h "(\n"
                            sequence_ $ map (hPutPatch h) ps
                            hPutStr h ")\n"
readSplitPS :: PackedString -> Maybe (Patch,PackedString)
readSplitPS s =
    case mylexPS s of
    Just (start,t) ->
        case read_patchesPS t of
        Just (ps,w) ->
            case mylexPS w of
            Just (end,x) -> if unpackPS end == ")" && unpackPS start == "("
                            then Just (Split ps, dropWhilePS Char.isSpace x)
                            else Nothing
readSplit :: ReadS Patch
readSplit s =
  [(Split ps, z) |
       ("(", t) <- mylex s,
       (ps, w) <- [read_more_patches [] t],
       (")", x) <- mylex w,
       z <- [dropWhile Char.isSpace x]]
hGetSplit h = do hLex h -- trust that the first token is "("
                 ps <- h_get_patches h
                 end <- hLex h
                 case end of
                          ")" -> return $ Split ps
\end{code}

\paragraph{Hunk}
Replace a hunk (set of contiguous lines) of text with a new
hunk.
\begin{verbatim}
hunk FILE LINE#
-LINE
...
+LINE
...
\end{verbatim}
\begin{code}
showHunk f line old new =
  "hunk " ++ f ++ " " ++(show line)++"\n" ++
  (foldl (\l r -> l++"-"++r++"\n") "" $ map unpackPS old) ++
  (foldl (\l r -> l++"+"++r++"\n") "" $ map unpackPS new)
hPutHunk h (FP f (Hunk l o n)) = do hPutStr h $ "hunk "++f++" "++show l++"\n"
                                    sequence_ $ map (hputpspre h '-') o
                                    sequence_ $ map (hputpspre h '+') n
hputpspre h c ps = do hPutChar h c
                      fixedhPutPS h ps
                      hPutChar h '\n'
fixedhPutPS h ps = if lengthPS ps == 0 then return () else hPutPS h ps
readHunkPS :: PackedString -> Maybe (Patch,PackedString)
readHunkPS s =
    case mylexPS s of
    Just (hun,s') ->
        case mylexPS s' of
        Just (fi,s''a) ->
          case readIntPS s''a of
          Just (l,s'') ->
            case lines_starting_withPS '-'$dropWhilePS Char.isSpace s'' of
            Just (old,s''') ->
                case lines_starting_withPS '+' s''' of
                Just (new,s''') ->
                    Just (FP (unpackPS fi) $ Hunk l old new, s''')
readHunk :: ReadS Patch
readHunk s =
    [(FP f $ Hunk line (map packString old) (map packString new), z) |
     ("hunk", t) <- mylex s,
     (f, w) <- mylex t,
     (line,  x) <- reads w,
     y <- [dropWhile Char.isSpace x],
     (old, zz) <- lines_starting_with '-' y,
     (new, z) <- lines_starting_with '+' zz ]
hGetHunk h = do hLex h -- trust that first token is "hunk"
                f <- hLex h
                line <- liftM read $ hLex h
                hSkipWhite h
                o <- hGetLinesStartingWith h '-'
                n <- hGetLinesStartingWith h '+'
                return $ FP f $ Hunk line o n
hGetLinesStartingWith :: Handle -> Char -> IO [PackedString]
hGetLinesStartingWith h c = do nex <- hLookAhead h
                               if nex /= c then return []
                                           else do hGetChar h
                                                   l <- hGetLinePS h
                                                   rest <- hGetLinesStartingWith h c
                                                   return $ l : rest
hSkipWhite h = do nex <- hLookAhead h
                  if Char.isSpace nex then do hGetChar h
                                              hSkipWhite h
                                      else return ()
\end{code}

\paragraph{Token replace}

Replace a token with a new token.  Note that this format means that the
white space must not be allowed within a token.  If you know of a practical
application of whitespace within a token, let me know and I may change
this.
\begin{verbatim}
replace FILENAME [REGEX] OLD NEW
\end{verbatim}
\begin{code}
showTok f t o n = "replace "++f++" ["++t++"] "++o++" "++n++"\n"
readTokPS :: PackedString -> Maybe (Patch,PackedString)
readTokPS s =
    case mylexPS s of
    Just (rep, s'a) ->
      case mylexPS s'a of
      Just (f, s') ->
        case mylexPS s' of
        Just (ught, s'') ->
            case mylexPS s'' of
            Just (o, s''') ->
                case mylexPS s''' of
                Just (n, s'''') ->
                    Just (FP (unpackPS f) $
                          TokReplace (drop_brackets $ unpackPS ught)
                          (unpackPS o) (unpackPS n), s'''')
readTok :: ReadS Patch
readTok s =
    [(FP f $ TokReplace (drop_brackets ught) o n, z) |
     ("replace", t) <- mylex s,
     (f, w) <- mylex t,
     (ught,  x) <- mylex w,
     (o, y) <- mylex x,
     (n, z) <- mylex y,
     is_in_brackets ught
    ]
is_in_brackets ('[':s) = -- ] (because the emacs haskell mode is lame...)
  case reverse s of
  ({-[-}']':s) -> True
  _ -> False
is_in_brackets _ = False
drop_brackets = reverse . tail . reverse . tail
hGetReplace h = do hLex h
                   f <- hLex h
                   stuffinbrackets <- hLex h
                   o <- hLex h
                   n <- hLex h
                   return $ FP f $ TokReplace (drop_brackets stuffinbrackets) o n
\end{code}

\paragraph{Binary file modification}

Modify a binary file
\begin{verbatim}
binary FILENAME
oldhex
*HEXHEXHEX
...
newhex
*HEXHEXHEX
...
\end{verbatim}
\begin{code}
-- This is a generic parser monad for convenience...
newtype GP a b = GenP (a -> Maybe (b,a))
instance Monad (GP a) where
    m >>= k          = GenP $ parse_then m k
    return x         = GenP (\a -> Just (x,a))
    fail s           = GenP (\_ -> Nothing)
parse_then :: GP a b -> (b -> GP a c) -> a -> Maybe (c,a)
parse_then (GenP f) g a = case f a of
                                 Nothing -> Nothing
                                 Just (b,a) -> parseGP (g b) a
parseGP :: GP a b -> a -> Maybe (b,a)
parseGP (GenP p) a = p a
skipGP :: (a -> a) -> GP a ()
skipGP s = GenP $ \a -> Just ((), s a)
assertGP b = unless b $ fail ""

showBinary :: String -> PackedString -> PackedString -> String
showBinary f o n = "binary "++f++"\noldhex"
                   ++unpackPS (str2hexPS o)
                   ++"newhex"++unpackPS (str2hexPS n)
readBinaryPS :: PackedString -> Maybe (Patch,PackedString)
readBinaryPS = parseGP $ do
  bin <- GenP mylexPS
  assertGP $ bin == packString "binary"
  fi <- GenP mylexPS
  newhex <- GenP mylexPS
  skipGP $ dropWhilePS Char.isSpace
  old <- GenP $ lines_starting_withPS '*'
  oldhex <- GenP mylexPS
  skipGP $ dropWhilePS Char.isSpace
  new <- GenP $ lines_starting_withPS '*'
  return $ binary (unpackPS fi)
             (hex2strPS $ concatPS old) (hex2strPS $ concatPS new)
readBinary :: ReadS Patch
readBinary s = catMaybes [readBinaryString s]
p2gp p = GenP $ \s -> case p s of {[] -> Nothing; [(x,s')] -> Just (x,s'); }
readBinaryString = parseGP $ do
  bin <- p2gp mylex
  assertGP $ bin == "binary"
  fi <- p2gp mylex
  newhex <- p2gp mylex
  skipGP $ dropWhile Char.isSpace
  old <- p2gp $ lines_starting_with '*'
  oldhex <- p2gp mylex
  skipGP $ dropWhile Char.isSpace
  new <- p2gp $ lines_starting_with '*'
  return $ binary fi (hex2strPS $ packString $ concat old)
                     (hex2strPS $ packString $ concat new)
hGetBinary h = error "still need to implement hGetBinary."

hex2strPS :: PackedString -> PackedString
hex2strPS = packString . hex2str
hex2str :: PackedString -> String
hex2str ps | nullPS ps = ""
           | lengthPS ps == 1 = error "odd number of chars in hex2str!"
           | is_boring c1 = hex2str $ tailPS ps
           | otherwise = chr (hex2int c2 + 16*(hex2int c1)):hex2str cs
           where c1 = headPS ps
                 c2 = headPS $ dropWhilePS is_boring $ tailPS ps
                 cs = tailPS $ dropWhilePS is_boring $ tailPS ps
                 is_boring c = c == '*' || Char.isSpace c
str2hexPS :: PackedString -> PackedString
str2hexPS = str2hex . unpackPS
str2hex :: String -> PackedString
str2hex s = packString $ (++"\n") $ ("\n*"++) $ break_every_80 1 $ str2hexNoPS s
str2hexNoPS :: String -> String
str2hexNoPS [] = ""
str2hexNoPS (c:cs) = (int2hex (h `div` 16)):(int2hex (h `mod` 16)):
                         str2hexNoPS cs
                             where h = ord $ c
break_every_80 _ "" = ""
break_every_80 n (c:cs)
    | n < 80 = c : break_every_80 (n+1) cs
    | otherwise = '\n':'*':c: break_every_80 1 cs
int2hex i | i < 10 = chr $ (ord '0') + i
          | i < 16 = chr $ (ord 'a') + i - 10
hex2int c = case ord c of
            i | i0 < 10 && i0 >= 0 -> i0
              | iA < 16 && iA >= 10 -> iA
              | otherwise -> error $ "bad hex value! '" ++ [c] ++ "'"
              where i0 = i - ord '0'
                    iA = 10 + i - ord 'a'
\end{code}

\paragraph{Add file}
Add an empty file to the tree.

\verb!addfile filename!
\begin{code}
showAddFile f   = "addfile " ++ f ++ "\n"
readAddFilePS :: PackedString -> Maybe (Patch,PackedString)
readAddFilePS s =
    case mylexPS s of
    Just (a,s') -> case mylexPS s' of
                   Just (f,s'') -> Just (FP (unpackPS f) AddFile, s'')
readAddFile :: ReadS Patch
readAddFile s =
  [(FP f AddFile, x) |
   ("addfile", t) <- mylex s,
   (f,   w) <- mylex t,
    x <- [dropWhile Char.isSpace w] ]
hGetAddFile h = do hLex h
                   f <- hLex h
                   return $ FP f AddFile
\end{code}

\paragraph{Remove file}
Delete a file from the tree.

\verb!rmfile filename!
\begin{code}
showRmFile f    = "rmfile " ++ f ++ "\n"
readRmFilePS :: PackedString -> Maybe (Patch,PackedString)
readRmFilePS s =
    case mylexPS s of
    Just (a,s') -> case mylexPS s' of
                   Just (f,s'') -> Just (FP (unpackPS f) RmFile, s'')
readRmFile :: ReadS Patch
readRmFile s =
  [(FP f RmFile, x) |
   ("rmfile", t) <- mylex s,
   (f,   w) <- mylex t,
    x <- [dropWhile Char.isSpace w] ]
hGetRmFile h = do hLex h
                  f <- hLex h
                  return $ FP f RmFile
\end{code}

\paragraph{Move}
Rename a file or directory.

\verb!move oldname newname!
\begin{code}
showMove d d' = "move " ++ d ++ " " ++ d' ++ "\n"
readMovePS :: PackedString -> Maybe (Patch,PackedString)
readMovePS s =
    case mylexPS s of
    Just (m,s') ->
        case mylexPS s' of
        Just (d,s'') ->
            case mylexPS s'' of
            Just (d',s''') -> Just (Move (unpackPS d) (unpackPS d'), s''')
readMove :: ReadS Patch
readMove s =
  [(Move d d', y) |
   ("move", t) <- mylex s,
   (d ,  w) <- mylex t,
   (d',  x) <- mylex w,
    y <- [dropWhile Char.isSpace x] ]
hGetMove h = do hLex h
                o <- hLex h
                n <- hLex h
                return $ Move o n
\end{code}

\paragraph{Change Pref}
Change one of the preference settings.  Darcs stores a number of simple
string settings.  Among these are the name of the test script and the name
of the script that must be called prior to packing in a make dist.
\begin{verbatim}
changepref prefname
oldval
newval
\end{verbatim}
\begin{code}
showChangePref p f t = "changepref " ++ p ++ "\n" ++ f ++ "\n" ++ t ++ "\n"
readChangePrefPS :: PackedString -> Maybe (Patch,PackedString)
readChangePrefPS s =
    case mylexPS s of
    Just (ch,s') ->
        case mylexPS s' of
        Just (p,s'') ->
            case breakPS (=='\n') $ tailPS $ dropWhilePS (==' ') s'' of
            (f,s''') ->
                case breakPS (=='\n') $ tailPS s''' of
                (t,s4) -> Just (ChangePref (u p) (u f) (u t), s4)
                where u = unpackPS
readChangePref :: ReadS Patch
readChangePref s =
  [(ChangePref p f t, dropWhile Char.isSpace y) |
   ("changepref", t) <- mylex s,
   (p ,  w) <- mylex t,
   (f,  _:x) <- [break (=='\n') $ tail $ dropWhile (==' ') w],
   (t,  y) <- [break (=='\n') x]]
\end{code}

\paragraph{Add dir}
Add an empty directory to the tree.

\verb!adddir filename!
\begin{code}
showAddDir d   = "adddir " ++ d ++ "\n"
readAddDirPS :: PackedString -> Maybe (Patch,PackedString)
readAddDirPS s =
    case mylexPS s of
    Just (a,s') -> case mylexPS s' of
                   Just (f,s'') -> Just (DP (unpackPS f) AddDir, s'')
readAddDir :: ReadS Patch
readAddDir s =
  [(DP d AddDir, x) |
   ("adddir", t) <- mylex s,
   (d,   w) <- mylex t,
    x <- [dropWhile Char.isSpace w] ]
hGetAddDir h = do hLex h
                  f <- hLex h
                  return $ DP f AddDir
\end{code}

\paragraph{Remove dir}
Delete a directory from the tree.

\verb!rmdir filename!
\begin{code}
showRmDir d    = "rmdir " ++ d ++ "\n"
readRmDirPS :: PackedString -> Maybe (Patch,PackedString)
readRmDirPS s =
    case mylexPS s of
    Just (a,s') -> case mylexPS s' of
                   Just (f,s'') -> Just (DP (unpackPS f) RmDir, s'')
readRmDir :: ReadS Patch
readRmDir s =
  [(DP d RmDir, x) |
   ("rmdir", t) <- mylex s,
   (d,   w) <- mylex t,
    x <- [dropWhile Char.isSpace w] ]
hGetRmDir h = do hLex h
                 f <- hLex h
                 return $ DP f RmDir
\end{code}

\paragraph{Merger patches}
Merge two patches.  The MERGERVERSION is included to allow some degree of
backwards compatibility if the merger algorithm needs to be changed.
\begin{verbatim}
merger MERGERVERSION
<first patch>
<second patch>
\end{verbatim}
\begin{code}
showMerger True  g p1 p2 = "merger "++g++" (\n" ++show p1++show p2++")\n"
showMerger False g p1 p2 = "regrem "++g++" (\n" ++show p1++show p2++")\n"
hPutMerger h (Merger b g p1 p2) = do if b then hPutStr h $ "merger "++g++" (\n"
                                          else hPutStr h $ "regrem "++g++" (\n"
                                     hPutPatch h p1
                                     hPutPatch h p2
                                     hPutStr h ")\n"
readMergerPS :: Bool -> PackedString -> Maybe (Patch,PackedString)
readMergerPS b s =
    case mylexPS $ snd $ fromJust $ mylexPS s of
    Just (g,s1) ->
        case mylexPS s1 of
        Just (start,s2) ->
            case readPatchPS s2 of
            Just (p1, s3) ->
                case readPatchPS s3 of
                Just (p2, s4) ->
                    case mylexPS s4 of
                    Just (end,s5) ->
                        if (unpackPS start) == "(" && (unpackPS end) == ")"
                        then Just (Merger b (unpackPS g) p1 p2, s5)
                        else Nothing
readMerger :: ReadS Patch
readMerger s =
    [(Merger True g p1 p2, y) |
     ("merger",s') <- mylex s,
     (g, s'') <- mylex s',
     ("(", t) <- mylex s'',
     (p1, u) <- readPatch t,
     (p2, v) <- readPatch u,
     (")", x) <- mylex v,
     y <- [dropWhile Char.isSpace x]
    ] ++
    [(Merger False g p1 p2, y) |
     ("regrem",s') <- mylex s,
     (g, s'') <- mylex s',
     ("(", t) <- mylex s'',
     (p1, u) <- readPatch t,
     (p2, v) <- readPatch u,
     (")", x) <- mylex v,
     y <- [dropWhile Char.isSpace x]
    ]
hGetMerger h = do m <- hLex h
                  g <- hLex h
                  verifyLex h "("
                  p1 <- hGetPatch h
                  p2 <- hGetPatch h
                  verifyLex h ")"
                  return $ Merger (m == "merger") g p1 p2
verifyLex h str = do s <- hLex h
                     if s == str then return ()
                         else do hUnLex h s
                                 ioError $ userError $ "aack, didn't match "++str++" with "++s
\end{code}

\paragraph{Named patches}

Named patches are diplayed as a `patch id' which is in square brackets,
followed by a patch.  Optionally, after the patch id (but before the patch
itself) can come a list of dependencies surrounded by angle brackets.  Each
dependency consists of a patch id.

\begin{code}
showNamed n [] p = show n ++ show p
showNamed n d p = show n++(foldl (++) "\n<"
                           (map ('\n':) $ map show d))++"\n> "++show p
hPutNamed h (NamedP n [] p) = do hPutStr h $ show n
                                 hPutStr h " < > "
                                 hPutPatch h p
hPutNamed h (NamedP n d p) = do hPutStr h $ show n
                                hPutStr h $ (foldl (++) "\n<" (map ('\n':) $map show d))
                                hPutStr h "\n> "
                                hPutPatch h p
readNamedPS s =
    case readPatchInfoPS s of
    Nothing -> error "Hello world!"
    Just (n,s2) ->
        case read_dependsPS s2 of
        Nothing -> error "Goodbye world!"
        Just (d, s3) ->
            case readPatchPS s3 of
            Nothing -> error $ "Yikes!" ++ unpackPS s3 ++ "\nfrom\n" ++ unpackPS s
            Just (p, s4) -> Just (NamedP n d p, s4)
readNamed s =
    [(NamedP n d p, v) |
     (n, t) <- reads s,
     (d, u) <- read_depends t,
     (p, v) <- readPatch u
    ]
read_dependsPS s = case mylexPS s of
                   Just (st,s') -> if unpackPS st == "<" then read_pisPS s'
                                   else Just ([],s)
read_pisPS s = case readPatchInfoPS s of
               Just (pi,s') ->
                   case read_pisPS s' of
                   Just (pis,s'') -> Just (pi:pis,s'')
               Nothing -> Just ([],tailPS $ dropWhilePS (/='>') s)
read_depends s = case mylex s of
                 [("<",s')] -> [read_pis s']
                 _ -> [([],s)]
read_pis :: String -> ([PatchInfo], String)
read_pis (s) = case reads s of
               [(pi,s')] ->
                   case read_pis s' of
                   (pis,s'') -> (pi:pis,s'')
               [] -> ([],tail $ dropWhile (/='>') s)
hGetNamed h = do pi <- hGetPatchInfo h
                 br <- hLex h
                 d <- if br == "<" then hGetPIs h else do { hUnLex h br; return []; }
                 nc <- hLookAhead h
                 p <- hGetPatch h
                 return $ NamedP pi d p
hGetPIs h = do d <- liftM (:[]) (hGetPatchInfo h)
                        `catch` (\ _ -> do endbr <- hLex h
                                           return [])
               if d == [] then return []
                   else do rest <- hGetPIs h
                           return $ d ++ rest
\end{code}

\begin{code}
mylexPS :: PackedString -> Maybe (PackedString,PackedString)
mylexPS s = case dropWhilePS Char.isSpace s of
            s' -> if nullPS s' then Just (nilPS,nilPS)
                  else Just $ breakPS Char.isSpace s'
mylex              :: ReadS String
mylex s          =  case dropWhile Char.isSpace s of
                      "" -> [("","")]
                      s' -> [(w, s'')]
                            where (w, s'') = break Char.isSpace s'

mygetline          :: ReadS String
mygetline s = case break (\c -> c == '\n') s of
              (w, s') -> [(w, if s' == "" then "" else tail s')]

lines_starting_with           :: Char -> ReadS [String]
lines_starting_with = read_more_lines_starting_with []

read_more_lines_starting_with :: [String] -> Char -> ReadS [String]
read_more_lines_starting_with lns tc "" = [(reverse lns, "")]
read_more_lines_starting_with lns tc (c:cs) =
    if c /= tc then [(reverse lns, c:cs)]
    else case mygetline cs of
         [(line, s')] -> read_more_lines_starting_with (line:lns) tc s'
lines_starting_withPS :: Char -> PackedString
                      -> Maybe ([PackedString],PackedString)
lines_starting_withPS c s =
    if nullPS s || headPS s /= c then Just ([],s)
    else case linesPS $ tailPS s of
         (l:_) -> case lines_starting_withPS c$dropPS (1+lengthPS l)$tailPS s of
              Just (ls,rest) -> Just (l:ls,rest)
readIntPS :: PackedString -> Maybe (Int, PackedString)
readIntPS s = case dropWhilePS Char.isSpace s of
              s' -> case spanPS Char.isDigit s' of
                    (n,s'') -> case reads $ unpackPS n of
                               [(nn,_)] -> Just (nn,s'')
                               _ -> Nothing
\end{code}

%FIXME: The following code needs to be moved.  It is a function
%``is\_similar'' which tells you if two patches are in the same category
%human-wise.  Currently it just returns true if they are filepatches on the
%same file.

\begin{code}
is_similar (FP f _) (FP f' _) = f == f'
is_similar p1 p2 = p1 == p2
is_addfile (FP _ AddFile) = True
is_addfile _ = False
\end{code}

%files or directories, changed by a patch
%we get it solely from the patch here
%instead of performing patch apply on a population
%we !could! achieve the same by applying a patch to a cleaned population
%and getting modified files and dirs
%but this should be significantly slower when the population grows large
%This could be useful for just presenting a summary of what a patch does
%(especially useful for larger repos)

\begin{code}
patchChanges :: Patch -> [(String,DirMark)]
patchChanges (NamedP _ _ p) = patchChanges p
patchChanges (Move f1 f2) = [(f1,MovedFile f2),(f2,MovedFile f1)]
patchChanges (DP d AddDir) = [(d,AddedDir)]
patchChanges (DP d RmDir) = [(d,RemovedDir)]
patchChanges (FP f AddFile) = [(f,AddedFile)]
patchChanges (FP f RmFile) = [(f,RemovedFile)]
patchChanges (FP f _) = [(f,ModifiedFile)]
patchChanges (Split ps) = concatMap patchChanges ps
patchChanges (ComP ps) = concatMap patchChanges ps
patchChanges (Merger b s p1 p2) = patchChanges p1 ++ patchChanges p2
patchChanges (ChangePref _ _ _) = []
\end{code}

%apply a patch to a population at a given time

\begin{code}
applyToPop :: PatchInfo -> Patch -> Population -> Population
applyToPop pi p (Pop _ tree)
 = Pop pi (applyToPopTree p tree)
   -- ``pi'' is global below!
 where applyToPopTree :: Patch -> PopTree -> PopTree
       applyToPopTree (NamedP _ _ p) tr = applyToPopTree p tr
       applyToPopTree p@(Merger _ _ _ _) tr
        = applyToPopTree (merger_equivalent p) tr
       applyToPopTree (ComP ps) tr =
        foldl (\t p -> applyToPopTree p t) tr ps
       applyToPopTree (Split ps) tr =
        foldl (\t p -> applyToPopTree p t) tr ps
       applyToPopTree p@(FP f _) tr = popChange (splitPS '/' (packString  f)) p tr
       applyToPopTree p@(DP d _) tr = popChange (splitPS '/' (packString  d)) p tr
       -- precondition: ``to'' does not exist yet!
       applyToPopTree p@(Move from to) tr
        = case (breakP (splitPS '/' (packString  from)) tr) of
           (tr',Just ins) -> let to' = (splitPS '/' (packString  to))
                                 ins' = case ins of
                                         PopDir i trs -> PopDir (Info {name = last to',
                                                                       modifiedBy = pi,
                                                                       modifiedHow = MovedDir from,
                                                                       createdBy = createdBy i,
                                                                       creationName = creationName i})
                                                                trs
                                         PopFile i -> PopFile (Info {name = last to',
                                                                     modifiedBy = pi,
                                                                     modifiedHow = MovedFile from,
                                                                     createdBy = createdBy i,
                                                                     creationName = creationName i})
                             in insertP to' tr' ins'
           _ -> tr -- ignore the move if ``from'' couldn't be found
       applyToPopTree _ tr = tr

       -- break a poptree fst: org tree with subtree removed,
       --                 snd: removed subtree
       breakP :: [PackedString] -> PopTree -> (PopTree,Maybe PopTree)
       breakP [parent,path] tr@(PopDir f trs)
        | parent == (name f) = case findRem path trs of
                                Just (trs',tr') -> (PopDir (setState f pi) trs',Just tr')
                                _ -> (tr,Nothing)
        | otherwise = (tr,Nothing)
        where findRem path [] = Nothing
              findRem path (d:trs)
               | path == pname d = Just (trs,d)
               | otherwise = do (trs',d') <- findRem path trs
                                return (d:trs',d')
       breakP (n:rest) tr@(PopDir f trs)
        | (name f) == n = case catMaybes inss of
                           [ins] -> (PopDir (setState f pi) trs', Just ins)
                           [] -> (tr,Nothing)
                           _ -> error "breakP: more than one break"
        | otherwise = (tr,Nothing)
          where (trs',inss) = unzip (map (breakP rest) trs)
       breakP _ tr = (tr,Nothing)
       -- insert snd arg into fst arg
       insertP :: [PackedString] -> PopTree -> PopTree -> PopTree
       insertP [parent,path] org@(PopDir f trs) tr
        | parent == (name f) = PopDir (setState f pi) (tr:trs)
        | otherwise = org
       insertP (n:rest) org@(PopDir f trs) tr
        | (name f) == n = PopDir (setState f pi) trs'
        | otherwise = org
          where trs' = map (\o -> insertP rest o tr) trs
       insertP _ org _ = org

       -- change a population according to a patch
       popChange :: [PackedString] -> Patch -> PopTree -> PopTree
       popChange [parent,path] (DP d AddDir) tr@(PopDir f trs)
        | parent == (name f) = PopDir (setState f pi) (new:trs)
        | otherwise = tr
              where new = PopDir (Info {name = path,
                                        modifiedBy = pi,
                                        modifiedHow = AddedDir,
                                        createdBy = Just pi,
                                        creationName = Just path}) []
       -- only mark a directory (and contents) as ``deleted'' do not delete it actually
       popChange [path] (DP d RmDir) tr@(PopDir f trs)
        | path == (name f) = PopDir (Info {name = path,
                                           modifiedBy = pi,
                                           modifiedHow = RemovedDir,
                                           createdBy = createdBy f,
                                           creationName = creationName f}) trs'
        | otherwise = tr
          where trs' = map markDel trs -- recursively ``delete'' the contents

       popChange [parent,path] (FP d AddFile) tr@(PopDir f trs)
        | parent == (name f) = PopDir (setState f pi) (new:trs)
        | otherwise = tr
              where new = PopFile (Info {name = path,
                                         modifiedBy = pi,
                                         modifiedHow = AddedFile,
                                         createdBy = Just pi,
                                         creationName = Just path})
       popChange [path] (FP d RmFile) tr@(PopFile f)
        | path == (name f) = PopFile (Info {name = path,
                                            modifiedBy = pi,
                                            modifiedHow = RemovedFile,
                                            createdBy = createdBy f,
                                            creationName = creationName f})
        | otherwise = tr
       popChange [path] (FP _ _) tr@(PopFile f)
        | path == (name f)
           = PopFile (Info {name = path,
                            modifiedBy = pi,
                            modifiedHow = ModifiedFile,
                            createdBy = createdBy f,
                            creationName = creationName f})
       popChange (n:rest) p tr@(PopDir f trs)
        | (name f) == n = PopDir (setState f pi) (map (popChange rest p) trs)
        | otherwise = tr
       popChange _ _ tr = tr
       markDel (PopDir f trs) = PopDir (Info {name = name f,
                                              modifiedBy = pi,
                                              modifiedHow = RemovedDir,
                                              createdBy = createdBy f,
                                              creationName = creationName f}) trs'
                where trs' = map markDel trs
       markDel (PopFile f) = PopFile (Info {name = name f,
                                            modifiedBy = pi,
                                            modifiedHow = RemovedFile,
                                            createdBy = createdBy f,
                                            creationName = creationName f})

pname :: PopTree -> PackedString
pname (PopDir i _) = name i
pname (PopFile i) = name i
\end{code}

