
module External (
    initCurl,
    copyFileOrUrl,
    fetchFile, fetchFilePS,
    sendEmail,
    signString,
    runPipe,
  ) where


import Monad
import System
import IO
import GHC.Handle ( openFd )
import Directory ( doesFileExist, removeFile )
import Maybe ( catMaybes )
import Posix ( createLink, getFileStatus, fileSize, sleep )

import Foreign
import Foreign.C

import RegexString

import FastPackedString
import Lock ( withTemp, withOpenTemp )
import Autoconf ( have_libcurl )
import Curl ( readUrl, readUrlPS )


{-
  Path resolving:
    * A URL contains the sequence "://".
    * A local filepath does not contain colons, except 
      as second character (windows drives).
    * A path that is neither a URL nor a local file
      is an ssh-path.

  Examples:
    /usr/repo/foo                - local file
    c:/src/darcs                 - local file
    http://abridgegame.org/darcs - URL
    peter@host:/path             - ssh
    host:/path                   - ssh


  This means that single-letter hosts in ssh-paths doesn't work,
  unless a username is provided.

  Perhaps ssh-paths should use "ssh://user@host/path"-syntax instead?
-}

match_not_file = mkRegex "..+:.+"
match_url = mkRegex ".+://.+"

is_file fou = matchRegex match_not_file fou == Nothing
is_url s = matchRegex match_url s /= Nothing


fetchFile :: String -> IO String
fetchFile fou | is_file fou = readFile fou
fetchFile fou = readRemote fou

fetchFilePS :: String -> IO PackedString
fetchFilePS fou | is_file fou = readFilePS fou
fetchFilePS fou = readRemotePS fou

-- rm_crlf's function should really be implemented in linesPS or a
-- replacement thereof, if it is done at all.  (Leaving this here as a note
-- until the issue of how to deal with carriage returns is settled.
rm_crlf [] = []
rm_crlf ('\r':'\n':xs) = '\n' : rm_crlf xs
-- rm_crlf ('\r':xs) = '\n' : rm_crlf xs -- for files from old macs?
rm_crlf (x:xs) = x : rm_crlf xs


initCurl :: IO ()
initCurl = return ()
-- initCurl = Curl.initCurl


readRemotePS :: String -> IO PackedString
readRemotePS s | is_url s =
 if have_libcurl then readUrlPS s
 else
  withTemp $ \tmp -> do
    -- better read command line from config-file..
    mwget <- lookForExecutable "wget"
    r <- case mwget of 
         Just wget -> system (wget ++ " --quiet -O \"" ++ tmp ++ "\" " ++ s)
         Nothing -> do
             mcurl <- lookForExecutable "curl"
             case mcurl of
               Just curl -> system (curl ++ " -s -o \"" ++ tmp ++ "\" " ++ s)
               Nothing -> error "Oooops, need curl or wget at the moment"
    when (r /= ExitSuccess) (fail $ "failed to fetch: " ++ s)
    readFilePS tmp
readRemotePS s = readSSH_PS s

readRemote :: String -> IO String
readRemote s | is_url s =
 if have_libcurl then Curl.readUrl s
 else
  withTemp $ \tmp -> do
    -- better read command line from config-file..
    mwget <- lookForExecutable "wget"
    r <- case mwget of 
         Just wget -> system (wget ++ " --quiet -O \"" ++ tmp ++ "\" " ++ s)
         Nothing -> do
             mcurl <- lookForExecutable "curl"
             case mcurl of
               Just curl -> system (curl ++ " -s -o \"" ++ tmp ++ "\" " ++ s)
               Nothing -> error "Oooops, need curl or wget at the moment"
    when (r /= ExitSuccess) (fail $ "failed to fetch: " ++ s)
    h <- openFile tmp ReadMode
    read_all_handle h
readRemote s = readSSH s


readSSH :: String -> IO String
readSSH path =
  withTemp $ \tmp -> do
    p <- try $ getEnv "SSH_PORT" -- or DARCS_SSH_PORT ?
    let port = either (const "") (("-P "++).show) p
    r <- system ("scp " ++ port ++ " \"" ++ path ++ "\" " ++ tmp)
    when (r /= ExitSuccess)
         (fail $ "(scp) failed to fetch: " ++ path)
    h <- openFile tmp ReadMode
    read_all_handle h

readSSH_PS :: String -> IO PackedString
readSSH_PS path =
  withTemp $ \tmp -> do
    p <- try $ getEnv "SSH_PORT" -- or DARCS_SSH_PORT ?
    let port = either (const "") (("-P "++).show) p
    r <- system ("scp " ++ port ++ " \"" ++ path ++ "\" " ++ tmp)
    when (r /= ExitSuccess)
         (fail $ "(scp) failed to fetch: " ++ path)
    readFilePS tmp


{-
   We can't read lazily since we must close
   the file before removing it.

   Note: I have the idea that at least under unix one can unlink a file
   while it is still open, which would mean that we *could* read lazily.
   Of course, there is also the problem of too many open files...
-}
read_all_handle :: Handle -> IO String
read_all_handle h = read_all h []
 where read_all h cs =
         do b <- hIsEOF h
            if b then hClose h >> return (reverse cs)
             else do
              c <- hGetChar h 
              read_all h (c:cs)



copyFileOrUrl :: FilePath -> FilePath -> IO ()
copyFileOrUrl fou out | is_file fou =
    createLink fou out
  `catch` \_ -> do 
    c <- readFile fou
    writeFile out c
copyFileOrUrl fou out = do
    c <- readRemote fou
    writeFile out c


sendEmail :: String -> String -> String -> String -> IO ()
sendEmail f t s body =
  withOpenTemp $ \(h,fn) -> do
    hPutStr h $ 
        "To: "      ++ t ++ "\n" ++
        "From: "    ++ f ++ "\n" ++
        "Subject: " ++ s ++ "\n" ++
        body
    hClose h
    r <- system ("/usr/sbin/sendmail " ++ t ++ " < " ++ fn)
    when (r /= ExitSuccess) $ fail ("failed to send mail to: " ++ t)


runPipe ss "" =
  withTemp $ \t -> do
    system (unwords ss ++ " >> " ++ t)
    h <- openFile t ReadMode
    read_all_handle h

runPipe ss inp =
 withOpenTemp $ \(hi,ti) -> do
  hPutStr hi inp
  hClose hi
  withTemp $ \to -> do
    system (unwords ss ++ " >> " ++ to ++ " < " ++ ti)
    ho <- openFile to ReadMode
    read_all_handle ho


signString :: String -> IO String
signString t = do
  has_usrbingpg <- doesFileExist "/usr/bin/gpg"
  if has_usrbingpg
     then runPipe ["/usr/bin/gpg","--clearsign"] t
     else do has_usrlocalbingpg <- doesFileExist "/usr/local/bin/gpg"
             if has_usrlocalbingpg
                then runPipe ["/usr/local/bin/gpg","--clearsign"] t
                else return t

lookForExecutable :: String -> IO (Maybe String)
lookForExecutable exe = do
  path <- get_path
  locations <- catMaybes `liftM` (sequence $ map (look_here exe) path)
  case locations of
      [] -> return Nothing
      (e:_) -> return $ Just e

look_here exe dir = do have <- doesFileExist $ dir ++ "/" ++ exe
                       if have then return $ Just $ dir ++ "/" ++ exe
                               else return Nothing

get_path :: IO [FilePath] -- FIXME should do this right!
get_path = return ["/bin","/usr/bin","/usr/local/bin","/opt/bin"]
