%  Copyright (C) 2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\section{darcs dist}
\begin{code}
module Dist ( dist ) where
import DarcsCommands
import DarcsArguments
import Directory
import IO
import System
import Monad ( when )
import Repository
import Patch
import SlurpDirectory
import RegexString
import RepoPrefs ( get_prefval )
import Lock ( withTempDir )
\end{code}

\options{dist}

\haskell{dist_description}

\begin{code}
dist_description =
 "Create a distribution tarball."
\end{code}

\haskell{dist_help} Basically, you will typically use it via a makefile
rule such as
\begin{verbatim}
dist:
    ./darcs dist --dist-name darcs-`./darcs --version`
\end{verbatim}
\verb!darcs dist! then simply creates a clean copy of the source tree,
which it then tars and gzips.  If you use programs such as autoconf or
automake, you really should run them on the clean tree before tarring it up
and distributing it.  You can do this using the pref value ``predist'',
which is a shell command that is run prior to tarring up the distribution:
\begin{verbatim}
% darcs setpref predist "autoconf && automake"
\end{verbatim}

\begin{code}
dist_help =
 "Dist is a handy tool for implementing a ``make dist'' target in your
makefile.  It creates a tarball of the recorded edition of your tree.
"
\end{code}

\begin{code}
dist = DarcsCommand {command_name = "dist",
                     command_help = dist_help,
                     command_description = dist_description,
                     command_extra_args = 0,
                     command_command = dist_cmd,
                     command_prereq = am_in_repo,
                     command_get_arg_possibilities = return [],
                     command_argdefaults = nodefaults,
                     command_darcsoptions = [distname_option, verbose]}
\end{code}

\begin{code}
dist_cmd opts _ = do
  dn <- get_dist_name opts
  verb <- return $ Verbose `elem` opts
  recorded <- slurp_recorded "."
  predist <- get_prefval "predist"
  formerdir <- getCurrentDirectory
  withTempDir ("_darcs/"++dn) $ \ddir -> do
    slurp_write recorded
    case predist of Nothing -> return ExitSuccess
                    Just pd -> system pd
    setCurrentDirectory (formerdir++"/_darcs")
    if verb
      then system("tar cvf - "++dn++" | gzip -c > "++formerdir++"/"++dn++".tar.gz")
      else system("tar cf  - "++dn++" | gzip -c > "++formerdir++"/"++dn++".tar.gz")
    putStr $ "Created dist as "++dn++".tar.gz\n"

guess_repo_name :: IO String
guess_repo_name = do
  pwd <- getCurrentDirectory
  case matchRegex (mkRegex ".*/([^/]*)$") pwd of
    Just (repo:_) -> return repo
    _ -> return "cantguessreponame"

get_dist_name :: [DarcsFlag] -> IO String
get_dist_name (DistName dn:_) = return dn
get_dist_name (_:fs) = get_dist_name fs
get_dist_name _ = guess_repo_name
\end{code}
