%  Copyright (C) 2002-2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\chapter{Diff}

\begin{code}
module Diff ( diff, sync, paranoid_diff, cmp ) where

import Posix ( EpochTime, setFileTimes, epochTime )
import FastPackedString
import IO
import Directory
import Monad ( liftM, when )
import List ( sort )

import SlurpDirectory
import Patch ( Patch, hunk, canonize, join_patches, reorder,
               submerge_in_dir, flatten, rmfile, rmdir,
               binary
             )
import RepoPrefs ( FileType(..) )
\end{code}

The diff function takes a recursive diff of two slurped-up directory trees.
The code involved is actually pretty trivial.  \verb!paranoid_diff! runs a
diff in which we don't make the assumption that files with the same
modification time are identical.

\begin{code}
diff :: (FilePath -> FileType) -> Slurpy -> Slurpy -> Maybe Patch
diff wt s1 s2
    | is_file s1 && is_file s2 &&
      (get_mtime s1 == undefined_time || get_mtime s1 /= get_mtime s2) &&
      get_length s1 == 0 && get_length s2 /= 0 =
          case wt n2 of
          TextFile -> Just $ hunk n2 1 [] c2 -- optimizing the 'add' case.
          BinaryFile -> Just $ binary n2 nilPS b2
    | is_file s1 && is_file s2 &&
      (get_mtime s1 == undefined_time || get_mtime s1 /= get_mtime s2) &&
      get_length s2 == 0 && get_length s1 /= 0 =
          case wt n2 of
          TextFile -> Just $ hunk n2 1 c1 [] -- optimizing the 'rm' case.
          BinaryFile -> Just $ binary n2 b1 nilPS
    | is_file s1 && is_file s2 &&
      (get_mtime s1 == undefined_time || get_mtime s1 /= get_mtime s2
       || get_length s1 /= get_length s2) =
          case wt n2 of
          TextFile -> canonize $! hunk n2 1 c1 c2
          BinaryFile -> Just $ binary n2 b1 b2
    | is_dir s1 && is_dir s2 =
        case recur_diff wt (get_dircontents s1) (get_dircontents s2) of
        [] -> Nothing
        ps -> Just $ reorder $ join_patches $ map (submerge_in_dir n2) ps
    | otherwise = Nothing
    where n2 = slurp_name s2
          c1 = fst $ get_filecontents s1
          c2 = fst $ get_filecontents s2
          b1 = getbin $ get_filecontents s1
          b2 = getbin $ get_filecontents s2
getbin :: FileContents -> PackedString
getbin (_,Just b) = b
getbin (c,Nothing) = unlinesPS c

recur_diff :: (FilePath -> FileType) -> [Slurpy] -> [Slurpy] -> [Patch]

recur_diff _ [] _ = []
recur_diff wt (s:ss) (s':ss')
    | s < s' = diff_removed wt s ++ recur_diff wt ss (s':ss')
    | s > s' = recur_diff wt (s:ss) ss'
    | s == s' =
        case diff wt s s' of
        Nothing -> rest
        Just p -> flatten p ++ rest
    where rest = recur_diff wt ss ss'

recur_diff wt (s:ss) [] = diff_removed wt s ++ recur_diff wt ss []
\end{code}

\begin{code}
paranoid_diff :: (FilePath -> FileType) -> Slurpy -> Slurpy -> Maybe Patch
paranoid_diff wt s1 s2
    | is_file s1 && is_file s2 && get_length s1 == 0 && get_length s2 /= 0 =
          case wt n2 of
          TextFile -> Just $ hunk n2 1 [] c2 -- optimizing the 'add' case.
          BinaryFile -> Just $ binary n2 nilPS b2
    | is_file s1 && is_file s2 && get_length s2 == 0 && get_length s1 /= 0 =
          case wt n2 of
          TextFile -> Just $ hunk n2 1 c1 [] -- optimizing the 'rm' case.
          BinaryFile -> Just $ binary n2 b1 nilPS
    | is_file s1 && is_file s2 =
          case wt n2 of
          TextFile -> canonize $! hunk n2 1 c1 c2
          BinaryFile -> Just $ binary n2 b1 b2
    | is_dir s1 && is_dir s2 =
        case p_recur_diff wt (get_dircontents s1) (get_dircontents s2) of
        [] -> Nothing
        ps -> Just $ reorder $ join_patches $ map (submerge_in_dir n2) ps
    | otherwise = Nothing
    where n2 = slurp_name s2
          c1 = fst $ get_filecontents s1
          c2 = fst $ get_filecontents s2
          b1 = getbin $ get_filecontents s1
          b2 = getbin $ get_filecontents s2

p_recur_diff :: (FilePath -> FileType) -> [Slurpy] -> [Slurpy] -> [Patch]

p_recur_diff _ [] _ = []
p_recur_diff wt (s:ss) (s':ss')
    | s < s' = diff_removed wt s ++ p_recur_diff wt ss (s':ss')
    | s > s' = p_recur_diff wt (s:ss) ss'
    | s == s' =
        case paranoid_diff wt s s' of
        Nothing -> rest
        Just p -> flatten p ++ rest
    where rest = p_recur_diff wt ss ss'
p_recur_diff wt (s:ss) [] = diff_removed wt s ++ p_recur_diff wt ss []
\end{code}

\begin{code}
diff_removed wt s
    | is_file s = case wt n of
                  TextFile -> [hunk n 1 (fst $ get_filecontents s) [],
                               rmfile n]
                  BinaryFile -> [binary n (getbin $ get_filecontents s) nilPS,
                                 rmfile n]
    | is_dir s = (map (submerge_in_dir n) $
                  foldl (++) [] $
                  map (diff_removed wt) $ get_dircontents s)
                 ++ [rmdir n]
    where n = slurp_name s
\end{code}

\begin{code}
sync :: String -> Slurpy -> Slurpy -> IO ()
sync path s1 s2
    | is_file s1 && is_file s2 &&
      (get_mtime s1 == undefined_time || get_mtime s1 /= get_mtime s2) &&
      get_length s1 == get_length s2 &&
      get_filecontents s1 == get_filecontents s2 =
        set_mtime n (get_mtime s2)
    | is_dir s1 && is_dir s2 = recur_sync n (get_dircontents s1) (get_dircontents s2)
    | otherwise = return ()
    where n = path++"/"++slurp_name s2

set_mtime fname ctime = do
    now <- epochTime
    setFileTimes fname now ctime

recur_sync _ [] _ = return ()
recur_sync path (s:ss) (s':ss')
    | s < s' = recur_sync path ss (s':ss')
    | s > s' = recur_sync path (s:ss) ss'
    | s == s' = do sync path s s'
                   recur_sync path ss ss'
recur_sync _ _ [] = return ()
\end{code}


\begin{code}
cmp :: FilePath -> FilePath -> IO Bool
cmp p1 p2 = do
  dir1 <- doesDirectoryExist p1
  dir2 <- doesDirectoryExist p2
  file1 <- doesFileExist p1
  file2 <- doesFileExist p2
  if dir1 && dir2
     then cmpdir p1 p2
     else if file1 && file2
          then cmpfile p1 p2
          else return False
cmpdir d1 d2 = do
  curdir <- getCurrentDirectory
  fn1 <- liftM (filter (\f->f/="." && f /="..")) $ getDirectoryContents d1
  fn2 <- liftM (filter (\f->f/="." && f /="..")) $ getDirectoryContents d2
  if sort fn1 /= sort fn2
     then return False
     else andIO $ map (\fn-> cmp (d1++"/"++fn) (d2++"/"++fn)) fn1
andIO :: [IO Bool] -> IO Bool
andIO (iob:iobs) = do b <- iob
                      if b then andIO iobs else return False
andIO [] = return True
cmpfile f1 f2 = do
  h1 <- openFile f1 ReadMode
  h2 <- openFile f2 ReadMode
  l1 <- hFileSize h1
  l2 <- hFileSize h2
  if l1 /= l2
     then do hClose h1
             hClose h2
             putStr $ "different file lengths for "++f1++" and "++f2++"\n"
             return False
     else do b <- hcmp h1 h2
             when (not b) $ putStr $ "files "++f1++" and "++f2++" differ\n"
             hClose h1
             hClose h2
             return b
hcmp h1 h2 = do
  c1 <- hGetPS h1 1024
  c2 <- hGetPS h2 1024
  if c1 == c2 && lengthPS c1 == 1024
     then hcmp h1 h2
     else if c1 == c2
          then return True
          else return False
\end{code}
