%  Copyright (C) 2002-2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\begin{code}
module DarcsArguments ( DarcsFlag( .. ), isa,
                        DarcsOption( .. ), option_from_darcsoption,
                        help, verbose, list_options, list_files,
                        notest, test,
                        list_registered_files, list_unregistered_files,
                        author, get_author,
                        patchname_option, distname_option,
                        all_patches, target, output, recursive,
                        edit_file, askdeps, ignoretimes,
                        sign, verify, edit_description,
                        reponame, tagname,
                        no_resolve_conflicts,
                        no_deps,
                        options_latex,
                        noskip_boring, allow_caseonly,
                      ) where
import System.Console.GetOpt
import Directory
import List
import System
import Monad ( liftM )
import IO ( hFlush, stdout )

import RepoPrefs ( boring_file_filter, darcsdir_filter, def_prefval )
\end{code}

\begin{code}
data DarcsFlag = Help | Verbose | ListOptions | NoTest | Test
               | Target String | Output String
               | Author String | PatchName String
               | DistName String | All | Recursive
               | AskDeps | IgnoreTimes | AnyOrder
               | Sign | Verify String | EditDescription
               | RepoName String | TagName String
               | NoResolve | Boring | AllowCaseOnly
               | DontGrabDeps
                 deriving ( Eq )
isa :: DarcsFlag -> (String -> DarcsFlag) -> Bool
f@(Target s) `isa` ft = f == (ft s)
f@(Output s) `isa` ft = f == (ft s)
f@(Author s) `isa` ft = f == (ft s)
f@(PatchName s) `isa` ft = f == (ft s)
f@(DistName s) `isa` ft = f == (ft s)
f@(Verify s) `isa` ft = f == (ft s)
f@(RepoName s) `isa` ft = f == (ft s)
f@(TagName s) `isa` ft = f == (ft s)
_ `isa` _ = False

data DarcsOption
    = DarcsArgOption [Char] [String] (String->DarcsFlag) String String
    | DarcsNoArgOption [Char] [String] DarcsFlag String
option_from_darcsoption (DarcsNoArgOption a b c h)
    = Option a b (NoArg c) h
option_from_darcsoption (DarcsArgOption a b c n h)
    = Option a b (ReqArg c n) h
\end{code}


\begin{code}
list_options = DarcsNoArgOption [] ["list-options"] ListOptions
               "simply list the command's arguments"
\end{code}

Every {\tt COMMAND} accepts \verb!--help! as an argument, which tells it to
provide a bit of help.  Among other things, this help always provides an
accurate listing of the options available with that command, and is
therefore frequently useful as a supplement to the somewhat spartan help
given in this manual.
\begin{verbatim}
% darcs COMMAND --help
\end{verbatim}
\begin{code}
help = DarcsNoArgOption ['h'] ["help"] Help
       "shows brief description of command and its arguments"
\end{code}

Most commands also accept the \verb!--verbose! option, which tells darcs to
provide additional output.  The amount of verbosity varies from command to
command.
\begin{code}
verbose = DarcsNoArgOption ['v'] ["verbose"] Verbose
          "give verbose output"
\end{code}

\begin{code}
notest = DarcsNoArgOption [] ["no-test"] NoTest "don't run the test script"
test = DarcsNoArgOption [] ["test"] Test "run the test script"
\end{code}

Darcs optimizes its operations by keeping track of the modification times
of your files.  This dramatically speeds up commands such as
\verb!whatsnew! and \verb!record! which would otherwise require reading
every file in the repo and comparing it with a reference version.  However,
there are times when this can cause problems, such as when running a series
of darcs commands from a script, in which case often a file will be
modified twice in the same second, which can lead to the second
modification going unnoticed.  The solution to such predicaments is the
\verb!--ignore-times! option, which instructs darcs not to trust the file
modification times, but instead to check each file's contents explicitely.
\begin{code}
ignoretimes = DarcsNoArgOption [] ["ignore-times"] IgnoreTimes
              "don't trust the file modification times"
\end{code}

\begin{code}
askdeps = DarcsNoArgOption [] ["ask-deps"] AskDeps "ask about dependencies"
\end{code}

Several commands need to be able to identify you.  Conventionally, you
provide an email address for this purpose.  The easiest way to do this is
to define an environment variable \verb!EMAIL! or \verb!DARCS_EMAIL! (with
the latter overriding the former).  You can also override this using the
\verb!--author! flag to any command.  Alternatively, you could set your
email address on a per-repository basis using the ``defaults'' mechanism
for ``ALL'' commands, as described in Appendix~\ref{repository_format}.

\begin{code}
author = DarcsArgOption ['A'] ["author"] Author "EMAIL" "specify author id"

get_author :: [DarcsFlag] -> IO String
get_author (Author a:_) = return a
get_author (_:flags) = get_author flags
get_author [] = do
  darcsemail <- getEnv "DARCS_EMAIL" `catch` (\e-> return "unknown")
  if darcsemail /= "unknown"
     then return darcsemail
     else do email <- getEnv "EMAIL" `catch` (\e-> return "unknown")
             if email /= "unknown"
                then return email
                else do putStr "What is your email address? "
                        hFlush stdout
                        getLine
\end{code}

\begin{code}
target = DarcsArgOption ['t'] ["to"] Target "EMAIL" "specify destination email"
\end{code}

\begin{code}
output = DarcsArgOption ['o'] ["output"] Output "FILE"
         "specify output filename"
\end{code}

\begin{code}
edit_description = DarcsNoArgOption [] ["edit-description"] EditDescription
                  "edit the patch bundle description"
\end{code}
\begin{code}
patchname_option = DarcsArgOption ['m'] ["patch-name"] PatchName "PATCHNAME"
                   "name of patch"
\end{code}

\begin{code}
distname_option = DarcsArgOption ['d'] ["dist-name"] DistName "DISTNAME"
                  "name of version"
\end{code}

\begin{code}
recursive h = DarcsNoArgOption ['r'] ["recursive"] Recursive h
\end{code}

\begin{code}
all_patches = DarcsNoArgOption ['a'] ["all"] All "answer yes to all patches"
\end{code}

\begin{code}
sign = DarcsNoArgOption ['s'] ["sign"] Sign "sign the patch with your gpg key"
\end{code}

\begin{code}
verify = DarcsArgOption ['V'] ["verify"] Verify "PUBRING"
         "verify that the patch was signed by a key in PUBRING"
\end{code}

\begin{code}
reponame = DarcsArgOption ['o'] ["repo-name"] RepoName "REPONAME"
           "name of output repository"
tagname = DarcsArgOption ['t'] ["tag-name"] TagName "TAGNAME"
          "name of version to pull"
no_deps = DarcsNoArgOption [] ["no-deps"] DontGrabDeps
                       "don't automatically fulfill dependencies"
\end{code}

\begin{code}
no_resolve_conflicts = DarcsNoArgOption [] ["no-resolve-conflicts"]
                       NoResolve "don't try to resolve conflicts"
\end{code}

\begin{code}
noskip_boring = DarcsNoArgOption [] ["boring"]
                Boring "don't skip boring files"
allow_caseonly = DarcsNoArgOption [] ["case-ok"]
                 AllowCaseOnly "don't refuse to add files differing only in case"
\end{code}

% The following code doesn't really belong here, but I'm not sure where to
% put it, as it certainly doesn't deserve its own module... or does it?

\begin{code}
a `catchall` b = a `catch` (\ _-> b)
a `ortryrunning` b = do ret <- a
                        if ret == ExitSuccess
                           then return ret
                           else b
edit_file :: String -> IO ExitCode
edit_file f = do
  ed <- get_editor
  system (ed++" "++f)
             `ortryrunning` system ("emacs "++f)
             `ortryrunning` system ("emacs -nw "++f)
             `ortryrunning` system ("nano "++f)
get_editor :: IO String
get_editor = getEnv "DARCSEDITOR" `catchall`
             getEnv "VISUAL" `catchall`
             getEnv "EDITOR" `catchall` return "vi"
\end{code}

\begin{code}
list_files :: IO [String]
list_files = do
    former_dir <- getCurrentDirectory
    skip_boring <- boring_file_filter
    fnames <- (filter (/= "_darcs") . skip_boring)
              `liftM` getDirectoryContents "."
    subdirlist <- do_this_list skip_boring fnames
    setCurrentDirectory former_dir
    return subdirlist

do_this_list :: ([String] -> [String]) -> [String] -> IO [String]
do_this_list _ [] = return []
do_this_list skip_boring (f:fs) = do
    isdir <- doesDirectoryExist f
    if isdir
       then do former_dir <- getCurrentDirectory
               setCurrentDirectory f
               fnames <- skip_boring `liftM` getDirectoryContents "."
               subdirlist <- do_this_list skip_boring fnames
               setCurrentDirectory former_dir
               rest <- do_this_list skip_boring fs
               return $ f : rest ++ map ((++) (f++"/")) subdirlist
               --return $ rest ++ map ((++) (f++"/")) subdirlist
       else do rest <- do_this_list skip_boring fs
               return $ f : rest

list_unregistered_files :: IO [String]
list_unregistered_files = do files <- list_files
                             registered_files <- list_registered_files
                             return $ files \\ registered_files

list_registered_files :: IO [String]
list_registered_files = do former_dir <- getCurrentDirectory
                           setCurrentDirectory "_darcs/current"
                           files <- list_files
                           setCurrentDirectory former_dir
                           return files
\end{code}

\begin{code}
options_latex opts = "\\begin{tabular}{lll}\n"++
                     unlines (map option_latex opts)++
                     "\\end{tabular}\n"
option_latex (DarcsNoArgOption a b c h) =
    show_short_options a ++ show_long_options b ++ h ++ "\\\\"
option_latex (DarcsArgOption a b c arg h) =
    show_short_options a ++
    show_long_options (map (++(" "++arg)) b) ++ h ++ "\\\\"
show_short_options :: [Char] -> String
show_short_options [] = "&"
show_short_options [c] = "\\verb!-"++[c]++"! &"
show_short_options (c:cs) = "\\verb!-"++[c]++"!,"++show_short_options cs

show_long_options [] = " &"
show_long_options [s] = "\\verb!--" ++ s ++ "! &"
show_long_options (s:ss)
    = "\\verb!--" ++ s ++ "!,"++ show_long_options ss
\end{code}
