/*
    GQ -- a GTK-based LDAP client
    Copyright (C) 1998-2001 Bert Vermeulen

    This program is released under the Gnu General Public License with
    the additional exemption that compiling, linking, and/or using
    OpenSSL is allowed.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include <config.h>

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>

#include "configfile.h"
#include "common.h"
#include "prefs.h"
#include "mainwin.h"
#include "util.h"
#include "template.h"
#include "errorchain.h"
#include "debug.h"


extern struct gq_config config;
extern struct tokenlist *token_bindtype[];

struct ldapserver *current_selected_server = NULL;
GtkWidget *current_edit_server_window = NULL;
GtkWidget *current_serverstab_serverclist = NULL;
GtkWidget *current_template_clist = NULL;


void server_edit_callback(GtkWidget *this, gpointer data)
{
     GtkWidget *window, *field;
     struct ldapserver *server, *servers;
     int server_name_changed;
     char *text;

     server = data;
     window = current_edit_server_window;

     /* Name */
     field = get_widget(window, "servername");
     text = gtk_entry_get_text(GTK_ENTRY(field));
     if(strcmp(text, server->name))
	  server_name_changed = 1;
     else
	  server_name_changed = 0;
     strncpy(server->name, text, MAX_SERVERNAME_LEN);

     /* make sure server name is unique */
     servers = config.ldapservers;
     while(servers->next) {
	  if(server != servers) {
	       if(!strcmp(servers->name, server->name)) {
		    /* already exists */
		    error_popup("Error adding new server",
				"A server by that name already exists\n\n"
				"Please choose another name");
		    return;
	       }
	  }
	  servers = servers->next;
     }

     /* LDAP host */
     field = get_widget(window, "ldaphost");
     text = gtk_entry_get_text(GTK_ENTRY(field));
     strncpy(server->ldaphost, text, MAX_HOSTNAME_LEN);

     /* LDAP port */
     field = get_widget(window, "ldapport");
     text = gtk_entry_get_text(GTK_ENTRY(field));
     server->ldapport = atoi(text);

     /* Base DN */
     field = get_widget(window, "basedn");
     text = gtk_entry_get_text(GTK_ENTRY(field));
     strncpy(server->basedn, text, MAX_DN_LEN);

     /* Bind DN */
     field = get_widget(window, "binddn");
     text = gtk_entry_get_text(GTK_ENTRY(field));
     strncpy(server->binddn, text, MAX_DN_LEN);

     /* Bind Password */
     field = get_widget(window, "bindpw");
     text = gtk_entry_get_text(GTK_ENTRY(field));
     strncpy(server->bindpw, text, MAX_BINDPW_LEN);

     /* Search attribute */
     field = get_widget(window, "searchattr");
     text = gtk_entry_get_text(GTK_ENTRY(field));
     strncpy(server->searchattr, text, MAX_ATTR_LEN);

     /* Maximum entries */
     field = get_widget(window, "maxentries");
     text = gtk_entry_get_text(GTK_ENTRY(field));
     server->maxentries = atoi(text);

     /* Cache connection */
     field = get_widget(window, "cacheconn");
     server->cacheconn = GTK_TOGGLE_BUTTON(field)->active ? 1 : 0;

     /* Enable TLS */
     field = get_widget(window, "enabletls");
     server->enabletls = GTK_TOGGLE_BUTTON(field)->active ? 1 : 0;

     /* Bind type */
     text = (char *) get_widget(window, "bindtype");
     server->bindtype = tokenize((struct tokenlist *)token_bindtype, text);

     /* connection info might have changed for this server -- close cached connection */
     close_connection(server, TRUE);

     if(server_name_changed) {

	  /* refresh clist in serverstab */
	  fill_serverlist_serverstab();

	  update_serverlist();

     }

     /* so the cancel button doesn't really cancel :-) */
     save_config();

     destroy_edit_server_window(NULL, NULL);

}


void destroy_edit_server_window(GtkWidget *this, gpointer data)
{

     gtk_widget_destroy(current_edit_server_window);
     current_edit_server_window = NULL;

}


gboolean destroy_edit_server_window_on_esc(GtkWidget *widget, GdkEventKey *event,
					   gpointer data)
{

     if(event && event->type == GDK_KEY_PRESS && event->keyval == GDK_Escape)
	  destroy_edit_server_window(NULL, NULL);

     return(TRUE);
}


void cancel_new_server_callback(gpointer data)
{

     delete_ldapserver(data);
     destroy_edit_server_window(NULL, NULL);

}


gboolean cancel_new_server_callback_on_esc(GtkWidget *widget, GdkEventKey *event,
				       gpointer data)
{

     if(event && event->type == GDK_KEY_PRESS && event->keyval == GDK_Escape)
	  cancel_new_server_callback(data);

     return(TRUE);
}


void bindtype_changed_callback(GtkEntry *entry, GtkWidget *window)
{

     gtk_object_set_data(GTK_OBJECT(window), "bindtype", gtk_entry_get_text(entry));

}


void create_edit_server_window(struct ldapserver *server)
{
     GtkWidget *editwindow, *notebook;
     GtkWidget *main_vbox, *vbox1, *vbox2, *hbox;
     GtkWidget *table1, *table2;
     GtkWidget *label, *entry, *button;
     GtkWidget *okbutton, *cancelbutton;
     int y, editing_new_server;
     char title[MAX_SERVERNAME_LEN + 8];
     char tmp[16];
     GtkWidget *bindtype;
     GList *bindtypes;

     if(current_edit_server_window)
	  return;

     if(server)
	  editing_new_server = 0;
     else
	  editing_new_server = 1;

     if(server == NULL) {
	  server = new_ldapserver();	  
	  init_ldapserver(server);
     }

     editwindow = gtk_window_new(GTK_WINDOW_DIALOG);
     current_edit_server_window = editwindow;

     if(editing_new_server)
	  sprintf(title, "New server");
     else
	  sprintf(title, "Server %s", server->name);
     gtk_window_set_title(GTK_WINDOW(editwindow), title);
     gtk_window_set_policy(GTK_WINDOW(editwindow), FALSE, FALSE, FALSE);

     main_vbox = gtk_vbox_new(FALSE, 0);
     gtk_container_border_width(GTK_CONTAINER(main_vbox), 10);
     gtk_widget_show(main_vbox);
     gtk_container_add(GTK_CONTAINER(editwindow), main_vbox);

     notebook = gtk_notebook_new();
     gtk_widget_show(notebook);
     GTK_WIDGET_UNSET_FLAGS(GTK_NOTEBOOK(notebook), GTK_CAN_FOCUS);
     gtk_box_pack_start(GTK_BOX(main_vbox), notebook, TRUE, TRUE, 0);


     /* "General" tab */

     vbox1 = gtk_vbox_new(FALSE, 20);
     gtk_object_set_data(GTK_OBJECT(editwindow), "vbox1", vbox1);
     gtk_widget_show(vbox1);
     gtk_container_border_width(GTK_CONTAINER(vbox1), 10);
     label = gtk_label_new("General");
     gtk_widget_show(label);
     gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox1, label);

     table1 = gtk_table_new(4, 2, FALSE);
     gtk_object_set_data(GTK_OBJECT(editwindow), "table1", table1);
     gtk_widget_show(table1);
     gtk_box_pack_start(GTK_BOX(vbox1), table1, FALSE, FALSE, 0);
     gtk_container_border_width(GTK_CONTAINER(table1), 10);
     gtk_table_set_row_spacings(GTK_TABLE(table1), 5);
     gtk_table_set_col_spacings(GTK_TABLE(table1), 13);

     /* Name */
     label = gtk_label_new("Name");
     gtk_misc_set_alignment(GTK_MISC(label), 0.0, .5);
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(table1), label, 0, 1, 0, 1,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
     gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_RIGHT);

     entry = gtk_entry_new();
     gtk_object_set_data(GTK_OBJECT(editwindow), "servername", entry);
     gtk_entry_set_text(GTK_ENTRY(entry), server->name);
     gtk_widget_show(entry);
     gtk_signal_connect(GTK_OBJECT(entry), "activate",
			GTK_SIGNAL_FUNC(server_edit_callback), server);
     gtk_table_attach(GTK_TABLE(table1), entry, 1, 2, 0, 1,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
     gtk_widget_grab_focus(entry);

     /* LDAP host */
     label = gtk_label_new("LDAP host");
     gtk_misc_set_alignment(GTK_MISC(label), 0.0, .5);
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(table1), label, 0, 1, 1, 2,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     entry = gtk_entry_new();
     gtk_object_set_data(GTK_OBJECT(editwindow), "ldaphost", entry);
     gtk_entry_set_text(GTK_ENTRY(entry), server->ldaphost);
     gtk_widget_show(entry);
     gtk_signal_connect(GTK_OBJECT(entry), "activate",
			GTK_SIGNAL_FUNC(server_edit_callback), server);
     gtk_table_attach(GTK_TABLE(table1), entry, 1, 2, 1, 2,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     /* Port */
     label = gtk_label_new("LDAP Port");
     gtk_misc_set_alignment(GTK_MISC(label), 0.0, .5);
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(table1), label, 0, 1, 2, 3,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     entry = gtk_entry_new();
     gtk_object_set_data(GTK_OBJECT(editwindow), "ldapport", entry);
     sprintf(tmp, "%d", server->ldapport);
     gtk_entry_set_text(GTK_ENTRY(entry), tmp);
     gtk_widget_show(entry);
     gtk_signal_connect(GTK_OBJECT(entry), "activate",
			GTK_SIGNAL_FUNC(server_edit_callback), server);
     gtk_table_attach(GTK_TABLE(table1), entry, 1, 2, 2, 3,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     /* Base DN */
     label = gtk_label_new("Base DN");
     gtk_misc_set_alignment(GTK_MISC(label), 0.0, .5);
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(table1), label, 0, 1, 3, 4,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     entry = gtk_entry_new();
     gtk_object_set_data(GTK_OBJECT(editwindow), "basedn", entry);
     gtk_entry_set_text(GTK_ENTRY(entry), server->basedn);
     gtk_widget_show(entry);
     gtk_signal_connect(GTK_OBJECT(entry), "activate",
			GTK_SIGNAL_FUNC(server_edit_callback), server);
     gtk_table_attach(GTK_TABLE(table1), entry, 1, 2, 3, 4,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);


     /* "Details" tab */

     vbox2 = gtk_vbox_new(FALSE, 0);
     gtk_object_set_data(GTK_OBJECT(editwindow), "vbox2", vbox2);
     gtk_widget_show(vbox2);
     gtk_container_border_width(GTK_CONTAINER(vbox2), 10);
     label = gtk_label_new("Details");
     gtk_widget_show(label);
     gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox2, label);

     table2 = gtk_table_new(8, 2, FALSE);
     gtk_object_set_data(GTK_OBJECT(editwindow), "table2", table2);
     gtk_widget_show(table2);
     gtk_box_pack_start(GTK_BOX(vbox2), table2, TRUE, TRUE, 0);
     gtk_container_border_width(GTK_CONTAINER(table2), 10);
     gtk_table_set_row_spacings(GTK_TABLE(table2), 5);
     gtk_table_set_col_spacings(GTK_TABLE(table2), 13);
     y = 0;

     /* Bind DN */
     label = gtk_label_new("Bind DN");
     gtk_misc_set_alignment(GTK_MISC(label), 0.0, .5);
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(table2), label, 0, 1, y, y + 1,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     entry = gtk_entry_new();
     gtk_object_set_data(GTK_OBJECT(editwindow), "binddn", entry);
     gtk_entry_set_text(GTK_ENTRY(entry), server->binddn);
     gtk_widget_show(entry);
     gtk_signal_connect(GTK_OBJECT(entry), "activate",
			GTK_SIGNAL_FUNC(server_edit_callback), server);
     gtk_table_attach(GTK_TABLE(table2), entry, 1, 2, y, y + 1,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
     y++;

     /* Bind Password */
     label = gtk_label_new("Bind Password");
     gtk_misc_set_alignment(GTK_MISC(label), 0.0, .5);
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(table2), label, 0, 1, y, y + 1,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     entry = gtk_entry_new();
     gtk_object_set_data(GTK_OBJECT(editwindow), "bindpw", entry);
     gtk_entry_set_text(GTK_ENTRY(entry), server->bindpw);
     gtk_widget_show(entry);
     gtk_signal_connect(GTK_OBJECT(entry), "activate",
			GTK_SIGNAL_FUNC(server_edit_callback), server);
     gtk_table_attach(GTK_TABLE(table2), entry, 1, 2, y, y + 1,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
     y++;

     /* Bind type */
     label = gtk_label_new("Bind type");
     gtk_misc_set_alignment(GTK_MISC(label), 0.0, .5);
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(table2), label, 0, 1, y, y + 1,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     bindtype = gtk_combo_new();
     bindtypes = g_list_append(NULL, detokenize((struct tokenlist *) token_bindtype,
						BINDTYPE_SIMPLE));
     bindtypes = g_list_append(bindtypes, detokenize((struct tokenlist *) token_bindtype,
						     BINDTYPE_KERBEROS));
     gtk_object_set_data(GTK_OBJECT(editwindow), "bindtype",
			 detokenize((struct tokenlist *) token_bindtype, BINDTYPE_SIMPLE));
     gtk_combo_set_popdown_strings(GTK_COMBO(bindtype), bindtypes);
     gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(bindtype)->entry), FALSE);
     gtk_signal_connect(GTK_OBJECT(GTK_COMBO(bindtype)->entry), "changed",
			GTK_SIGNAL_FUNC(bindtype_changed_callback),
			(gpointer) editwindow);
     gtk_widget_show(bindtype);
     gtk_list_select_item(GTK_LIST(GTK_COMBO(bindtype)->list), server->bindtype);
     gtk_table_attach(GTK_TABLE(table2), bindtype, 1, 2, y, y + 1,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
     y++;

     /* Search attribute */
     label = gtk_label_new("Search Attribute");
     gtk_misc_set_alignment(GTK_MISC(label), 0.0, .5);
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(table2), label, 0, 1, y, y + 1,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     entry = gtk_entry_new();
     gtk_object_set_data(GTK_OBJECT(editwindow), "searchattr", entry);
     gtk_entry_set_text(GTK_ENTRY(entry), server->searchattr);
     gtk_widget_show(entry);
     gtk_signal_connect(GTK_OBJECT(entry), "activate",
			GTK_SIGNAL_FUNC(server_edit_callback), server);
     gtk_table_attach(GTK_TABLE(table2), entry, 1, 2, y, y + 1,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
     y++;

     /* Maximum entries */
     label = gtk_label_new("Maximum entries");
     gtk_misc_set_alignment(GTK_MISC(label), 0.0, .5);
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(table2), label, 0, 1, y, y + 1,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     entry = gtk_entry_new();
     gtk_object_set_data(GTK_OBJECT(editwindow), "maxentries", entry);
     sprintf(tmp, "%d", server->maxentries);
     gtk_entry_set_text(GTK_ENTRY(entry), tmp);
     gtk_widget_show(entry);
     gtk_signal_connect(GTK_OBJECT(entry), "activate",
			GTK_SIGNAL_FUNC(server_edit_callback), server);
     gtk_table_attach(GTK_TABLE(table2), entry, 1, 2, y, y + 1,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

     /* Cache connections */
     button = gtk_check_button_new_with_label("Cache connection");
     gtk_object_set_data(GTK_OBJECT(editwindow), "cacheconn", button);
     if(server->cacheconn)
	  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(button), TRUE);
     GTK_WIDGET_UNSET_FLAGS(GTK_CHECK_BUTTON(button), GTK_CAN_FOCUS);
     gtk_widget_show(button);
     gtk_box_pack_start(GTK_BOX(vbox2), button, FALSE, TRUE, 2);

     /* Enable TLS */
     button = gtk_check_button_new_with_label("Enable TLS");
     gtk_object_set_data(GTK_OBJECT(editwindow), "enabletls", button);
     if(server->enabletls)
	  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(button), TRUE);
     GTK_WIDGET_UNSET_FLAGS(GTK_CHECK_BUTTON(button), GTK_CAN_FOCUS);
     gtk_widget_show(button);
     gtk_box_pack_start(GTK_BOX(vbox2), button, FALSE, TRUE, 0);


     /* OK and Cancel buttons */
     hbox = gtk_hbox_new(FALSE, 13);
     gtk_container_border_width(GTK_CONTAINER(hbox), 10);
     gtk_widget_show(hbox);
     gtk_box_pack_start(GTK_BOX(main_vbox), hbox, FALSE, TRUE, 0);

     okbutton = gtk_button_new_with_label("  OK  ");
     gtk_widget_show(okbutton);
     gtk_signal_connect(GTK_OBJECT(okbutton), "clicked",
			GTK_SIGNAL_FUNC(server_edit_callback),
			(gpointer) server);
     gtk_box_pack_start(GTK_BOX(hbox), okbutton, FALSE, TRUE, 10);
     GTK_WIDGET_SET_FLAGS(okbutton, GTK_CAN_DEFAULT);
     gtk_widget_grab_default(okbutton);

     cancelbutton = gtk_button_new_with_label(" Cancel ");
     gtk_widget_show(cancelbutton);
     if(editing_new_server) {
	  gtk_signal_connect_object(GTK_OBJECT(cancelbutton), "clicked",
				    GTK_SIGNAL_FUNC(cancel_new_server_callback),
				    (gpointer) server);
	  gtk_signal_connect_object(GTK_OBJECT(editwindow), "delete_event",
				    GTK_SIGNAL_FUNC(cancel_new_server_callback),
				    (gpointer) server);
	  gtk_signal_connect(GTK_OBJECT(editwindow), "key_press_event",
			     GTK_SIGNAL_FUNC(cancel_new_server_callback_on_esc),
			     (gpointer) server);
     }
     else {
	  gtk_signal_connect(GTK_OBJECT(cancelbutton), "clicked",
			     GTK_SIGNAL_FUNC(destroy_edit_server_window),
			     NULL);
	  gtk_signal_connect(GTK_OBJECT(editwindow), "delete_event",
			     GTK_SIGNAL_FUNC(destroy_edit_server_window),
			     NULL);
	  gtk_signal_connect(GTK_OBJECT(editwindow), "key_press_event",
			     GTK_SIGNAL_FUNC(destroy_edit_server_window_on_esc),
			     NULL);
     }

     gtk_box_pack_end(GTK_BOX(hbox), cancelbutton, FALSE, TRUE, 10);

     gtk_widget_show(editwindow);

}


void serverstab_deletebutton_callback(GtkWidget *widget, GtkWidget *widget2)
{

     if(current_selected_server) {
	  delete_ldapserver(current_selected_server);
	  current_selected_server = NULL;

	  save_config();

	  fill_serverlist_serverstab();
	  update_serverlist();
     }

}


void serverstab_newbutton_callback(GtkWidget *widget, GtkWidget *clist)
{

     create_edit_server_window(NULL);

}


void serverstab_editbutton_callback(GtkWidget *widget, GtkWidget *clist)
{

     /* quietly ignore editbutton if no server selected */
     if(current_selected_server)
	  create_edit_server_window(current_selected_server);

}


void server_selected_callback(GtkWidget *clist, gint row, gint column,
			      GdkEventButton *event, gpointer data)
{
     struct ldapserver *server;

     server = gtk_clist_get_row_data(GTK_CLIST(clist), row);

     /* never actually getting GDK_BUTTON_PRESS here for some reason */
     if(event->type == GDK_BUTTON_PRESS || event->type == GDK_BUTTON_RELEASE)
	  current_selected_server = server;
     else if(event->type == GDK_2BUTTON_PRESS) {
	  current_selected_server = server;
	  create_edit_server_window(server);
     }

}


void server_unselected_callback(GtkWidget *clist, gint row, gint column,
			      GdkEventButton *event, gpointer data)
{

     current_selected_server = NULL;

}


void fill_serverlist_serverstab(void)
{
     GtkWidget *clist;
     struct ldapserver *server;
     int row;
     char *serverlist[1];

     clist = current_serverstab_serverclist;
     if(!GTK_IS_CLIST(clist))
	  return;

     gtk_clist_freeze(GTK_CLIST(clist));
     gtk_clist_clear(GTK_CLIST(clist));

     row = 0;
     server = config.ldapservers;
     while(server) {
	  serverlist[0] = server->name;
	  gtk_clist_append(GTK_CLIST(clist), serverlist);
	  gtk_clist_set_row_data(GTK_CLIST(clist), row, (gpointer) server);
	  server = server->next;
	  row++;
     }

     gtk_clist_thaw(GTK_CLIST(clist));

}


void prefs_okbutton_callback(GtkWidget *this)
{
     GtkWidget *widget;
     GtkCombo *combo;
     int type;
     char *search_options[] = {
	  "st_bw",
	  "st_ew",
	  "st_co",
	  "st_eq",
	  ""
     }, *ldif_format[] = {
	  "format_umich",
	  "format_v1",
	  ""
     };

     /* Search type */
     for(type = 0; search_options[type][0]; type++) {
	  widget = gtk_object_get_data(GTK_OBJECT(this),
				   search_options[type]);
	  if(GTK_TOGGLE_BUTTON(widget)->active)
	       break;
     }
     config.search_argument = type;

     /* Show DN */
     widget = gtk_object_get_data(GTK_OBJECT(this), "showdn");
     config.showdn = GTK_TOGGLE_BUTTON(widget)->active ? 1 : 0;

     /* Show OC */
     widget = gtk_object_get_data(GTK_OBJECT(this), "showoc");
     config.showoc = GTK_TOGGLE_BUTTON(widget)->active ? 1 : 0;

     /* Show RDN only */
     widget = gtk_object_get_data(GTK_OBJECT(this), "show_rdn_only");
     config.show_rdn_only = GTK_TOGGLE_BUTTON(widget)->active ? 1 : 0;

     /* Sorting: search mode */
     widget = gtk_object_get_data(GTK_OBJECT(this), "sort_search");
     config.sort_search = GTK_TOGGLE_BUTTON(widget)->active ? 1 : 0;

     /* Sorting: browse mode */
     widget = gtk_object_get_data(GTK_OBJECT(this), "sort_browse");
     config.sort_browse = GTK_TOGGLE_BUTTON(widget)->active ? 1 : 0;

     /* LDIF: format */
     for(type = 0; ldif_format[type][0]; type++) {
	  widget = gtk_object_get_data(GTK_OBJECT(this), ldif_format[type]);
	  if(GTK_TOGGLE_BUTTON(widget)->active)
	       break;
     }
     config.ldifformat = type;

     combo = gtk_object_get_data(GTK_OBJECT(this), "schemaserver");
     if(combo)
	  strncpy(config.schemaserver, gtk_entry_get_text(GTK_ENTRY(combo->entry)), MAX_SERVERNAME_LEN);

     save_config();

     gtk_widget_destroy(this);

}


void create_prefs_window(void)
{
     GtkWidget *prefswindow;
     GtkWidget *label, *vbox2;
     GtkWidget *notebook;
     GtkWidget *vbox_search_options, *vbox_browse_options, *vbox_servers, *vbox_templates, *vbox_ldif;
     GtkWidget *hbox_buttons, *okbutton, *cancelbutton;

     prefswindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);
     gtk_widget_set_usize(prefswindow, 420, 370);
     gtk_container_border_width(GTK_CONTAINER(prefswindow), 5);
     gtk_window_set_title(GTK_WINDOW(prefswindow), "Preferences");
     gtk_window_set_policy(GTK_WINDOW(prefswindow), FALSE, FALSE, FALSE);
     gtk_signal_connect_object(GTK_OBJECT(prefswindow), "key_press_event",
			       GTK_SIGNAL_FUNC(close_on_esc),
			       (gpointer) prefswindow);

     vbox2 = gtk_vbox_new(FALSE, 0);
     gtk_widget_show(vbox2);
     gtk_container_add(GTK_CONTAINER(prefswindow), vbox2);

     notebook = gtk_notebook_new();
     gtk_widget_show(notebook);
     gtk_box_pack_start(GTK_BOX(vbox2), notebook, TRUE, TRUE, 0);

     /* Search Options tab */
     vbox_search_options = gtk_vbox_new(FALSE, 0);
     gtk_container_border_width(GTK_CONTAINER(vbox_search_options), 10);
     create_search_optionstab(vbox_search_options, prefswindow);
     gtk_widget_show(vbox_search_options);
     label = gtk_label_new("Search Options");
     gtk_widget_show(label);
     GTK_WIDGET_UNSET_FLAGS(GTK_NOTEBOOK(notebook), GTK_CAN_FOCUS);
     gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox_search_options, label);

     /* Browse Options tab */
     vbox_browse_options = gtk_vbox_new(FALSE, 0);
     gtk_container_border_width(GTK_CONTAINER(vbox_browse_options), 10);
     create_browse_optionstab(vbox_browse_options, prefswindow);
     gtk_widget_show(vbox_browse_options);
     label = gtk_label_new("Browse Options");
     gtk_widget_show(label);
     gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox_browse_options, label);

     /* Servers tab */
     vbox_servers = gtk_vbox_new(FALSE, 0);
     create_serverstab(vbox_servers);
     gtk_widget_show(vbox_servers);
     label = gtk_label_new("Servers");
     gtk_widget_show(label);
     gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox_servers, label);

#ifdef HAVE_LDAP_STR2OBJECTCLASS
     /* Templates tab */
     vbox_templates = gtk_vbox_new(FALSE, 0);
     create_templatestab(vbox_templates, prefswindow);
     gtk_widget_show(vbox_templates);
     label = gtk_label_new("Templates");
     gtk_widget_show(label);
     gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox_templates, label);
#else
     vbox_templates = NULL;
#endif

     /* LDIF tab */
     vbox_ldif = gtk_vbox_new(FALSE, 0);
     gtk_container_border_width(GTK_CONTAINER(vbox_ldif), 10);
     create_ldiftab(vbox_ldif, prefswindow);
     gtk_widget_show(vbox_ldif);
     label = gtk_label_new("LDIF");
     gtk_widget_show(label);
     gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox_ldif, label);

     /* OK and Cancel buttons outside notebook */
     hbox_buttons = gtk_hbox_new(FALSE, 0);
     gtk_widget_show(hbox_buttons);
     gtk_box_pack_start(GTK_BOX(vbox2), hbox_buttons, FALSE, TRUE, 0);
     gtk_container_border_width(GTK_CONTAINER(hbox_buttons), 10);

     okbutton = gtk_button_new_with_label("  OK  ");
     gtk_widget_show(okbutton);
     gtk_signal_connect_object(GTK_OBJECT(okbutton), "pressed",
			       GTK_SIGNAL_FUNC(prefs_okbutton_callback),
			       GTK_OBJECT(prefswindow));
     gtk_box_pack_start(GTK_BOX(hbox_buttons), okbutton, FALSE, TRUE, 0);
     GTK_WIDGET_SET_FLAGS(okbutton, GTK_CAN_DEFAULT);
     gtk_widget_grab_focus(okbutton);
     gtk_widget_grab_default(okbutton);

     cancelbutton = gtk_button_new_with_label("  Cancel  ");
     gtk_widget_show(cancelbutton);
     gtk_box_pack_end(GTK_BOX(hbox_buttons), cancelbutton, FALSE, TRUE, 0);
     gtk_signal_connect_object(GTK_OBJECT(cancelbutton), "pressed",
			       GTK_SIGNAL_FUNC(gtk_widget_destroy),
			       GTK_OBJECT(prefswindow));

     gtk_widget_show(prefswindow);

}


void create_serverstab(GtkWidget *target)
{
     GtkWidget *vbox1, *vbox2, *hbox1, *hbox2, *scrwin;
     GtkWidget *button_new, *button_edit, *button_delete;
     GtkWidget *server_clist;

     vbox1 = gtk_vbox_new(FALSE, 0);
     gtk_widget_show(vbox1);
     gtk_container_add(GTK_CONTAINER(target), vbox1);
     gtk_container_border_width(GTK_CONTAINER(vbox1), 10);

     hbox2 = gtk_hbox_new(FALSE, 25);
     gtk_widget_show(hbox2);
     gtk_box_pack_start(GTK_BOX(vbox1), hbox2, TRUE, TRUE, 0);

     /* scrolled window to hold the server clist */
     scrwin = gtk_scrolled_window_new(NULL, NULL);
     gtk_widget_show(scrwin);
     gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrwin),
				    GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
     gtk_box_pack_start(GTK_BOX(hbox2), scrwin, TRUE, TRUE, 0);

     server_clist = gtk_clist_new(1);
     GTK_WIDGET_UNSET_FLAGS(server_clist, GTK_CAN_FOCUS);
     current_serverstab_serverclist = server_clist;
     gtk_widget_set_usize(scrwin, 200, 300);
     gtk_clist_set_selection_mode(GTK_CLIST(server_clist),
				  GTK_SELECTION_SINGLE);
     gtk_clist_set_shadow_type(GTK_CLIST(server_clist), GTK_SHADOW_ETCHED_IN);
     gtk_widget_show(server_clist);
     gtk_signal_connect(GTK_OBJECT(server_clist), "select_row",
			GTK_SIGNAL_FUNC(server_selected_callback), NULL);
     gtk_signal_connect(GTK_OBJECT(server_clist), "unselect_row",
			GTK_SIGNAL_FUNC(server_unselected_callback), NULL);
     gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrwin), server_clist);
     fill_serverlist_serverstab();

     vbox2 = gtk_vbox_new(FALSE, 10);
     gtk_widget_show(vbox2);
     gtk_box_pack_start(GTK_BOX(hbox2), vbox2, TRUE, TRUE, 0);

     /* New button */
     button_new = gtk_button_new_with_label("New");
     GTK_WIDGET_UNSET_FLAGS(button_new, GTK_CAN_FOCUS);
     gtk_object_set_data(GTK_OBJECT(target), "button_new", button_new);
     gtk_widget_show(button_new);
     gtk_signal_connect(GTK_OBJECT(button_new), "clicked",
			GTK_SIGNAL_FUNC(serverstab_newbutton_callback),
			NULL);
     gtk_box_pack_start(GTK_BOX(vbox2), button_new, FALSE, TRUE, 0);

     /* Edit button */
     button_edit = gtk_button_new_with_label("Edit");
     GTK_WIDGET_UNSET_FLAGS(button_edit, GTK_CAN_FOCUS);
     gtk_object_set_data(GTK_OBJECT(target), "button_edit", button_edit);
     gtk_widget_show(button_edit);
     gtk_signal_connect(GTK_OBJECT(button_edit), "clicked",
			GTK_SIGNAL_FUNC(serverstab_editbutton_callback),
			server_clist);
     gtk_box_pack_start(GTK_BOX(vbox2), button_edit, FALSE, TRUE, 0);

     /* Delete button */
     button_delete = gtk_button_new_with_label("Delete");
     GTK_WIDGET_UNSET_FLAGS(button_delete, GTK_CAN_FOCUS);
     gtk_object_set_data(GTK_OBJECT(target), "button_delete", button_delete);
     gtk_widget_show(button_delete);
     gtk_signal_connect(GTK_OBJECT(button_delete), "clicked",
			GTK_SIGNAL_FUNC(serverstab_deletebutton_callback),
			NULL);
     gtk_box_pack_start(GTK_BOX(vbox2), button_delete, FALSE, TRUE, 0);

     hbox1 = gtk_hbox_new(FALSE, 10);
     gtk_widget_show(hbox1);
     gtk_box_pack_start(GTK_BOX(vbox1), hbox1, FALSE, TRUE, 0);
     gtk_container_border_width(GTK_CONTAINER(hbox1), 10);

}


#ifdef HAVE_LDAP_STR2OBJECTCLASS
void create_templatestab(GtkWidget *target, GtkWidget *targetwindow)
{
     GtkWidget *vbox1, *vbox2, *hbox1, *hbox2, *scrwin;
     GtkWidget *button_new, *button_edit, *button_delete;
     GtkWidget *template_clist, *label, *combo;

     vbox1 = gtk_vbox_new(FALSE, 0);
     gtk_widget_show(vbox1);
     gtk_container_add(GTK_CONTAINER(target), vbox1);
     gtk_container_border_width(GTK_CONTAINER(vbox1), 10);

     hbox1 = gtk_hbox_new(FALSE, 10);
     gtk_widget_show(hbox1);
     gtk_box_pack_start(GTK_BOX(vbox1), hbox1, FALSE, FALSE, 0);


     /* Schema server */
     label = gtk_label_new("Last resort schema server");
     gtk_widget_show(label);
     gtk_box_pack_start(GTK_BOX(hbox1), label, FALSE, FALSE, 0);

     combo = gtk_combo_new();
     gtk_object_set_data(GTK_OBJECT(targetwindow), "schemaserver", combo);
     GTK_WIDGET_UNSET_FLAGS(GTK_COMBO(combo)->entry, GTK_CAN_FOCUS);
     fill_serverlist_combo(combo);
     if(strlen(config.schemaserver))
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(combo)->entry), config.schemaserver);
     gtk_widget_show(combo);
     gtk_box_pack_start(GTK_BOX(hbox1), combo, FALSE, FALSE, 0);

     hbox2 = gtk_hbox_new(FALSE, 25);
     gtk_widget_show(hbox2);
     gtk_box_pack_start(GTK_BOX(vbox1), hbox2, TRUE, TRUE, 10);

     /* scrolled window to hold the server clist */
     scrwin = gtk_scrolled_window_new(NULL, NULL);
     gtk_widget_set_usize(scrwin, 200, 300);
     gtk_widget_show(scrwin);
     gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrwin),
				    GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
     gtk_box_pack_start(GTK_BOX(hbox2), scrwin, TRUE, TRUE, 0);

     template_clist = gtk_clist_new(1);
     current_template_clist = template_clist;
     gtk_object_set_data(GTK_OBJECT(targetwindow), "templatelist", template_clist);
     GTK_WIDGET_UNSET_FLAGS(template_clist, GTK_CAN_FOCUS);
     gtk_clist_set_selection_mode(GTK_CLIST(template_clist),
				  GTK_SELECTION_SINGLE);
     gtk_clist_set_shadow_type(GTK_CLIST(template_clist), GTK_SHADOW_ETCHED_IN);
     gtk_signal_connect(GTK_OBJECT(template_clist), "select_row",
			GTK_SIGNAL_FUNC(template_selected_callback), targetwindow);
     gtk_widget_show(template_clist);
     gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrwin), template_clist);

     fill_clist_templates(template_clist);

     vbox2 = gtk_vbox_new(FALSE, 10);
     gtk_widget_show(vbox2);
     gtk_box_pack_start(GTK_BOX(hbox2), vbox2, TRUE, TRUE, 0);

     /* New button */
     button_new = gtk_button_new_with_label("New");
     GTK_WIDGET_UNSET_FLAGS(button_new, GTK_CAN_FOCUS);
     gtk_object_set_data(GTK_OBJECT(target), "button_new", button_new);
     gtk_widget_show(button_new);
     gtk_signal_connect(GTK_OBJECT(button_new), "clicked",
			GTK_SIGNAL_FUNC(template_new_callback),
			targetwindow);
     gtk_box_pack_start(GTK_BOX(vbox2), button_new, FALSE, TRUE, 0);

     /* Edit button */
     button_edit = gtk_button_new_with_label("Edit");
     GTK_WIDGET_UNSET_FLAGS(button_edit, GTK_CAN_FOCUS);
     gtk_object_set_data(GTK_OBJECT(target), "button_edit", button_edit);
     gtk_widget_show(button_edit);
     gtk_signal_connect(GTK_OBJECT(button_edit), "clicked",
			GTK_SIGNAL_FUNC(template_edit_callback),
			targetwindow);
     gtk_box_pack_start(GTK_BOX(vbox2), button_edit, FALSE, TRUE, 0);

     /* Delete button */
     button_delete = gtk_button_new_with_label("Delete");
     GTK_WIDGET_UNSET_FLAGS(button_delete, GTK_CAN_FOCUS);
     gtk_object_set_data(GTK_OBJECT(target), "button_delete", button_delete);
     gtk_widget_show(button_delete);
     gtk_signal_connect(GTK_OBJECT(button_delete), "clicked",
			GTK_SIGNAL_FUNC(template_delete_callback),
			targetwindow);
     gtk_box_pack_start(GTK_BOX(vbox2), button_delete, FALSE, TRUE, 0);

}

#endif     

void create_search_optionstab(GtkWidget *target, GtkWidget *targetwindow)
{
     GtkWidget *stframe, *vbox_st, *viewframe, *vbox_view;
     GtkWidget *stradiobutton, *dnbutton, *ocbutton;
     GtkWidget *hbox_options, *sort_search_button;
     GSList *stgroup;

     hbox_options = gtk_hbox_new(TRUE, 10);
     gtk_widget_show(hbox_options);
     gtk_box_pack_start(GTK_BOX(target), hbox_options, FALSE, TRUE, 5);

     /* Search type frame in Options tab */
     stframe = gtk_frame_new(" Search type ");
     gtk_widget_show(stframe);
     gtk_box_pack_start(GTK_BOX(hbox_options), stframe, FALSE, TRUE, 0);
     vbox_st = gtk_vbox_new(TRUE, 0);
     gtk_container_border_width(GTK_CONTAINER(vbox_st), 5);
     gtk_widget_show(vbox_st);
     gtk_container_add(GTK_CONTAINER(stframe), vbox_st);

     stradiobutton = gtk_radio_button_new_with_label(NULL, "Begins with");
     if(config.search_argument == SEARCHARG_BEGINS_WITH)
	  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(stradiobutton), TRUE);
     GTK_WIDGET_UNSET_FLAGS(GTK_RADIO_BUTTON(stradiobutton), GTK_CAN_FOCUS);
     gtk_object_set_data(GTK_OBJECT(targetwindow), "st_bw", stradiobutton);
     gtk_box_pack_start(GTK_BOX(vbox_st), stradiobutton, TRUE, TRUE, 3);
     gtk_widget_show(stradiobutton);

     stgroup = gtk_radio_button_group(GTK_RADIO_BUTTON(stradiobutton));
     stradiobutton = gtk_radio_button_new_with_label(stgroup, "Ends with");
     if(config.search_argument == SEARCHARG_ENDS_WITH)
	  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(stradiobutton), TRUE);
     GTK_WIDGET_UNSET_FLAGS(GTK_RADIO_BUTTON(stradiobutton), GTK_CAN_FOCUS);
     gtk_object_set_data(GTK_OBJECT(targetwindow), "st_ew", stradiobutton);
     gtk_box_pack_start(GTK_BOX(vbox_st), stradiobutton, TRUE, TRUE, 3);
     gtk_widget_show(stradiobutton);

     stgroup = gtk_radio_button_group(GTK_RADIO_BUTTON(stradiobutton));
     stradiobutton = gtk_radio_button_new_with_label(stgroup, "Contains");
     if(config.search_argument == SEARCHARG_CONTAINS)
	  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(stradiobutton), TRUE);
     GTK_WIDGET_UNSET_FLAGS(GTK_RADIO_BUTTON(stradiobutton), GTK_CAN_FOCUS);
     gtk_object_set_data(GTK_OBJECT(targetwindow), "st_co", stradiobutton);
     gtk_box_pack_start(GTK_BOX(vbox_st), stradiobutton, TRUE, TRUE, 3);
     gtk_widget_show(stradiobutton);

     stgroup = gtk_radio_button_group(GTK_RADIO_BUTTON(stradiobutton));
     stradiobutton = gtk_radio_button_new_with_label(stgroup, "Equals");
     if(config.search_argument == SEARCHARG_EQUALS)
	  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(stradiobutton), TRUE);
     GTK_WIDGET_UNSET_FLAGS(GTK_RADIO_BUTTON(stradiobutton), GTK_CAN_FOCUS);
     gtk_object_set_data(GTK_OBJECT(targetwindow), "st_eq", stradiobutton);
     gtk_box_pack_start(GTK_BOX(vbox_st), stradiobutton, TRUE, TRUE, 3);
     gtk_widget_show(stradiobutton);

#if 0
     /* Sorting frame in Options tab */
     sortframe = gtk_frame_new(" Sorting ");
     gtk_widget_show(sortframe);
     gtk_box_pack_start(GTK_BOX(hbox_options), sortframe, FALSE, TRUE, 0);
     vbox_sort = gtk_vbox_new(FALSE, 0);
     gtk_container_border_width(GTK_CONTAINER(vbox_sort), 5);
     gtk_widget_show(vbox_sort);
     gtk_container_add(GTK_CONTAINER(sortframe), vbox_sort);

     /* Sort in search mode button */
     sort_search_button = gtk_check_button_new_with_label("Search mode");
     gtk_object_set_data(GTK_OBJECT(targetwindow), "sort_search", sort_search_button);
     if(config.sort_search)
	  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(sort_search_button), TRUE);
     GTK_WIDGET_UNSET_FLAGS(GTK_CHECK_BUTTON(sort_search_button), GTK_CAN_FOCUS);
     gtk_widget_show(sort_search_button);
     gtk_box_pack_start(GTK_BOX(vbox_sort), sort_search_button, FALSE, FALSE, 5);
#endif
     /* View frame in Options tab */
     viewframe = gtk_frame_new(" View ");
     gtk_widget_show(viewframe);
     gtk_box_pack_start(GTK_BOX(target), viewframe, TRUE, TRUE, 5);
     vbox_view = gtk_vbox_new(TRUE, 0);
     gtk_container_border_width(GTK_CONTAINER(vbox_view), 5);
     gtk_widget_show(vbox_view);
     gtk_container_add(GTK_CONTAINER(viewframe), vbox_view);

     /* Show Distinguished Name checkbox */
     dnbutton = gtk_check_button_new_with_label("Show Distinguished Name");
     gtk_object_set_data(GTK_OBJECT(targetwindow), "showdn", dnbutton);
     if(config.showdn)
	  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(dnbutton), TRUE);
     GTK_WIDGET_UNSET_FLAGS(GTK_CHECK_BUTTON(dnbutton), GTK_CAN_FOCUS);
     gtk_widget_show(dnbutton);
     gtk_box_pack_start(GTK_BOX(vbox_view), dnbutton, FALSE, TRUE, 5);

     /* Show Objectclass checkbox */
     ocbutton = gtk_check_button_new_with_label("Show Objectclass");
     gtk_object_set_data(GTK_OBJECT(targetwindow), "showoc", ocbutton);
     if(config.showoc)
	  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(ocbutton), TRUE);
     GTK_WIDGET_UNSET_FLAGS(GTK_CHECK_BUTTON(ocbutton), GTK_CAN_FOCUS);
     gtk_widget_show(ocbutton);
     gtk_box_pack_start(GTK_BOX(vbox_view), ocbutton, FALSE, TRUE, 5);

     /* Sort in search mode button */
     sort_search_button = gtk_check_button_new_with_label("Sort results");
     gtk_object_set_data(GTK_OBJECT(targetwindow), "sort_search", sort_search_button);
     if(config.sort_search)
	  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(sort_search_button), TRUE);
     GTK_WIDGET_UNSET_FLAGS(GTK_CHECK_BUTTON(sort_search_button), GTK_CAN_FOCUS);
     gtk_widget_show(sort_search_button);
     gtk_box_pack_start(GTK_BOX(vbox_view), sort_search_button, FALSE, TRUE, 5);
}


void create_browse_optionstab(GtkWidget *target, GtkWidget *targetwindow)
{
     GtkWidget *viewframe, *vbox_view;
     GtkWidget *show_rdn_only_button, *sort_browse_button;

  /* View frame in Options tab */
     viewframe = gtk_frame_new(" View ");
     gtk_widget_show(viewframe);
     gtk_box_pack_start(GTK_BOX(target), viewframe, FALSE, TRUE, 5);
     vbox_view = gtk_vbox_new(TRUE, 0);
     gtk_container_border_width(GTK_CONTAINER(vbox_view), 5);
     gtk_widget_show(vbox_view);
     gtk_container_add(GTK_CONTAINER(viewframe), vbox_view);

  /* show rdn only button */
     show_rdn_only_button = gtk_check_button_new_with_label("Show Relative Distinguished Name only");
     gtk_object_set_data(GTK_OBJECT(targetwindow), "show_rdn_only", show_rdn_only_button);
     if(config.show_rdn_only)
	  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(show_rdn_only_button), TRUE);
     GTK_WIDGET_UNSET_FLAGS(GTK_CHECK_BUTTON(show_rdn_only_button), GTK_CAN_FOCUS);
     gtk_widget_show(show_rdn_only_button);
     gtk_box_pack_start(GTK_BOX(vbox_view), show_rdn_only_button, FALSE, FALSE, 5);

  /* Sort in browse mode button */
     sort_browse_button = gtk_check_button_new_with_label("Sort results");
     gtk_object_set_data(GTK_OBJECT(targetwindow), "sort_browse", sort_browse_button);
     if(config.sort_browse)
	  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(sort_browse_button), TRUE);
     GTK_WIDGET_UNSET_FLAGS(GTK_CHECK_BUTTON(sort_browse_button), GTK_CAN_FOCUS);
     gtk_widget_show(sort_browse_button);
     gtk_box_pack_start(GTK_BOX(vbox_view), sort_browse_button, FALSE, FALSE, 5);


}


void create_ldiftab(GtkWidget *target, GtkWidget *targetwindow)
{
     GtkWidget *formatframe;
     GtkWidget *formatradio;
     GSList *formatgroup;
     GtkWidget *vbox1;

     /* Format frame */
     formatframe = gtk_frame_new(" Format ");
     gtk_widget_show(formatframe);
     gtk_box_pack_start(GTK_BOX(target), formatframe, FALSE, TRUE, 5);

     vbox1 = gtk_vbox_new(FALSE, 0);
     gtk_container_border_width(GTK_CONTAINER(vbox1), 10);
     gtk_container_add(GTK_CONTAINER(formatframe), vbox1);
     gtk_widget_show(vbox1);

     formatradio = gtk_radio_button_new_with_label(NULL, "UMich/OpenLDAP style (no comments/version)");
     if(config.ldifformat == LDIF_UMICH)
	  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(formatradio), TRUE);
     GTK_WIDGET_UNSET_FLAGS(GTK_RADIO_BUTTON(formatradio), GTK_CAN_FOCUS);
     gtk_object_set_data(GTK_OBJECT(targetwindow), "format_umich", formatradio);
     gtk_box_pack_start(GTK_BOX(vbox1), formatradio, TRUE, TRUE, 3);
     gtk_widget_show(formatradio);

     formatgroup = gtk_radio_button_group(GTK_RADIO_BUTTON(formatradio));
     formatradio = gtk_radio_button_new_with_label(formatgroup, "LDIF Version 1 (RFC2849)");
     if(config.ldifformat == LDIF_V1)
	  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(formatradio), TRUE);
     GTK_WIDGET_UNSET_FLAGS(GTK_RADIO_BUTTON(formatradio), GTK_CAN_FOCUS);
     gtk_object_set_data(GTK_OBJECT(targetwindow), "format_v1", formatradio);
     gtk_box_pack_start(GTK_BOX(vbox1), formatradio, TRUE, TRUE, 3);
     gtk_widget_show(formatradio);

}


#ifdef HAVE_LDAP_STR2OBJECTCLASS
void template_new_callback(GtkWidget *widget, GtkWidget *target)
{
     GtkWidget *servercombo;
     struct ldapserver *server;
     char *servername;

     if( (servercombo = gtk_object_get_data(GTK_OBJECT(target), "schemaserver")) == NULL)
	  return;

     if( (servername = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(servercombo)->entry))) == NULL)
	  return;

     if( (server = server_by_name(servername)) == NULL)
	  return;

     create_template_edit_window(server, NULL);

}


void template_edit_callback(GtkWidget *widget, GtkWidget *target)
{
     GtkWidget *servercombo, *templatelist;
     struct ldapserver *server;
     char *servername, *templatename;

     if( (servercombo = gtk_object_get_data(GTK_OBJECT(target), "schemaserver")) == NULL)
	  return;

     if( (servername = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(servercombo)->entry))) == NULL)
	  return;

     if( (server = server_by_name(servername)) == NULL)
	  return;

     if( (templatelist = gtk_object_get_data(GTK_OBJECT(target), "templatelist")) == NULL)
	  return;

     if( (templatename = get_clist_selection(templatelist)) == NULL)
	  return;

     create_template_edit_window(server, templatename);

}


void template_selected_callback(GtkWidget *clist, gint row, gint column,
			      GdkEventButton *event, gpointer data)
{

     if(event->type == GDK_2BUTTON_PRESS)
	  template_edit_callback(NULL, data);

}


void template_delete_callback(GtkWidget *widget, GtkWidget *target)
{
     GList *list;
     GtkWidget *templatelist;
     struct gq_template *tmpl;
     char *templatename;

     if( (templatelist = gtk_object_get_data(GTK_OBJECT(target), "templatelist")) == NULL)
	  return;

     if( (templatename = get_clist_selection(templatelist)) == NULL)
	  return;

     if( (tmpl = find_template_by_name(templatename))) {
	  config.templates = g_list_remove(config.templates, tmpl);
     }

     if( (list = GTK_CLIST(templatelist)->selection))
	  gtk_clist_remove(GTK_CLIST(templatelist), GPOINTER_TO_INT(list->data));

     save_config();

}
#endif

GtkWidget* get_widget(GtkWidget *widget, gchar *widget_name)
{
     GtkWidget *found_widget;

     if(widget->parent)
	  widget = gtk_widget_get_toplevel(widget);
     found_widget = (GtkWidget*) gtk_object_get_data(GTK_OBJECT(widget),
						     widget_name);
     if(!found_widget)
	  g_warning("Widget not found: %s", widget_name);

     return found_widget;
}

