package org.clearsilver;

import java.io.FileNotFoundException;
import java.io.IOException;

/** This class is a wrapper around the HDF C API.  Many features of the C API
 *  are not yet exposed through this wrapper.
 */
public class HDF {
  int hdfptr;  // stores the C HDF* pointer
  HDF root;    // If this is a child HDF node, points at the root node of
               // the tree.  For root nodes this is null.  A child node needs
               // to hold a reference on the root to prevent the root from
               // being GC-ed.
  static {
    try {
      System.loadLibrary("clearsilver-jni");
    } catch ( UnsatisfiedLinkError e ) {
      System.out.println("Could not load 'clearsilver-jni'");
      System.exit(1);
    }
  }

  /** Constructs an empty HDF dataset */
  public HDF() {
    hdfptr = _init();
    root = null;
  }

  /** Constructs an HDF child node.  Used by other methods in this class when
   * a child node needs to be constructed.
   */
  private HDF(int hdfptr, HDF parent) {
    this.hdfptr = hdfptr;
    this.root = (parent.root != null) ? parent.root : parent;
  }

  /** Clean up allocated memory if neccesary. close() allows application
   *  to force clean up.
   */
  public void close() {
    // Only root nodes have ownership of the C HDF pointer, so only a root
    // node needs to dealloc hdfptr.dir
    if ( root == null) {
      if (hdfptr != 0) {
        _dealloc(hdfptr);
        hdfptr = 0;
      }
    }
  }

  /** Call close() just in case when deallocating Java object.
   */
  // Should be protected access (like Object).
  protected void finalize() throws Throwable {
    close();
    super.finalize();
  }

  /** Loads the contents of the specified HDF file from disk into the current
   *  HDF object.  The loaded contents are merged with the existing contents.
   */
  public boolean readFile(String filename) throws IOException,
         FileNotFoundException {
    if (hdfptr == 0) {
      throw new NullPointerException("HDF is closed.");
    }
    return _readFile(hdfptr, filename);
  }

  /** Retrieves the integer value at the specified path in this HDF node's
   *  subtree.  If the value does not exist, or cannot be converted to an
   *  integer, default_value will be returned. */
  public int getIntValue(String hdfname, int default_value) {
    if (hdfptr == 0) {
      throw new NullPointerException("HDF is closed.");
    }
    return _getIntValue(hdfptr,hdfname,default_value);
  }

  /** Retrieves the value at the specified path in this HDF node's subtree.
  */
  public String getValue(String hdfname, String default_value) {
    if (hdfptr == 0) {
      throw new NullPointerException("HDF is closed.");
    }
    return _getValue(hdfptr,hdfname,default_value);
  }

  /** Sets the value at the specified path in this HDF node's subtree. */
  public void setValue(String hdfname, String value) {
    if (hdfptr == 0) {
      throw new NullPointerException("HDF is closed.");
    }
    _setValue(hdfptr,hdfname,value);
  }

  /** Remove the specified subtree. */
  public void removeTree(String hdfname) {
    if (hdfptr == 0) {
      throw new NullPointerException("HDF is closed.");
    }
    _removeTree(hdfptr,hdfname);
  }

  /** Links the src hdf name to the dest. */
  public void setSymLink(String hdf_name_src, String hdf_name_dest) {
    if (hdfptr == 0) {
      throw new NullPointerException("HDF is closed.");
    }
    _setSymLink(hdfptr,hdf_name_src,hdf_name_dest);
  }


  /** Retrieves the HDF object that is the root of the subtree at hdfpath, or
   *  null if no object exists at that path. */
  public HDF getObj(String hdfpath) {
    if (hdfptr == 0) {
      throw new NullPointerException("HDF is closed.");
    }
    int obj_ptr = _getObj(hdfptr, hdfpath);
    if ( obj_ptr == 0 ) {
      return null;
    }
    return new HDF(obj_ptr, this);
  }

  /** Retrieves the HDF for the first child of the root of the subtree at hdfpath, or
   *  null if no child exists of that path or if the path doesn't exist. */
  public HDF getChild(String hdfpath) {
    if (hdfptr == 0) {
      throw new NullPointerException("HDF is closed.");
    }
    int obj_ptr = _getChild(hdfptr, hdfpath);
    if ( obj_ptr == 0 ) {
      return null;
    }
    return new HDF(obj_ptr, this);
  }

  /** Retrieves the HDF object that is the root of the subtree at
   *  hdfpath, create the subtree if it doesn't exist */
  public HDF getOrCreateObj(String hdfpath) {
    if (hdfptr == 0) {
      throw new NullPointerException("HDF is closed.");
    }
    int obj_ptr = _getObj(hdfptr, hdfpath);
    if ( obj_ptr == 0 ) {
      // Create a node
      _setValue(hdfptr, hdfpath, "");
      obj_ptr = _getObj( hdfptr, hdfpath );
      if ( obj_ptr == 0 ) {
        return null;
      }
    }
    return new HDF(obj_ptr, this);
  }

  /** Returns the name of this HDF node.   The root node has no name, so
   *  calling this on the root node will return null. */
  public String objName() {
    if (hdfptr == 0) {
      throw new NullPointerException("HDF is closed.");
    }
    return _objName(hdfptr);
  }

  /** Returns the value of this HDF node, or null if this node has no value.
   *  Every node in the tree can have a value, a child, and a next peer. */
  public String objValue() {
    if (hdfptr == 0) {
      throw new NullPointerException("HDF is closed.");
    }
    return _objValue(hdfptr);
  }

  /** Returns the child of this HDF node, or null if there is no child.
   *  Use this in conjunction with objNext to walk the HDF tree.  Every node
   *  in the tree can have a value, a child, and a next peer. */
  public HDF objChild() {
    if (hdfptr == 0) {
      throw new NullPointerException("HDF is closed.");
    }
    int child_ptr = _objChild(hdfptr);
    if ( child_ptr == 0 ) {
      return null;
    }
    return new HDF(child_ptr, this);
  }

  /** Returns the next sibling of this HDF node, or null if there is no next
   *  sibling.  Use this in conjunction with objChild to walk the HDF tree.
   *  Every node in the tree can have a value, a child, and a next peer. */
  public HDF objNext() {
    if (hdfptr == 0) {
      throw new NullPointerException("HDF is closed.");
    }
    int next_ptr = _objNext(hdfptr);
    if ( next_ptr == 0 ) {
      return null;
    }
    return new HDF(next_ptr, this);
  }

  public void copy(String hdfpath, HDF src) {
    if (hdfptr == 0 || src.hdfptr == 0) {
      throw new NullPointerException("HDF is closed.");
    }
    _copy(hdfptr, hdfpath, src.hdfptr);
  }

  /**
   * Generates a string representing the content of the HDF tree rooted at
   * this node.
   */
  public String dump() {
    if (hdfptr == 0) {
      throw new NullPointerException("HDF is closed.");
    }
    return _dump(hdfptr);
  }

  private static native int _init();
  private static native void _dealloc(int ptr);
  private static native boolean _readFile(int ptr, String filename);
  private static native int _getIntValue(int ptr, String hdfname,
                                         int default_value);
  private static native String _getValue(int ptr, String hdfname,
                                         String default_value);
  private static native void _setValue(int ptr, String hdfname,
                                       String hdf_value);
  private static native void _removeTree(int ptr, String hdfname);
  private static native void _setSymLink(int ptr, String hdf_name_src,
                                       String hdf_name_dest);
  private static native int _getObj(int ptr, String hdfpath);
  private static native int _getChild(int ptr, String hdfpath);
  private static native int _objChild(int ptr);
  private static native int _objNext(int ptr);
  private static native String _objName(int ptr);
  private static native String _objValue(int ptr);
  private static native void _copy(int destptr, String hdfpath, int srcptr);

  private static native String _dump(int ptr);
}
