/*
 *  main.c
 *  
 *
 *  Created by Charles Mercadal on Thu Oct 16 2003.
 *
 */

#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <termios.h>
#include <signal.h>


#include "common.h"
#include "psinfo.h"
#include "output.h"


static void printstats(void);
static void runtillinput(void);
static void usage(void);
static void windowresize(int);


static bool sflag;
static int sleeptime;
struct psinfo procs;

int main(int argc, char *argv[])
{
    bool gflag;
    extern char *optarg;
    extern int optind;
    int ch;

    
    sleeptime = 2;
    gflag = sflag = false;
    
    while ((ch = getopt(argc, argv, "w:gs")) != -1) 
        switch (ch) {
            case 'w':
                sleeptime = atoi(optarg);
                break;
            case 'g':
                gflag = true;
                break;
            case 's':
                sflag = true;
                break;
            case '?':
            default:
                usage();
        }
    
    argc -= optind;
    argv += optind;
    
    if (!*argv)	/* no pids specified */
        usage();
    
    setinfo(&procs, atoi(*argv), gflag);	/* get process name, etc. */
    
    if (!procs.valid) {
        (void)fprintf(stderr, "%s:  no such process id (%s)\n", *(argv - optind), *argv);
        exit(1);
    }
    
    (void)signal(SIGWINCH, windowresize);	/* singwinch -> resize */
    
    initscreen();			/* curses setup */
    paintscreen(&procs);	/* draw screen */
    
    runtillinput();	/* event loop */
    
    resetdisplay();
    
    if (!procs.valid)	/* did runtillinput reutrn because process is no longer valid? */
        (void)fprintf(stderr, "%s:  process ended (%s)\n", *(argv - optind), *argv);
    
    removeinfo(&procs);
    
    if (!sflag)
        printstats();
    
    return 0;
}


void usage(void)
{

    (void)fprintf(stderr, "usage:  pscpug [-gs] [-w time] pid\n");
    exit(1);
}


void runtillinput(void)
{
    struct termios attributes;


    /* immediate mode on input */
	tcgetattr(0, &attributes);
    attributes.c_lflag &= (~ICANON);
    attributes.c_cc[VMIN] = 1;
    tcsetattr(0, TCSANOW, &attributes);
    
    /* 
     * non-blocking I/O on stdin
     * This should instead be done with poll on stdin, 
     * but I had an issue on OS X 10.4
     */
    fcntl(STDIN_FILENO, F_SETFL, O_NONBLOCK);
    
    while (procs.valid) {
        if (getchar() != -1)
            break;
        
        updateinfo(&procs);
        printinfo(&procs);
        
        sleep(sleeptime);
    }
}


void windowresize(int sig)
{

    paintscreen(&procs);
}


void printstats(void)
{

    (void)printf("Statistics:\n");
    (void)printf("Average %%CPU:  %3.1f%%\n", (procs.statistics.cumulative / procs.statistics.increments));
    (void)printf("Maximum %%CPU:  %3.1f%%\n", procs.statistics.maxuse);
    (void)printf("Minimum %%CPU:  %3.1f%%\n", procs.statistics.minuse);
}
