/*
 * GQview
 * (C) 2002 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */


#include "gqview.h"
#include "info.h"

#include "filelist.h"
#include "image.h"
#include "ui_fileops.h"

#include <pwd.h>
#include <grp.h>


#define IMAGE_SIZE_W 200
#define IMAGE_SIZE_H 200


typedef struct _TabData TabData;
struct _TabData
{
	void (*func_free)(gpointer data);
	void (*func_sync)(InfoData *id, gpointer data);
	void (*func_image)(InfoData *id, gpointer data);
	gpointer data;
};


/*
 *-------------------------------------------------------------------
 * table utils
 *-------------------------------------------------------------------
 */

static GtkWidget *table_add_line(GtkWidget *table, gint x, gint y,
				 const gchar *description, const gchar *text)
{
	GtkWidget *label;

	if (!text) text = "";

	label = gtk_label_new(description);
	gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
	gtk_table_attach(GTK_TABLE(table), label,
			 x, x + 1, y, y + 1,
			 GTK_FILL, 0,
			 2, 2);
	gtk_widget_show(label);

	label = gtk_label_new(text);
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
	gtk_table_attach(GTK_TABLE(table), label,
			 x + 1, x + 2, y, y + 1,
			 GTK_FILL, 0,
			 2, 2);
	gtk_widget_show(label);
	return label;
}

/*
 *-------------------------------------------------------------------
 * file attributes tab
 *-------------------------------------------------------------------
 */

typedef struct _InfoTabPerm InfoTabPerm;
struct _InfoTabPerm
{
	GtkWidget *label_user;
	GtkWidget *label_group;
	GtkWidget *label_perms;
};

static void info_tab_perm_free(gpointer data)
{
	InfoTabPerm *tab = data;

	g_free(tab);
}

static gchar *mode_number(mode_t m)
{
	int mb, mu, mg, mo;

	mb = mu = mg = mo = 0;

	if (m & S_ISUID) mb |= 4;
	if (m & S_ISGID) mb |= 2;
	if (m & S_ISVTX) mb |= 1;

	if (m & S_IRUSR) mu |= 4;
	if (m & S_IWUSR) mu |= 2;
	if (m & S_IXUSR) mu |= 1;

	if (m & S_IRGRP) mg |= 4;
	if (m & S_IWGRP) mg |= 2;
	if (m & S_IXGRP) mg |= 1;

	if (m & S_IROTH) mo |= 4;
	if (m & S_IWOTH) mo |= 2;
	if (m & S_IXOTH) mo |= 1;

	return g_strdup_printf("%d%d%d%d", mb, mu, mg, mo);
}

static void info_tab_perm_sync(InfoData *id, gpointer data)
{
	InfoTabPerm *tab = data;
	struct stat st;

	if (stat(id->path, &st) < 0)
		{
		gtk_label_set(GTK_LABEL(tab->label_user), "");
		gtk_label_set(GTK_LABEL(tab->label_group), "");
		gtk_label_set(GTK_LABEL(tab->label_perms), "");
		}
	else
		{
		struct passwd *user;
		struct group *grp;
		gchar pbuf[12];
		gchar *pmod;
		gchar *buf;

		user = getpwuid(st.st_uid);
		gtk_label_set(GTK_LABEL(tab->label_user), (user) ? user->pw_name : "");

		grp = getgrgid(st.st_gid);
		gtk_label_set(GTK_LABEL(tab->label_group), (grp) ? grp->gr_name : "");

		pbuf[0] = (st.st_mode & S_IRUSR) ? 'r' : ' ';
		pbuf[1] = (st.st_mode & S_IWUSR) ? 'w' : ' ';
		pbuf[2] = (st.st_mode & S_IXUSR) ? 'x' : ' ';
		pbuf[3] = '-';
		pbuf[4] = (st.st_mode & S_IRGRP) ? 'r' : ' ';
		pbuf[5] = (st.st_mode & S_IWGRP) ? 'w' : ' ';
		pbuf[6] = (st.st_mode & S_IXGRP) ? 'x' : ' ';
		pbuf[7] = '-';
		pbuf[8] = (st.st_mode & S_IROTH) ? 'r' : ' ';
		pbuf[9] = (st.st_mode & S_IWOTH) ? 'w' : ' ';
		pbuf[10] = (st.st_mode & S_IXOTH) ? 'x' : ' ';
		pbuf[11] = '\0';

		pmod = mode_number(st.st_mode);
		buf = g_strdup_printf("%s (%s)", pbuf, pmod);
		gtk_label_set(GTK_LABEL(tab->label_perms), buf);
		g_free(buf);
		g_free(pmod);
		}
}

static TabData *info_tab_perm_new(InfoData *id)
{
	TabData *td;
	InfoTabPerm *tab;
	GtkWidget *table;
	GtkWidget *label;

	tab = g_new0(InfoTabPerm, 1);

	table = gtk_table_new(6, 8, FALSE);
	gtk_container_set_border_width(GTK_CONTAINER(table), 5);

	tab->label_user = table_add_line(table, 0, 0, _("Owner:"), NULL);
	tab->label_group = table_add_line(table, 0, 1, _("Group:"), NULL);
	tab->label_perms = table_add_line(table, 0, 2, "", NULL);

	label = gtk_label_new(_("Misc."));
	gtk_notebook_append_page(GTK_NOTEBOOK(id->notebook), table, label);
	gtk_widget_show(table);

	/* register */
	td = g_new0(TabData, 1);
	td->func_free = info_tab_perm_free;
	td->func_sync = info_tab_perm_sync;
	td->func_image = NULL;
	td->data = tab;

	return td;
}

/*
 *-------------------------------------------------------------------
 * general tab
 *-------------------------------------------------------------------
 */

typedef struct _InfoTabGeneral InfoTabGeneral;
struct _InfoTabGeneral
{
	GtkWidget *label_file_time;
	GtkWidget *label_file_size;
	GtkWidget *label_dimensions;
	GtkWidget *label_image_size;
	GtkWidget *label_compression;
	gint compression_done;
	gint byte_size;
};

static void info_tab_general_image(InfoData *id, gpointer data)
{
	InfoTabGeneral *tab = data;
	gchar *buf;
	gint mem_size;

	if (id->image->unknown) return;

	buf = g_strdup_printf("%d x %d", id->image->image_width, id->image->image_height);
	gtk_label_set(GTK_LABEL(tab->label_dimensions), buf);
	g_free(buf);

	mem_size = id->image->image_width * id->image->image_width * 3;
	buf = text_from_size_abrev(mem_size);
	gtk_label_set(GTK_LABEL(tab->label_image_size), buf);
	g_free(buf);

	if (!tab->compression_done && mem_size > 0)
		{
		buf = g_strdup_printf("%.1f%%", (float)tab->byte_size / mem_size * 100.0);
		gtk_label_set(GTK_LABEL(tab->label_compression), buf);
		g_free(buf);

		tab->compression_done = TRUE;
		}
}

static void info_tab_general_sync(InfoData *id, gpointer data)
{
	InfoTabGeneral *tab = data;
	gchar *buf;

	gtk_label_set(GTK_LABEL(tab->label_file_time), text_from_time(filetime(id->path)));

	tab->byte_size = filesize(id->path);

	buf = text_from_size(tab->byte_size);
	gtk_label_set(GTK_LABEL(tab->label_file_size), buf);
	g_free(buf);

	gtk_label_set(GTK_LABEL(tab->label_dimensions), "");
	gtk_label_set(GTK_LABEL(tab->label_image_size), "");

	gtk_label_set(GTK_LABEL(tab->label_compression), "");

	tab->compression_done = FALSE;
}

static void info_tab_general_free(gpointer data)
{
	InfoTabGeneral *tab = data;

	g_free(tab);
}

static TabData *info_tab_general_new(InfoData *id)
{
	TabData *td;
	InfoTabGeneral *tab;
	GtkWidget *table;
	GtkWidget *label;

	tab = g_new0(InfoTabGeneral, 1);

	table = gtk_table_new(2, 8, FALSE);
	gtk_container_set_border_width(GTK_CONTAINER(table), 5);

	tab->label_file_time = table_add_line(table, 0, 0, _("File date:"), NULL);
	tab->label_file_size = table_add_line(table, 0, 1, _("File size:"), NULL);

	tab->label_dimensions = table_add_line(table, 0, 2, _("Dimensions:"), NULL);
	tab->label_image_size = table_add_line(table, 0, 3, _("Image size:"), NULL);

	tab->label_compression = table_add_line(table, 0, 4, _("Compress ratio:"), NULL);

	label = gtk_label_new(_("General"));
	gtk_notebook_append_page(GTK_NOTEBOOK(id->notebook), table, label);
	gtk_widget_show(table);

	/* register */
	td = g_new0(TabData, 1);
	td->func_free = info_tab_general_free;
	td->func_sync = info_tab_general_sync;
	td->func_image = info_tab_general_image;
	td->data = tab;

	return td;
}

/*
 *-------------------------------------------------------------------
 * tabs
 *-------------------------------------------------------------------
 */

static void info_tabs_sync(InfoData *id, gint image)
{
	GList *work;

	work = id->tab_list;
	while (work)
		{
		TabData *td = work->data;
		work = work->next;

		if (image)
			{
			if (td->func_image) td->func_image(id, td->data);
			}
		else
			{
			if (td->func_sync) td->func_sync(id, td->data);
			}
		}
}

static void info_tabs_free(InfoData *id)
{
	GList *work;

	work = id->tab_list;
	while (work)
		{
		TabData *td = work->data;
		work = work->next;

		if (td->func_free) td->func_free(td->data);
		g_free(td);
		}
	g_list_free(id->tab_list);
	id->tab_list = NULL;
}

static void info_tabs_init(InfoData *id)
{
	id->tab_list = g_list_append(id->tab_list, info_tab_general_new(id));
	id->tab_list = g_list_append(id->tab_list, info_tab_perm_new(id));
}

/*
 *-------------------------------------------------------------------
 * sync
 *-------------------------------------------------------------------
 */

static void info_window_sync(InfoData *id, const gchar *path)
{

	if (!path) return;

	gtk_entry_set_text(GTK_ENTRY(id->name_entry), filename_from_path(path));

	if (id->label_count)
		{
		gchar *buf;
		buf = g_strdup_printf("Image %d of %d",
				      g_list_index(id->list, (gpointer)path) + 1,
				      g_list_length(id->list));
		gtk_label_set(GTK_LABEL(id->label_count), buf);
		g_free(buf);
		}

	info_tabs_sync(id, FALSE);

	image_change_path(id->image, path, 0.0);
}

/*
 *-------------------------------------------------------------------
 * base window
 *-------------------------------------------------------------------
 */

static void info_window_image_update_cb(ImageWindow *imd, gpointer data)
{
	InfoData *id = data;

	info_tabs_sync(id, TRUE);
}

static void info_window_back_cb(GtkWidget *widget, gpointer data)
{
	InfoData *id = data;
	GList *work;

	work = g_list_find(id->list, (gpointer)id->path);
	if (!work || !work->prev) return;

	work = work->prev;
	id->path = work->data;

	info_window_sync(id, id->path);

	gtk_widget_set_sensitive(id->button_back, (work->prev != NULL));
	gtk_widget_set_sensitive(id->button_next, TRUE);
}

static void info_window_next_cb(GtkWidget *widget, gpointer data)
{
	InfoData *id = data;
	GList *work;

	work = g_list_find(id->list, (gpointer)id->path);
	if (!work || !work->next) return;

	work = work->next;
	id->path = work->data;

	info_window_sync(id, id->path);

	gtk_widget_set_sensitive(id->button_next, (work->next != NULL));
	gtk_widget_set_sensitive(id->button_back, TRUE);
}

static void info_window_close_cb(GtkWidget *widget, gpointer data)
{
	InfoData *id = data;

	gtk_widget_destroy(id->window);
}

static gint info_window_delete_cb(GtkWidget *widget, GdkEventAny *event, gpointer data)
{
	InfoData *id = data;

	gtk_widget_destroy(id->window);
	return TRUE;
}

static void info_window_destroy_cb(GtkWidget *widget, gpointer data)
{
	InfoData *id = data;

	info_tabs_free(id);
	path_list_free(id->list);
	g_free(id);
}

void info_window_new(const gchar *path, GList *list)
{
	InfoData *id;
	GtkWidget *main_vbox;
	GtkWidget *main_hbox;
	GtkWidget *hbox;
	GtkWidget *label;
	GtkWidget *button;
	GtkWidget *arrow;

	if (!path && !list) return;

	if (!list)
		{
		list = g_list_append(NULL, g_strdup(path));
		}

	id = g_new0(InfoData, 1);

	id->list = list;
	id->path = (gchar *)id->list->data;

	id->window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	window_set_icon(id->window, NULL, NULL);

	gtk_window_set_policy(GTK_WINDOW(id->window), FALSE, TRUE, FALSE);
	gtk_window_set_title(GTK_WINDOW(id->window), _("Image properties - GQview"));
	gtk_window_set_wmclass(GTK_WINDOW(id->window), "properties", "GQview");
	gtk_container_border_width(GTK_CONTAINER(id->window), 5);

	gtk_signal_connect(GTK_OBJECT(id->window), "delete_event",
			   GTK_SIGNAL_FUNC(info_window_delete_cb), id);
	gtk_signal_connect(GTK_OBJECT(id->window), "destroy",
			   GTK_SIGNAL_FUNC(info_window_destroy_cb), id);

	main_hbox = gtk_hbox_new(FALSE, 5);
	gtk_container_add(GTK_CONTAINER(id->window), main_hbox);
	gtk_widget_show(main_hbox);

	id->image = image_new(FALSE);
	image_set_update_func(id->image, info_window_image_update_cb, id);

	gtk_widget_set_usize(id->image->widget, IMAGE_SIZE_W, IMAGE_SIZE_H);
	gtk_box_pack_start(GTK_BOX(main_hbox), id->image->widget, TRUE, TRUE, 0);
	gtk_widget_show(id->image->widget);

	main_vbox = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(main_hbox), main_vbox, TRUE, TRUE, 0);
	gtk_widget_show(main_vbox);

	hbox = gtk_hbox_new(FALSE, 5);
	gtk_box_pack_start(GTK_BOX(main_vbox), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);

	label = gtk_label_new(_("Filename:"));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
	gtk_widget_show(label);

	id->name_entry = gtk_entry_new_with_max_length(512);
	gtk_widget_set_sensitive(id->name_entry, FALSE);
	gtk_box_pack_start(GTK_BOX(hbox), id->name_entry, TRUE, TRUE, 0);
	gtk_widget_show(id->name_entry);

	id->notebook = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(id->notebook), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(main_vbox), id->notebook, TRUE, TRUE, 5);
	gtk_widget_show(id->notebook);

	hbox = gtk_hbox_new(FALSE, 5);
	gtk_box_pack_start(GTK_BOX(main_vbox), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);

	id->button_back = gtk_button_new();
	arrow = gtk_arrow_new(GTK_ARROW_LEFT, GTK_SHADOW_NONE);
	gtk_container_add(GTK_CONTAINER(id->button_back), arrow);
	gtk_widget_show(arrow);

	gtk_signal_connect(GTK_OBJECT(id->button_back), "clicked",
			   GTK_SIGNAL_FUNC(info_window_back_cb), id);
	gtk_widget_set_sensitive(id->button_back, FALSE);
	gtk_box_pack_start(GTK_BOX(hbox), id->button_back, FALSE, FALSE, 0);
	gtk_widget_show(id->button_back);

	id->button_next = gtk_button_new();
	arrow = gtk_arrow_new(GTK_ARROW_RIGHT, GTK_SHADOW_NONE);
	gtk_container_add(GTK_CONTAINER(id->button_next), arrow);
	gtk_widget_show(arrow);

	gtk_signal_connect(GTK_OBJECT(id->button_next), "clicked",
			   GTK_SIGNAL_FUNC(info_window_next_cb), id);
	gtk_widget_set_sensitive(id->button_next, (id->list->next != NULL));
	gtk_box_pack_start(GTK_BOX(hbox), id->button_next, FALSE, FALSE, 0);
	gtk_widget_show(id->button_next);

	if (id->list->next)
		{
		id->label_count = gtk_label_new("");
		gtk_box_pack_start(GTK_BOX(hbox), id->label_count, FALSE, FALSE, 0);
		gtk_widget_show(id->label_count);
		}

	button = gtk_button_new_with_label(_("Close"));
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			   GTK_SIGNAL_FUNC(info_window_close_cb), id);
	gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	gtk_widget_show(button);

	/* set up tabs */

	info_tabs_init(id);

	/* fill it */

	info_window_sync(id, id->path);

	/* finish */

	gtk_widget_show(id->window);
}

