static char rcsid[] = "$Id: PctestIpv4Raw.cc,v 1.8 2000/01/20 20:30:31 bmah Exp $";
//
// $Id: PctestIpv4Raw.cc,v 1.8 2000/01/20 20:30:31 bmah Exp $
//
// PctestIpv4Raw.cc
// Bruce A. Mah <bmah@ca.sandia.gov>
//
// This work was first produced by an employee of Sandia National
// Laboratories under a contract with the U.S. Department of Energy.
// Sandia National Laboratories dedicates whatever right, title or
// interest it may have in this software to the public. Although no
// license from Sandia is needed to copy and use this software,
// copying and using the software might infringe the rights of
// others. This software is provided as-is. SANDIA DISCLAIMS ANY
// WARRANTY OF ANY KIND, EXPRESS OR IMPLIED.
//
// Class of IPv4 tests using raw sockets
//

#include <sys/types.h>
#include <netdb.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/udp.h>
#include <netinet/ip_icmp.h>
#include <arpa/inet.h>

#include "pc.h"
#include "PctestIpv4Raw.h"
#include "TestRecord.h"

//
// GetSocketOut
//
// Input:  None
//
// Output:  In return value, returns socket number.
//
// Get output socket of an appropriate type.  We'll make this virtual 
// shortly.
//
int PctestIpv4Raw::GetSocketOut() {

    int sock;
    int rc;
    
    sock = socket(AF_INET, SOCK_RAW, IPPROTO_UDP);
    if (sock < 0) {
	perror("socket");
	return sock;
    }
    
    int hdrinclOption;
    hdrinclOption = 1;
    rc = setsockopt(sock, IPPROTO_IP, IP_HDRINCL, 
		    (const char *) &hdrinclOption,
		    sizeof(hdrinclOption));
    if (rc < 0) {
	perror("setsockopt(IP_HDRINCL)");
	return rc;
    }

#ifdef linux
    // Linux needs SO_BSDCOMPAT enabled on our UDP socket, to avoid
    // getting ICMP errors when we send packets out.
    int bsdcompatOption;
    bsdcompatOption = 1;
    rc = setsockopt(sock, SOL_SOCKET, SO_BSDCOMPAT, 
		    (const char *) &bsdcompatOption,
		    sizeof(bsdcompatOption));
    if (rc < 0) {
	perror("setsockopt(SO_BSDCOMPAT)");
	return rc;
    }
#endif /* linux */

    return sock;
}

//
// PctestIpv4Raw::Test
//
// Input:
//
// Output:
//
// A negative icmpCode indicates a timeout.
//
int PctestIpv4Raw::Test(TestRecord &tr)
{

    struct timeval tvBefore, tvAfter;
    struct timeval timeout;
    int rc;			// syscall return code
    fd_set readFds;		// reading file descriptors
    int done = 0;

    // If the requested sending size is too small or too large, 
    // then return an error.  The caller should have figured out the 
    // minimum sending size by calling Pctest::GetMinSize().
    if ((tr.size < GetMinSize()) || (tr.size > IP_MAXPACKET)) {
	return -1;
    }

    // Make up a UDP packet to send out.  Start with an IP header.
    // See Section 25.3 of UNIX Network Programming, Second Edition,
    // for why we need to twiddle the byte-orders of some fields but
    // not others, depending on what OS we run.
    extern unsigned int Tos;
    struct ip ipHeader;
    memset(&ipHeader, 0, sizeof(ipHeader));
#ifdef __osf__
    // Tru64 <netinet/ip.h> doesn't declar ip_hl if __STDC__ == 1
    ipHeader.ip_vhl = (sizeof(ip) >> 2) | (4 << 4);
#else    
    ipHeader.ip_hl = sizeof(ip) >> 2;
    ipHeader.ip_v = 4;
#endif /* __osf__ */
    ipHeader.ip_tos = Tos;
#ifdef linux
    ipHeader.ip_len = htons(tr.size);
#else
    ipHeader.ip_len = tr.size;
#endif /* linux */
    ipHeader.ip_id = htons(0);
#ifdef linux
    ipHeader.ip_off = htons(IP_DF);
#else
    ipHeader.ip_off = IP_DF;
#endif /* linux */
    ipHeader.ip_ttl = tr.hops;
    ipHeader.ip_p = IPPROTO_UDP;
    ipHeader.ip_sum = 0;
    memset(&(ipHeader.ip_src), 0, sizeof(struct in_addr));
    memcpy(&(ipHeader.ip_dst), &(targetSocketAddress.sin_addr), sizeof(struct in_addr));

    // Make up UDP header;
    int udpPayloadSize = tr.size - sizeof(ip) - sizeof(udphdr);
    struct udphdr udpHeader;
    memset(&udpHeader, 0, sizeof(udpHeader));
    udpHeader.uh_sport = htons(30003); //XXX
    udpHeader.uh_dport = htons(destPort++);
    udpHeader.uh_ulen = htons(udpPayloadSize + sizeof(udpHeader));
    udpHeader.uh_sum = htons(0); // Let the UDP checksum be 0

    // UDP payload
    char *udpPayload;
    udpPayload = GeneratePayload(udpPayloadSize);
    if (udpPayload == NULL) {
	fprintf(stderr, "Couldn't allocate space for payload\n");
	return -1;
    }

    // Build the packet now.
    char *ipPacket;
    int ipPacketSize;
    ipPacketSize = sizeof(ip) + sizeof(udphdr) + udpPayloadSize;
    ipPacket = new char[ipPacketSize];
    memcpy(ipPacket, &ipHeader, sizeof(ipHeader));
    memcpy(ipPacket + sizeof(ipHeader), &udpHeader, sizeof(udpHeader));
    memcpy(ipPacket + sizeof(ipHeader) + sizeof(udpHeader),
	   udpPayload, udpPayloadSize);

    // Set timeout value and socket select parameters
    extern int Timeout;
    timeout.tv_sec = Timeout;
    timeout.tv_usec = 0;
    FD_ZERO(&readFds);
    FD_SET(socketIn, &readFds);

    // Timestamp before 
    gettimeofday(&tvBefore, NULL);

    // Send UDP packet
    rc = sendto(socketOut, ipPacket, ipPacketSize, 0,
		(struct sockaddr *) &targetSocketAddress,
		sizeof(struct sockaddr_in));
    delete [] udpPayload;
    delete [] ipPacket;
    if (rc < 0) {
	perror("sendto");
	return rc;
    }

    // We need to check the socket until we get a valid packet.
    // So we might end up doing this select/read several times.
    do {

	// Select and wait for an ICMP response or a timeout
	rc = select(FD_SETSIZE, &readFds, NULL, NULL, &timeout);
	if (rc < 0) {
	    perror("select");
	    return rc;
	}

	// Timestamp after and update test record timestamp fields
	gettimeofday(&tvAfter, NULL);
	tr.tv.tv_sec = tvAfter.tv_sec - tvBefore.tv_sec - syscallTime.tv_sec;
	tr.tv.tv_usec = tvAfter.tv_usec - tvBefore.tv_usec - syscallTime.tv_usec;
	while (tr.tv.tv_usec < 0) {
	    tr.tv.tv_usec += 1000000;
	    tr.tv.tv_sec--;
	}

	// Read response from socket
	if (rc == 1) {
	    IF_DEBUG(2, fprintf(stderr, "icmp packet received\n"));

	    const int icmpPacketSize = 1024;
	    char icmpPacket[icmpPacketSize];
	
	    rc = read(socketIn, icmpPacket, icmpPacketSize);
	    if (rc < 0) {
		perror("read");
		return rc;
	    }

	    // Now parse the packet, doing a little error checking along
	    // the way.  By the end, we'll have ipHeader and icmpHeader
	    // pointing to valid structures within the packet, and
	    // ipHeader2 pointing to the IP header of the generating
	    // IP packet..
	    ip *ipHeaderIn, *ipHeaderIn2;
	    icmp *icmpHeaderIn;
	    udphdr *udpHeaderIn;

	    // Check protocol in IP header
	    ipHeaderIn = (ip *) icmpPacket;
	    if (ipHeaderIn->ip_p != proto) {
		IF_DEBUG(0, fprintf(stderr, "Received unknown protocol %d in (supposedly) ICMP packet\n", ipHeaderIn->ip_p));
		return -1;
	    }

#ifdef __osf__
	    // Tru64 <netinet/ip.h> doesn't declar ip_hl if __STDC__ == 1
	    icmpHeaderIn = (icmp *) (((char *) ipHeaderIn) +
				     ((ipHeaderIn->ip_vhl & 0x0f) << 2));
#else
	    icmpHeaderIn = (icmp *) (((char *) ipHeaderIn) + (ipHeaderIn->ip_hl << 2));
#endif /* __osf__ */
	    IF_DEBUG(3, fprintf(stderr, "icmp type = %d, code = %d\n", 
				icmpHeaderIn->icmp_type, icmpHeaderIn->icmp_code));

	    // Check ICMP type.  Most types (such as echo request/reply,
	    // router adverts, etc.) we ignore.
	    if ((icmpHeaderIn->icmp_type != ICMP_TIMXCEED) && 
		(icmpHeaderIn->icmp_type != ICMP_UNREACH)) {
		IF_DEBUG(3, fprintf(stderr, "ignoring icmp packet\n"));
		continue;
	    }
	    
	    // Check for a valid (to us) IP header within the packet.
	    // For "time exceeded" or "destination unreachable", this
	    // header will be 8 bytes past the ICMP header.
	    ipHeaderIn2 = (ip *) ((char *) icmpHeaderIn + 8);

	    // Check to be sure that we have enough of the packet to hold
	    // a valid IP header? XXX

	    // Additional checking here...must be UDP
	    if (ipHeaderIn2->ip_p != IPPROTO_UDP) {
		IF_DEBUG(3, fprintf(stderr, "ignoring icmp packet for non-udp\n"));
		continue;
	    }

	    // Align UDP header template, check port numbers and
	    // payload length for us. XXX
#ifdef __osf__
	    // Tru64 <netinet/ip.h> doesn't declar ip_hl if __STDC__ == 1
	    udpHeaderIn = (udphdr *) (((char *) ipHeaderIn2) +
				      ((ipHeaderIn2->ip_vhl & 0x0f) << 2));
#else
	    udpHeaderIn = (udphdr *) (((char *) ipHeaderIn2) + (ipHeaderIn2->ip_hl << 2));
#endif /* __osf__ */

	    // Check destination UDP port number (we don't know the
	    // source) and UDP (header+payload) length
	    if ((udpHeaderIn->uh_dport != udpHeader.uh_dport) || 
		(ntohs(udpHeaderIn->uh_ulen) != udpPayloadSize + sizeof(udphdr))) {
		IF_DEBUG(3, fprintf(stderr, "ignoring icmp packet for unknown udp packet\n"));
		continue;
	    }

	    // Fill in return fields
	    tr.targetAddress = new char[sizeof(in_addr)];
	    memcpy(tr.targetAddress, &targetAddress, sizeof(in_addr));
	    tr.targetAddressLength = sizeof(in_addr);

	    tr.icmpDestAddress = new char[sizeof(in_addr)];
	    memcpy(tr.icmpDestAddress, &(ipHeaderIn->ip_dst), sizeof(in_addr));
	    tr.icmpDestAddressLength = sizeof(in_addr);

	    tr.icmpSourceAddress = new char[sizeof(in_addr)];
	    memcpy(tr.icmpSourceAddress, &(ipHeaderIn->ip_src), sizeof(in_addr));
	    tr.icmpSourceAddressLength = sizeof(in_addr);

	    tr.icmpType = icmpHeaderIn->icmp_type;
	    tr.icmpCode = icmpHeaderIn->icmp_code;

	    done = 1;

	}
	else {
	    IF_DEBUG(2, fprintf(stderr, "timeout\n"));

	    tr.targetAddress = new char[sizeof(in_addr)];
	    memcpy(tr.targetAddress, &targetAddress, sizeof(in_addr));
	    tr.targetAddressLength = sizeof(in_addr);

	    tr.icmpDestAddress = new char[sizeof(in_addr)];
	    memset(tr.icmpDestAddress, 0, sizeof(in_addr));
	    tr.icmpDestAddressLength = sizeof(in_addr);

	    tr.icmpSourceAddress = new char[sizeof(in_addr)];
	    memset(tr.icmpSourceAddress, 0, sizeof(in_addr));
	    tr.icmpSourceAddressLength = sizeof(in_addr);

	    tr.icmpType = -1;
	    tr.icmpCode = -1;

	    done = 1;

	}
    } while (!done);

    return 0;

}

