/* 

                          Firewall Builder

                 Copyright (C) 2004 NetCitadel, LLC

  Author:  Vadim Kurland vadim@fwbuilder.org

  $Id: RCS.cpp,v 1.62 2006/10/21 06:53:25 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "config.h"
#include "global.h"
#include "utils.h"

// need this for FS_SEPARATOR
#include "fwbuilder/libfwbuilder-config.h"
#include "fwbuilder/Tools.h"

//#include "FWWindow.h"

#include <RCS.h>
#include <qdir.h>
#include <qregexp.h>
#include <qmessagebox.h>
#include <qapplication.h>
#include <qeventloop.h>
#include <qtextcodec.h>

#include <stdlib.h>

#if defined(_WIN32)
#  include <stdio.h>
#  include <sys/stat.h>
#  include <fcntl.h>
#  include <time.h>
#  include <sys/timeb.h>
#else
#  include <unistd.h>
#  include <string.h>
#  include <errno.h>
#  if defined(TM_IN_SYS_TIME)
#    include <sys/time.h>
#  else
#    include <time.h>
#  endif
#endif

#include <iostream>

using namespace std;
using namespace libfwbuilder;

QString     RCS::rcs_file_name     = "";
QString     RCS::rlog_file_name    = "";
QString     RCS::rcsdiff_file_name = "";
QString     RCS::ci_file_name      = "";
QString     RCS::co_file_name      = "";

RCSEnvFix*  RCS::rcsenvfix         = NULL;

/***********************************************************************
 *
 * class Revision
 *
 ***********************************************************************/
Revision::Revision()
{
}

Revision::Revision(const QString &file, const QString &r)
{
    filename = file;
    rev      = r;
}

Revision::Revision(const Revision &r)
{
    filename   = r.filename ;
    rev        = r.rev      ;
    date       = r.date     ;
    author     = r.author   ;
    locked_by  = r.locked_by;
    log        = r.log      ;
}

void Revision::operator=(const Revision &r)
{
    filename   = r.filename ;
    rev        = r.rev      ;
    date       = r.date     ;
    author     = r.author   ;
    locked_by  = r.locked_by;
    log        = r.log      ;
}

bool Revision::operator<(const Revision &r)
{
    for(int i=1; ; i++)
    {
        QString v1=  rev.section(".",i,i);
        QString v2=r.rev.section(".",i,i);
        if (v1=="" && v2=="") return false;
        if (v1==v2) continue;
        if (v1=="" && v2!="") return true;
        if (v1!="" && v2=="") return false;
        if (v1.toInt()>v2.toInt()) return false;
        if (v1.toInt()<v2.toInt()) return true;
        i++;
    }
    return true;
}

bool Revision::operator==(const Revision &r)
{
    return rev==r.rev;
}

bool Revision::operator!=(const Revision &r)
{
    return rev!=r.rev;
}

/***********************************************************************
 *
 * class RCSEnvFix
 *
 ***********************************************************************/
RCSEnvFix::RCSEnvFix()
{
    int tzoffset;
    QString tzsign = "";

    time_t clock; 
    struct tm *ltm; 
    time(&clock); 
    ltm = (struct tm *) localtime(&clock);

#if defined(HAVE_STRUCT_TM_TM_ZONE)
/*
 * struct tm has member tm_zone and should have member tm_gmtoff
 * autoconf checks only for tm_zone, but these two member fields come
 * together These fields are present in *BSD struct tm (including Mac
 * OS X). Linux manual does not mention them, but they are present
 * there as well. These files are missing in Windows version of time.h
 *
 * long tm_gmtoff is offset from UTC in seconds, it is negative in timezones
 * west of GMT
 */

    tzoffset = ltm->tm_gmtoff/60;
    if (tzoffset<0)
    {
        tzoffset = -1*tzoffset;
        tzsign = "-";
    } else {
        tzsign = "+";
    }
    
#else
// global variable timezone has seconds West of GMT (positive in
// timezones west of GMT)

    tzoffset = ((ltm->tm_isdst>0)?timezone-3600:timezone)/60;
    if (tzoffset<0)
    {
        tzoffset = -1*tzoffset;
        tzsign = "+";
    } else {
        tzsign = "-";
    }

#endif

    TZOffset.sprintf("%02d:%02d",tzoffset/60,tzoffset%60);
    TZOffset = tzsign + TZOffset;

    if (fwbdebug)
	qDebug("tzoffset: %d TZOffset: '%s'",tzoffset,TZOffset.ascii());

#ifdef _WIN32
/* need this crap because Windows does not set environment variable TZ
 * by default, but rcs absolutely requires it. Even though I am using
 * option "-z" with all RCS commands, stupid RCS on windows does not
 * work if env var TZ is not set
 */
    env.push_back( QString("TZ=GMT")+TZOffset );

/*
 * NB: need to prepend installation directory in front of PATH on
 * windows, otherwise ci fails when GUI is launched by windows
 * explorer through file extension association. When the program is
 * launched from menu "Start", its working directory is the dir. where
 * it is installed. Since windows implies a '.' in front of PATH,
 * everything works. When the program is started with some other
 * directory as current dir, RCS tools fail without any error message.
 */
    env.push_back( QString("PATH=%1;%2").arg(appRootDir.c_str()).arg(getenv("PATH")) );
#endif

/* also need to set env variable USER for rcs tools, but if the user name
 * contains spaces, replace them with underscores (like "John Smith")
 */
    QString uname=getUserName();

    env.push_back( QString("USER=")+uname);
    env.push_back( QString("LOGNAME=")+uname);
}

QStringList* RCSEnvFix::getEnv()
{
    if (env.empty()) return NULL;
    return &env;
}

/***********************************************************************
 *
 * class RCS
 *
 ***********************************************************************/
RCS::RCS(const QString &file)
{
    if (rcsenvfix==NULL) rcsenvfix = new RCSEnvFix();

    if (rcs_file_name=="")
    {
#ifdef _WIN32
        string ts;
        ts     = appRootDir+FS_SEPARATOR+RCS_FILE_NAME      ;
        rcs_file_name     = ts.c_str();

        ts    = appRootDir+FS_SEPARATOR+RLOG_FILE_NAME     ;
        rlog_file_name    = ts.c_str();

        ts = appRootDir+FS_SEPARATOR+RCSDIFF_FILE_NAME  ;
        rcsdiff_file_name = ts.c_str();

        ts      = appRootDir+FS_SEPARATOR+CI_FILE_NAME       ;
        ci_file_name      = ts.c_str();

        ts      = appRootDir+FS_SEPARATOR+CO_FILE_NAME       ;
        co_file_name      = ts.c_str();
#else
        rcs_file_name     = RCS_FILE_NAME      ;
        rlog_file_name    = RLOG_FILE_NAME     ;
        rcsdiff_file_name = RCSDIFF_FILE_NAME  ;
        ci_file_name      = CI_FILE_NAME       ;
        co_file_name      = CO_FILE_NAME       ;
#endif
    }

    filename      = file;
    checked_out   = false;
    locked        = false;
    inrcs         = false;
    tracking_file = false;
    ro            = false;
    temp          = false;

    ciproc = new QProcess();
    proc = new QProcess();
    proc->setCommunication( QProcess::Stdin|QProcess::Stdout|QProcess::Stderr|QProcess::DupStderr);

    connect(proc, SIGNAL(readyReadStdout()), this,  SLOT(readFromStdout() ) );
    connect(proc, SIGNAL(readyReadStderr()), this,  SLOT(readFromStderr() ) );


    try
    {
        QString  rcspath=filename.left( filename.findRev("/") );
        QDir     rcsdir;
        rcsdir.cd(rcspath);

/*
 * rlog is started with environment defined by RCSEnvFix, which does
 * not have env. var LANG so it always runs in english
 */
        QStringList split_log = QStringList::split(QRegExp("------|======"),rlog());
                                                   
        QString head_section = split_log[0];

        QRegExp head_rx("head:\\s+([0-9\\.]+)\\s*\\n");
        int pos = head_rx.search( head_section );
        if (pos>-1) head = head_rx.cap(1);

        QStringList::iterator i;
        for (i=split_log.begin(),++i; i!=split_log.end(); ++i)
        {
            QString section = *i;
            if (section.length()==0) continue;

            int match = -1;

            Revision r(filename);
            r.rev = "";
            r.log = "";

            QRegExp rev_rx("revision\\s+([0-9\\.]+)");
            match = rev_rx.search( section );
            if (match>-1)
            {
                r.rev = rev_rx.cap(1);
            }

            QRegExp lock_rx("revision\\s+([0-9\\.]+)\\s+locked by:\\s+(\\S+);");
            lock_rx.setMinimal(true);
            match = lock_rx.search( section );
            if (match>-1)
            {
                r.locked_by = lock_rx.cap(2);
                locked      = true;
                locked_by   = lock_rx.cap(2);
                locked_rev  = r.rev;
            }

// older implementation copied revision and "locked by" to r.log
// we'll do the same here to maintain compatibility
            QRegExp rev2_rx("(revision.+)\\n");
            rev2_rx.setMinimal(true);
            match = rev2_rx.search( section );
            if (match>-1)
            {
                r.log += rev2_rx.cap(1) + "\n";
            }


            QRegExp date_rx("date:\\s+([^;]+);\\s+author:\\s+(\\S+);");
            date_rx.setMinimal(true);
            match = date_rx.search( section );
            if (match>-1)
            {
                r.date   = date_rx.cap(1);
                r.author = date_rx.cap(2);
            }

            QRegExp log_rx("date:.*\\n(.*)$");
            log_rx.setMinimal(true);
            match = log_rx.search( section );
            if (match>-1)
                r.log += log_rx.cap(1);

            r.log.replace('\r',"");

            if (r.rev != "")
            {
                revisions.push_back(r);
                if (fwbdebug) qDebug("revision %s: '%s'",r.rev.ascii(),r.log.ascii());
            }

        }

#if 0
        for ()
        {
            if ( (*i).find("head: ")==0)
            {
                head=(*i).section(QRegExp("\\s+"),1,1);
                continue;
            }

            if ( (*i).find(QRegExp("^=========="))==0 ) break;

            if ((*i).find(QRegExp("^revision\\s+[0-9\\.]+"))==0)
            {
                if (fwbdebug) qDebug("revision '%s'",(*i).ascii());

                Revision r(filename);

                r.rev       = (*i).section(QRegExp("\\s+"),1,1);
                QString lb  = (*i).section(QRegExp("[\\s;]+"),4,4);
                if (lb!="")
                {       
                    r.locked_by = lb;
                    locked      = true;
                    locked_by   = lb;
                    locked_rev  = r.rev;
                }

                r.log="";

                for ( ;   (*i).find(QRegExp("^=========="))==-1 &&
                          (*i).find(QRegExp("^----------"))==-1 &&
                          i!=rcslog.end(); ++i )
                {
                    if ((*i).find(QRegExp("^date:.*author:.*state:"))==0)
                    {
                        r.date      = (*i).section(QRegExp("[\\s;]+"),1,2);
                        r.author    = QString("    ")+(*i).section(QRegExp("[\\s;]+"),4,4);
                        continue;
                    }
                    if ((*i).find(QRegExp("^branches:"))==0) continue;
                    r.log= r.log + *i + "\n";
                }
                r.log.replace('\r',"");
                revisions.push_back(r);
                if (fwbdebug) qDebug("revision %s: '%s'",r.rev.ascii(),r.log.ascii());

                if (i==rcslog.end()) break;
            }
        }
//        qBubbleSort( revisions.begin() , revisions.end() );
#endif

        inrcs         = true;
        tracking_file = true;
        selectedRev   = head;
    }
    catch (FWException &ex)
    {
        inrcs         = false;
        tracking_file = true;
    }
}

RCS::~RCS()
{
    delete proc;
}

QStringList* RCS::getEnv()
{
    if (rcsenvfix==NULL) rcsenvfix = new RCSEnvFix();
    return rcsenvfix->getEnv();
}

RCSEnvFix*   RCS::getRCSEnvFix()
{
    if (rcsenvfix==NULL) rcsenvfix = new RCSEnvFix();
    return rcsenvfix;
}


void RCS::readFromStdout()
{
    stdoutBuffer=stdoutBuffer + QString(proc->readStdout());
}

void RCS::readFromStderr()
{
    stderrBuffer=stderrBuffer + QString(proc->readStderr());
}

/*********************************************************************
 *  trivial RCS integration
 */

void RCS::abandon()
{
    if (!isInRCS()) return;

/* check out head revision and unlock it */
    proc->clearArguments();

    proc->addArgument( co_file_name );
    proc->addArgument( "-q" );
    proc->addArgument( "-f" );
    proc->addArgument( QString("-z") + rcsenvfix->getTZOffset() );
    proc->addArgument( QString("-u") );
    proc->addArgument( filename );

    stdoutBuffer="";
    stderrBuffer="";

    if (fwbdebug) qDebug("starting co with environment '%s'",
                         rcsenvfix->getEnv()->join("\n").ascii());

    if (proc->start( rcsenvfix->getEnv() ) )
    {
        while (proc->isRunning()) ; // cxx_sleep(1);
        if (proc->normalExit() && proc->exitStatus()==0)
        {
            checked_out = false;
            locked      = false;
            selectedRev = head;
            return;
        }
    }
/* error. */

    selectedRev = "";

    checked_out=false;
    
    QString err = tr("Error checking file out: %1").arg(stderrBuffer);
    QMessageBox::critical(app->activeWindow(), "Firewall Builder", err, tr("&Continue") );

    throw(FWException(err.latin1()));
}

/**
 *  initial RCS checkin
 */
void RCS::add() throw(libfwbuilder::FWException)
{
    proc->clearArguments();

    int      i=filename.findRev("/");
    QString  rcspath=filename.left(i);
    QDir     rcsdir;
    rcsdir.cd(rcspath);

    if (!rcsdir.exists("RCS")) rcsdir.mkdir("RCS");

    proc->addArgument( rcs_file_name );
    proc->addArgument( "-q" );
    proc->addArgument( "-i" );
    proc->addArgument( "-kb" );
    proc->addArgument( QString("-z") + rcsenvfix->getTZOffset() );
    proc->addArgument( "-t-\"Initial checkin\"" );
    proc->addArgument( filename );

    stdoutBuffer="";
    stderrBuffer="";

    if (proc->start( rcsenvfix->getEnv() ) )
    {
        while (proc->isRunning()) ; // cxx_sleep(1);
        if (proc->normalExit() && proc->exitStatus()==0)
        {
            proc->clearArguments();
            proc->addArgument( ci_file_name );
            proc->addArgument( "-q" );
            proc->addArgument( "-u" );
            proc->addArgument( QString("-z") + rcsenvfix->getTZOffset() );
            proc->addArgument( filename );

            stdoutBuffer="";
            stderrBuffer="";

            if (proc->start( rcsenvfix->getEnv() ) )
            {
                while (proc->isRunning()) ; // cxx_sleep(1);
                if (proc->normalExit() && proc->exitStatus()==0)
                {
                    inrcs       = true;
                    selectedRev = "1.1";
                    head        = "1.1";
                    return;
                }
            }
        }
    }
    QByteArray outp = proc->readStdout();
    QString msg=QObject::tr("Fatal error during initial RCS checkin of file %1 :\n %2\nExit status %3")
	.arg(filename).arg(outp.data()).arg(proc->exitStatus());
    throw(FWException( msg.latin1()  ));
}

bool RCS::isInRCS()
{
    if (tracking_file) return inrcs;

    proc->clearArguments();

    proc->addArgument( rlog_file_name );
    proc->addArgument( QString("-z") + rcsenvfix->getTZOffset() );
    proc->addArgument( "-R" );
    proc->addArgument( filename );

    stdoutBuffer="";
    stderrBuffer="";

    if (!proc->start( rcsenvfix->getEnv() ) ) throw(FWException("Fatal error running rlog "));

    while (proc->isRunning()) ; // cxx_sleep(1);

    if (proc->normalExit() && proc->exitStatus()==1)
    {
/* exist status '1' means the file is not in RCS */
        inrcs=false;
	if (fwbdebug)
	{
          QByteArray outp = proc->readStdout();
          qDebug("Error running rlog: %s",outp.data());
	}
        return false;
    }
    inrcs=true;
    return true;
}

bool RCS::co(bool force) throw(libfwbuilder::FWException)
{
    return co(selectedRev,force);
}

/**
 *  RCS checkout
 *
 * possible situations:
 *
 * 1. file is not in RCS - do nothing, return false
 *
 * 2. need to open file read-only
 *
 *   2.1 requested revision is emty or the head: no need to
 *   checkout, just return true
 *
 *   2.2 need to open read-only, older revision: do checkout of that
 *   revision into temporary file without locking, change file name,
 *   set flag 'temp'
 *
 * 3. need to open read-write, but file is locked
 *
 *   3.1 file is locked by the same user: offer user a choice
 *   open read-only or continue editing or cancel
 *
 *   3.2 file is locked by another user: offer a choice open read-only
 *   or cancel
 *
 * 4. need to open read-write, any revision: do normal checkout and
 * lock
 *
 */
bool RCS::co(const QString &rev,bool force) throw(libfwbuilder::FWException)
{
/* first check if filename is already in RCS */

    if (!isInRCS()) return false;

    if (ro)
    {
        if (rev==head || rev=="") return true;

/* check out requested revision to stdout 
 *
 * TODO: right now it loads the whole file into memory, then writes it
 * to the temp file. It should be more efficient to read and write in
 * chunks.
 *
 */
        proc->clearArguments();

        proc->addArgument( co_file_name );
        proc->addArgument( QString("-q") );
        proc->addArgument( QString("-kb") );
        proc->addArgument( QString("-z") + rcsenvfix->getTZOffset() );
        proc->addArgument( QString("-p")+rev );
        proc->addArgument( filename );

        stdoutBuffer="";
        stderrBuffer="";

        if (fwbdebug) qDebug("starting co with environment '%s'",
                             rcsenvfix->getEnv()->join("\n").ascii());

        if (proc->start( rcsenvfix->getEnv() ) )
        {
            while (proc->isRunning())
            {
                QByteArray ba = proc->readStdout();
                if (ba.size()!=0)  stdoutBuffer=stdoutBuffer + QString(ba);
//                cerr << "read " << ba.size() << " bytes from stdout" << endl;
//                cxx_sleep(1);
            }
            if (proc->normalExit() && proc->exitStatus()==0)
            {

#ifdef _WIN32
                char tname[1024];
                strncpy(tname, filename.left(filename.findRev("/")+1).latin1(),sizeof(tname)-20);
                strcat(tname,"tmpXXXXXX");
                _mktemp(tname);
                int fd = _open(tname, _O_RDWR|_O_CREAT|_O_EXCL|_O_BINARY , _S_IREAD|_S_IWRITE );
#else
                char tname[PATH_MAX];
                strncpy(tname, filename.latin1(), sizeof(tname)-20 );
                strcat(tname,"_temp_XXXXXX");
                int fd = mkstemp(tname);
#endif 
		if (fd<0)
		{
                    QString err = tr("Error creating temporary file ")+tname+QString(" :\n")+strerror(errno);
                    QMessageBox::critical(app->activeWindow(), "Firewall Builder", err, tr("&Continue") );
                    throw(FWException(err.latin1()));
		}
#ifdef _WIN32
                if (_write(fd,stdoutBuffer.latin1(),stdoutBuffer.length() )<0)
		{
		    _close(fd);
#else
                if ( write(fd,stdoutBuffer.latin1(),stdoutBuffer.length() )<0)
		{
		    close(fd);
#endif
                    QString err = tr("Error writing to temporary file ")+tname+QString(" :\n")+strerror(errno);
                    QMessageBox::critical(app->activeWindow(), "Firewall Builder", err, tr("&Continue") );
                    throw(FWException(err.latin1()));
                }
                close(fd);

                filename    = tname;
                temp        = true;
                checked_out = false;
                locked      = false;
                selectedRev = rev;
                return true;
            }
        }

        selectedRev = head;

        QString err = tr("Error checking file out: %1").arg(stderrBuffer);
        QMessageBox::critical(app->activeWindow(), "Firewall Builder", err, tr("&Continue") );
        throw(FWException(err.latin1()));

    } else
    {
        QString me=getUserName();
        if (locked)
        {
/* the file is already locked, can not just check it out like that */

            if (me!=locked_by)
            {
                switch (QMessageBox::warning(
                            app->activeWindow(),"Firewall Builder", 
                            tr("File is opened and locked by %1.\nYou can only open it read-only.")
                            .arg(locked_by),
                            "Open &read-only", "&Cancel", QString::null,
                            0, 1 ) )
                {
                case 0:  ro=true;   return false;
                case 1:  throw(FWException("cancel opening file"));  break;
                }
            }

            if (force) goto checkout;

            switch ( QMessageBox::warning(app->activeWindow(), "Firewall Builder",
                                          tr("Revision %1 of this file has been checked out and locked by you earlier.\n\
The file may be opened in another copy of Firewall Builder or was left opened\n\
after the program crashed.").arg(locked_rev),
                                           tr("Open &read-only"), tr("&Open and continue editing"), tr("&Cancel"),
                                          0, 2 ) )
            {
            case 0:  ro=true;  return false;
            case 1:
/* continue working with the file */
                checked_out = true;
                locked      = true;
                selectedRev = locked_rev;
                return true;
            case 2:  throw(FWException("cancel opening file"));  break;
            }
        }

/* if the user wanted specific revision and it should be opened
 * read-only, we need to check it out into a temporary file without
 * locking
 */

 checkout:

/* check out and lock */
        proc->clearArguments();

        proc->addArgument( co_file_name );
        proc->addArgument( "-q" );
        if (force)  proc->addArgument( "-f" );
        proc->addArgument( QString("-l")+rev );
        proc->addArgument( QString("-z") + rcsenvfix->getTZOffset() );
        proc->addArgument( filename );

        stdoutBuffer="";
        stderrBuffer="";

        if (fwbdebug) qDebug("starting co with environment '%s'",
                             rcsenvfix->getEnv()->join("\n").ascii());

        if (proc->start( rcsenvfix->getEnv() ) )
        {
            while (proc->isRunning()) ; // cxx_sleep(1);
            if (proc->normalExit() && proc->exitStatus()==0)
            {
                checked_out = true;
                locked      = true;
                selectedRev = rev;
                return true;
            }
        }
/* error. */

        selectedRev = head;

        QString err = tr("Error checking file out: %1").arg(stderrBuffer);
        QMessageBox::critical(app->activeWindow(), "Firewall Builder", err, tr("&Continue") );

        throw(FWException(err.latin1()));
    }
    return false;
}


bool RCS::ci( const QString &_lm,
              bool unlock) throw(libfwbuilder::FWException)
{
/* first check if filename is already in RCS */
    if (!isInRCS()) return false;

    QString logmsg = _lm;

    if (logmsg.isEmpty()) logmsg="_";  // otherwise ci adds "*** empty log message ***"

    if (fwbdebug)
        qDebug("RCS::ci  log message (%d characters): '%s'",
               logmsg.length(), logmsg.ascii());

    ciproc->setCommunication( QProcess::Stdin|QProcess::Stdout|QProcess::Stderr|QProcess::DupStderr);

    ciproc->clearArguments();

    ciproc->addArgument( ci_file_name );
    if (unlock) ciproc->addArgument( "-u" );
    else        ciproc->addArgument( "-l" );
    ciproc->addArgument( QString("-z") + rcsenvfix->getTZOffset() );

    ciproc->addArgument( filename );

    stdoutBuffer="";
    stderrBuffer="";

    if (fwbdebug) qDebug("starting ci with environment '%s'",
                         rcsenvfix->getEnv()->join("\n").ascii());

    QCString rcslog = logmsg.utf8();

    QString obuf;

    connect(ciproc, SIGNAL(processExited()), this,  SLOT(completeCI() ) );
    connect(ciproc, SIGNAL(wroteToStdin()),  this,  SLOT(completeCI() ) );

/*
 * under some circumstances, ci may exit immediately (e.g. when there
 * were no changes done to the file and it won't expect any rcs log
 * record on stdin). In this case slot completeCI is called
 * immediately, even before we have a chance to enter event loop. We
 * need to make sure we do not enter event loop if this happens. We
 * use flag ciRunning to check for that.
 *
 * Also it seems on windows all data is sent to the process and slot
 * is called while we still are inside launch, so that once we exit
 * from it, all is done and there is no need to enter event loop.
 */    
    ciRunning=true;
    if (!ciproc->start(rcsenvfix->getEnv() ) )
    {
        if (fwbdebug) qDebug("Checkin error: file=%s error=%s",
                             filename.latin1(),obuf.latin1());

        throw( FWException( (obuf+"\n"+
                             ciproc->arguments().join(" ")+"\n"+
                             rcsenvfix->getEnv()->join("\n")).ascii() ) );
    }

/* make a copy, omitting trailing '\0' so it won't get sent to ci */
    QByteArray rcslogCopy;
    rcslogCopy.duplicate((const char*)(rcslog),rcslog.length());

    ciproc->writeToStdin(rcslogCopy);
    ciproc->writeToStdin("\n.\n");

    if (fwbdebug) qDebug("launch succeeded, entering event loop");

    if (ciRunning)
        QApplication::eventLoop()->enterLoop();

    if (fwbdebug) qDebug("all data sent to ci");
    do
    {
        cxx_sleep(1);
        QString s=QString(ciproc->readStdout());
        obuf = obuf + s;
        if (fwbdebug) qDebug("stdoutBuffer='%s'",obuf.ascii());
    } while (ciproc->isRunning());

    if (fwbdebug) qDebug("all data collected from stdout/stderr");

    while (ciproc->isRunning())
        cxx_sleep(1);

    if (fwbdebug) qDebug("ci exited");

    if (ciproc->normalExit() && ciproc->exitStatus()==0)
    {
        if (fwbdebug) qDebug("ci exited normally");
        if (unlock)
        {
            checked_out = false;
            locked      = false;
        }
        return true;
    }
        
    return true;
}

void RCS::completeCI()
{
    ciRunning=false;

    int loopLevel = QApplication::eventLoop()->loopLevel();

    if (fwbdebug) qDebug("completeCI: loop level %d",loopLevel);

    if (loopLevel>1) QApplication::eventLoop()->exitLoop();
}

/**
 * rlog - run rlog in the background and collect RCS log
 *
 * As it turns out, we can not trust rlog option "-zLT" to properly
 * convert timezone information on Windows. This might be abug in the
 * ported rlog. When timezone is east of GMT, ci properly converts
 * when file is checked in, but rlog uses wrong sing and substracts
 * offset instead of adding it. Suppose we are in Japan time zone
 * (GMT+9), and file is checked in at 15:00 local time. Ci properly
 * writes checkin time as 6:00 GMT, but rlog reports it as 21:00 on a
 * previous day (it does -9 hours instead of +9 hours ). Option
 * "-z+09:00" works properly
 *
 */
QString RCS::rlog() throw(libfwbuilder::FWException)
{
    proc->clearArguments();

    proc->addArgument( rlog_file_name );
    proc->addArgument( QString("-z") + rcsenvfix->getTZOffset() );
//    proc->addArgument( "-zLT" );
    proc->addArgument( filename );

    if (fwbdebug)
	qDebug("Running rlog:  %s",proc->arguments().join(" ").ascii());

    stdoutBuffer="";
    stderrBuffer="";

    if (!proc->start( rcsenvfix->getEnv() ) )
        throw(FWException("Fatal error running rlog "));

    if (fwbdebug) qDebug("Running rlog");

    while (proc->isRunning()) 
    {
	QByteArray ba = proc->readStdout();
	if (ba.size()!=0)
        {
            /* I've discovered that sometimes log records contain zero
             * characters; this may be because of incorrect encoding
             * while log records were created.
             */
            unsigned int pos=0;
            int pp=-1;
            while (pos<ba.size() && (pp=ba.find('\0',pos))!=-1)
            {
                pos=pp;
                ba.at(pos)='\n';
                ++pos;
            }

            ba.at(ba.size()-1)='\0';

            stdoutBuffer=stdoutBuffer + QString(ba);

            if (fwbdebug)
                qDebug("******** Reading rlog: bufsize=%d, %s",
                       ba.size(),(const char*)(QCString(ba)));

        }
    }

    QString rlogTxt = QString::fromUtf8(stdoutBuffer.ascii());

    if (proc->normalExit() && proc->exitStatus()==0)
        return rlogTxt;

    QString msg=QObject::tr("Fatal error running rlog for %1").arg(filename);
    throw( FWException( msg.latin1() ) );
}

QStringList RCS::rcsdiff(const QString &rev) throw(libfwbuilder::FWException)
{
    isDiff();
    return QStringList::split("\n",stdoutBuffer );
}

bool        RCS::isDiff(const QString &rev) throw(libfwbuilder::FWException)
{
    proc->clearArguments();

    proc->addArgument( rcsdiff_file_name );
    proc->addArgument( "-q" );
    if (rev!="") proc->addArgument( QString("-r")+rev );
    else
    {
        if (selectedRev!="") proc->addArgument( QString("-r")+selectedRev );
    }
    proc->addArgument( QString("-z") + rcsenvfix->getTZOffset() );
    proc->addArgument( filename );

    stdoutBuffer="";
    stderrBuffer="";

    if (proc->start( rcsenvfix->getEnv() ) ) 
    {
        while (proc->isRunning())
        {
            QByteArray ba = proc->readStdout();
            if (ba.size()!=0)  stdoutBuffer=stdoutBuffer + QString(ba);
        }
    } else
        throw(FWException("Fatal error running rcsdiff "));

//    while (proc->isRunning()) ; // cxx_sleep(1);

    if (proc->normalExit()) return (proc->exitStatus()!=0);
    QString msg=QObject::tr("Fatal error running rcsdiff for file %1").arg(filename);
    throw( FWException( msg.latin1() ) );
}

QString RCS::getHead()
{
    if (isInRCS()) return head;
    return "";
}

QString RCS::getSelectedRev()
{
    if (isInRCS()) return selectedRev;
    return "";
}

