/* 

                          Firewall Builder

                 Copyright (C) 2003 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@fwbuilder.org

  $Id: ObjectManipulator.cpp,v 1.159 2006/10/22 18:20:04 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/



#include "config.h"
#include "global.h"
#include "utils.h"

#include "ObjectManipulator.h"
#include "ObjectEditor.h"
#include "ObjectTreeViewItem.h"
#include "ObjectTreeView.h"
#include "FWObjectClipboard.h"
#include "FWObjectPropertiesFactory.h"
#include "FWBSettings.h"
#include "newFirewallDialog.h"
#include "newHostDialog.h"
#include "listOfLibraries.h"
#include "findDialog.h"
#include "newGroupDialog.h"
#include "FindObjectWidget.h"

#include <qobject.h>
#include <qobjectlist.h>
#include <qlistview.h>
#include <qimage.h>
#include <qpixmapcache.h>
#include <qheader.h>
#include <qwidgetstack.h>
#include <qpushbutton.h>
#include <qtabwidget.h>
//#include <qtoolbar.h>
#include <qaction.h>
#include <qlabel.h>
#include <qcombobox.h>
#include <qsplitter.h>
#include <qtoolbutton.h>
#include <qlayout.h>
#include <qmessagebox.h>
#include <qpopupmenu.h>
#include <qtextbrowser.h>
#include <qapplication.h>
#include <qcursor.h>
#include <qtooltip.h>
#include <qlineedit.h>
#include <qcombobox.h>
#include <qstatusbar.h>
#include <qeventloop.h>


#include "DialogFactory.h"
#include "FWBTree.h"
#include "FWWindow.h"
#include "ConfirmDeleteObjectDialog.h"

#include "fwbuilder/Library.h"
#include "fwbuilder/Firewall.h"
#include "fwbuilder/Host.h"
#include "fwbuilder/Network.h"
#include "fwbuilder/IPv4.h"
#include "fwbuilder/DNSName.h"
#include "fwbuilder/AddressTable.h"
#include "fwbuilder/AddressRange.h"
#include "fwbuilder/ObjectGroup.h"

#include "fwbuilder/Resources.h"
#include "fwbuilder/FWReference.h"
#include "fwbuilder/Interface.h"
#include "fwbuilder/RuleSet.h"
#include "fwbuilder/RuleElement.h"

#ifdef USE_INTERFACE_POLICY
#  include "fwbuilder/InterfacePolicy.h"
#endif

#include "fwbuilder/CustomService.h"
#include "fwbuilder/IPService.h"
#include "fwbuilder/ICMPService.h"
#include "fwbuilder/TCPService.h"
#include "fwbuilder/UDPService.h"
#include "fwbuilder/ServiceGroup.h"
#include "fwbuilder/TagService.h"

#include "fwbuilder/Interval.h"
#include "fwbuilder/IntervalGroup.h"
#include "fwbuilder/Management.h"

#include <iostream>
#include <algorithm>

using namespace std;
using namespace libfwbuilder;

#define OBJTREEVIEW_WIDGET_NAME  "ObjTreeView"


HistoryItem::~HistoryItem() {}

ObjToolTip::ObjToolTip(ObjectTreeView *w) : QToolTip(w->viewport(),0)
{
    otv=w;
}

void ObjToolTip::maybeTip(const QPoint &p)
{
    if (st->getObjTooltips())
    {
        QListViewItem *itm = otv->itemAt( p );
        if (itm==NULL) return;
        ObjectTreeViewItem *otvi=dynamic_cast<ObjectTreeViewItem*>(itm);
        assert(otvi);
        FWObject *obj=otvi->getFWObject();
        QRect     cr =otv->itemRect(itm);

        tip(cr,
          FWObjectPropertiesFactory::getObjectPropertiesDetailed(obj,true,true));
    }
}

ObjectManipulator::ObjectManipulator( QWidget *parent ) :
    ObjectManipulator_q( parent, tr("Object Manipulator") )
{
    treeWidth    = -1;
    treeHeight   = -1;
    currentObj   = NULL;
    active       = false;
    currentTreeView=NULL;

//    setFocusPolicy( QWidget::StrongFocus  );

/* Adding pop-down menu to the button "New" */

    QString icon_path="/FWBuilderResources/Type/";

    QPopupMenu* newObjectPopup = new QPopupMenu( this );

    addPopupMenuItem( this, newObjectPopup, icon_path+Library::TYPENAME+"/icon-tree",         tr( "New &Library" ),        SLOT( newLibrary()        ) );
    newObjectPopup->insertSeparator();                                                     

    addPopupMenuItem( this, newObjectPopup, icon_path+Firewall::TYPENAME+"/icon-tree",      tr( "New &Firewall" ),       SLOT( newFirewall()       ) );
    addPopupMenuItem( this, newObjectPopup, icon_path+Host::TYPENAME+"/icon-tree",          tr( "New &Host" ),           SLOT( newHost()           ) );
    addPopupMenuItem( this, newObjectPopup, icon_path+Interface::TYPENAME+"/icon-tree",     tr( "New &Interface" ),      SLOT( newInterface()      ) );

    addPopupMenuItem( this, newObjectPopup, icon_path+Network::TYPENAME+"/icon-tree",       tr( "New &Network" ),        SLOT( newNetwork()        ) );
    addPopupMenuItem( this, newObjectPopup, icon_path+IPv4::TYPENAME+"/icon-tree",          tr( "New &Address" ),        SLOT( newAddress()        ) );
    addPopupMenuItem( this, newObjectPopup, icon_path+DNSName::TYPENAME+"/icon-tree",       tr( "New &DNS Name" ),       SLOT( newDNSName()        ) );
    addPopupMenuItem( this, newObjectPopup, icon_path+AddressTable::TYPENAME+"/icon-tree",  tr( "New A&ddress Table" ),       SLOT( newAddressTable()        ) );
    addPopupMenuItem( this, newObjectPopup, icon_path+AddressRange::TYPENAME+"/icon-tree",  tr( "New Address &Range" ),  SLOT( newAddressRange()   ) );
    addPopupMenuItem( this, newObjectPopup, icon_path+ObjectGroup::TYPENAME+"/icon-tree",   tr( "New &Object Group" ),   SLOT( newObjectGroup()    ) );
    newObjectPopup->insertSeparator();                                                     
    addPopupMenuItem( this, newObjectPopup, icon_path+CustomService::TYPENAME+"/icon-tree", tr( "New &Custom Service" ), SLOT( newCustom()         ) );
    addPopupMenuItem( this, newObjectPopup, icon_path+IPService::TYPENAME+"/icon-tree",     tr( "New &IP Service" ),     SLOT( newIP()             ) );
    addPopupMenuItem( this, newObjectPopup, icon_path+ICMPService::TYPENAME+"/icon-tree",   tr( "New IC&MP Service" ),   SLOT( newICMP()           ) );
    addPopupMenuItem( this, newObjectPopup, icon_path+TCPService::TYPENAME+"/icon-tree",    tr( "New &TCP Service" ),    SLOT( newTCP()            ) );
    addPopupMenuItem( this, newObjectPopup, icon_path+UDPService::TYPENAME+"/icon-tree",    tr( "New &UDP Service" ),    SLOT( newUDP()            ) );
    addPopupMenuItem( this, newObjectPopup, icon_path+TagService::TYPENAME+"/icon-tree",          tr( "New &TagService" ),           SLOT( newTagService()            ) );
    addPopupMenuItem( this, newObjectPopup, icon_path+ServiceGroup::TYPENAME+"/icon-tree",  tr( "New &Service Group" ),  SLOT( newServiceGroup()   ) );
    newObjectPopup->insertSeparator();                                                     
    addPopupMenuItem( this, newObjectPopup, icon_path+Interval::TYPENAME+"/icon-tree",      tr( "New Ti&me Interval" ),  SLOT( newInterval()       ) );

//    QToolButton *btn = (QToolButton*)toolBar->child("newObjectAction_action_button");

    newButton->setPopup( newObjectPopup );
    newButton->setPopupDelay( 0 );


#if defined(Q_WS_X11)
/* do something that makes sense only on X11 */

#elif defined(Q_OS_WIN32) || defined(Q_OS_CYGWIN)
/* do something that only works on windows */

#elif defined(Q_OS_MAC)

#endif

//    backwardAction->setEnabled( false );

//    setMinimumSize( QSize( 0, 174 ) );
//    splitter3->setMinimumSize( QSize( 0, 118 ) );
//    treeFrame->setMinimumSize( QSize( 200, 0 ) );
//    splitter3->setResizeMode( treeFrame, QSplitter::KeepSize );
}


QString ObjectManipulator::getTreeLabel( FWObject *obj )
{
    QString name;

    if (Interface::isA(obj))
    {
        name=Interface::constcast(obj)->getLabel().c_str();
        if (name=="")  name=QString::fromUtf8(obj->getName().c_str());
        QString q;
        if (Interface::constcast(obj)->isDyn())        q=" dyn";
        if (Interface::constcast(obj)->isUnnumbered()) q=" unnum";
        if (Interface::constcast(obj)->isBridgePort()) q=" bridge port";
        if (Interface::constcast(obj)->isExt())        q=q+" ext";
        if (q!="") name=name+" ("+q+")";
    }
    else
    {       
        name=QString::fromUtf8(obj->getName().c_str());
        if (Library::isA(obj) && obj->isReadOnly())
	    name=name+QObject::tr(" ( read only )");
    }

#if 0
    if (name=="")
    {  // no name, use type description string instead
        name= Resources::global_res->getObjResourceStr(obj,"description").c_str();
    }
#endif
    return name;
}

ObjectTreeViewItem* ObjectManipulator::insertObject( ObjectTreeViewItem *itm,
                                                     FWObject *obj )
{
    if (FWReference::cast(obj)!=NULL) return NULL;
    if (Resources::global_res->getObjResourceBool(obj,"hidden") ) return NULL;
    if (RuleSet::cast(obj)!=NULL) return NULL;

    ObjectTreeViewItem *nitm=NULL;

    QString icn_filename;

    if (FWBTree::isSystem(obj))  icn_filename="folder1.png";
    else
        icn_filename=Resources::global_res->getObjResourceStr(obj, "icon-tree").c_str();

    if (Resources::global_res->getResourceBool(
            string("/FWBuilderResources/Type/") +
            obj->getTypeName() + "/hidden") ) return NULL;

    nitm=new ObjectTreeViewItem( itm );
    nitm->setLib("");
    nitm->setText( 0, getTreeLabel(obj) );
    QPixmap pm;
    if ( ! QPixmapCache::find( icn_filename, pm) )
    {
        pm = QPixmap::fromMimeSource( icn_filename );
        QPixmapCache::insert( icn_filename, pm);
        if (fwbdebug)
            qDebug("Created pixmap from %s  w=%d h=%d depth=%d",
                   icn_filename.ascii(),
                   pm.width(),
                   pm.height(),
                   pm.depth());
    }
    nitm->setPixmap( 0, pm );
    nitm->setDragEnabled(true);

    nitm->setProperty("id",   obj->getId().c_str()   );
    nitm->setProperty("type", obj->getTypeName().c_str() );
    nitm->setFWObject( obj );

    allItems[obj] = nitm;

    return nitm;
}


void ObjectManipulator::insertSubtree( ObjectTreeViewItem *itm,
                                       FWObject *obj )
{
    ObjectTreeViewItem *nitm = insertObject(itm, obj);

    if (nitm==NULL) return;
    if ( FWBTree::isSystem(obj) ) nitm->setOpen( st->getExpandTree() );

    for (list<FWObject*>::iterator m=obj->begin(); m!=obj->end(); m++) 
    {
        FWObject *o1=*m;
        if (FWReference::cast(o1)!=NULL) continue;
        insertSubtree( nitm, o1 );
    }
}

void ObjectManipulator::showDeletedObjects(bool f)
{
    try
    {
        FWObject *dobj = mw->db()->findInIndex( FWObjectDatabase::getDeletedObjectsId());

        if (fwbdebug)
            qDebug("ObjectManipulator::showDeletedObjects f=%d  dobj=%p",f, dobj);

        if (dobj==NULL)
        {
            dobj=mw->db()->create(Library::TYPENAME);
            dobj->setId(mw->db()->getDeletedObjectsId());
            dobj->setName("Deleted Objects");
            dobj->setReadOnly(false);
            mw->db()->add(dobj);
        }

        int idx = getIdxForLib(dobj);

        if (fwbdebug)
            qDebug("ObjectManipulator::showDeletedObjects idx=%d",idx);

        if (f)
        {
            if (idx>=0) return;
            addTreePage( dobj );
            openLib( dobj );
        } else
        {
            if (idx<0) return;

            QListView *otv = idxToTrees[idx];

            if (fwbdebug)
                qDebug("ObjectManipulator::showDeletedObjects otv=%p",otv);

            assert(otv!=NULL);
            widgetStack->removeWidget( otv );
            removeLib(idx);
        }
    }
    catch(FWException &ex)
    {
/* we get exception if file is opened read-only and there is no "deleted
 * objects" library yet
 */
    }
}

void ObjectManipulator::removeObjectFromTreeView(FWObject *obj )
{
    QListView *objTreeView = idxToTrees[ getIdxForLib(getCurrentLib()) ];
    dynamic_cast<ObjectTreeView*>(objTreeView)->clearLastSelected();

    ObjectTreeViewItem *itm = allItems[obj];
    allItems[obj]=NULL;
//    allItems.erase(obj);
    delete itm;
}

void ObjectManipulator::updateLibColor(FWObject *lib)
{
    QListView *objTreeView = idxToTrees[ getIdxForLib(lib) ];

    QString clr=lib->getStr("color").c_str();
    if (clr=="" || clr=="#000000" || clr=="black") clr="#FFFFFF";
    objTreeView->setPaletteBackgroundColor( QColor( clr ) );
}

int ObjectManipulator::getIdxForLib(FWObject* lib)
{
    for (int i=0; i<libs->count(); i++)
        if ( idxToLibs[i]->getId() == lib->getId() ) return i;

    return -1;
}

void ObjectManipulator::updateLibName(FWObject *lib)
{
    int              oldidx = getIdxForLib(lib);
    QListView  *objTreeView = idxToTrees[oldidx];    
    QString      newlibname = QString::fromUtf8(lib->getName().c_str());

    if (libs->text(oldidx)!=newlibname)
    {
        removeLib(oldidx);
//        libs->removeItem( oldidx );
//        idxToLibs.erase(oldidx);
//        idxToTrees.erase(oldidx);

        addLib(lib,objTreeView);
    
    }
}


/*
 * TODO: make this signal/slot. Dialogs just emit signal
 * 'updateObject_sign', which objectManipulator should have connected
 * to its slot which would do what updateObjName does now, and more.
 */
void ObjectManipulator::updateObjName(FWObject *obj,
                                      const QString &oldName,
                                      bool  askForAutorename)
{
    if (obj->getName()==oldName.ascii()) return;

    if (obj!=currentObj) openObject(obj);

    QListViewItem *itm = allItems[obj];
    assert(itm!=NULL);

    if (fwbdebug)
    {
        qDebug("ObjectManipulator::updateObjName  changing name %s -> %s",
               oldName.latin1(), QString::fromUtf8(obj->getName().c_str()).latin1());
    }

    if ((QString::fromUtf8(obj->getName().c_str())!=oldName) &&
         (Host::isA(obj) || Firewall::isA(obj) || Interface::isA(obj)))
    {
        if (fwbdebug)
            qDebug("ObjectManipulator::updateObjName  autorename");
        autorename(obj,askForAutorename);
        if (fwbdebug)
            qDebug("ObjectManipulator::updateObjName  autorename done");
    }

    itm->setText(0, getTreeLabel( obj ) );

    if (!Library::isA(obj)) itm->parent()->sort();

/* need to update name of the firewall in the drop-down list */
    if (Firewall::isA(obj))
    {
        mw->updateFirewallName(obj,oldName);
    }

    // reopenFirewalls is called from FirewallDialog::applyChanges()
    //if (QString::fromUtf8(obj->getName().c_str())!=oldName) 
    //{
    //  QTimer::singleShot( 0, mw, SLOT(reopenFirewall()) );
    //}

    info();  // need to update info in case user edited comments and other attributes.
}

/* 
 * variant specifically used for interfaces that have name and a label
 */ 
void ObjectManipulator::updateObjName(FWObject *obj,
                                      const QString &oldName,
                                      const QString &oldLabel,
                                      bool  askForAutorename)
{
    if (obj!=currentObj) openObject(obj);

    QListViewItem *itm = allItems[obj];
    assert(itm!=NULL);

    if (fwbdebug)
    {
        qDebug("ObjectManipulator::updateObjName  changing name %s -> %s",
               oldName.latin1(), QString::fromUtf8(obj->getName().c_str()).latin1());
    }

    if ((QString::fromUtf8(obj->getName().c_str())!=oldName) && Interface::isA(obj))
        autorename(obj,askForAutorename);

    itm->setText(0, getTreeLabel( obj ) );
    itm->parent()->sort();

    Interface *i = Interface::cast(obj);
    if  ((i!=NULL && i->getLabel()!=oldLabel.latin1()) ||
         (QString::fromUtf8(obj->getName().c_str())!=oldName))
    {
        //mw->reopenFirewall();
        mw->scheduleRuleSetRedraw();
    }

    info();  // need to update info in case user edited comments and other attributes.
}

void ObjectManipulator::autorename(FWObject *obj,bool ask)
{
    if (Host::isA(obj) || Firewall::isA(obj))
    {
        if (!ask || QMessageBox::warning(
                this,"Firewall Builder", 
                tr(
"The name of the object '%1' has changed. The program can also\n"
"rename IP address objects that belong to this object,\n"
"using standard naming scheme 'host_name:interface_name:ip'.\n"
"This makes it easier to distinguish what host or a firewall\n"
"given IP address object belongs to when it is used in \n"
"the policy or NAT rule. The program also renames MAC address\n"
"objects using scheme 'host_name:interface_name:mac'.\n"
"Do you want to rename child IP and MAC address objects now?\n"
"(If you click 'No', names of all address objects that belong to\n"
"%1 will stay the same.)")
                .arg(QString::fromUtf8(obj->getName().c_str()))
                .arg(QString::fromUtf8(obj->getName().c_str())),
                tr("&Yes"), tr("&No"), QString::null,
                0, 1 )==0 )
        {
            list<FWObject*> il = obj->getByType(Interface::TYPENAME);
            for (list<FWObject*>::iterator i=il.begin(); i!=il.end(); ++i)
            {
                autorename(*i,IPv4::TYPENAME,"ip");
                autorename(*i,physAddress::TYPENAME,"mac");
            }
        }
    }

    if (Interface::isA(obj))
    {
        if (!ask || QMessageBox::warning(
                this,"Firewall Builder", 
                tr(
"The name of the interface '%1' has changed. The program can also\n"
"rename IP address objects that belong to this interface,\n"
"using standard naming scheme 'host_name:interface_name:ip'.\n"
"This makes it easier to distinguish what host or a firewall\n"
"given IP address object belongs to when it is used in \n"
"the policy or NAT rule. The program also renames MAC address\n"
"objects using scheme 'host_name:interface_name:mac'.\n"
"Do you want to rename child IP and MAC address objects now?\n"
"(If you click 'No', names of all address objects that belong to\n"
"%1 will stay the same.)")
                .arg(QString::fromUtf8(obj->getName().c_str()))
                .arg(QString::fromUtf8(obj->getName().c_str())),
                tr("&Yes"), tr("&No"), QString::null,
                0, 1 )==0 )
        {
            autorename(obj,IPv4::TYPENAME,"ip");
            autorename(obj,physAddress::TYPENAME,"mac");
        }
    }
}

void ObjectManipulator::autorename(FWObject *obj,
                                   const string &objtype,
                                   const string &namesuffix)
{
    FWObject      *hst = obj->getParent();
    list<FWObject*> ol = obj->getByType(objtype);
    int           sfxn = 1;

    for (list<FWObject*>::iterator j=ol.begin(); j!=ol.end(); ++j,sfxn++)
    {
        QString sfx;
        if (ol.size()==1) sfx="";
        else              sfx.setNum(sfxn);
        QString nn = QString("%1:%2:%3%4")
            .arg(QString::fromUtf8(hst->getName().c_str()))
            .arg(QString::fromUtf8(obj->getName().c_str()))
            .arg(namesuffix.c_str())
            .arg(sfx);

        (*j)->setName(string(nn.utf8()));
        QListViewItem *itm1 = allItems[ *j ];
        assert(itm1!=NULL);
        itm1->setText(0, getTreeLabel( *j ) );
        itm1->parent()->sort();
    }
    ol.clear();
}

void ObjectManipulator::clearObjects()
{
    if (fwbdebug) qDebug("ObjectManipulator::clearObjects  start");

    invalidateDialog();
    while (history.size()!=0) history.pop();

    if (fwbdebug) qDebug("ObjectManipulator::clearObjects  history size: %d",
                         history.size());

    int N=libs->count();

    if (fwbdebug) qDebug("ObjectManipulator::clearObjects  %d libs", N);

    for (int i=N-1; i>=0; i--)
    {
        QListView *otv = idxToTrees[i];
        assert(otv!=NULL);
        widgetStack->removeWidget( otv );
        delete otv;
        removeLib(i);
    }

    if (fwbdebug) qDebug("ObjectManipulator::clearObjects  idxToLibs size: %d",
                         idxToLibs.size());
    if (fwbdebug) qDebug("ObjectManipulator::clearObjects  idxToTrees size: %d",
                         idxToTrees.size());

    idxToLibs.clear();
    idxToTrees.clear();

    if (fwbdebug) qDebug("ObjectManipulator::clearObjects  done");
}

void ObjectManipulator::loadObjects()
{
    loadObjects( mw->db() );
}

void ObjectManipulator::loadObjects(FWObjectDatabase *db)
{
    if (fwbdebug) qDebug("ObjectManipulator::loadObjects  start");

    if (libs->count()!=0) clearObjects();

    FWObject *firstUserLib=NULL;
    list<FWObject*> ll = mw->db()->getByType( Library::TYPENAME );

//    ll.sort(FWObjectNameCmpPredicate());

    for (FWObject::iterator i=ll.begin(); i!=ll.end(); i++)
    {
        FWObject *lib = (*i);

        if (fwbdebug)
            qDebug("ObjectManipulator::loadObjects  lib %p %s %s",
                   lib, lib->getId().c_str(), lib->getName().c_str() );

        if ( lib->getId()==DELETED_LIB &&
             ! st->getBool("UI/ShowDeletedObjects")) continue;

        if ( lib->getId()!=STANDARD_LIB &&
             lib->getId()!=TEMPLATE_LIB &&
             firstUserLib==NULL) firstUserLib=*i;

        addTreePage( lib );

        if (fwbdebug) qDebug("ObjectManipulator::loadObjects  added lib %s",
                             lib->getName().c_str());
    }

    if (firstUserLib==NULL) firstUserLib=ll.front();
    openLib( firstUserLib );
}

void ObjectManipulator::addLib( FWObject *lib,QListView* otv)
{
    QString newlibname = QString::fromUtf8(lib->getName().c_str());
    int              N = libs->count();
    int            idx = 0;
    vector<FWObject*>::iterator  i1=idxToLibs.begin();
    vector<QListView*>::iterator i2=idxToTrees.begin();
    for ( ; idx<N; ++idx,++i1,++i2)
        if ( libs->text(idx) > newlibname ) break;

    string icn=Resources::global_res->getObjResourceStr(lib,"icon-tree").c_str();
    QPixmap pm;
    if ( ! QPixmapCache::find( icn.c_str(), pm) )
    {
        pm = QPixmap::fromMimeSource( icn.c_str() );
        QPixmapCache::insert( icn.c_str(), pm);
    }
    libs->insertItem( pm, newlibname, idx);
//    idx=libs->count()-1;

    libs->setCurrentItem(idx);

    idxToLibs.insert(i1,lib);
    if (otv!=NULL) idxToTrees.insert(i2,otv);
    
}

void ObjectManipulator::addTreePage( FWObject *lib)
{
    ObjectTreeView *objTreeView = new ObjectTreeView( widgetStack,
                                                      OBJTREEVIEW_WIDGET_NAME );
    new ObjToolTip(objTreeView);

    addLib(lib,objTreeView);

    objTreeView->setSizePolicy(
        QSizePolicy( QSizePolicy::Preferred,
                     (QSizePolicy::SizeType)7,
                     0,
                     0,
                     objTreeView->sizePolicy().hasHeightForWidth() )
    );

    widgetStack->addWidget( objTreeView );

//    objTreeView->setSelectionMode( QListView::Extended );
    
    updateLibColor( lib );
//    updateLibName( lib );

    connect(objTreeView,SIGNAL( editCurrentObject_sign() ),
             this,        SLOT( editSelectedObject()) );

    connect(objTreeView,SIGNAL( switchObjectInEditor_sign(libfwbuilder::FWObject*) ),
             this,        SLOT( switchObjectInEditor(libfwbuilder::FWObject*)) );

    connect( objTreeView, SIGNAL( deleteObject_sign(libfwbuilder::FWObject*) ),
             this,        SLOT( deleteObj() ) );

    connect( objTreeView, SIGNAL( objectDropped_sign(libfwbuilder::FWObject*) ),
             this,        SLOT( openObject(libfwbuilder::FWObject*) ) );

    connect( objTreeView, SIGNAL( contextMenuRequested(QListViewItem*,const QPoint&,int) ),
             this,        SLOT( contextMenu(QListViewItem*,const QPoint&,int) ) );

//    connect( objTreeView, SIGNAL( currentChanged(QListViewItem*) ),
//             this,        SLOT( selectionChanged() ) );

    connect( objTreeView, SIGNAL( selectionChanged() ),
             this,        SLOT( selectionChanged() ) );


    ObjectTreeViewItem *itm1=new ObjectTreeViewItem( objTreeView );

    itm1->setLib("");
    itm1->setOpen(TRUE);

/* need to enable dragging in order to avoid object highlighting in
 * the tree when user drags mouse cursor */

    itm1->setDragEnabled(true);

    itm1->setText( 0 , getTreeLabel( lib ) );
    if (lib->isReadOnly())
    {
        QPixmap pm;
        if ( ! QPixmapCache::find( "lock.png", pm) )
        {
            pm = QPixmap::fromMimeSource( "lock.png" );
            QPixmapCache::insert( "lock.png", pm);
        }
        itm1->setPixmap(0, pm );
    } else
    {   
        string icn=Resources::global_res->getObjResourceStr(lib,"icon-tree").c_str();
        QPixmap pm;
        if ( ! QPixmapCache::find( icn.c_str(), pm) )
        {
            pm = QPixmap::fromMimeSource( icn.c_str() );
            QPixmapCache::insert( icn.c_str(), pm);
        }
        itm1->setPixmap( 0, pm);
    }

    itm1->setProperty("id",   lib->getId().c_str()   );
    itm1->setProperty("type", lib->getTypeName().c_str() );
    itm1->setFWObject( lib );
    allItems[lib] = itm1;

//    objTreeView->setSelected( itm1, true );

    for (list<FWObject*>::iterator m=lib->begin(); m!=lib->end(); m++) 
        insertSubtree( itm1, (*m) );
    objTreeView->updateTreeItems();
}

void ObjectManipulator::removeLib(FWObject* lib)
{
    removeLib( getIdxForLib(lib) );
}

void ObjectManipulator::removeLib(int id)
{
    int              N = libs->count();
    int            idx = 0;
    vector<FWObject*>::iterator  i1=idxToLibs.begin();
    vector<QListView*>::iterator i2=idxToTrees.begin();
    for ( ; idx<N; ++idx,++i1,++i2)
    {
        if ( idx==id )
        {
            libs->removeItem( idx );
            idxToLibs.erase(i1);
            idxToTrees.erase(i2);
        }
    }
}

void ObjectManipulator::makeNameUnique(FWObject* parent,FWObject* obj)
{
    int      suffix=1;
    QString  basename=QString::fromUtf8(obj->getName().c_str());
    QString  newname=basename;

/* check if there is another object with the same name */
    while (parent->findObjectByName(obj->getTypeName(),newname.latin1())!=NULL)
    {
/* there is a duplicate */
        newname=QString("%1-%2").arg(basename).arg(suffix);
        suffix++;
    }
    obj->setName(string(newname.utf8()));
}

void ObjectManipulator::contextMenu(QListViewItem *item,
                                    const QPoint &pos, int col)
{
/* in extended selection mode there may be several selected items */

    if (fwbdebug)
        qDebug("ObjectManipulator::contextMenu  selectedObjects.size=%d",
               getCurrentObjectTree()->getNumSelected());

    ObjectTreeViewItem *otvi=dynamic_cast<ObjectTreeViewItem*>(item);
    if (otvi==NULL)  return;  // happens when user clicks outside an item

    if (!getCurrentObjectTree()->isSelected(otvi->getFWObject()))
        openObject( otvi , true );

    if (currentObj==NULL)  currentObj=otvi->getFWObject();

    QPopupMenu *popup=new QPopupMenu(this);

    int edtID =popup->insertItem( tr("Edit"), this, SLOT( editSelectedObject()));

    QPopupMenu *duptargets  = new QPopupMenu(popup);
    QPopupMenu *movetargets = new QPopupMenu(popup);

/* we add " ... to library ..." submenu to the "Move " menu item only
 * if user did not select a library, or if they selected several
 * objects. Method moveObj knows that library should not be moved
 * into another library.
 */
    bool libSelected = 
        (getCurrentObjectTree()->getNumSelected()==1 && 
         Library::isA(getCurrentObjectTree()->getSelectedObjects().front()));

    int libid = 0;

    FWObject *cl=getCurrentLib();
    int moveTargets=0;
    vector<FWObject*>::iterator i;
    for (i=idxToLibs.begin(); i!=idxToLibs.end(); ++i,++libid)
    {
        FWObject *lib   = *i;

        /* can't move to the same library. Will use menu item 'create
         * here' to duplicate to the same library 
         */
        if (lib==cl) continue;

        if ( lib->getId()==STANDARD_LIB ||
             lib->getId()==TEMPLATE_LIB ||
             lib->getId()==DELETED_LIB  ||
             lib->isReadOnly())
            continue;
        int did=duptargets->insertItem(
            tr("place in library %1").arg(QString::fromUtf8(lib->getName().c_str()))
        );

        duptargets->connectItem( did, this, SLOT( duplicateObj(int)) );
        duptargets->setItemParameter(did, libid );

        if (!libSelected)
        {
            int mid=movetargets->insertItem(
                tr("to library %1").arg(QString::fromUtf8(lib->getName().c_str()))
            );

            movetargets->connectItem( mid, this, SLOT( moveObj(int)) );
            movetargets->setItemParameter(mid, libid );

            moveTargets++;
        }
    }

    int did=duptargets->insertItem(tr("place here"));
    duptargets->connectItem( did, this, SLOT( duplicateObjUnderSameParent()) );

    int dupID =popup->insertItem( tr("Duplicate ...") , duptargets );
    int movID;

    if (moveTargets!=0)
    {
        movID=popup->insertItem( tr("Move ...") ,      movetargets );
    } else
    {
        movID=popup->insertItem( tr("Move ..."), this,
                                 SLOT( moveObj(int)) );
        popup->setItemParameter(movID, -1 );
    }


    popup->insertSeparator();

    int copyID=popup->insertItem( tr("Copy") ,      this ,
                                  SLOT( copyObj() ) );
    int cutID =popup->insertItem( tr("Cut") ,       this ,
                                  SLOT( cutObj() ) );
    int pasteID =popup->insertItem( tr("Paste") ,     this ,
                                    SLOT( pasteObj() ) );

    popup->insertSeparator();

    int delID =popup->insertItem( tr("Delete") ,    this ,
                                  SLOT( deleteObj() ) );

    int newID1=-1;
    int newID2=-1;

    if (getCurrentObjectTree()->getNumSelected()==1)
    {
        popup->insertSeparator();

        if ( (Firewall::isA(currentObj) || Host::isA(currentObj)) && 
             ! currentObj->isReadOnly() )
            newID1=popup->insertItem( tr("Add Interface"),     this ,
                               SLOT( newInterface() ) );

        if (Interface::isA(currentObj) && ! currentObj->isReadOnly())
        {
            newID1=popup->insertItem( tr("Add IP Address"),    this ,
                               SLOT( newInterfaceAddress() ) );
            newID2=popup->insertItem( tr("Add MAC Address"),   this ,
                               SLOT( newPhysicalAddress() ) );
        }

        if (currentObj->getPath(true)=="Firewalls")
            newID1=popup->insertItem( tr("New Firewall"),      this ,
                               SLOT( newFirewall() ) );

        if (currentObj->getPath(true)=="Objects/Addresses")
        {
            newID1=popup->insertItem( tr("New Address"),       this ,
                               SLOT( newAddress() ) );
        }
        if (currentObj->getPath(true)=="Objects/DNS Names")
        {
            newID1=popup->insertItem( tr("New DNS Name"),       this ,
                               SLOT( newDNSName() ) );
        }

        if (currentObj->getPath(true)=="Objects/Address Tables")
        {
            newID1=popup->insertItem( tr("New Address Table"),  this ,
                               SLOT( newAddressTable() ) );
        }

        if (currentObj->getPath(true)=="Objects/Address Ranges")
            newID1=popup->insertItem( tr("New Address Range"), this ,
                               SLOT( newAddressRange() ) );

        if (currentObj->getPath(true)=="Objects/Hosts")
            newID1=popup->insertItem( tr("New Host"),          this ,
                               SLOT( newHost() ) );

        if (currentObj->getPath(true)=="Objects/Networks")
            newID1=popup->insertItem( tr("New Network"),       this ,
                               SLOT( newNetwork() ) );

        if (currentObj->getPath(true)=="Objects/Groups")
            newID1=popup->insertItem( tr("New Group"),         this ,
                               SLOT( newObjectGroup() ) );

        if (currentObj->getPath(true)=="Services/Custom")
            newID1=popup->insertItem( tr("New Custom Service"),this ,
                               SLOT( newCustom() ) );

        if (currentObj->getPath(true)=="Services/IP")
            newID1=popup->insertItem( tr("New IP Service"),    this ,
                               SLOT( newIP() ) );

        if (currentObj->getPath(true)=="Services/ICMP")
            newID1=popup->insertItem( tr("New ICMP Service"),  this ,
                               SLOT( newICMP() ) );

        if (currentObj->getPath(true)=="Services/TCP")
            newID1=popup->insertItem( tr("New TCP Service"),   this ,
                               SLOT( newTCP() ) );

        if (currentObj->getPath(true)=="Services/UDP")
            newID1=popup->insertItem( tr("New UDP Service"),   this ,
                               SLOT( newUDP() ) );

        if (currentObj->getPath(true)=="Services/TagServices")
            newID1=popup->insertItem( tr("New TagService"),   this ,
                               SLOT( newTagService() ) );

        if (currentObj->getPath(true)=="Services/Groups")
            newID1=popup->insertItem( tr("New Group"),         this ,
                               SLOT( newServiceGroup() ) );

        if (currentObj->getPath(true)=="Time")
            newID1=popup->insertItem( tr("New Time Interval"), this ,
                               SLOT( newInterval() ) );

        popup->insertSeparator();
        popup->insertItem( tr("Find") , this , SLOT( findObject()));
        popup->insertItem( tr("Where used") , this , SLOT( findWhereUsedSlot()));
/*
        if (Firewall::cast(currentObj)!=NULL)
        {
            popup->insertSeparator();
            popup->insertItem( tr("Compile") , this , SLOT( compile()));
            popup->insertItem( tr("Install") , this , SLOT( install()));
        }
        */
    } else
    {

        popup->insertItem( tr("Group"), this ,
                           SLOT( groupObjects() ) );

    }
    
    if (Firewall::cast(currentObj)!=NULL || ObjectGroup::cast(currentObj)!=NULL)
    {
        popup->insertSeparator();
        popup->insertItem( tr("Compile") , this , SLOT( compile()));
        popup->insertItem( tr("Install") , this , SLOT( install()));
        
//        popup->insertSeparator();
//        popup->insertItem( tr("Simulate install") , this , SLOT( simulateInstall()));
    }

    popup->insertSeparator();
    int lcID=popup->insertItem( tr("Lock"), this ,
                       SLOT( lockObject() ) );
    int unlcID=popup->insertItem( tr("Unlock"), this ,
                       SLOT( unlockObject() ) );
    popup->setItemEnabled(lcID,  getCurrentObjectTree()->isLockable());
    popup->setItemEnabled(unlcID,  getCurrentObjectTree()->isUnlockable());
    
    if (fwbdebug)
    {
/* keep this for debugging  */
        popup->insertSeparator();
        popup->insertItem( tr("dump") , this , SLOT( dumpObj()));
    }

    if (getCurrentObjectTree()->getNumSelected()==1)
    {
        popup->setItemEnabled(edtID,  ! FWBTree::isSystem(currentObj) );
    } else
        popup->setItemEnabled(edtID,  false);

    bool dupMenuItem=true;
    bool moveMenuItem=true;
    bool copyMenuItem=true;
    bool pasteMenuItem=true;
    bool delMenuItem=true;
    bool newMenuItem=true;
    bool inDeletedObjects = false;

    getMenuState( (moveTargets>0),
                  dupMenuItem,moveMenuItem,copyMenuItem,pasteMenuItem,
                  delMenuItem,newMenuItem,inDeletedObjects);

    popup->setItemEnabled(dupID, dupMenuItem); 
    popup->setItemEnabled(movID, moveMenuItem);
    popup->setItemEnabled(copyID,copyMenuItem);
    popup->setItemEnabled(pasteID,pasteMenuItem);
    
    popup->setItemEnabled(cutID, copyMenuItem);
    popup->setItemEnabled(delID, delMenuItem);

    popup->setItemEnabled(newID1,newMenuItem);
    popup->setItemEnabled(newID2,newMenuItem);


    if (inDeletedObjects) popup->changeItem(movID, tr("Undelete...") );

    popup->exec( pos );
}

void ObjectManipulator::getMenuState(bool haveMoveTargets,
                                     bool &dupMenuItem,
                                     bool &moveMenuItem,
                                     bool &copyMenuItem,
                                     bool &pasteMenuItem,
                                     bool &delMenuItem,
                                     bool &newMenuItem,
                                     bool &inDeletedObjects)
{
    dupMenuItem=true;
    moveMenuItem=true;
    copyMenuItem=true;
    pasteMenuItem=true;
    delMenuItem=true;
    newMenuItem=true;

    inDeletedObjects = false;

    FWObject *delObjLib =
        mw->db()->findInIndex( FWObjectDatabase::getDeletedObjectsId());

    vector<FWObject*> so = getCurrentObjectTree()->getSelectedObjects();
    for (vector<FWObject*>::iterator i=so.begin();  i!=so.end(); ++i)
    {
        FWObject *obj= *i;

        QString objPath = obj->getPath(true).c_str();

        copyMenuItem  = copyMenuItem && FWBTree::getCopyMenuState(objPath);
        pasteMenuItem = pasteMenuItem &&
            FWBTree::getPasteMenuState(objPath) &&
            (FWObjectClipboard::obj_clipboard->size()!=0);
        delMenuItem   = delMenuItem && FWBTree::getDeleteMenuState(objPath);

        if (pasteMenuItem)
        {
            /*
             *  enable Paste menu item only if object can be pasted
             */
            vector<string>::iterator i;
            for (i= FWObjectClipboard::obj_clipboard->begin();
                 i!=FWObjectClipboard::obj_clipboard->end(); ++i)
            {
                FWObject *co= mw->db()->findInIndex(*i);
                FWObject *nobj=pasteTo( obj , co , false, true);
                pasteMenuItem = pasteMenuItem && (nobj!=NULL);
            }
        }

        dupMenuItem=
            (dupMenuItem && ! FWBTree::isSystem(obj) && ! Library::isA(obj) );

        inDeletedObjects = (delObjLib!=NULL && obj->isChildOf(delObjLib));
        dupMenuItem = dupMenuItem && !inDeletedObjects;

// can't move system objects
        moveMenuItem = moveMenuItem && ! FWBTree::isSystem(obj);

// can't move interfaces unless parent host object is also selected
        if ( Interface::isA(obj) &&
             std::find(so.begin(),so.end(),obj->getParent())==so.end())
            moveMenuItem = false;

// can't move ip addresses if parent is interface
        if (IPv4::isA(obj) && Interface::isA(obj->getParent()))
            moveMenuItem = false;

// can't move physAddress objects
        moveMenuItem = moveMenuItem && ! physAddress::isA(obj);

// can't move read-only objects
        moveMenuItem = moveMenuItem && ! obj->isReadOnly();

// can't move libraries unless in deleted objects
        if (Library::isA(obj) && ! inDeletedObjects) moveMenuItem = false;

// can't move if there is only one user-defined library in the tree
// but we dont care about number of libraries if this will become
// 'undelete' operation
        if (!haveMoveTargets && ! inDeletedObjects) moveMenuItem = false;

//        copyMenuItem= (copyMenuItem &&
//                       ! FWBTree::isSystem(currentObj) &&
//                       ! Library::isA(currentObj));
//        delMenuItem= (delMenuItem && ! FWBTree::isSystem(currentObj));

        newMenuItem= (newMenuItem && ! obj->isReadOnly() );
        Interface *intf = Interface::cast(obj);
        if (intf &&
            (intf->isDyn() ||
             intf->isUnnumbered() ||
             intf->isBridgePort())
        )
            newMenuItem = false;

    }
}

void ObjectManipulator::find()
{
    if (getCurrentObjectTree()->getNumSelected()==0) return;

    FWObject *obj=getCurrentObjectTree()->getSelectedObjects().front();
    if (obj==NULL) return;
    fd->setObject( obj );
    fd->show();
}
void ObjectManipulator::findObject()
{
    if (getCurrentObjectTree()->getNumSelected()==0) return;

    FWObject *obj=getCurrentObjectTree()->getSelectedObjects().front();
    if (obj==NULL) return;
    mw->findObject( obj );
}

void ObjectManipulator::dumpObj()
{
    if (getCurrentObjectTree()->getNumSelected()==0) return;

    FWObject *obj=getCurrentObjectTree()->getSelectedObjects().front();
    if (obj==NULL) return;
    obj->dump(false,false);
}

void ObjectManipulator::compile()
{
    if (getCurrentObjectTree()->getNumSelected()==0) return;

    vector<FWObject*> so = getCurrentObjectTree()->getSimplifiedSelection();
    set<Firewall*> fo;
    filterFirewallsFromSelection(so,fo);
    
    //FWObject *obj=getCurrentObjectTree()->getSelectedObjects().front();
    //if (obj==NULL) return;
    //mw->showFirewall(obj);
    mw->compile(fo);
}
void ObjectManipulator::filterFirewallsFromSelection(vector<FWObject*> &so,set<Firewall*> &fo)
{
    Firewall *fw;
    ObjectGroup *gr;
    for (vector<FWObject*>::iterator i=so.begin();  i!=so.end(); ++i)
    {
        fw= Firewall::cast( *i );
        if (fw!=NULL)
        {
            fo.insert(fw);
            continue;
        }
        gr=ObjectGroup::cast( *i);
        if (gr!=NULL)
        {
            extractFirewallsFromGroup(gr,fo);
        }
    }
    
}
void ObjectManipulator::extractFirewallsFromGroup(ObjectGroup *gr,set<Firewall*> &fo)
{
   Firewall *f;
   set<FWObject*> oset;
   mw->db()->findObjectsInGroup(gr,oset);
   
   set<FWObject*>::iterator i;
   for(i=oset.begin();i!=oset.end();++i)   
   {
       f=Firewall::cast(*i);
       if (f!=NULL) fo.insert(f);
   }
}
void ObjectManipulator::install()
{
    if (getCurrentObjectTree()->getNumSelected()==0) return;

    //FWObject *obj=getCurrentObjectTree()->getSelectedObjects().front();
    //if (obj==NULL) return;
    //mw->showFirewall(obj);

    vector<FWObject*> so = getCurrentObjectTree()->getSimplifiedSelection();
    set<Firewall*> fo;
    filterFirewallsFromSelection(so,fo);
    
    
    
    mw->install(fo);
}

FWObject* ObjectManipulator::duplicateObject(FWObject *targetLib,
                                             FWObject *obj,
                                             const QString &name,
                                             bool  askForAutorename)
{
    if (!isTreeReadWrite(this, targetLib)) return NULL;

    openLib(targetLib);

    FWObject *o=NULL;

    QString n;
    if (!name.isEmpty()) n=name;
    else                 n=QString::fromUtf8(obj->getName().c_str());

    o=createObject(obj->getTypeName().c_str(), n, obj);
    if (o)
    {
      openObject(o);
      if (!o->isReadOnly() && (Host::isA(o) || Firewall::isA(o) || Interface::isA(o)) )
        autorename(o,askForAutorename);
    }
    return o;
}

void ObjectManipulator::duplicateObj(int libid)
{
    if (getCurrentObjectTree()->getNumSelected()==0) return;

    ObjectTreeView* ot=getCurrentObjectTree();
    ot->freezeSelection(true);
    FWObject *obj;
    FWObject *nobj = NULL;
    vector<FWObject*> so = getCurrentObjectTree()->getSimplifiedSelection();
    for (vector<FWObject*>::iterator i=so.begin();  i!=so.end(); ++i)
    {
        obj= *i;

        if ( FWBTree::isSystem(obj) || Interface::isA(obj) ) continue;

        FWObject *cl   = idxToLibs[libid];

        nobj = duplicateObject(cl,obj,"",false);

        if (nobj->getTypeName()==Firewall::TYPENAME)
        {
            mw->addFirewallToList(nobj);
            mw->showFirewall(nobj);
        }
    }
    editObject(nobj);
    ot->freezeSelection(false);
}

void ObjectManipulator::duplicateObjUnderSameParent()
{
    if (getCurrentObjectTree()->getNumSelected()==0) return;

    ObjectTreeView* ot=getCurrentObjectTree();
    ot->freezeSelection(true);
    FWObject *obj;
    FWObject *o = NULL;
    vector<FWObject*> so = getCurrentObjectTree()->getSimplifiedSelection();
    for (vector<FWObject*>::iterator i=so.begin();  i!=so.end(); ++i)
    {
        obj= *i;

        o=NULL;

        QString n=QString::fromUtf8(obj->getName().c_str());

        openObject(o=createObject(obj->getParent(),
                                  obj->getTypeName().c_str(), n, obj));

        if (Host::isA(o) || Firewall::isA(o) || Interface::isA(o))
            autorename(o,false);

        if (Firewall::isA(o))
        {
            mw->addFirewallToList(o);
            mw->showFirewall(o);
        }
    }
    if (o!=NULL)   editObject(o);
    ot->freezeSelection(false);
}

void ObjectManipulator::moveObject(FWObject *targetLib, FWObject *obj)
{
    FWObject *cl=getCurrentLib();
    if (cl==targetLib) return;

//    bool inDeletedObjects = (obj->getParent()->getId()==FWObjectDatabase::getDeletedObjectsId());

//    QString parentType;
//    QString parentName;
    FWObject *grp = NULL;

    if (FWObjectDatabase::isA(targetLib))        grp = targetLib;
    else
    {
        grp=FWBTree::getStandardSlotForObject(targetLib,
                                              obj->getTypeName().c_str());
    }

    if (fwbdebug)
        qDebug("ObjectManipulator::moveObject  grp= %p", grp);

    if (grp==NULL) grp=targetLib;

    if (fwbdebug)
        qDebug("ObjectManipulator::moveObject  grp= %s",grp->getName().c_str());

    if (!grp->isReadOnly())
    {
        obj->ref();
        obj->ref();

        if (fwbdebug)
            qDebug("ObjectManipulator::moveObject  removing from the widget");

        ObjectTreeViewItem *itm = allItems[obj];
        if (itm->parent()==NULL) return;

        itm->parent()->takeItem(itm);

        if (fwbdebug)
            qDebug("ObjectManipulator::moveObject  removing from the tree");

        obj->getParent()->remove(obj);

        if (fwbdebug)
            qDebug("ObjectManipulator::moveObject  adding to the tree");

        grp->add(obj);
        obj->unref();

        if (fwbdebug)
            qDebug("ObjectManipulator::moveObject  adding to the widget");

        if (allItems[grp]==NULL)
        {
/* adding to the root, there is not such tree item */
            if (Library::isA(obj))
            {
                addTreePage(obj);
                openLib(obj);
            } else
            {
/* it screwed up, just print debugging message */
                if (fwbdebug)
                    qDebug("ObjectManipulator::moveObject  no place in the tree corresponding to the object %p %s",grp,grp->getName().c_str());
            }
        } else
            allItems[grp]->insertItem(itm);

        if (Firewall::cast(obj)!=NULL)
        {
            mw->addFirewallToList(obj);
            mw->showFirewall(obj);
        }
    }

//    if (fwbdebug)
//        qDebug("ObjectManipulator::moveObject  open lib cl %s",
//               cl->getName().c_str());
//    openLib(cl);

    if (fwbdebug)
        qDebug("ObjectManipulator::moveObject  all done");
}

/*
 *  targetLibName is the name of the target library in Unicode
 */
void ObjectManipulator::moveObject(const QString &targetLibName,
                                   FWObject *obj)
{
    list<FWObject*> ll = mw->db()->getByType( Library::TYPENAME );
    for (FWObject::iterator i=ll.begin(); i!=ll.end(); i++)
    {
        FWObject *lib=*i;
        if (targetLibName==QString::fromUtf8(lib->getName().c_str()))
        {
            if (fwbdebug)
                qDebug("ObjectManipulator::moveObject  found lib %s",
                       lib->getName().c_str() );

            moveObject(lib,obj);
        }
    }
}

void ObjectManipulator::moveObj(int libid)
{
    if (getCurrentObjectTree()->getNumSelected()==0) return;

    ObjectTreeView* ot=getCurrentObjectTree();
    ot->freezeSelection(true);
    FWObject *obj;

    FWObject *targetLib   = idxToLibs[libid];

    vector<FWObject*> so = getCurrentObjectTree()->getSimplifiedSelection();
    for (vector<FWObject*>::iterator i=so.begin();  i!=so.end(); ++i)
    {
        obj= *i;

        if (fwbdebug)
        {
            qDebug("ObjectManipulator::moveObj  obj=%p  obj: %s",
                   obj, obj->getName().c_str() );
        }
        if (Library::isA(obj))
        {
/* We can only move library to the root of the tree. This case only
 * happens when user tries to undelete a library. 
 */
            moveObject(mw->db(),obj);
        } else
        {
            if (obj->isChildOf(targetLib)) continue;

            if ( FWBTree::isSystem(obj) ||
                 Interface::isA(obj)    ||
                 Interface::isA(obj->getParent())) continue;

            moveObject(targetLib,obj);
        }
    }
    ot->freezeSelection(false);
}

void ObjectManipulator::copyObj()
{
    if (getCurrentObjectTree()->getNumSelected()==0) return;
    FWObject *obj;
    FWObjectClipboard::obj_clipboard->clear();

    vector<FWObject*> so = getCurrentObjectTree()->getSimplifiedSelection();

    for (vector<FWObject*>::iterator i=so.begin();  i!=so.end(); ++i)
    {
        obj= *i;
        if ( ! FWBTree::isSystem(obj) )
            FWObjectClipboard::obj_clipboard->add( obj );
    }
}

void ObjectManipulator::cutObj()
{
    copyObj();
    deleteObj();   // works with the list getCurrentObjectTree()->getSelectedObjects()
}

void ObjectManipulator::pasteObj()
{
    if (getCurrentObjectTree()->getNumSelected()==0) return;
    FWObject *obj=getCurrentObjectTree()->getSelectedObjects().front();
    if (obj==NULL) return;

    vector<string>::iterator i;
    for (i= FWObjectClipboard::obj_clipboard->begin();
         i!=FWObjectClipboard::obj_clipboard->end(); ++i)
    {
        FWObject *co= mw->db()->findInIndex(*i);
        FWObject *nobj=pasteTo( obj , co );
        if (nobj!=NULL)
        {
            if (Firewall::isA(nobj)) mw->addFirewallToList(nobj);
            if (Firewall::isA(obj))  mw->showFirewall(obj);
        }
    }
}

FWObject*  ObjectManipulator::pasteTo(FWObject *target,FWObject *obj,
                                      bool openobj,bool validateOnly)
{
    FWObject *ta=target;
    if (IPv4::isA(ta)) ta=ta->getParent();
    try
    {
/* clipboard holds a copy of the object */
//        if (ta->getTypeName()==obj->getTypeName()) ta=ta->getParent();

        Host      *hst  = Host::cast(ta);   // works for firewall, too
        Interface *intf = Interface::cast(ta);

        if (FWBTree::isSystem(ta))
        {
            if (!FWBTree::validateForInsertion(ta,obj))
            {
                if (validateOnly) return NULL;

                QMessageBox::warning(
                    this,"Firewall Builder", 
                    QObject::tr("Impossible to insert object %1 (type %2) into %3\nbecause of incompatible type.")
                    .arg(obj->getName().c_str())
                    .arg(obj->getTypeName().c_str())
                    .arg(target->getName().c_str()),
                    "&Continue", QString::null, QString::null,
                    0, 1 );
                
                return obj;
            }
        }
        
        if ( FWBTree::isSystem(ta) ||
            (hst!=NULL  && hst->validateChild(obj)) ||
            (intf!=NULL && intf->validateChild(obj))
        )
        {
            if (validateOnly) return obj;

/* add a copy of the object to system group */

            FWObject *nobj=
                mw->db()->create(obj->getTypeName());
            assert (nobj!=NULL);
            nobj->ref();
            nobj->duplicate(obj,true);   // creates new object ID

            makeNameUnique(ta,nobj);
            ta->add( nobj );
            insertSubtree( allItems[ta], nobj);

            if (openobj) openObject(nobj);

            return nobj;
        }

        Group *grp=Group::cast(ta);

        if (grp!=NULL && grp->validateChild(obj))
        {
            if (validateOnly) return obj;

/* check for duplicates. We just won't add an object if it is already there */
            string cp_id=obj->getId();
            list<FWObject*>::iterator j;
            for(j=grp->begin(); j!=grp->end(); ++j)     
            {
                FWObject *o1=*j;
                if(cp_id==o1->getId()) return o1;

                FWReference *ref;
                if( (ref=FWReference::cast(o1))!=NULL &&
                    cp_id==ref->getPointerId()) return o1;
            }

            grp->addRef(obj);
            if (openobj) openObject(grp);
        }
    }
    catch(FWException &ex)
    {
        if (validateOnly) return NULL;

        QMessageBox::warning(
            this,"Firewall Builder", 
            ex.toString().c_str(),
            "&Continue", QString::null,QString::null,
            0, 1 );
    }

    if (validateOnly) return NULL;
    return obj;
}

void ObjectManipulator::lockObject()
{
    
    if (fwbdebug)
        qDebug("ObjectManipulator::lockObject selected %d objects ",
               getCurrentObjectTree()->getNumSelected());

    if (getCurrentObjectTree()->getNumSelected()==0) return;

    FWObject *obj;

    vector<FWObject*> so = getCurrentObjectTree()->getSimplifiedSelection();
    for (vector<FWObject*>::iterator i=so.begin();  i!=so.end(); ++i)
    {
        obj= *i;
        obj->setReadOnly(true);
    }
    getCurrentObjectTree()->setLockFlags();
    getCurrentObjectTree()->updateTreeItems();
}

void ObjectManipulator::unlockObject()
{
    if (fwbdebug)
        qDebug("ObjectManipulator::unlockObject selected %d objects ",
               getCurrentObjectTree()->getNumSelected());

    if (getCurrentObjectTree()->getNumSelected()==0) return;

    FWObject *obj;

    vector<FWObject*> so = getCurrentObjectTree()->getSimplifiedSelection();
    for (vector<FWObject*>::iterator i=so.begin();  i!=so.end(); ++i)
    {
        obj= *i;
        obj->setReadOnly(false);
    }
    getCurrentObjectTree()->setLockFlags();
    getCurrentObjectTree()->updateTreeItems();
}

void ObjectManipulator::deleteObj()
{
    if (fwbdebug)
        qDebug("ObjectManipulator::deleteObj selected %d objects ",
               getCurrentObjectTree()->getNumSelected());

    if (getCurrentObjectTree()->getNumSelected()==0) return;

    FWObject *obj;
    bool emptyingTrash      = false;
    bool emptyingTrashInLib = false;

    FWObject *delObjLib = mw->db()->findInIndex(FWObjectDatabase::getDeletedObjectsId());
    if (fwbdebug)
        qDebug("ObjectManipulator::deleteObj  delObjLib=%p",delObjLib);

    vector<FWObject*> so = getCurrentObjectTree()->getSimplifiedSelection();

    if (delObjLib!=NULL)
    {
        for (vector<FWObject*>::iterator i=so.begin();  i!=so.end(); ++i)
        {
            obj= *i;
            emptyingTrash |= obj->isChildOf(delObjLib);
        }
    }

    emptyingTrashInLib = emptyingTrash && mw->editingLibrary();

/* Ask user iff:
 *
 * we are emptying trash while editing library file (.fwl)
 *    else
 * 
 * if we are not emptying Trash (i.e. not deleting "Deleted objects" library)
 *    and
 * (we delete more than one object
 *    or
 * we delete one object and it is not a library (because in this case
 * we ask them later anyway))
 */

    QString msg;

    if (emptyingTrashInLib)
    {
        msg = tr(
    "Emptying the 'Deleted Objects' in a library file is not recommended.\n"
    "When you remove deleted objects from a library file, Firewall Builder\n"
    "loses ability to track them. If a group or a policy rule in some\n"
    "data file still uses removed object from this library, you may encounter\n"
    "unusual and unexpected behavior of the program.\n"
    "Do you want to delete selected objects anyway ?"
        );
        if (QMessageBox::warning(
                this,"Firewall Builder", msg,
                tr("&Yes"), tr("&No"), QString::null,
                0, 1 )!=0) return;
    } else
    {

        if (fwbdebug)
            qDebug("ObjectManipulator::deleteObj emptyingTrash=%d so.size=%d  so.front()->type=%s",
                   emptyingTrash,
                   so.size(),
                   so.front()->getTypeName().c_str() );
        

        if (!emptyingTrash && (so.size()>1 || !Library::isA(so.front())))
        {
            /*
            msg = tr(
                "When you delete an object, it is removed from the tree and\n"
                "all groups and firewall policy rules that reference it.\n"
                "Do you want to delete selected objects ?"
            );
            if (QMessageBox::warning(
                    this,"Firewall Builder", msg,
                    tr("&Yes"), tr("&No"), QString::null,
                    0, 1 )!=0) return;
                    */
            QApplication::setOverrideCursor( QCursor( Qt::WaitCursor) );
            ConfirmDeleteObjectDialog * dlg= new ConfirmDeleteObjectDialog(this);
            
            dlg->load(so);

            QApplication::restoreOverrideCursor();
            if(dlg->exec()==QDialog::Rejected ) return;
        }
    }


/* need to work with a copy of the list of selected objects because
 * some of the methods we call below clear list
 * getCurrentObjectTree()->getSelectedObjects()
 */

    vector<FWObject*> so2 = so;

    if (fwbdebug)
    {
        for (vector<FWObject*>::iterator i=so2.begin();  i!=so2.end(); ++i)
        {
            obj= *i;
            qDebug("ObjectManipulator::deleteObj will delete obj=%p ( %s %s ) ",
                   obj, obj->getTypeName().c_str(), obj->getName().c_str());
        }
    }

    try
    {
        for (vector<FWObject*>::iterator i=so2.begin();  i!=so2.end(); ++i)
        {
            obj= *i;

//        openObject(obj,false);

            if ( ! FWBTree::isSystem(obj) )
            {
                if (Library::isA(obj))
                {
                    list<FWObject*> ll=mw->db()->getByType(Library::TYPENAME);
                    if (ll.size()==1)  return;

                    if (QMessageBox::warning(
                            this,"Firewall Builder", 
                            tr(
"When you delete a library, all objects that belong to it\n"
"disappear from the tree and all groups and rules that reference them.\n"
"You won't be able to reverse this operation later.\n"
"Do you still want to delete library %1?")
                            .arg(QString::fromUtf8(obj->getName().c_str())),
                            tr("&Yes"), tr("&No"), QString::null,
                            0, 1 )!=0 ) continue;
                }

                if (oe->isVisible() && oe->getOpened()==obj) oe->hide();

                delObj(obj);
            }
        }
    }
    catch(FWException &ex)
    {
    }
}

void ObjectManipulator::delObj(FWObject *obj,bool openobj)
{
    if (obj->getId()==STANDARD_LIB || obj->getId()==DELETED_LIB) return;

    mw->findObjectWidget->reset();
    try
    {
        if (fwbdebug)
            qDebug("ObjectManipulator::delObj  delete obj %p %s",
                   obj,obj->getName().c_str());

        FWObject *parent=obj->getParent();
        FWObject *delObjLib = mw->db()->findInIndex( DELETED_LIB );

        if (fwbdebug)
            qDebug("ObjectManipulator::delObj  deleted obj lib %p",
                   delObjLib);

        bool islib  = Library::isA(obj);
//        bool isintf = (Interface::isA(obj) && Firewall::isA(parent));
        bool isfw   = Firewall::isA(obj);
        bool isDelObj = (delObjLib!=NULL && obj->isChildOf(delObjLib));
        
        if (!islib && !isDelObj && obj->getId()!=TEMPLATE_LIB)
            updateLastModifiedTimestampForAllFirewalls(obj);

        if (fwbdebug)
            qDebug("ObjectManipulator::delObj  delete islib=%d isfw=%d isDelObj=%d",islib,isfw,isDelObj);

/*
 * TODO: we have to remove not only the object, but also all its child
 * objects from the database, as well as all references to them. This
 * logic should really be in FWObject::removeAllInstances(FWObject*);
 */

/* remove from our internal tables before it is removed from the
 * object tree so we could use obj->getId()
 */
        if (islib && !isDelObj)
        {
            int idx = getIdxForLib(obj);
            QListView *otv = idxToTrees[idx];
            assert(otv!=NULL);
            widgetStack->removeWidget( otv );
            removeLib(idx);

            list<FWObject*> fl;
            findFirewalls(obj, fl);
            for (list<FWObject*>::iterator i=fl.begin(); i!=fl.end(); i++)
                mw->deleteFirewall( *i );
        }

        if (isfw) mw->deleteFirewall(obj);


//        removeObjectFromTreeView(obj);

        QApplication::setOverrideCursor( QCursor( Qt::WaitCursor) );

        if (islib && obj->isReadOnly()) obj->setReadOnly(false);

        if (obj->getId()==TEMPLATE_LIB) // special case
        {
            if (fwbdebug) qDebug("ObjectManipulator::delObj:   special case: deleting template library");
            mw->db()->removeAllInstances(obj);
        } else
        {
            if (fwbdebug) qDebug("ObjectManipulator::delObj:   recursively deleting library and all its objects");
            mw->db()->recursivelyRemoveObjFromTree(obj, false);
            if (islib) parent=mw->db()->getFirstByType(Library::TYPENAME);
        }

        QApplication::restoreOverrideCursor();
        if (fwbdebug) qDebug("ObjectManipulator::delObj:   done");

        removeObjectFromTreeView(obj);
        mw->scheduleRuleSetRedraw();

        if (!isDelObj)
        {
            if (allItems[delObjLib]!=NULL)
                insertSubtree( allItems[delObjLib], obj );
        } else
            FWObjectClipboard::obj_clipboard->clear();

        if (openobj)
        {
            if (isfw)
            {
                std::list<Firewall*> fwlist;
                findAllFirewalls(fwlist);
                if (fwlist.size()>0)
                {
                    FWObject *first_fw = fwlist.front();
                    if (first_fw!=NULL)
                    {
                        mw->showFirewall( first_fw );
                        openObject( first_fw );
                    }
                }
                //QTimer::singleShot( 0, mw, SLOT(reopenFirewall()) );
            } else {
                openObject(parent);
            }
        }
    }
    catch(FWException &ex)
    {
        if (fwbdebug) qDebug("ObjectManipulator::delObj: catch:  restoreOverrideCursor");
        QApplication::restoreOverrideCursor();
        QMessageBox::warning(
            this,"Firewall Builder", 
            ex.toString().c_str(),
            "&Continue", QString::null,QString::null,
            0, 1 );
        throw(ex);
    }
}

void ObjectManipulator::groupObjects()
{
    if (getCurrentObjectTree()->getNumSelected()==0) return;

    FWObject *co = getCurrentObjectTree()->getSelectedObjects().front();

    newGroupDialog ngd( this );

    if (ngd.exec()==QDialog::Accepted)
    {
        QString objName = ngd.obj_name->text();
        QString libName = ngd.libs->currentText();

        QString type = ObjectGroup::TYPENAME;
        if (Service::cast(co)!=NULL)  type=ServiceGroup::TYPENAME;
        if (Interval::cast(co)!=NULL) type=IntervalGroup::TYPENAME;

        FWObject *newgrp=NULL;

        list<FWObject*> ll = mw->db()->getByType( Library::TYPENAME );
        for (FWObject::iterator i=ll.begin(); i!=ll.end(); i++)
        {
            FWObject *lib=*i;
            if (libName==QString::fromUtf8(lib->getName().c_str()))
            {
/* TODO: need to show a dialog and say that chosen library is read-only.
 * this is not critical though since newGroupDialog fills the pull-down
 * only with names of read-write libraries
 */
                if (lib->isReadOnly()) return;
                FWObject *parent = FWBTree::getStandardSlotForObject(lib,type);
                if (parent==NULL)
                {
                    if (fwbdebug)
                        qDebug("ObjectManipulator::groupObjects(): could not find standard slot for object of type %s in library %s",
                               type.ascii(),lib->getName().c_str());
                    return;
                }
                newgrp = createObject(parent,type,objName);

                break;
            }
        }
        if (newgrp==NULL) return;

        FWObject *obj;

        ObjectTreeView* ot=getCurrentObjectTree();
        ot->freezeSelection(true);

        vector<FWObject*> so = getCurrentObjectTree()->getSimplifiedSelection();

        for (vector<FWObject*>::iterator i=so.begin();  i!=so.end(); ++i)
        {
            obj= *i;
            newgrp->addRef(obj);
        }
        ot->freezeSelection(false);
    
        openObject(newgrp);
        editObject(newgrp);
    }
}

void ObjectManipulator::info()
{
    if (fwbdebug) qDebug("ObjectManipulator::info()");

    if (currentObj)
    {
        mw->info(currentObj);
        active=true;
    }
}


void ObjectManipulator::restoreSelection(bool same_widget)
{
    if (fwbdebug)
        qDebug("ObjectManipulator::restoreSelection  same_widget=%d",same_widget);

    select();
    openObject( oe->getOpened(), false);
}

void ObjectManipulator::editSelectedObject()
{
    if (getCurrentObjectTree()->getNumSelected()==0) return;

    FWObject *obj=getCurrentObjectTree()->getSelectedObjects().front();
    if (obj==NULL) return;
    editObject(obj);
}

bool ObjectManipulator::editObject(FWObject *obj)
{
    if (!oe->isVisible()) oe->show();
    return switchObjectInEditor(obj);
}

bool ObjectManipulator::switchObjectInEditor(FWObject *obj)
{
    if (fwbdebug) qDebug("ObjectManipulator::switchObjectInEditor");

    mw->unselectRules();

    if (!oe->isVisible()) return false;

    if (!mw->requestEditorOwnership(this,
                                    obj,
                                    ObjectEditor::optNone,
                                    true))
        return false;

    select();

    if (obj!=oe->getOpened())
    {
        oe->open(obj);
        currentObj=obj;
        active=true;
        openObject(obj);  // position the tree so that obj is visible
    }
    return true;      // successfully (re)opened obj in the editor
}


void ObjectManipulator::openObject(ObjectTreeViewItem *otvi,
                                   bool register_in_history)
{
    openObject(otvi->getFWObject(),register_in_history);
}

/* This method is called from the GroupObjectDialog when user double
 * clicks on the object in a group, so first we should check if this
 * object is shown in the tree and if not, find and open it.
 */
void ObjectManipulator::openObject(FWObject *obj, bool register_in_history)
{
    if (fwbdebug)
        qDebug("ObjectManipulator::openObject   obj=%s",
               (obj)?obj->getName().c_str():"NULL");

    if (obj==NULL) return;

    raise();
    FWObject *o=obj;
    if (FWReference::cast(o)!=NULL) o=FWReference::cast(o)->getPointer();

    ObjectTreeViewItem *otvi=allItems[o];
// this changes selection and thus calls slot slectionChanged
    showObjectInTree(otvi);

    libs->setCurrentItem( getIdxForLib( obj->getLibrary() ) );
    updateCreateObjectMenu( obj->getLibrary() );
}

void ObjectManipulator::selectionChanged()
{
    if (fwbdebug)
        qDebug("ObjectManipulator::selectionChanged");
    QListView *qlv= getCurrentObjectTree();
    if (qlv==NULL) return;

    ObjectTreeViewItem* otvi=
        dynamic_cast<ObjectTreeViewItem*>(qlv->currentItem());
    if (otvi==NULL) return;

    FWObject *obj=otvi->getFWObject();
    if (obj==NULL) return;

    FWObject *o=obj;
//    if (FWReference::cast(o)!=NULL) o=FWReference::cast(o)->getPointer();

    if (history.empty() || otvi!=history.top().item() )
    {
        mw->backAction->setEnabled( true );
        history.push( HistoryItem(otvi,o->getId().c_str()) );
    }

    currentObj = obj;

    bool dupMenuItem=true;
    bool moveMenuItem=true;
    bool copyMenuItem=true;
    bool pasteMenuItem=true;
    bool delMenuItem=true;
    bool newMenuItem=true;
    bool inDeletedObjects = false;

    getMenuState(false,
                 dupMenuItem,moveMenuItem,copyMenuItem,pasteMenuItem,
                 delMenuItem,newMenuItem,inDeletedObjects);

    mw->editCopyAction->setEnabled(copyMenuItem);
    mw->editDeleteAction->setEnabled(delMenuItem);
    mw->editCutAction->setEnabled(copyMenuItem);
    mw->editPasteAction->setEnabled(pasteMenuItem);

    active=true;

/*
    mw->unselectRules();
    if (oe->validateAndSave())
    {
        oe->selectionChanged(obj);
    }
*/
    info();
}

/*
 * I could use default value for the parameter register_in_history,
 * but that caused problems when this method was used as a slot
 */
void ObjectManipulator::openObject(QListViewItem *item)
{
    ObjectTreeViewItem *otvi=dynamic_cast<ObjectTreeViewItem*>(item);
    openObject(otvi,true);
}

void ObjectManipulator::openObject(FWObject *obj)
{
    openObject(obj,true);
}

void ObjectManipulator::showObjectInTree(ObjectTreeViewItem *otvi)
{
    if (otvi==NULL) return;
    widgetStack->raiseWidget( currentTreeView=otvi->getTree() );
    otvi->getTree()->clearSelection();
    otvi->getTree()->ensureItemVisible( otvi );
    otvi->getTree()->setCurrentItem( otvi );
    otvi->getTree()->setSelected( otvi, true );
}

void ObjectManipulator::invalidateDialog()
{
    currentObj=NULL;
}

void ObjectManipulator::libChanged(int ln)
{
    QListView *lv = idxToTrees[ln];
    assert(lv!=NULL);

    ObjectTreeViewItem *otvi=dynamic_cast<ObjectTreeViewItem*>(lv->currentItem());
    assert(otvi!=NULL);

    currentObj=otvi->getFWObject();
    showObjectInTree( otvi );

    info();

    updateCreateObjectMenu( idxToLibs[ln] );
    return;
}

void ObjectManipulator::updateCreateObjectMenu(FWObject* lib)
{
    bool      f   =
        lib->getId()==STANDARD_LIB ||
        lib->getId()==TEMPLATE_LIB ||
        lib->getId()==DELETED_LIB  ||
        lib->isReadOnly();

    newButton->setEnabled( !f );
    QAction *noa = (QAction*)(mw->child("newObjectAction"));
    noa->setEnabled( !f );
}

void ObjectManipulator::back()
{
//    if (!validateDialog()) return;

    if (!history.empty())
    {
        history.pop();

/* skip objects that have been deleted */
        while ( ! history.empty())
        {
            if (mw->db()->findInIndex( history.top().id().latin1())!=NULL) break;
            history.pop();
        }

        if (history.empty())
        {
            mw->backAction->setEnabled( false );
            return;
        }

        openObject( history.top().item(), false );

        if (oe->isVisible())
        {
            ObjectTreeViewItem *otvi=history.top().item();
            switchObjectInEditor(otvi->getFWObject());
        }
    }
}

FWObject*  ObjectManipulator::getCurrentLib()
{
    return idxToLibs[ libs->currentItem() ];
}

ObjectTreeView* ObjectManipulator::getCurrentObjectTree()
{
    return currentTreeView;
}

void ObjectManipulator::openLib(FWObject *obj)
{
    openObject(obj->getLibrary(),false);
}

void ObjectManipulator::newObject()
{
//    QToolButton *btn = (QToolButton*)(mw->toolBar)->child("newObjectAction_action_button");
    newButton->openPopup();
}

FWObject* ObjectManipulator::createObject(const QString &objType,
                                          const QString &objName,
                                          FWObject *copyFrom)
{
    if (!validateDialog()) return NULL;

    if (fwbdebug) qDebug("ObjectManipulator::createObject   check 1");

    FWObject *lib  = getCurrentLib();
    int       i = 0;

    if (fwbdebug)
    {
        qDebug("lib: %s %s",lib->getName().c_str(), lib->getId().c_str());
        qDebug("lib: isReadOnly=%d isLoaded=%d",
               lib->isReadOnly(), addOnLibs->isLoaded( lib->getName().c_str() ) );
        qDebug("libs->count()=%d", libs->count() );
    }

    while ( lib->getId()==STANDARD_LIB ||
            lib->getId()==TEMPLATE_LIB ||
            lib->getId()==DELETED_LIB  ||
            lib->isReadOnly() )
    {
        if (i>=libs->count())
        {
//            if (fwbdebug)
//                qDebug("ObjectManipulator::createObject   return NULL");
//            return NULL;
            lib  = getCurrentLib();
            break;
        }

        lib= idxToLibs[i];

        if (fwbdebug)
        {
            qDebug("i=%d",i);
            qDebug("lib: %s %s",lib->getName().c_str(), lib->getId().c_str());
            qDebug("lib: isReadOnly=%d isLoaded=%d",
                   lib->isReadOnly(), addOnLibs->isLoaded( lib->getName().c_str() ) );
        }
        i++;
    }

    FWObject *parent=FWBTree::getStandardSlotForObject(lib, objType);
    if (parent==NULL)
    {

      QMessageBox::warning(this,"Firewall Builder", 
                           QObject::tr(
"Type '%1': new object can not be created because\n"
"corresponding branch is missing in the object tree.\n"
"Please repair the tree using command 'fwbedit -s -f file.fwb'.")
                           .arg(objType),
                           "&Continue", QString::null, QString::null,
                           0, 1 );
                

      return NULL;
    }
    return actuallyCreateObject(parent,objType,objName,copyFrom);
}

FWObject* ObjectManipulator::createObject(FWObject *parent,
                                       const QString &objType,
                                       const QString &objName,
                                       FWObject *copyFrom)
{
    if (!validateDialog()) return NULL;

    FWObject *lib  = getCurrentLib();
    int       i = 0;

    assert(parent!=NULL);

    if (fwbdebug)
    {
        qDebug("ObjectManipulator::createObject 2: parent=%s",
               parent->getName().c_str());
        qDebug("ObjectManipulator::createObject 2: objType=%s  objName=%s",
               objType.latin1(), objName.latin1());
    }

    while ( lib->getId()==STANDARD_LIB ||
            lib->getId()==TEMPLATE_LIB ||
            lib->getId()==DELETED_LIB  ||
            lib->isReadOnly() )
    {
        if (i>=libs->count())
        {
            lib=getCurrentLib();
            break;
        }
        lib= idxToLibs[i];
        i++;
    }

    if (parent==NULL) parent=lib;

    return actuallyCreateObject(parent,objType,objName,copyFrom);
}

FWObject* ObjectManipulator::actuallyCreateObject(FWObject *parent,
                                                  const QString &objType,
                                                  const QString &objName,
                                                  FWObject *copyFrom)
{
    if (!isTreeReadWrite(this, parent)) return NULL;
    FWObject *nobj = mw->db()->create(objType.latin1());
    assert(nobj!=NULL);

    if (copyFrom!=NULL) nobj->duplicate(copyFrom,true);
    if (nobj->isReadOnly()) nobj->setReadOnly(false);
    
    nobj->setName( string(objName.utf8()) );
    makeNameUnique(parent,nobj);
    
    parent->add(nobj);
    insertSubtree(allItems[parent], nobj);

    return nobj;
}

void ObjectManipulator::newLibrary()
{
    if (!validateDialog()) return;

    FWObject *nlib = FWBTree::createNewLibrary(mw->db());

    addTreePage( nlib );

    openObject( nlib );
    editObject(nlib);
}

void ObjectManipulator::newFirewall()
{
    newFirewallDialog *nfd=new newFirewallDialog();
    if (oe->isVisible()) oe->hide();
    nfd->exec();
    FWObject *o = nfd->getNewFirewall();
    delete nfd;

    if (o!=NULL)
    {
        openObject(o);

        mw->addFirewallToList(o);
        mw->showFirewall(o);
//        updateLastModifiedTimestampForAllFirewalls(o);

        editObject(o);
    }
}

void ObjectManipulator::newHost()
{
    newHostDialog *nhd=new newHostDialog();
    if (oe->isVisible()) oe->hide();
    nhd->exec();
    FWObject *o = nhd->getNewHost();
    delete nhd;

    if (o!=NULL)
    {
        openObject(o);
        editObject(o);
    }
}

void ObjectManipulator::newInterface()
{
    if ( currentObj->isReadOnly() ) return;

    FWObject *i=NULL;

    if (Host::isA(currentObj) || Firewall::isA(currentObj))
        i=createObject(currentObj,Interface::TYPENAME,tr("New Interface"));

    if (Interface::isA(currentObj))
        i=createObject(currentObj->getParent(),Interface::TYPENAME,tr("New Interface"));

    if (i==NULL) return;

#ifdef USE_INTERFACE_POLICY
    if (Firewall::isA(i->getParent())) i->add(new InterfacePolicy());
#endif

    openObject( i );
    
    if (Firewall::isA(i->getParent())) mw->showFirewall(i->getParent());
    updateLastModifiedTimestampForAllFirewalls(i);

    editObject(i);
}

void ObjectManipulator::newNetwork()
{
    FWObject *o=createObject(Network::TYPENAME,tr("New Network"));
    if (o!=NULL)
    {
        openObject(o);
        editObject(o);
    }
}

void ObjectManipulator::newAddress()
{
    if ( currentObj->isReadOnly() ) return;

    FWObject *o;
/*
 * Oleg reports that his expectation was that "New Address" should
 * always create an address object even if current selected object in
 * the tree is an interface. I tend to agree with him, this was a
 * usability issue because behavior of the program was different
 * depending on which object was selected in the tree. I am changing
 * it and will make it so "New Address" will always create a new
 * Address object uner Objects/Addresses. Interface address can be
 * created using context pop-up menu.
 *                                         12/19/04 --vk
 */
    o=createObject(IPv4::TYPENAME,tr("New Address"));

#if 0
    if (Interface::isA(currentObj))
    {
        Interface *intf = Interface::cast(currentObj);
        if (intf &&
            (intf->isDyn() || intf->isUnnumbered() || intf->isBridgePort())
        ) return;
	QString iname=QString("%1:%2:ip")
	    .arg(QString::fromUtf8(currentObj->getParent()->getName().c_str()))
	    .arg(QString::fromUtf8(currentObj->getName().c_str()));
        o=createObject(currentObj, IPv4::TYPENAME, iname);
    }
    else
    {
        o=createObject(IPv4::TYPENAME,tr("New Address"));
    }
#endif


    if (o!=NULL)
    {
        openObject(o);
        editObject(o);
    }
}

void ObjectManipulator::newDNSName()
{
    FWObject *o;
    o=createObject(DNSName::TYPENAME,tr("New DNS Name"));
    if (o!=NULL)
    {
        openObject(o);
        editObject(o);
    }
}

void ObjectManipulator::newAddressTable()
{
    FWObject *o;
    o=createObject(AddressTable::TYPENAME,tr("New Address Table"));
    if (o!=NULL)
    {
        openObject(o);
        editObject(o);
    }
}

void ObjectManipulator::newInterfaceAddress()
{
    if ( currentObj->isReadOnly() ) return;

    if (Interface::isA(currentObj))
    {
        Interface *intf = Interface::cast(currentObj);
        if (intf &&
            (intf->isDyn() || intf->isUnnumbered() || intf->isBridgePort())
        ) return;
	QString iname=QString("%1:%2:ip")
	    .arg(QString::fromUtf8(currentObj->getParent()->getName().c_str()))
	    .arg(QString::fromUtf8(currentObj->getName().c_str()));
        FWObject *o=createObject(currentObj, IPv4::TYPENAME, iname);
        if (o!=NULL)
        {
            openObject(o);
            editObject(o);
            updateLastModifiedTimestampForAllFirewalls(o);
        }
    }
}

void ObjectManipulator::newTagService()
{
    FWObject *o;
    o=createObject(TagService::TYPENAME,tr("New TagService"));
    if (o!=NULL)
    {
        openObject(o);
        editObject(o);
    }
}
void ObjectManipulator::newPhysicalAddress()
{
    if ( currentObj->isReadOnly() ) return;

    if (Interface::isA(currentObj))
    {
        Interface *intf = Interface::cast(currentObj);
        if (intf->getByType(physAddress::TYPENAME).empty())
        {
            QString iname=QString("%1:%2:mac")
                .arg(QString::fromUtf8(currentObj->getParent()->getName().c_str()))
                .arg(QString::fromUtf8(currentObj->getName().c_str()));
            FWObject *o=createObject(currentObj,physAddress::TYPENAME,iname);
            if (o!=NULL)
            {
                openObject(o);
                editObject(o);
                updateLastModifiedTimestampForAllFirewalls(o);
            }
        }
    }
}

void ObjectManipulator::newAddressRange()
{
    FWObject *o;
    o=createObject(AddressRange::TYPENAME,tr("New Address Range"));
    if (o!=NULL)
    {
        openObject(o);
        editObject(o);
    }
}

void ObjectManipulator::newObjectGroup()
{
    FWObject *o;
    o=createObject(ObjectGroup::TYPENAME,tr("New Object Group"));
    if (o!=NULL)
    {
        openObject(o);
        editObject(o);
    }
}


void ObjectManipulator::newCustom()
{
    FWObject *o;
    o=createObject(CustomService::TYPENAME,tr("New Custom Service"));
    if (o!=NULL)
    {
        openObject(o);
        editObject(o);
    }
}

void ObjectManipulator::newIP()
{
    FWObject *o;
    o=createObject(IPService::TYPENAME,tr("New IP Service"));
    if (o!=NULL)
    {
        openObject(o);
        editObject(o);
    }
}

void ObjectManipulator::newICMP()
{
    FWObject *o;
    o=createObject(ICMPService::TYPENAME,tr("New ICMP Service"));
    if (o!=NULL)
    {
        openObject(o);
        editObject(o);
    }
}

void ObjectManipulator::newTCP()
{
    FWObject *o;
    o=createObject(TCPService::TYPENAME,tr("New TCP Service"));
    if (o!=NULL)
    {
        openObject(o);
        editObject(o);
    }
}

void ObjectManipulator::newUDP()
{
    FWObject *o;
    o=createObject(UDPService::TYPENAME,tr("New UDP Service"));
    if (o!=NULL)
    {
        openObject(o);
        editObject(o);
    }
}

void ObjectManipulator::newServiceGroup()
{
    FWObject *o;
    o=createObject(ServiceGroup::TYPENAME,tr("New Service Group"));
    if (o!=NULL)
    {
        openObject(o);
        editObject(o);
    }
}


void ObjectManipulator::newInterval()
{
    FWObject *o;
    o=createObject(Interval::TYPENAME,tr("New Time Interval"));
    if (o!=NULL)
    {
        openObject(o);
        editObject(o);
    }
}



bool ObjectManipulator::validateDialog()
{
    if (currentObj==NULL) return true;
    if (!oe->isVisible()) return true;
    return oe->validateAndSave();
}

void ObjectManipulator::select()
{
    if (fwbdebug)
        qDebug("ObjectManipulator::select()");

    if (currentObj==NULL) return;
    ObjectTreeViewItem *otvi=allItems[currentObj];
    if (otvi)
    {
        otvi->listView()->setSelected(otvi,true);
        active=true;
        otvi->listView()->setFocus();
        otvi->listView()->update();
    }
}

void ObjectManipulator::unselect()
{
    if (currentObj==NULL) return;

    for (int i=0; i<libs->count(); i++)
        idxToTrees[i]->clearSelection();

    active=false;
}

bool ObjectManipulator::isSelected()
{
    return active;
}

list<Firewall *> ObjectManipulator::findFirewallsForObject(FWObject *o)
{
    if (fwbdebug)
        qDebug("ObjectManipulator::findFirewallsForObject");

    list<Firewall *> fws;
    set<FWObject *> resset;
    QTime tt;
    tt.start();
    FWObject *f=o;
    while (f!=NULL && !Firewall::isA(f)) f=f->getParent();
    if (f) fws.push_back(Firewall::cast(f));
    mw->db()->findWhereUsed(o,mw->db(),resset);
    
    set<FWObject *>::iterator i=resset.begin();
    for ( ;i!=resset.end();++i)
    {
        RuleElement *re=RuleElement::cast(*i);
        if (re==NULL) continue;
        
        Rule *r=Rule::cast(re->getParent());
        if (r && !r->isDisabled())
        {
            f=r;
            while (f!=NULL && !Firewall::isA(f)) f=f->getParent();
            if (f && std::find(fws.begin(),fws.end(),f)==fws.end()) 
                fws.push_back(Firewall::cast(f));
        }
    }
   
    if (fwbdebug)
        qDebug(QString("Program spent %1 ms searching for firewalls.")
               .arg(tt.elapsed()));
    
    return fws;
}

void ObjectManipulator::updateLastModifiedTimestampForOneFirewall(FWObject *o)
{
    if (fwbdebug) qDebug("ObjectManipulator::updateLastModifiedTimestampForOneFirewall");

    if (o==NULL) return;

    Firewall *f = Firewall::cast(o);
    if (f==NULL) return;

    f->updateLastModifiedTimestamp();
    getCurrentObjectTree()->updateTreeItems ();
    info();
}


void ObjectManipulator::updateLastModifiedTimestampForAllFirewalls(FWObject *o)
{
    if (fwbdebug) qDebug("ObjectManipulator::updateLastModifiedTimestampForAllFirewalls");

    if (o==NULL) return;

    QStatusBar *sb = mw->statusBar();
    sb->message( tr("Searching for firewalls affected by the change...") );
    QApplication::eventLoop()->processEvents(QEventLoop::ExcludeUserInput,100);

    if (fwbdebug) qDebug("ObjectManipulator::updateLastModifiedTimestampForAllFirewalls:   setOverrideCursor");
    QApplication::setOverrideCursor(QCursor( Qt::WaitCursor));

    list<Firewall *> fws = findFirewallsForObject(o);
    if (fws.size())
    {
        Firewall *f;
        for (list<Firewall *>::iterator i=fws.begin();
                i!=fws.end();
                ++i)
        {
            f=*i;
            f->updateLastModifiedTimestamp();
        }
        
        getCurrentObjectTree()->updateTreeItems ();
        info();
    }
    if (fwbdebug) qDebug("ObjectManipulator::updateLastModifiedTimestampForAllFirewalls:   restoreOverrideCursor");
    QApplication::restoreOverrideCursor();
    sb->clear();
    QApplication::eventLoop()->processEvents(QEventLoop::ExcludeUserInput,100);
}

void ObjectManipulator::updateLastInstalledTimestamp(FWObject *o)
{
    if (fwbdebug) qDebug("ObjectManipulator::updateLastInstalledTimestamp");
    if (o==NULL) return;
    Firewall * f=(Firewall *)o;
    if (f!=NULL)
    {
        bool visualUpdate=f->needsInstall();
        f->updateLastInstalledTimestamp();
        if (visualUpdate) 
        {
            getCurrentObjectTree()->updateTreeItems ();
        }
        info();
    }
}
void ObjectManipulator::updateLastCompiledTimestamp(FWObject *o)
{
    if (fwbdebug) qDebug("ObjectManipulator::updateLastCompiledTimestamp");

    if (o==NULL) return;
    Firewall * f=(Firewall *)o;
    if (f!=NULL)
    {
        f->updateLastCompiledTimestamp();
        info();
    }
}

void ObjectManipulator::simulateInstall()
{
    if (fwbdebug) qDebug("ObjectManipulator::simulateInstall");
    
    if (getCurrentObjectTree()->getNumSelected()==0) return;
    
    Firewall *fw;

    vector<FWObject*> so = getCurrentObjectTree()->getSimplifiedSelection();
    for (vector<FWObject*>::iterator i=so.begin();  i!=so.end(); ++i)
    {
        fw= Firewall::cast( *i );
        if (fw!=NULL)
        {
            fw->updateLastCompiledTimestamp();
            fw->updateLastInstalledTimestamp();
        }
    }
    getCurrentObjectTree()->updateTreeItems (); 
    
}

void ObjectManipulator::findAllFirewalls (list<Firewall *> &fws)
{
    if (fwbdebug) qDebug("ObjectManipulator::findAllFirewalls");

    std::list<FWObject*> fwlist;
    findByObjectType(mw->db(),Firewall::TYPENAME,fwlist);
    for (list<FWObject*>::iterator m=fwlist.begin(); m!=fwlist.end(); m++)
        fws.push_back( Firewall::cast(*m) );
}

FWObject* ObjectManipulator::getSelectedObject()
{
    return currentObj;
}

void ObjectManipulator::findWhereUsedSlot()
{
    if (getCurrentObjectTree()->getNumSelected()==0) return;

    FWObject *obj=getCurrentObjectTree()->getSelectedObjects().front();
    if (obj==NULL) return;
    mw->findWhereUsed(obj);
    
}

