/* 

                          Firewall Builder

                 Copyright (C) 2003 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@fwbuilder.org

  $Id: GroupObjectDialog.cpp,v 1.54 2006/10/22 04:39:36 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include "config.h"
#include "global.h"
#include "utils.h"

#include "fwbuilder/FWObjectDatabase.h"
#include "FWWindow.h"
#include "FWBTree.h"
#include "FWBSettings.h"
#include "FWObjectPropertiesFactory.h"
#include "GroupObjectDialog.h"
#include "ObjectListViewItem.h"
#include "ObjectIconViewItem.h"
#include "ObjectManipulator.h"
#include "FWObjectDrag.h"
#include "FWObjectClipboard.h"

#include "fwbuilder/Library.h"
#include "fwbuilder/Group.h"
#include "fwbuilder/Resources.h"

#include <qlineedit.h>
#include <qtextedit.h>
#include <qiconview.h>
#include <qlistview.h>
#include <qtoolbutton.h>
#include <qcombobox.h>
#include <qevent.h>
#include <qwidgetstack.h>
#include <qlayout.h>
#include <qpushbutton.h>
#include <qpopupmenu.h>
#include <qtooltip.h>
#include <qscrollview.h>
#include <qpixmapcache.h>

#include <iostream>
#include <algorithm>

using namespace std;
using namespace libfwbuilder;

enum GroupObjectDialog::viewType GroupObjectDialog::vt = GroupObjectDialog::Icon;

#define LIST_VIEW_MODE "list"
#define ICON_VIEW_MODE "icon"

IconViewObjToolTip::IconViewObjToolTip(QIconView *w): QToolTip(w->viewport(),0)
{
    view=w;
}

void IconViewObjToolTip::maybeTip(const QPoint &pos)
{
    if (st->getObjTooltips())
    {
        int cx,cy;

        view->viewportToContents(pos.x(),pos.y(),cx,cy);

        FWObject  *obj=NULL;
        QRect      cr;

        QIconViewItem      *itm   = view->findItem( QPoint(cx,cy) );
        if (itm==NULL) return;
        ObjectIconViewItem *oivi  = dynamic_cast<ObjectIconViewItem*>(itm);
        assert(oivi!=NULL);
        obj     = oivi->getFWObject();
        cr      = itm->rect();

        if (obj==NULL) return;

        cr.moveTopLeft( view->contentsToViewport( cr.topLeft() ) );

        tip(cr,
       FWObjectPropertiesFactory::getObjectPropertiesDetailed(obj,true,true));
    }
}


ListViewObjToolTip::ListViewObjToolTip(QListView *w): QToolTip(w->viewport(),0)
{
    view=w;
}

void ListViewObjToolTip::maybeTip(const QPoint &pos)
{
    if (st->getObjTooltips())
    {
        QListViewItem      *itm   = view->itemAt( pos );
        if (itm==NULL) return;
        ObjectListViewItem *otvi  = dynamic_cast<ObjectListViewItem*>(itm);
        assert(otvi!=NULL);

        FWObject *obj     = otvi->getFWObject();
        QRect     cr      = view->itemRect(itm);

        if (obj==NULL) return;

//        cr.moveTopLeft( view->contentsToViewport( cr.topLeft() ) );

        tip(cr,
       FWObjectPropertiesFactory::getObjectPropertiesDetailed(obj,true,true));
    }
}



GroupObjectDialog::GroupObjectDialog(QWidget *parent) : GroupObjectDialog_q(parent)
{
    obj=NULL;
    selectedObject=NULL;

    iconView = new ObjectIconView( WStackPage, "iconView" );
    iconView->setItemsMovable( FALSE );

    listView = new ObjectListView( WStackPage, "listView" );
    listView->addColumn( tr( "Name" ) );
    listView->addColumn( tr( "Properties" ) );
    listView->setAcceptDrops( TRUE );


    new IconViewObjToolTip(iconView);
    new ListViewObjToolTip(listView);


    //QGridLayout *WStackPageLayout = new QGridLayout( WStackPage, 1, 1, 0, 0, "WStackPageLayout"); 

    //WStackPageLayout->addWidget( iconView, 0, 0 );
    //WStackPageLayout->addWidget( listView, 0, 0 );
    //objectViewsStack->addWidget( WStackPage, 0 );

    objectViewsStack->addWidget(iconView);
    objectViewsStack->addWidget(listView);
    objectViewsStack->raiseWidget(iconView);

    
    setTabOrder( obj_name, iconView );
    setTabOrder( iconView, listView );
    setTabOrder( listView, comment );

    listView->setSelectionMode(QListView::Extended);
    iconView->setSelectionMode(QIconView::Extended);
    
    //listView->hide();
    //iconView->show();
    

    iconViewBtn->setAutoRaise(false);
    listViewBtn->setAutoRaise(false);

    //apply->setEnabled( false );

    connect( iconView, SIGNAL( doubleClicked(QIconViewItem*) ),
             this,     SLOT( openObject(QIconViewItem*) ) );

    connect( iconView, SIGNAL( returnPressed(QIconViewItem*) ),
             this,     SLOT( openObject(QIconViewItem*) ) );

    connect( iconView, SIGNAL( currentChanged(QIconViewItem*) ),
             this,     SLOT( iconViewCurrentChanged(QIconViewItem*) ) );

    connect( iconView, SIGNAL (selectionChanged()),
            this,      SLOT (iconViewSelectionChanged())); 

    connect( iconView, SIGNAL( dropped(QDropEvent*,const QValueList<QIconDragItem>&) ),
             this,     SLOT( dropped(QDropEvent*) ) );

    connect( iconView, SIGNAL( contextMenuRequested(QIconViewItem*,const QPoint&) ),
             this,     SLOT( iconContextMenu(QIconViewItem*,const QPoint&) ) );

    connect( iconView, SIGNAL( delObject_sign() ),
             this,     SLOT( deleteObj() ) );



    connect( listView, SIGNAL( doubleClicked(QListViewItem*) ),
             this,     SLOT( openObject(QListViewItem*) ) );
    connect( listView, SIGNAL( returnPressed(QListViewItem*) ),
             this,     SLOT( openObject(QListViewItem*) ) );
    connect( listView, SIGNAL( currentChanged(QListViewItem*) ),
             this,     SLOT( listViewCurrentChanged(QListViewItem*) ) );
    connect( listView, SIGNAL (selectionChanged()),
            this,      SLOT (listViewSelectionChanged())); 

    connect( listView, SIGNAL( dropped(QDropEvent*) ),
             this,     SLOT( dropped(QDropEvent*) ) );
    connect( listView, SIGNAL( contextMenuRequested(QListViewItem*,const QPoint&,int) ),
             this,     SLOT( listContextMenu(QListViewItem*,const QPoint&,int) ) );
    connect( listView, SIGNAL( delObject_sign() ),
             this,     SLOT( deleteObj() ) );

    QString s    = st->getGroupViewColumns();
    int     col0 = s.section(',',0,0).toInt();
    int     col1 = s.section(',',1,1).toInt();

    listView->setColumnWidth(0,col0);
    listView->setColumnWidth(1,col1);

    QString mode=st->getGroupViewMode();
    if (mode==ICON_VIEW_MODE) switchToIconView();
    if (mode==LIST_VIEW_MODE) switchToListView();

}

void GroupObjectDialog::iconViewSelectionChanged()
{
    selectedObjects.clear();
    QIconViewItem *item;
    for ( item = iconView->firstItem(); item; item = item->nextItem() )
    {
        if(item->isSelected())
        {
            ObjectIconViewItem *oivi=dynamic_cast<ObjectIconViewItem*>(item);
            assert(oivi!=NULL);
            FWObject *o = oivi->getFWObject();
            if (o!=NULL)
                selectedObjects.push_back(o);
        }
        
    }
        
}
void GroupObjectDialog::listViewSelectionChanged()
{
    selectedObjects.clear();
    QListViewItemIterator it(listView);
    while(it.current())
    {
        if(it.current()->isSelected())
        {
            QListViewItem *itm= it.current();
            ObjectListViewItem *otvi=dynamic_cast<ObjectListViewItem*>(itm);
            assert(otvi!=NULL);

            FWObject *o=otvi->getFWObject();
            assert(o!=NULL);
            selectedObjects.push_back(o);
        }
        ++it;
    }
}

void GroupObjectDialog::iconViewCurrentChanged(QIconViewItem *itm)
{
    if (itm==NULL)
    {
        selectedObject=NULL;
        return;
    }

    ObjectIconViewItem *oivi=dynamic_cast<ObjectIconViewItem*>(itm);
    assert(oivi!=NULL);

    FWObject *o = oivi->getFWObject();
    if (o!=NULL)
        selectedObject=o;

}


void GroupObjectDialog::listViewCurrentChanged(QListViewItem *itm)
{
    if (itm==NULL)
    {
        selectedObject=NULL;
        return;
    }

    ObjectListViewItem *otvi=dynamic_cast<ObjectListViewItem*>(itm);
    assert(otvi!=NULL);

    FWObject *o = otvi->getFWObject();
    if (o!=NULL)
        selectedObject=o;

}

/*
 * used to add an object for paste and drop operations
 */
void GroupObjectDialog::insertObject(FWObject *o)
{
    assert(o!=NULL);
    Group *g = dynamic_cast<Group*>(obj);
    assert(g!=NULL);

    if ( ! g->validateChild(o) || g->isReadOnly() ) return;

    if (fwbdebug)
        qDebug("Adding object %s to the group %s",
               o->getName().c_str(), g->getName().c_str());

/* avoid duplicates */
    string cp_id=o->getId();

    map<string, ObjectListViewItem*>::iterator i;
    for (i=allListViewItems.begin(); i!=allListViewItems.end(); ++i)
    {
        string go=(*i).first;
        //if (FWReference::cast(go)!=NULL) go=FWReference::cast(go)->getPointer();
        //if (o==go || o->getId()==go->getId()) return;
	if(go==cp_id) return;
    }

    addIcon(o, ! FWBTree::isSystem(obj) );

    
    changed();
}

void GroupObjectDialog::addIcon(FWObject *fwo)
{
    FWObject *o=fwo;
    bool      ref=false;
    if (FWReference::cast(o)!=NULL)
    {
        o=FWReference::cast(o)->getPointer();
        ref=true;
    }

    addIcon(o,ref);
}

void GroupObjectDialog::addIcon(FWObject *o,bool ref)
{
    if (Resources::global_res->getResourceBool(
            string("/FWBuilderResources/Type/") +
            o->getTypeName() + "/hidden") ) return;

    QString obj_name=QString::fromUtf8(o->getName().c_str());
    
    QString icn_filename =
        Resources::global_res->getObjResourceStr(o, (ref)?"icon-ref":"icon").c_str();

    QPixmap pm;
    if ( ! QPixmapCache::find( icn_filename, pm) )
    {
        pm = QPixmap::fromMimeSource( icn_filename );
        QPixmapCache::insert( icn_filename, pm);
    }

    ObjectIconViewItem *ivitm = new ObjectIconViewItem(iconView,obj_name,pm );
    ivitm->setProperty("id",   o->getId().c_str()   );
    ivitm->setProperty("type", o->getTypeName().c_str() );
    ivitm->setFWObject( o );

    allIconViewItems[o->getId()]=ivitm;

    icn_filename = Resources::global_res->getObjResourceStr(o, "icon-tree").c_str();

    ObjectListViewItem *tvitm=new ObjectListViewItem( listView );
    tvitm->setText( 0, obj_name );
    tvitm->setText( 1, FWObjectPropertiesFactory::getObjectProperties(o) );
    tvitm->setPixmap( 0, pm);
    tvitm->setDragEnabled(true);

    tvitm->setProperty("id",   o->getId().c_str()   );
    tvitm->setProperty("type", o->getTypeName().c_str() );
    tvitm->setFWObject( o );

    allListViewItems[o->getId()]=tvitm;
}

void GroupObjectDialog::loadFWObject(FWObject *o)
{
    obj=o;
    Group *g = Group::cast(obj);
    assert(g!=NULL);

    init=true;

    allIconViewItems.clear();
    allListViewItems.clear();

    fillLibraries(libs,obj);

    obj_name->setText( QString::fromUtf8(g->getName().c_str()) );
    comment->setText( QString::fromUtf8(g->getComment().c_str()) );

    obj_name->setEnabled( !FWBTree::isSystem(obj) );
    libs->setEnabled(     !FWBTree::isSystem(obj) );
    comment->setEnabled(  !FWBTree::isSystem(obj) );

    listView->clear();

    iconView->clear();
    iconView->setAutoArrange( true );
    iconView->setResizeMode( QIconView::Adjust );
    iconView->setGridX(50);
    iconView->setGridY(40);
    iconView->setItemsMovable( FALSE );

    switch (vt)
    {
    case Icon:
        if ( ! iconViewBtn->isOn() ) iconViewBtn->toggle();
        //listView->hide();
        //iconView->show();
        objectViewsStack->raiseWidget(iconView);
        break;

    case List:
        if ( ! listViewBtn->isOn() ) listViewBtn->toggle();
        //listView->show();
        //iconView->hide();
        objectViewsStack->raiseWidget(listView);
        break;
    }

    for (FWObject::iterator i=g->begin(); i!=g->end(); i++)
        addIcon( *i );

    
    //apply->setEnabled( false );

    obj_name->setEnabled(!o->isReadOnly() && !FWBTree::isSystem(o));
    setDisabledPalette(obj_name);

    comment->setEnabled(!o->isReadOnly() && !FWBTree::isSystem(o));
    setDisabledPalette(comment);

    libs->setEnabled(!o->isReadOnly() && !FWBTree::isSystem(o));
    setDisabledPalette(libs);

//    listView->setEnabled(!o->isReadOnly());
    setDisabledPalette(listView);

//    iconView->setEnabled(!o->isReadOnly());
    setDisabledPalette(iconView);


    init=false;
}
    
void GroupObjectDialog::changed()
{
    //if (!init) apply->setEnabled( true );

    emit changed_sign();
}

void GroupObjectDialog::validate(bool *res)
{
    *res=true;
}

void GroupObjectDialog::isChanged(bool *res)
{
    //*res=(!init && apply->isEnabled());
}

void GroupObjectDialog::libChanged()
{
    changed();
}

void GroupObjectDialog::applyChanges()
{
    if (fwbdebug)
        qDebug("GroupObjectDialog::applyChanges");

    if (!isTreeReadWrite(this,obj)) return;
    if (!validateName(this,obj,obj_name->text())) return;

    string oldname=obj->getName();
    obj->setName( string(obj_name->text().utf8()) );
    obj->setComment( string(comment->text().utf8()) );

    init=true;

    set<FWObject*> oldobj;
    set<FWObject*> newobj;
    map<string, ObjectListViewItem*>::iterator i;
    for (i=allListViewItems.begin(); i!=allListViewItems.end(); ++i)
    {
        newobj.insert( mw->db()->findInIndex((*i).first) );
    }

    for (FWObject::iterator j=obj->begin(); j!=obj->end(); ++j)
    {
        FWObject *o= *j;
        if (FWReference::cast(o)!=NULL)
            o=FWReference::cast(o)->getPointer();
        oldobj.insert( o );
    }

    set<FWObject*> diff;
    set_difference( oldobj.begin(), oldobj.end(),
                    newobj.begin(), newobj.end(), 
                    inserter(diff,diff.begin()));
/* diff contains objects present in oldobj but not in newobj - these objects
   were deleted from the group */

    for (set<FWObject*>::iterator k=diff.begin(); k!=diff.end(); ++k)
    {
        if (FWBTree::isSystem(obj)) om->delObj(*k, false);
        else               obj->removeRef( *k );
    }

    diff.clear();

    set_difference( newobj.begin(), newobj.end(), 
                    oldobj.begin(), oldobj.end(),
                    inserter(diff,diff.begin()));
/* diff contains objects present in newobj but not in oldobj - these objects
   were added to the group */

    for (set<FWObject*>::iterator k1=diff.begin(); k1!=diff.end(); ++k1)
    {
        if (FWBTree::isSystem(obj))  om->pasteTo(obj,*k1, false);
        else                obj->addRef( *k1);
    }

    om->updateObjName(obj,QString::fromUtf8(oldname.c_str()));

    init=true;

/* move to another lib if we have to */
    if (! FWBTree::isSystem(obj) && libs->currentText() != QString(obj->getLibrary()->getName().c_str()))
        om->moveObject(libs->currentText(), obj);

    init=false;

    //apply->setEnabled( false );
    om->updateLastModifiedTimestampForAllFirewalls(obj);

    if (fwbdebug)
        qDebug("GroupObjectDialog::applyChanges done");
}

void GroupObjectDialog::discardChanges()
{
    loadFWObject(obj);
}

void GroupObjectDialog::switchToIconView()
{
    if (vt == Icon) return;
    vt = Icon;

    if ( ! iconViewBtn->isOn() ) iconViewBtn->toggle();

    //listView->hide();
    //iconView->show();
    objectViewsStack->raiseWidget(iconView);

    st->setGroupViewMode(ICON_VIEW_MODE);
}

void GroupObjectDialog::switchToListView()
{
    if (vt == List) return;
    vt = List;

    if ( ! listViewBtn->isOn() ) listViewBtn->toggle();

    //listView->show();
    //iconView->hide();
    objectViewsStack->raiseWidget(listView);

    st->setGroupViewMode(LIST_VIEW_MODE);
}

void GroupObjectDialog::openObject()
{
    if (selectedObject!=NULL)
    {
        om->openObject( selectedObject );
        om->editObject( selectedObject );
    }
}

void GroupObjectDialog::openObject(QListViewItem *itm)
{
    ObjectListViewItem *otvi=dynamic_cast<ObjectListViewItem*>(itm);
    assert(otvi!=NULL);

    FWObject *o = otvi->getFWObject();
    if (o!=NULL)
    {
        om->openObject( o );
        om->editObject( o );
    }
}

void GroupObjectDialog::openObject(QIconViewItem *itm)
{
    ObjectIconViewItem *oivi=dynamic_cast<ObjectIconViewItem*>(itm);
    assert(oivi!=NULL);

    FWObject *o = oivi->getFWObject();
    if (o!=NULL)
    {
        om->openObject( o );
        om->editObject( o );
    }
}

void GroupObjectDialog::dropped(QDropEvent *ev)
{
    list<FWObject*> ol;
    if (FWObjectDrag::decode(ev, ol))
    {
        if (ol.size()==0) return;
        for (list<FWObject*>::iterator i=ol.begin(); i!=ol.end(); ++i)
            insertObject( *i );
    }
}

void GroupObjectDialog::iconContextMenu(QIconViewItem *itm,const QPoint &p)
{
    FWObject *o=NULL;
    ObjectIconViewItem *oivi=dynamic_cast<ObjectIconViewItem*>(itm);
    if (oivi!=NULL) o = oivi->getFWObject();
    selectedObject=o;

    setupPopupMenu(p);
}


void GroupObjectDialog::listContextMenu(QListViewItem *itm,const QPoint &p,int col)
{
    FWObject *o=NULL;
    ObjectListViewItem *otvi=dynamic_cast<ObjectListViewItem*>(itm);
    if (otvi!=NULL) o = otvi->getFWObject();
    selectedObject=o;

    setupPopupMenu(p);
}

void GroupObjectDialog::setupPopupMenu(const QPoint &pos)
{
    QPopupMenu *popup=new QPopupMenu(this);

    if (selectedObject!=NULL)
    {
        if (selectedObject->isReadOnly() )
            popup->insertItem( tr("Open") ,  this , SLOT( openObject())    );
        else
            popup->insertItem( tr("Edit") ,  this , SLOT( openObject())    );
    }

    int copyID =popup->insertItem( tr("Copy") ,  this , SLOT( copyObj())   );
    int cutID  =popup->insertItem( tr("Cut") ,   this , SLOT( cutObj())    );
    int pasteID=popup->insertItem( tr("Paste") , this , SLOT( pasteObj())  );
    int delID  =popup->insertItem( tr("Delete") ,this , SLOT( deleteObj()) );

    popup->setItemEnabled(copyID,
                          selectedObject!=NULL &&
                          ! FWBTree::isSystem(selectedObject) );
    popup->setItemEnabled(cutID,
                          selectedObject!=NULL && 
                          ! FWBTree::isSystem(obj) &&
                          ! obj->isReadOnly() );
    popup->setItemEnabled(pasteID,
                          ! FWBTree::isSystem(obj) &&
                          ! obj->isReadOnly() );
    popup->setItemEnabled(delID,
                          selectedObject!=NULL &&
                          ! FWBTree::isSystem(obj) &&
                          ! obj->isReadOnly() );

    popup->exec( pos );
}

void GroupObjectDialog::copyObj()
{
    FWObjectClipboard::obj_clipboard->clear();
    for(vector<FWObject*>::iterator it=selectedObjects.begin();
            it!=selectedObjects.end(); ++it)
    {
        FWObject* selectedObject=*it;
        
        if (selectedObject!=NULL && ! FWBTree::isSystem(selectedObject) ) 
        {
            FWObject *o=selectedObject;
            if (FWReference::cast(o)!=NULL)
                o=FWReference::cast(o)->getPointer();
            

            FWObjectClipboard::obj_clipboard->add( o );
        }
        
    }
}

void GroupObjectDialog::cutObj()
{
    copyObj();
    deleteObj();
}

void GroupObjectDialog::pasteObj()
{
    vector<string>::iterator i;
    
    for (i= FWObjectClipboard::obj_clipboard->begin();
         i!=FWObjectClipboard::obj_clipboard->end(); ++i)
    {
        insertObject( mw->db()->findInIndex(*i) );
    }

//    if (FWObjectClipboard::obj_clipboard->getObject()==NULL) return;
//    insertObject( FWObjectClipboard::obj_clipboard->getObject() );
}

void GroupObjectDialog::deleteObj()
{
    vector<FWObject*> tv;
    FWObject* selectedObject;
    copy (selectedObjects.begin(),selectedObjects.end(),inserter(tv,tv.begin()));
    
    for(vector<FWObject*>::iterator it=tv.begin();
            it!=tv.end(); ++it)
    {
        selectedObject=(*it);
        
        if (selectedObject!=NULL &&  ! FWBTree::isSystem(obj) )
        {
            Group *g = dynamic_cast<Group*>(obj);
            assert(g!=NULL);

            FWObject *o=selectedObject;
            assert(o!=NULL);
            if (FWReference::cast(o)!=NULL)
                o=FWReference::cast(o)->getPointer();

            selectedObject=NULL;

    //        g->removeRef(o);

            assert(allListViewItems[o->getId()]!=NULL);
            delete allListViewItems[o->getId()];
            allListViewItems.erase(o->getId());

            assert(allIconViewItems[o->getId()]!=NULL);
            delete allIconViewItems[o->getId()];
            allIconViewItems.erase(o->getId());
            
        }
    }
    changed();
}

/* ObjectEditor class connects its slot to this signal and does all
 * the verification for us, then accepts (or not) the event. So we do
 * nothing here and defer all the processing to ObjectEditor
 */
void GroupObjectDialog::closeEvent(QCloseEvent *e)
{

    QString s = QString("%1,%2")
        .arg(listView->columnWidth(0))
        .arg(listView->columnWidth(1));

    st->setGroupViewColumns(s);

    emit close_sign(e);

}

void GroupObjectDialog::selectObject(FWObject *o)
{
//    ObjectListViewItem* list_item=allListViewItems[o->getId()];
    ObjectIconViewItem* icon_item=allIconViewItems[o->getId()];

    iconView->setSelected(icon_item,true,false);
}
