/* GKrellM
|  Copyright (C) 1999-2001 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|  Latest versions might be found at:  http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* 3/13/2000  Patch from Tom Gilbert <gilbertt@btinternet.com> allows window
|			  managers to differentiate options dialog from GKrellM itself.
*/

#include "gkrellm.h"
#include "gkrellm_private_proto.h"
#include <dirent.h>



typedef struct
	{
	GtkText		*text;
	gchar		*string;
	gint		translate;
	}
	InfoText;

static GList	*info_text_list;


#define		N_FONT_OVERRIDES	3

GtkWidget		*config_window;

static GdkFont	*text_font_normal,
				*text_font_bold,
				*text_font_italic;

static gint		rebuild_flag;

  /* These names nust track the order of MON_ defines in gkrellm.h
  */
static gchar	*chart_monitor_name[N_CHART_MONITORS] =
	{
	N_("CPU"), N_("Proc"), N_("Disk"), N_("Net"), N_("Inet"),
	};

/* void gtk_adjustment_set_value( GtkAdjustment *adjustment, gfloat value); */

  /* Popup a message close to a widget (should be one with a window, eg
  |  drawing_area).
  */
static void
message_window(gchar *title, gchar *message, GtkWidget *widget, gint mainwin)
	{
	GtkWidget	*top_win;
	GtkWidget	*window;
	GtkWidget	*scrolled;
	GtkWidget	*vbox, *vbox1, *hbox;
	GtkWidget	*label;
	GtkWidget	*separator;
	GtkWidget	*button;
	gchar		*s;
	gint		x, y, w, h, w_main, nlines, dummy;

	if (!message)
		return;
	top_win = gkrellm_get_top_window();
	window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	if (widget == NULL)
		widget = top_win;

    gtk_window_set_policy(GTK_WINDOW(window), TRUE, TRUE, FALSE);
    gtk_window_set_title(GTK_WINDOW(window), title);
    gtk_container_border_width(GTK_CONTAINER(window), 8);

    vbox = gtk_vbox_new(FALSE, 2);
    gtk_container_add(GTK_CONTAINER(window),vbox);

	for (nlines = 0, s = message; *s; ++s)
		if (*s == '\n')
			++nlines;
	if (nlines > 20)
		{
		scrolled = gtk_scrolled_window_new(NULL, NULL);
		gtk_widget_set_usize(scrolled, 0, 300);	/* A guess */
		gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
				GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
		gtk_box_pack_start(GTK_BOX(vbox), scrolled, TRUE, TRUE, 0);
		vbox1 = gtk_vbox_new(FALSE, 2);
		gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolled),
				vbox1);
		}
	else
		vbox1 = vbox;
	label = gtk_label_new(message);
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(vbox1), label, FALSE, FALSE,0);

	separator = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox), separator, FALSE, FALSE, 4);

	hbox = gtk_hbox_new(TRUE, 2);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 2);
    button = gtk_button_new_with_label(_("  OK  "));
    gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);

	gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
		(GtkSignalFunc) gtk_widget_destroy, GTK_OBJECT(window));
	gtk_widget_show_all(window);
	gdk_window_get_origin(widget->window, &x, &y);
	gdk_window_get_size(window->window, &w, &h);
	if (mainwin)
		{
		gdk_window_get_origin(top_win->window, &x, &dummy);
		gdk_window_get_size(top_win->window, &w_main, &dummy);
		x = (x < w_display / 2) ? x + w_main : x - w - 4;
		}
	else
		x -= w/2;

	if (y + h > h_display)
		y = h_display - h;
	if (y < 25)
		y = 25;
	gtk_widget_set_uposition(window, x, y);
	}

void
gkrellm_message_window(gchar *title, gchar *message, GtkWidget *widget)
	{
	message_window(title, message, widget, 1);
	}

void
gkrellm_config_message_window(gchar *title, gchar *message, GtkWidget *widget)
	{
	message_window(title, message, widget, 0);
	}

static void
add_info_text(InfoText *info)
	{
	GdkFont	*font	= NULL;
	gchar	*s;

#if defined(ENABLE_NLS)
	s = info->translate ? gettext(info->string) : info->string;
#else
	s = info->string;
#endif

	if (strncmp(s, "<b>", 3) == 0)
		{
		font = text_font_bold;
		s += 3;
		}
	else if (strncmp(s, "<i>", 3) == 0)
		{
		font = text_font_italic;
		s += 3;
		}
	else
		font = text_font_normal;

	gtk_text_insert(GTK_TEXT(info->text), font, NULL, NULL, s, -1);
	}

void
gkrellm_add_info_text_string(GtkWidget *text, gchar *string)
	{
	InfoText	*info;

	info = g_new0(InfoText, 1);
	info->text = GTK_TEXT(text);
	info->string = g_strdup(string);
	info->translate = FALSE;
	info_text_list = g_list_append(info_text_list, info);
	add_info_text(info);
	}

void
gkrellm_add_info_text(GtkWidget *text, gchar **string, gint n_strings)
	{
	InfoText	*info;
	gint		i;

	for (i = 0; i < n_strings; ++i)
		{
		info = g_new0(InfoText, 1);
		info->text = GTK_TEXT(text);
		info->string = g_strdup(string[i]);
		info->translate = TRUE;
		info_text_list = g_list_append(info_text_list, info);
		add_info_text(info);
		}
	}

void
gkrellm_launch_button_cb(DecalButton *button)
	{
	Launcher    *launch;
	gchar		*command;

	launch = (Launcher *) button->data;
	command = g_strconcat(launch->command, " &", NULL);
	system(command);
	g_free(command);
	}

void
gkrellm_configure_tooltip(Panel *p, Launcher *launch)
	{
	if (launch->tooltip == NULL)
		{
		if (*launch->tooltip_comment != '\0' && *launch->command != '\0')
			{
			launch->tooltip = gtk_tooltips_new();
			gtk_tooltips_set_tip(launch->tooltip, p->drawing_area,
					launch->tooltip_comment, NULL);
			gtk_tooltips_set_delay(launch->tooltip, 750);
			}
		return;
		}
	if (*launch->tooltip_comment != '\0' && *launch->command != '\0')
		{
		gtk_tooltips_set_tip(launch->tooltip, p->drawing_area,
					launch->tooltip_comment, NULL);
		gtk_tooltips_enable(launch->tooltip);
		}
	else
		gtk_tooltips_disable(launch->tooltip);
	}

void
gkrellm_apply_launcher(GtkWidget **launch_entry, GtkWidget **tooltip_entry,
			Panel *p, Launcher *launch, void (*func)())
	{
	gchar	*command, *tip_comment;

	command = gkrellm_entry_get_text(launch_entry);
	tip_comment = gkrellm_entry_get_text(tooltip_entry);
	if (   !strcmp(command, launch->command)
		&& !strcmp(tip_comment, launch->tooltip_comment)
	   )
		return;
	if (*command != '\0')		
		{
		if (launch->button == NULL)
			{
			if (launch->type == METER_PANEL_TYPE)
				launch->button = gkrellm_put_label_in_meter_button(p, func,
										launch, launch->pad);
			else
				launch->button = gkrellm_put_label_in_panel_button(p, func,
										launch, launch->pad);
			}
		gkrellm_dup_string(&launch->command, command);
		}
	else
		{
		if (launch->button)
			gkrellm_destroy_button(launch->button);
		launch->button = NULL;
		launch->pipe = NULL;		/* Close?? */
		gkrellm_dup_string(&launch->command, "");
		}
	gkrellm_dup_string(&launch->tooltip_comment, tip_comment);
	gkrellm_configure_tooltip(p, launch);
	}

GtkWidget *
gkrellm_launcher_table_new(GtkWidget *vbox, gint n_launchers)
	{
	GtkWidget	*table;

	table = gtk_table_new(2 * n_launchers, 3, TRUE /*homogeneous*/);
	gtk_table_set_col_spacings(GTK_TABLE(table), 2);
	gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, 2);
	return table;
	}

void
gkrellm_config_launcher(GtkWidget *table, gint n, GtkWidget **launch_entry,
			GtkWidget **tooltip_entry, gchar *desc, Launcher *launch)
	{
	gchar		buf[64];
	GtkWidget	*label, *hbox;
	gint		row;

	row = (tooltip_entry ? 2 : 1) * n;
	hbox = gtk_hbox_new(FALSE, 0);
    /* Attach left right top bottom */
    gtk_table_attach_defaults(GTK_TABLE(table), hbox, 0, 1, row, row + 1);

	snprintf(buf, sizeof(buf), _("%s command:"), desc);
    label = gtk_label_new(buf);
	gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
	gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 4);

    *launch_entry = gtk_entry_new_with_max_length(255);
    gtk_table_attach_defaults(GTK_TABLE(table), *launch_entry,
                1, 3, row, row + 1);
	gtk_entry_set_text(GTK_ENTRY(*launch_entry),
					(launch && launch->command) ? launch->command : "");

	if (tooltip_entry)
		{
		hbox = gtk_hbox_new(FALSE, 0);
	    gtk_table_attach_defaults(GTK_TABLE(table), hbox,
					0, 1, row + 1, row + 2);
		snprintf(buf, sizeof(buf), _("%s comment:"), desc);
	    label = gtk_label_new(buf);
		gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
		gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 4);
	    *tooltip_entry = gtk_entry_new_with_max_length(255);
	    gtk_table_attach_defaults(GTK_TABLE(table), *tooltip_entry,
	                1, 2, row + 1, row + 2);
		gtk_entry_set_text(GTK_ENTRY(*tooltip_entry),
					(launch && launch->tooltip_comment)
					? launch->tooltip_comment : "");
		}
	}

  /* FIXME: this guy is called on panels at create events
  |  when this situation has occured:  the GKrellM rebuild has destroyed
  |  the decal button list.  But existing launchers have not had their
  |  button pointer set to NULL!  This could cause a problem with
  |  code that tries to check for button pointers in the create routines.
  */
void
gkrellm_setup_launcher(Panel *p, Launcher *launch, gint type, gint pad)
	{
	if (launch->command == NULL)
		launch->command = g_strdup("");
	if (launch->tooltip_comment == NULL)
		launch->tooltip_comment = g_strdup("");
	launch->type = type;
	launch->pad = pad;
	if (p)
		{
		gkrellm_configure_tooltip(p, launch);
		if (*(launch->command) != '\0')
			launch->button = gkrellm_put_label_in_meter_button(p,
					gkrellm_launch_button_cb, launch, launch->pad);
		else
			launch->button = NULL;	/* In case dangling pointer, see above */
		}
	}

void
insert_expanded_filler(GtkWidget *box)
	{
	GtkWidget	*label;

	label = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(box), label, TRUE, TRUE, 0);
	}

void
gkrellm_check_button(GtkWidget *box, GtkWidget **button, gboolean active,
			gboolean expand, gint pad, gchar *s)
	{
	*button = gtk_check_button_new_with_label(s);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(*button), active);
	gtk_box_pack_start(GTK_BOX(box), *button, expand, expand, pad);
	}

void
gkrellm_spin_button(GtkWidget *box, GtkWidget **spin_button, gfloat value,
		gfloat low, gfloat high, gfloat step0, gfloat step1,
		gint digits, gint width,
		void (*cb_func)(), gpointer data, gboolean right_align, gchar *string)
	{
	GtkWidget		*hbox	= NULL,
					*label;
	GtkSpinButton	*spin;
	GtkAdjustment	*adj;

	if (string && box)
		{
	    hbox = gtk_hbox_new (FALSE, 2);
    	gtk_box_pack_start(GTK_BOX(box), hbox, TRUE, TRUE, 0);
		box = hbox;
		}
    adj = (GtkAdjustment *) gtk_adjustment_new (value,
								low, high, step0, step1, 0.0);
    *spin_button = gtk_spin_button_new(adj, 0.5, digits);
	if (width > 0)
		gtk_widget_set_usize(*spin_button, width, 0);
    spin = GTK_SPIN_BUTTON(*spin_button);
    gtk_spin_button_set_numeric(spin, TRUE);
	if (data == NULL)
		data = (gpointer) spin;
	if (cb_func)
		gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
				GTK_SIGNAL_FUNC(cb_func), data);
	if (box)
		{
		if (right_align && string)
			{
			label = gtk_label_new(string);
			gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
			gtk_box_pack_start(GTK_BOX(box), label, TRUE, TRUE, 2);
			}
		gtk_box_pack_start(GTK_BOX(box), *spin_button, FALSE, TRUE, 0);
		if (!right_align && string)
			{
			label = gtk_label_new(string);
			gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
			gtk_box_pack_start(GTK_BOX(box), label, TRUE, TRUE, 2);
			}
		}
	}

static void
create_about_tab(GtkWidget *vbox)
	{
	GtkWidget	*label;
	gchar		*buf;

	label = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);

	buf = g_strdup_printf(_("GKrellM %d.%d.%d\nGNU Krell Monitors\n\n"
				"Copyright (c) 1999-2001 by Bill Wilson\n"
				"bill@gkrellm.net\n"
				"http://gkrellm.net\n\n"
				"Released under the GNU Public License"),
				GKRELLM_VERSION_MAJOR, GKRELLM_VERSION_MINOR,
				GKRELLM_VERSION_REV);
	label = gtk_label_new(buf);
	g_free(buf);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);

	label = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);

#if defined(__FreeBSD__)
	buf = g_strdup_printf(_("FreeBSD code was contributed by:\n"
							"Hajimu UMEMOTO <ume@mahoroba.org>"));
	label = gtk_label_new(buf);
	g_free(buf);
	gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);
#endif
#if defined(__NetBSD__)
	buf = g_strdup_printf(_("NetBSD code was contributed by:\n"
							"Anthony Mallet <metall@ficus.yi.org>"));
	label = gtk_label_new(buf);
	g_free(buf);
	gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);
#endif
	}

/* ------------------General Settings---------------------------------*/
static
GtkWidget	*enable_hst_button,
			*hostname_short_button,
			*save_position_button,
			*large_info_font_button,
			*fixed_scale_spin_button,
			*update_HZ_spin_button;
#if defined(TRACK_GTK_THEMES)
GtkWidget	*track_gtk_button;
#endif

static gchar	*text_fonts[2][3] =
{
{"-adobe-helvetica-medium-r-normal-*-*-100-*-*-*-*-*,-*-*-medium-r-*-*-10-*-*-*-*-*-*-*",
"-adobe-helvetica-bold-r-normal-*-*-100-*-*-*-*-*-*,-*-*-bold-r-*-*-10-*-*-*-*-*-*-*,-*-*-medium-r-*-*-10-*-*-*-*-*-*-*",
"-adobe-helvetica-bold-o-normal-*-*-100-*-*-*-*-*,-*-*-bold-o-*-*-10-*-*-*-*-*-*-*,-*-*-medium-r-*-*-10-*-*-*-*-*-*-*"},

{"-adobe-helvetica-medium-r-normal-*-*-120-*-*-*-*-*,-*-*-medium-r-*-*-12-*-*-*-*-*-*-*",
"-adobe-helvetica-bold-r-normal-*-*-120-*-*-*-*-*-*,-*-*-bold-r-*-*-12-*-*-*-*-*-*-*,-*-*-medium-r-*-*-12-*-*-*-*-*-*-*",
"-adobe-helvetica-bold-o-normal-*-*-120-*-*-*-*-*,-*-*-bold-o-*-*-12-*-*-*-*-*-*-*,-*-*-medium-r-*-*-12-*-*-*-*-*-*-*"}
};

static void
init_info_font(gint load)
	{
	gint	i;

	if (text_font_normal)
		gdk_font_unref(text_font_normal);
	if (text_font_bold)
		gdk_font_unref(text_font_bold);
	if (text_font_italic)
		gdk_font_unref(text_font_italic);
	text_font_normal = NULL;
	text_font_bold = NULL;
	text_font_italic = NULL;	
	if (load)
		{
		i = UC.large_info_font ? 1 : 0;
		if (text_font_normal == NULL)
			text_font_normal = gdk_fontset_load(text_fonts[i][0]);
		if (text_font_bold == NULL)
			text_font_bold = gdk_fontset_load(text_fonts[i][1]);
		if (text_font_italic == NULL)
			text_font_italic = gdk_fontset_load(text_fonts[i][2]);
		}
	}

static void
cb_info_font(GtkWidget *widget, gpointer data)
	{
	InfoText	*info;
	GtkText		*tt		= NULL;
	GList		*list;
	gint		n;

	UC.large_info_font = GTK_TOGGLE_BUTTON(large_info_font_button)->active;
	init_info_font(TRUE);
	for (list = info_text_list; list; list = list->next)
		{
		info = (InfoText *) list->data;
		if (info->text == tt)
			continue;
		tt = info->text;
		gtk_text_freeze(info->text);
		n = gtk_text_get_length(info->text);
		gtk_text_backward_delete(info->text, n);
		gtk_text_thaw(info->text);
		}
	for (list = info_text_list; list; list = list->next)
		{
		info = (InfoText *) list->data;
		add_info_text(info);
		}
	}

static void
apply_general_config()
	{
	gint	n;

#if defined(TRACK_GTK_THEMES)
	UC.track_gtk_rcfiles = GTK_TOGGLE_BUTTON(track_gtk_button)->active;
#endif
	UC.enable_hostname = GTK_TOGGLE_BUTTON(enable_hst_button)->active;
	UC.hostname_short = GTK_TOGGLE_BUTTON(hostname_short_button)->active;

	UC.large_info_font = GTK_TOGGLE_BUTTON(large_info_font_button)->active;
	UC.save_position = GTK_TOGGLE_BUTTON(save_position_button)->active;
	UC.fixed_scale = gtk_spin_button_get_value_as_int(
					GTK_SPIN_BUTTON(fixed_scale_spin_button));
	n = UC.update_HZ;
	UC.update_HZ = gtk_spin_button_get_value_as_int(
					GTK_SPIN_BUTTON(update_HZ_spin_button));
	if (n != UC.update_HZ)
		start_timer(UC.update_HZ);
	}

static gchar	*general_info_text[]	=
{
N_("<b>Krells\n"),
N_("Krells are the horizontally moving indicators below each chart and\n"
"on meter style monitors.  Depending on the monitor, they show fast\n"
"response data rates, a percentage of some capacity, or something else.\n"),

N_("<b>\nCharts\n"),
N_("Charts can be configured to auto scale or to have a fixed number of\n"
	"1 to 5 grids. In either case, chart resolution is set by specifying\n"
	"a value per grid.\n"),
N_("<b>\tCPU charts - "),
N_("Data is plotted as a percentage.  In auto scale mode,\n"
	"\tresolution is 20% per grid and the krell full scale value is 100%.\n"
	"\tFixed scale grid resolution is 100 divided by the number of grids.\n"
	"\tUser time is plotted in the \"in\" color, sys time in the \"out\" color.\n"),

N_("<b>\tProc chart - "),
N_("The krell shows process forks with a full scale value\n"
	"\tof 10 forks.  The chart has a resolution of 10 forks/sec per grid in\n"
	"\tauto scale mode and 50 forks/second max on fixed scale charts.\n"
	"\tThe process load resolution per grid is best left at 1.0 for auto\n"
	"\tscaling, but can be set as high as 5 if you use fixed scale charts\n"
	"\twith only 1 or 2 grids.  Forks are plotted in the \"in\" color, load\n"
	"\tis plotted in the \"out\" color.\n"),

N_("<b>\tInet charts - "),
N_("The krell for this chart samples TCP port hits once\n"
	"\tper second with a full scale value of 5 hits.  Grid resolutions for\n"
	"\tthe minute and hour chart are not linked to the krell full scale\n"
	"\tvalue.  See Internet->Info\n"),

N_("<b>\tDisk and Net charts - "),

N_("In autoscale mode a grid resolution should\n"
	"\tbe selected that is about 1/5 the expected maximum data rate\n"
	"\tsince the krell full scale value is set to 5 times the grid resolution.\n"
	"\tFor fixed scaling, the krell full scale value is set to the selected\n"
	"\tnumber of grids times the grid resolution.  The \"in\" color is used\n"
	"\tfor disk read data and net receive bytes, while disk write data and\n"
	"\tnet transmit bytes are plotted in the \"out\" color.\n"
	"\tExample: Your ppp0 net interface is a 56k modem and data transfer\n"
	"\trates are typically less than 10000 bytes/sec.  You could select\n"
	"\tauto scaling charts and set grid resolution to 2000 bytes/sec per\n"
	"\tgrid, or you could select 2 fixed grids and set grid resolution to\n"
	"\t5000 bytes/sec per grid. In both cases the krell full scale value is\n"
	"\t10000 bytes/sec rate.\n"),

N_("<b>\nCommands\n"),
N_("\tMany monitors can be configured to launch commands.  Just enter the\n"
   "\tcommand where you see a \"command\" entry and also a comment if you\n"
   "\twant a tooltip to appear for the command.  After a command is entered,\n"
   "\tfor a monitor, a button for launching it will become visible when you\n"
   "\tmove the mouse into the panel area of the monitor.\n\n"),

N_("<b>\nMouse Button Actions:\n"),
N_("<b>\tLeft "),
N_("clicking on charts will toggle a display of some extra info.\n"),

N_("<b>\nNotes on Options\n"),
N_("Remember screen location - This is useful if you do not configure\n"
	"\tyour window manager to start GKrellM at a specific location.  This\n"
	"\toption can be overridden with the -g geometry command line option.\n"
	"Krell and LED updates per second - Set this to a lower value if you\n"
	"\trun a slower CPU or run GKrellM over a network and want to save\n"
	"\tsome bandwith.\n")
};

static void
create_general_tab(GtkWidget *tab_vbox)
	{
	GtkWidget		*tabs;
	GtkWidget		*vbox;
	GtkWidget		*hbox;
	GtkWidget		*scrolled;
	GtkWidget		*text;
	gint			i;

	tabs = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(tabs), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(tab_vbox), tabs, TRUE, TRUE, 0);

/* --Options tab */
	vbox = gkrellm_create_tab(tabs, _("Options"));
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_container_add(GTK_CONTAINER(vbox), hbox);

	gkrellm_check_button(hbox, &enable_hst_button, UC.enable_hostname,
			TRUE, 0, _("Hostname display"));

	gkrellm_check_button(hbox, &hostname_short_button, UC.hostname_short,
			TRUE, 10, _("Short hostname"));

	gkrellm_check_button(vbox, &save_position_button, UC.save_position, TRUE,0,
		_("Remember screen location at exit and move to it at next startup"));

#if defined(TRACK_GTK_THEMES)
	gkrellm_check_button(vbox, &track_gtk_button, UC.track_gtk_rcfiles, TRUE,0,
		_("Track Gtk theme changes for similarly named themes"));
#endif

	gkrellm_check_button(vbox, &large_info_font_button, UC.large_info_font,
			TRUE, 0, _("Larger font for Info/Help pages"));
	gtk_signal_connect(GTK_OBJECT(large_info_font_button), "toggled",
			cb_info_font, NULL);

	gkrellm_spin_button(vbox, &update_HZ_spin_button, (gfloat) UC.update_HZ,
			2.0, 10.0, 1.0, 1.0, 0, 50, NULL, NULL, FALSE,
			_("Krell and LED updates per second."));

	gkrellm_spin_button(vbox, &fixed_scale_spin_button, (gfloat) UC.fixed_scale,
			0.0, 5.0, 1.0, 1.0, 0, 50, NULL, NULL, FALSE,
	_("0: Autoscale charts    1-5: Number of grids for fixed scale charts"));

/* --Info tab */
	vbox = gkrellm_create_tab(tabs, _("Info"));
	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start(GTK_BOX(vbox), scrolled, TRUE, TRUE, 0);
	text = gtk_text_new(NULL, NULL);
	for (i = 0; i < sizeof(general_info_text)/sizeof(gchar *); ++i)
		gkrellm_add_info_text_string(text, _(general_info_text[i]));
	gtk_text_set_editable(GTK_TEXT(text), FALSE);
	gtk_container_add(GTK_CONTAINER(scrolled), text);
	}



/* ------------------Sizes Settings----------------------------------*/

static GtkWidget	*heights_equal_button,
					*spin_button_w,
					*spin_button_h[N_CHART_MONITORS];

static void
cb_equal_button()
	{
	gint	value, i;

	value = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spin_button_h[0]));
	if (value)
		for (i = 1; i < N_CHART_MONITORS; ++i)
			gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_button_h[i]),
					(gfloat) value);
	}

static void
cb_size_changed(GtkWidget *widget, GtkSpinButton *spin)
	{
	gint	i, value;

	if (   spin != GTK_SPIN_BUTTON(spin_button_w)
		&& GTK_TOGGLE_BUTTON(heights_equal_button)->active
	   )
		{
		value = gtk_spin_button_get_value_as_int(spin);
		for (i = 0; i < N_CHART_MONITORS; ++i)
			gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_button_h[i]),
							(gfloat) value);
		}
	rebuild_flag = TRUE;
	}

static void
apply_sizes_config()
	{
	gint	i;

	UC.chart_width =
		gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spin_button_w));
	for (i = 0; i < N_CHART_MONITORS; ++i)
		UC.chart_height[i] =
		   gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spin_button_h[i]));
	}

static void
create_sizes_tab(GtkWidget *vbox)
	{
	GtkWidget		*frame;
	GtkWidget		*hbox, *hbox1;
	GtkWidget		*vbox1;
	gint			i, heights_equal;

	heights_equal = TRUE;
	for (i = 1; i < N_CHART_MONITORS; ++i)
		if (UC.chart_height[i] != UC.chart_height[0])
			heights_equal = FALSE;

	hbox1 = gtk_hbox_new (FALSE, 5);
	gtk_container_set_border_width(GTK_CONTAINER(hbox1), 5);
	gtk_container_add(GTK_CONTAINER(vbox), hbox1);

	vbox1 = gtk_vbox_new(FALSE, 5);
	gtk_box_pack_end(GTK_BOX(hbox1), vbox1, TRUE, TRUE, 0);
	
	frame = gtk_frame_new (_("Monitor Width"));
	gtk_box_pack_start(GTK_BOX(vbox1), frame, TRUE, TRUE, 0);
	hbox = gtk_hbox_new (FALSE, 5);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 5);
	gtk_container_add(GTK_CONTAINER(frame), hbox);

	gkrellm_spin_button(hbox, &spin_button_w, (gfloat) UC.chart_width,
			(gfloat) CHART_WIDTH_MIN, (gfloat) CHART_WIDTH_MAX,
			1.0, 5.0, 0, 60, cb_size_changed, NULL, FALSE, _("All Monitors"));

	frame = gtk_frame_new (_("Chart Heights"));
	gtk_box_pack_start(GTK_BOX(hbox1), frame, TRUE, TRUE, 0);
	vbox1 = gtk_vbox_new (FALSE, 5);
	gtk_container_set_border_width(GTK_CONTAINER(vbox1), 2);
	gtk_container_add(GTK_CONTAINER(frame), vbox1);

	for (i = 0; i < N_CHART_MONITORS; ++i)
		{
		gkrellm_spin_button(vbox1, &spin_button_h[i],
			(gfloat) UC.chart_height[i], (gfloat) GK.chart_height_min,
			(gfloat) GK.chart_height_max, 1.0, 5.0, 0, 60,
			cb_size_changed, NULL, FALSE, _(chart_monitor_name[i]));
		}

	gkrellm_check_button(vbox, &heights_equal_button, heights_equal, FALSE, 0,
				_("Keep heights equal."));
	gtk_signal_connect(GTK_OBJECT(heights_equal_button),"clicked",
				(GtkSignalFunc) cb_equal_button, NULL);
	}

/* ------------------Themes Tab----------------------------------*/

#define	THEME_RELOAD	0
#define	THEME_NEXT		1
#define	THEME_PREV		2
#define	THEME_ALT_NEXT	3
#define	THEME_ALT_PREV	4

static GList		*themes_list;
static GList		*theme_position_in_list;

static GtkWidget	*theme_clist;
static GtkWidget	*theme_spin_button;
static GtkWidget	*theme_alt_label;
static GtkWidget	*default_theme_entry;
static GtkWidget	*font_button[N_FONT_OVERRIDES];
static gint			theme_width, author_width;
static gint			theme_modified;

static gchar *
get_theme_author(gchar *path)
	{
	static gchar	buf[128];
	FILE			*f;
	gchar			*s, *q, *rcfile, line[128];

	buf[0] = '\0';
	if (!path || *path == '\0')
		return buf;
	rcfile = g_strdup_printf("%s/%s", path, GKRELLMRC);
	f = fopen(rcfile, "r");
	g_free(rcfile);
	if (f == NULL)
		return buf;
	while (fgets(line, sizeof(line), f))
		{
		if (   (s = strtok(line, " :=\t\n")) == NULL
			|| strcmp(s, "author") != 0
		   )
			continue;
		s = strtok(NULL, "\n");		/* Rest of line is Author string */
		if (s)
			{
			while (   *s == ' ' || *s == '\t' || *s == '"' || *s == '='
					|| *s == ':')
				++s;
			q = strchr(s, (int) '"');
			if (q)
				*q = '\0';
			strcpy(buf, s);
			break;
			}
		}
	fclose(f);
	return buf;
	}

static void
add_themes_to_list(GList **list, gchar *theme_dir)
	{
	DIR				*dir;
	struct dirent	*dentry;
	gchar			*path;

	if ((dir = opendir(theme_dir)) == NULL)
		return;
	while ((dentry = readdir(dir)) != NULL)
		{
		if (dentry->d_name[0] != '.' && dentry->d_ino > 0)
			{
			if (basename_in_list(*list, dentry->d_name))
				{
				if (GK.debug_level & DEBUG_MISC)
					printf("Ignoring duplicate theme %s/%s\n",
						theme_dir, dentry->d_name);
				continue;
				}
			path = g_strdup_printf("%s/%s", theme_dir, dentry->d_name);
			if (isdir(path, NULL))
				*list = g_list_append(*list, path);
			else
				g_free(path);
			}
		}
	closedir(dir);
	}

static void
find_theme_position_in_list()
	{
	theme_position_in_list = string_in_list(themes_list,
			(*(GK.theme_path)) ? GK.theme_path : "Default");
	if (! theme_position_in_list)	/* Shouldn't happen */
		theme_position_in_list = themes_list;
	}

void
make_themes_list()
	{
	gchar	*theme_dir;

	free_glist_and_data(&themes_list);
	themes_list = g_list_append(themes_list, g_strdup("Default"));
	theme_dir = g_strdup_printf("%s/%s", gkrellm_homedir(),
					GKRELLM_THEMES_DIR);
	add_themes_to_list(&themes_list, theme_dir);
	g_free(theme_dir);

	add_themes_to_list(&themes_list, LOCAL_THEMES_DIR);
	add_themes_to_list(&themes_list, SYSTEM_THEMES_DIR);
	themes_list = g_list_sort(themes_list, (GCompareFunc) strcmp_basename);

	if (GK.command_line_theme)
		themes_list = g_list_prepend(themes_list,
							g_strdup(GK.command_line_theme));

	find_theme_position_in_list();
	}

static void
cb_theme_selected(GtkWidget *clist, gint row, gint column,
					GdkEventButton *bevent, gpointer data)
	{
	gchar			*theme;

	gtk_spin_button_set_value(GTK_SPIN_BUTTON(theme_spin_button), 0.0);
	if ((theme = gtk_clist_get_row_data(GTK_CLIST(clist), row)) != NULL)
		gtk_entry_set_text(GTK_ENTRY(default_theme_entry), theme);

	gkrellm_dup_string(&GK.theme_path, theme);
	find_theme_position_in_list();
	theme_modified = TRUE;
	build_gkrellm();
	}

static void
scan_for_themes(GtkWidget *clist)
	{
	GList	*list;
	gchar	*path, *name;
	gchar	*buf[3];
	gint	row, l;

	make_themes_list();
	buf[2] = NULL;
	theme_width = 10;
	author_width = 10;

	for (list = themes_list; list; list = list->next)
		{
		path = (gchar *) list->data;
		if ((name = strrchr(path, '/')) == NULL)
			name = path;
		else
			++name;
		buf[0] = name;
		if (!strcmp(name, "Default"))
			path = "";
		buf[1] = get_theme_author(path);
		row = gtk_clist_append(GTK_CLIST(clist), buf);
		gtk_clist_set_row_data_full(GTK_CLIST(clist), row,
						g_strdup(path), (GtkDestroyNotify) g_free);
		if ((l = strlen(name)) > theme_width)
			theme_width = l;
		if ((l = strlen(buf[1])) > author_width)
			author_width = l;
		}
	}

static void
cb_font_override()
	{
	gint	i;

	for (i = 0; i < N_FONT_OVERRIDES; ++i)
		if (GTK_TOGGLE_BUTTON(font_button[i])->active)
			{
			if (GK.font_override != i)
				{
				GK.font_override = i;
				theme_modified = TRUE;
				build_gkrellm();
				}
			}
	}

static void
cb_theme_alternative_changed(GtkWidget *widget, GtkSpinButton *spin)
	{
	gint	i;

	i = gtk_spin_button_get_value_as_int(spin);
	if (i > GK.theme_n_alternatives)
		{
		i = GK.theme_n_alternatives;
		gtk_spin_button_set_value(spin, (gfloat) i);
		}
	if (i != GK.theme_alternative)
		{
		GK.theme_alternative = i;
		theme_modified = TRUE;
		build_gkrellm();
		}
	}

void
set_theme_alternatives_label()
	{
	GtkSpinButton	*spin;
	gchar			buf[64];

	if (!config_window)
		return;
	spin = GTK_SPIN_BUTTON(theme_spin_button);
	gtk_spin_button_set_value(spin, (gfloat) GK.theme_alternative);
	snprintf(buf, sizeof(buf), _("%d total theme alternatives"),
					GK.theme_n_alternatives);
	gtk_label_set_text(GTK_LABEL(theme_alt_label), buf);
	}

void
save_theme_config()
	{
	FILE	*f;
	gchar	*path;

	/* Assume gkrellm -t is for testing and don't save theme config changes.
	|  Similary for GK.demo.
	*/
	if (!theme_modified || GK.command_line_theme || GK.demo)
		return;

	path = gkrellm_make_config_file_name(gkrellm_homedir(),
					GKRELLM_THEME_CONFIG);

	if (GK.debug_level & DEBUG_MISC)
		printf(_("Writting new %s: <%s>\n"), path, GK.theme_path);

	if ((f = fopen(path, "w")) != NULL)
		{
		fprintf(f, "%s\n%d\n%d\n",
				GK.theme_path, GK.theme_alternative, GK.font_override);
		fclose(f);
		}
	g_free(path);
	theme_modified = FALSE;
	}

#if defined(TRACK_GTK_THEMES)
  /* Some Gtk themes have a suffix */
static gchar *gtk_suffix[]	=
	{
	"-gtk", "_gtk", "-GTK", "_GTK"
	};
#endif

  /* Called from cb_client_event.  Triggered by a Gtk theme switch or if
  |  an app sends _GKRELLM_READ_THEME event.
  */
void
read_theme_event(gint gkrellm_read_theme)
	{
	if (gkrellm_read_theme)
		{
		g_free(GK.theme_path);
		GK.theme_path = NULL;	/* Force reread of GKRELLM_THEME_CONFIG */
		}
#if defined(TRACK_GTK_THEMES)
	else if (UC.track_gtk_rcfiles)
		/* This code currently not accessible (UC.track_gtk_rcfiles is not
		|  setable) I've heard a master theme switcher (E, Gtk, xmms, gkrellm)
		|  may be in the works which would be a better general solution.
		*/
		{
		FILE	*f;
		gchar	*gtkrc_path, *s, *g;
		gchar	buf[512];
		gint	i;
		GList	*list;

		gtkrc_path = g_strdup_printf("%s/.gtkrc", gkrellm_homedir());
		if ((f = fopen(gtkrc_path, "r")) != NULL)
			{
				/* # -- THEME AUTO-WRITTEN DO NOT EDIT		*/
			fgets(buf, sizeof(buf), f);
				/* include "/home/bill/.themes/XX/gtk/gtkrc	*/
			fgets(buf, sizeof(buf), f);
			fclose(f);
			if ((s = strstr(buf, "/gtk/gtkrc")) != NULL)
				{
				*s = '\0';
				if ((s = strrchr(buf, (int) '/')) != NULL)
					{
					++s;
					list = basename_in_list(themes_list, s);
					if (!list)
						{
						for (i = 0; i < sizeof(gtk_suffix)/sizeof(gchar *); ++i)
							if ((g = strstr(s, gtk_suffix[i])) != NULL)
								{
								*g = '\0';
								list = basename_in_list(themes_list, s);
								break;
								}
						}
					if (list)
						{
						s = (gchar *) list->data;
						gkrellm_dup_string(&GK.theme_path, strcmp(s, "Default")
								? s : "");
						theme_modified = TRUE;
						save_theme_config();
						}
					}
				}
			}
		}
#endif
	build_gkrellm();
	}

static void
cb_load_theme(gpointer data, guint action, GtkWidget *widget)
	{
	gchar			*path;
	gint			row;

	if (GK.no_config)
		return;
	if (!themes_list)
		make_themes_list();
	if (action == THEME_ALT_NEXT || action == THEME_ALT_PREV)
		{
		GK.theme_alternative += ((action == THEME_ALT_NEXT) ? 1 : -1);
		if (GK.theme_alternative > GK.theme_n_alternatives)
			{
			GK.theme_alternative = 0;
			action = THEME_NEXT;
			}
		if (GK.theme_alternative < 0)
			{
			GK.theme_alternative = 100;
			action = THEME_PREV;
			}
		theme_modified = TRUE;
		}
	else if (action != THEME_RELOAD)
		GK.theme_alternative = 0;

	if (action == THEME_NEXT || action == THEME_PREV)
		{
		if (action == THEME_NEXT)
			{
			theme_position_in_list = theme_position_in_list->next;
			if (!theme_position_in_list)
				theme_position_in_list = themes_list;
			}
		else
			{
			theme_position_in_list = theme_position_in_list->prev;
			if (!theme_position_in_list)
				theme_position_in_list = g_list_last(themes_list);
			}
		if (config_window)
			{
			row = g_list_position(themes_list, theme_position_in_list);
			if (gtk_clist_row_is_visible(GTK_CLIST(theme_clist), row)
						!= GTK_VISIBILITY_FULL)
			gtk_clist_moveto(GTK_CLIST(theme_clist), row, 0, 0.5, 0.0);
			gtk_clist_select_row(GTK_CLIST(theme_clist), row, 0);
			return;		/* cb_theme_selected will build_gkrellm()	*/
			}
		path = (gchar *) theme_position_in_list->data;
		gkrellm_dup_string(&GK.theme_path, strcmp(path, "Default") ? path :"");
		theme_modified = TRUE;
		}
	build_gkrellm();
	}

static gchar	*themes_titles[] = {N_("Theme"), N_("Author") };

static void
create_theme_tab(GtkWidget *vbox)
	{
	GtkWidget		*hbox;
	GtkWidget		*label;
	GtkWidget		*scrolled;
	GSList			*group;
	gchar			*s;
	gint            i;

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
	label = gtk_label_new(_("Theme:"));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 3);

	default_theme_entry = gtk_entry_new_with_max_length(128);
	if (GK.theme_path)
		gtk_entry_set_text(GTK_ENTRY(default_theme_entry), GK.theme_path);
	gtk_box_pack_start(GTK_BOX(hbox), default_theme_entry, TRUE, TRUE,0);

	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start(GTK_BOX(vbox), scrolled, TRUE, TRUE, 0);
    for (i = 0; i < 2; i++)
		themes_titles[i]=_(themes_titles[i]);
	theme_clist = gtk_clist_new_with_titles(2, themes_titles);
	gtk_signal_connect (GTK_OBJECT(theme_clist), "select_row",
			(GtkSignalFunc) cb_theme_selected, NULL);

	gtk_clist_set_shadow_type(GTK_CLIST(theme_clist), GTK_SHADOW_OUT);
	gtk_clist_column_titles_passive(GTK_CLIST(theme_clist));
	gtk_container_add (GTK_CONTAINER (scrolled), theme_clist);

	scan_for_themes(theme_clist);

	/* How can I get a string width for the default gtk font? */
	gtk_clist_set_column_width(GTK_CLIST(theme_clist), 0, theme_width * 7 + 8);
	gtk_clist_set_column_width(GTK_CLIST(theme_clist), 1, author_width * 7);

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	gkrellm_spin_button(hbox, &theme_spin_button, (gfloat)GK.theme_alternative,
			0.0, 100.0, 1.0, 5.0, 0, 60,
			cb_theme_alternative_changed, NULL, FALSE, NULL);
	theme_alt_label = gtk_label_new("");
	gtk_box_pack_start (GTK_BOX (hbox), theme_alt_label, TRUE, TRUE, 4);
	gtk_misc_set_alignment(GTK_MISC(theme_alt_label), 0, 0.5);
	set_theme_alternatives_label();

	label = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 0);
	label = gtk_label_new(_("Fonts:"));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
	gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
	font_button[0] = gtk_radio_button_new_with_label(NULL, _("Theme"));
	gtk_box_pack_start(GTK_BOX(hbox), font_button[0], FALSE, FALSE, 2);
	group = gtk_radio_button_group(GTK_RADIO_BUTTON(font_button[0]));

	font_button[1] = gtk_radio_button_new_with_label(group, _("Alt1"));
	gtk_box_pack_start(GTK_BOX(hbox), font_button[1], FALSE, FALSE, 2);
	group = gtk_radio_button_group(GTK_RADIO_BUTTON(font_button[1]));

	font_button[2] = gtk_radio_button_new_with_label(group, _("Alt2"));
	gtk_box_pack_start(GTK_BOX(hbox), font_button[2], FALSE, FALSE, 2);
	gtk_toggle_button_set_active(
			GTK_TOGGLE_BUTTON(font_button[GK.font_override]), TRUE);

	for (i = 0; i < N_FONT_OVERRIDES; ++i)
		gtk_signal_connect_object(GTK_OBJECT(font_button[i]), "toggled",
				GTK_SIGNAL_FUNC(cb_font_override), NULL);

	s = g_strdup_printf(_("Untar your theme tar files in %s/%s"),
						gkrellm_homedir(), GKRELLM_THEMES_DIR);
	label = gtk_label_new(s);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);

	label = gtk_label_new(
			_("Download themes from the GKrellM theme site at www.muhri.net"));
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
	}


/* -------------------------------------------------------------------*/
static GtkCTreeNode	*node_plugins,
					*node_builtins;
static GtkWidget	*notebook_config;
static GtkWidget	*config_ctree;
static gint			config_page;

static gint			expand_builtins,
					expand_plugins;


static void
close_config()
	{
	InfoText	*info;
	GList		*list;

	save_plugins_enable_list();
	expand_builtins = GTK_CTREE_ROW(node_builtins)->expanded;
	expand_plugins = GTK_CTREE_ROW(node_plugins)->expanded;
	gtk_widget_destroy(config_window);
	init_info_font(FALSE);
	config_window = NULL;
	for (list = info_text_list; list; list = list->next)
		{
		info = (InfoText *) list->data;
		g_free(info->string);
		}
	free_glist_and_data(&info_text_list);
	}


static void
apply_config()
	{
	GList	*list;
	Monitor	*mon;

	apply_general_config();

	for (list = gkrellm_monitor_list; list; list = list->next)
		{
		mon = (Monitor *) list->data;
		if (mon->apply_config && mon->private->enabled)
			(*(mon->apply_config))();
		}

	apply_sizes_config();

	save_theme_config();
	save_user_config();

	gkrellm_pack_side_frames();
	save_plugins_enable_list();

	if (rebuild_flag)
		build_gkrellm();
	}

static void
OK_config()
	{
	apply_config();
	close_config();
	}

  /* Obsolete */
GtkWidget *
create_tab(GtkWidget *tabs, char *name)
	{
	return gkrellm_create_tab(tabs, name);
	}

GtkWidget *
gkrellm_create_tab(GtkWidget *tabs, char *name)
	{
	GtkWidget	*label;
	GtkWidget	*vbox;

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER(vbox), 3);

	label = gtk_label_new(name);
	gtk_notebook_append_page(GTK_NOTEBOOK(tabs), vbox, label);

	return vbox;
	}

static GtkWidget *
create_config_page(GtkNotebook *notebook, gchar *text, GtkCTree *ctree,
		GtkCTreeNode *node_parent, GtkCTreeNode **node_result)
	{
	GtkWidget		*vbox;
	GtkCTreeNode	*node;
	gchar			*title[1] = {text};

	vbox = gtk_vbox_new(FALSE, 0);
	node = gtk_ctree_insert_node(ctree, node_parent, NULL, title, 0,
			NULL, NULL, NULL, NULL, FALSE, FALSE);
	gtk_ctree_node_set_row_data(ctree, node, GINT_TO_POINTER(config_page++));
	gtk_notebook_append_page(notebook, vbox, NULL);
	if (node_result)
		*node_result = node;
	return vbox;
	}

void
add_plugin_config_page(Monitor *mon)
	{
	GtkWidget		*vbox;
	GtkCTreeNode	*node;

	if (config_window && mon->create_config)
		{
		vbox = create_config_page(GTK_NOTEBOOK(notebook_config), mon->name,
			GTK_CTREE(config_ctree), node_plugins, &node);
		(*(mon->create_config))(vbox);

		gtk_widget_show_all(vbox);
		mon->private->notebook_config_widget = notebook_config;
		mon->private->notebook_config_page =
				g_list_length(GTK_NOTEBOOK(notebook_config)->children) - 1;
		mon->private->ctree_config_node = node;
		}
	else
		mon->private->notebook_config_widget = NULL;
	}

void
remove_plugin_config_page(Monitor *mon)
	{
	GtkCTreeNode	*node;
	GList			*list;
	Monitor			*tmon;
	gint			page;

	if (mon->private->notebook_config_widget)
		{
		node = mon->private->ctree_config_node;
		while (node)
			{
			page = GPOINTER_TO_INT(gtk_ctree_node_get_row_data(
						GTK_CTREE(config_ctree), node));
			gtk_ctree_node_set_row_data(GTK_CTREE(config_ctree), node,
						GINT_TO_POINTER(page - 1));
			node = GTK_CTREE_ROW(node)->sibling;
			}
		gtk_ctree_remove_node(GTK_CTREE(config_ctree),
				mon->private->ctree_config_node);
		gtk_notebook_remove_page(
				GTK_NOTEBOOK(mon->private->notebook_config_widget),
				mon->private->notebook_config_page);
		--config_page;

		/* When a config_page is removed, any greater plugin config_page must
		|  be decremented
		*/
		for (list = gkrellm_monitor_list; list; list = list->next)
			{
			tmon = (Monitor *) list->data;
			if (mon->private->notebook_config_page
						< tmon->private->notebook_config_page)
				tmon->private->notebook_config_page -= 1;
			}
		}
	mon->private->notebook_config_widget = NULL;
	}

static void
cb_select_monitor_config(GtkWidget *ctree, GtkCTreeNode *node)
	{
	gint	row;

	if (!GTK_CLIST(ctree)->selection)
		return;
	row = GPOINTER_TO_INT(gtk_ctree_node_get_row_data(GTK_CTREE(ctree), node));
	gtk_notebook_set_page(GTK_NOTEBOOK(notebook_config), row);
	}


void
create_config_window()
	{
	GtkWidget		*vbox,
					*main_vbox,
					*config_hbox,
					*hbox;
	GtkWidget		*ctree;
	GtkWidget		*scrolled;
	GtkWidget		*button;
	GtkCTreeNode	*node;
	GList			*list;
	Monitor			*mon;
	gchar			*title[1] = { _("Monitors") };
	gchar			*config_name, *window_title;
	
	if (config_window)
		{
		gdk_window_raise(config_window->window);
		return;
		}

	init_info_font(TRUE);
	rebuild_flag = FALSE;
	config_page = 0;

	config_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_signal_connect(GTK_OBJECT(config_window), "delete_event",
			(GtkSignalFunc) close_config, NULL);
	gtk_window_set_policy(GTK_WINDOW(config_window), FALSE, FALSE, TRUE);

	config_name = gkrellm_make_config_file_name(NULL, "GKrellM");
	window_title = g_strdup_printf("%s %s", config_name, _("Configuration"));
	gtk_window_set_title(GTK_WINDOW(config_window), window_title);
	g_free(config_name);
	g_free(window_title);

	gtk_window_set_wmclass(GTK_WINDOW(config_window),
					"Gkrellm_conf", "Gkrellm");
	gtk_container_border_width(GTK_CONTAINER(config_window), 2);
	
	main_vbox = gtk_vbox_new(FALSE, 4);
	gtk_container_add(GTK_CONTAINER(config_window), main_vbox);

	config_hbox = gtk_hbox_new(FALSE, 4);
	gtk_box_pack_start(GTK_BOX(main_vbox), config_hbox, TRUE, TRUE, 0);

	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start(GTK_BOX(config_hbox), scrolled, FALSE, FALSE, 0);

	ctree = gtk_ctree_new_with_titles(1, 0, title);
	config_ctree = ctree;
	gtk_ctree_set_indent(GTK_CTREE(ctree), 16);	
	gtk_clist_column_titles_passive(GTK_CLIST(ctree));
	gtk_widget_set_usize(ctree, 150, 0);
	gtk_container_add(GTK_CONTAINER(scrolled), ctree);
/*	gtk_box_pack_start(GTK_BOX(config_hbox), ctree, FALSE, FALSE, 0); */
	gtk_signal_connect(GTK_OBJECT(ctree), "tree_select_row",
			(GtkSignalFunc) cb_select_monitor_config, NULL);

	notebook_config = gtk_notebook_new();
	gtk_notebook_set_show_tabs(GTK_NOTEBOOK(notebook_config), FALSE);
/*	gtk_notebook_set_show_border(GTK_NOTEBOOK(notebook_config), FALSE); */
	gtk_box_pack_start(GTK_BOX(config_hbox), notebook_config, TRUE, TRUE, 0);
/*	gtk_object_set_user_data(GTK_OBJECT(ctree), notebook_config);	*/

	vbox = create_config_page(GTK_NOTEBOOK(notebook_config), _("General"),
			GTK_CTREE(ctree), NULL, &node);
	create_general_tab(vbox);
	gtk_ctree_select(GTK_CTREE(ctree), node);

	vbox = create_config_page(GTK_NOTEBOOK(notebook_config), _("Builtins"),
			GTK_CTREE(ctree), NULL, &node_builtins);
	for (list = gkrellm_monitor_list; list; list = list->next)
		{
		mon = (Monitor *) list->data;
		if (MONITOR_ID(mon) == MON_PLUGIN || ! mon->create_config)
			continue;
		vbox = create_config_page(GTK_NOTEBOOK(notebook_config), mon->name,
			GTK_CTREE(ctree), node_builtins, NULL);
		(*(mon->create_config))(vbox);
		}
	if (expand_builtins)
		gtk_ctree_expand(GTK_CTREE(ctree), node_builtins);

	vbox = create_config_page(GTK_NOTEBOOK(notebook_config), _("Plugins"),
			GTK_CTREE(ctree), NULL, &node_plugins);
	create_plugin_config(vbox);

	vbox = create_config_page(GTK_NOTEBOOK(notebook_config), _("Sizes"),
			GTK_CTREE(ctree), NULL, NULL);
	create_sizes_tab(vbox);

	vbox = create_config_page(GTK_NOTEBOOK(notebook_config), _("Themes"),
			GTK_CTREE(ctree), NULL, NULL);
	create_theme_tab(vbox);

	vbox = create_config_page(GTK_NOTEBOOK(notebook_config), _("About"),
			GTK_CTREE(ctree), NULL, NULL);
	create_about_tab(vbox);

	/* Add plugin notebook pages last since they may need special add/remove
	|  actions if system wide plugins are enabled/disabled.
	*/
	for (list = gkrellm_monitor_list; list; list = list->next)
		{
		mon = (Monitor *) list->data;
		if (   ! mon->create_config
			|| ! mon->private->enabled
			|| MONITOR_ID(mon) != MON_PLUGIN
		   )
			continue;
		add_plugin_config_page(mon);
		}
	if (expand_plugins)
		gtk_ctree_expand(GTK_CTREE(ctree), node_plugins);

	hbox = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(main_vbox), hbox, FALSE, FALSE, 0);

	button = gtk_button_new_with_label(_("OK"));
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
				(GtkSignalFunc) OK_config, NULL);
	gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 10);

	button = gtk_button_new_with_label(_("Apply"));
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
				(GtkSignalFunc) apply_config, NULL);
	gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 10);

	button = gtk_button_new_with_label(_("Close"));
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
				(GtkSignalFunc) close_config, NULL);
	gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 10);

	gtk_widget_show_all(config_window);
	}

static void
cb_debug(gpointer data, guint action, GtkWidget *widget)
	{
	gchar	*new;

	if (GK.debug_level & action)
		{
		GK.debug_level &= ~action;
		new = "OFF";
		}
	else
		{
		GK.debug_level |= action;
		new = "ON";
		}
	if (action == DEBUG_IMAGES)
		printf(_("==> Turning _%s_ theme images debugging.\n"), new);
	if (action == DEBUG_GKRELLMRC)
		printf(_("==> Turning _%s_ gkrellmrc debugging.\n"), new);
	if (action == DEBUG_STYLES)
		printf(_("==> Turning _%s_ style debugging.\n"), new);
	}


static GtkItemFactoryEntry	no_config_items[] =
	{
{"/-",				   NULL,	NULL,					0, "<Separator>"},
{N_("/Quit"),		   NULL,	gtk_main_quit,			0, "<Item>"},
{"/-",				   NULL,	NULL,					0, "<Separator>"},
	};

static GtkItemFactoryEntry	menu_items[] =
	{
{"/-",				   NULL,	NULL,					0, "<Separator>"},
{N_("/Configuration"), "F1",	create_config_window, 	0, "<Item>"},
{N_("/Theme/Prev"),		"p",	cb_load_theme,		THEME_ALT_PREV,	"<Item>"},
{N_("/Theme/Next"),		"n",	cb_load_theme,		THEME_ALT_NEXT,	"<Item>"},
{"/-",				   NULL,	NULL,					0, "<Separator>"},
{N_("/Quit"),		   NULL,	gtk_main_quit,			0, "<Item>"},
{"/-",				   NULL,	NULL,					0, "<Separator>"},
	};

static GtkItemFactoryEntry	debug_items[] =
  {
	{"/-",				    NULL,	NULL,					0, "<Separator>"},
	{N_("/Theme prev"), 	"u",	cb_load_theme,	THEME_PREV,	"<Item>"},
	{N_("/Theme next"),		"d",	cb_load_theme,	THEME_NEXT,	"<Item>"},
	{N_("/Reload Theme"),	"F5",	cb_load_theme,	THEME_RELOAD, "<Item>"},
	{N_("/Debug Images"),	"F6",	cb_debug,	DEBUG_IMAGES, 	"<Item>"},
	{N_("/Debug gkrellmrc"),"F7",	cb_debug,	DEBUG_GKRELLMRC,"<Item>"},
	{N_("/Debug styles"),	"F8",	cb_debug,	DEBUG_STYLES,	"<Item>"},
  };

static GtkItemFactory	*debug_factory;


GtkItemFactory *
create_item_factory_popup()
	{
	GtkWidget		*top_win;
	GtkItemFactory	*item_factory;
	GtkAccelGroup	*accel_group;
	gint			n,i;

	top_win = gkrellm_get_top_window();
	accel_group = gtk_accel_group_new ();
	gtk_window_add_accel_group(GTK_WINDOW(top_win), accel_group);

	n = sizeof(debug_items) / sizeof (GtkItemFactoryEntry);
	debug_factory = gtk_item_factory_new(GTK_TYPE_MENU, "<Main>", accel_group);
	gtk_item_factory_create_items(debug_factory, n, debug_items, NULL);

	if (GK.no_config)
		{
		n = sizeof(no_config_items) / sizeof (GtkItemFactoryEntry);
		item_factory = gtk_item_factory_new(GTK_TYPE_MENU, "<Main>",
				accel_group);
		for(i = 0; i < n; i++)
			no_config_items[i].path = _(no_config_items[i].path);
		gtk_item_factory_create_items(item_factory, n, no_config_items, NULL);
		}
	else
		{
		n = sizeof(menu_items) / sizeof (GtkItemFactoryEntry);
		item_factory = gtk_item_factory_new(GTK_TYPE_MENU, "<Main>",
				accel_group);
		for(i = 0; i < n; i++)
			menu_items[i].path = _(menu_items[i].path);
		gtk_item_factory_create_items(item_factory, n, menu_items, NULL);
		}
	return item_factory;
	}

