/*  $Id$
    BurgerSpaceEngine.h

    burgerspace - A hamburger-smashing video game.
    Copyright (C) 2001 Pierre Sarrazin <sarrazip@iname.com>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
    02111-1307, USA.
*/

#ifndef H_BurgerSpaceEngine
#define H_BurgerSpaceEngine

#include "IngredientGroup.h"
#include "IngredientSprite.h"

#include "GameEngine.h"
#include "Sprite.h"

#include <string>

using namespace std;

typedef char **XPM;


class BurgerSpaceEngine : public GameEngine
/*  A hamburger-smashing video game.
*/
{
public:

    typedef PixmapArray::Pixmap Pixmap;


    enum { RIGHT, UP, LEFT, DOWN };
    /*	Possible directions of movement, in trigonometric order.
	This enumeration is garanteed to start at zero.
	The order of the constants is guarateed, so they are 0, 1, 2, 3.
    */

    BurgerSpaceEngine(int initLevelNumber) throw(int);
    /*  See base class.

	The game will start at level 'initLevelNumber'.
	This must be a positive integer.

	Throws an integer code if an error occurs.
    */

    virtual ~BurgerSpaceEngine();
    /*  Nothing interesting.
    */

    virtual void processKey(SDLKey keysym, bool pressed);
    /*  Inherited.
    */

    virtual bool tick();
    /*  Inherited.
    */

protected:

    // Inherited:
    bool animatePlayer();
    void animateAutomaticCharacters();
    void detectCollisions();
    void restoreBackground();
    void drawSprites();

private:

    ///////////////////////////////////////////////////////////////////////////
    //
    //  LOCAL TYPES, CLASSES AND CONSTANTS
    //
    //


    enum Key
    {
	NOKEY = 0,

	// The next four constants are guaranteed to have the values 1 to 4:
	UP_ARROW,
	DOWN_ARROW,
	LEFT_ARROW,
	RIGHT_ARROW,

	LETTER_P,
	ENTER,
	SPACE,
	CTRL,
	ALT,
	ESC,

	NUM_SUPPORTED_KEYS
    };

    class Level
    {
    public:
	Couple sizeInTiles;
	Couple sizeInPixels;
	Couple positionInPixels;

	Level();
	~Level();

	void init(int no, int nCols, int nRows, Couple posInPixels);
	void setLevelNo(int no);
	int  getLevelNo() const;
	void setTileMatrixEntry(int colNo, int rowNo,
				XPM xpm, Pixmap pixmap);
	bool *getDirtMatrixRow(int rowNo);
	Pixmap *getTileMatrixRow(int rowNo);
	XPM getXPMAtPixel(Couple pos) const;
	void takeSpriteDirt(const Sprite &s);

    private:
	int levelNo;
	XPM *xpmMatrix;   // vector of XPM pointers
	Pixmap *tileMatrix;  // vector of X11 pixmaps
	bool *dirtMatrix;    // true means dirty
    };

    class IngInit
    {
    public:
	enum IngType
	{
	    BOTTOM_BUN, MEAT, LETTUCE, RED_STUFF, YELLOW_STUFF, TOP_BUN
	};

	int xInitTile, yInitTile, yTargetTile, rank;
	IngType type;
    };

    
    struct IntPair
    {
	int first, second;
    };

    
    struct IntQuad
    {
	int first, second, third, fourth;
    };


    ///////////////////////////////////////////////////////////////////////////
    //
    //  DATA MEMBERS
    //
    //

    static const char
	*levelDescriptor1[],
	*levelDescriptor2[],
	*levelDescriptor3[],
	*levelDescriptor4[],
	*levelDescriptor5[],
	*levelDescriptor6[];

    static const char **levelDescriptorTable[];

    static const IngInit
	tableIngredientsLevel1[],
	tableIngredientsLevel2[],
	tableIngredientsLevel3[],
	tableIngredientsLevel4[],
	tableIngredientsLevel5[],
	tableIngredientsLevel6[];
    
    static const IngInit *tableOfTablesOfIngredientsLevel[];

    static const IntQuad enemyStartingHeights[];
    static const IntPair playerStartingPos[];


    int initLevelNo;
    int cumulLevelNo;

    bool paused;
    unsigned long tickCount;

    Pixmap tilePixmaps[5];


    /*  PLAYER:
    */
    Couple initPlayerPos;  // initial player position in a level
    PixmapArray playerPA;
    Sprite *playerSprite;
    int lastPlayerDirection;

    PixmapArray pepperPA;
    SpriteList  pepperSprites;


    /*  ENEMIES:
    */
    unsigned long timeForNewEnemy;
    		// tick count at which a new enemy must be created;
		// 0 means none

    PixmapArray eggPA;
    SpriteList  eggSprites;
    PixmapArray hotdogPA;
    SpriteList  hotdogSprites;
    PixmapArray picklePA;
    SpriteList  pickleSprites;


    /*  INGREDIENTS:
    */
    PixmapArray topBunPA;
    PixmapArray lettucePA;
    PixmapArray meatPA;
    PixmapArray redStuffPA;
    PixmapArray yellowStuffPA;
    PixmapArray bottomBunPA;
    IngredientSpriteList ingredientSprites;  // owns the contained objects
    IngredientGroupList  ingredientGroups;


    /*  TREATS (icecream, etc, that the player eats to get a pepper):
    */
    PixmapArray treatPA;
    SpriteList treatSprites;
    int timeForTreat;


    /*  DIGITS:
    */
    PixmapArray digitPA;
    SpriteList scoreSprites;


    /*  KEYBOARD COMMANDS:
    */
    bool escapePressed;
    bool leftArrowPressed;
    bool rightArrowPressed;
    bool upArrowPressed;
    bool downArrowPressed;
    bool spacePressed, spacePressedBefore;
    bool ctrlPressed, ctrlPressedBefore;
    bool letterPPressed, letterPPressedBefore;

    int    numHamburgersToDo;
    long   thePeakScore;  // player's best score yet in this game

    long   theScore;      // player's score in points
    bool   updateScore;   // indicates if player's score needs to be rewritten
    Couple scoreAreaPos;
    Couple scoreAreaSize;

    bool   celebrationMode;  // used when player has just won the level

    int    numLives;  // number of player lives left
    bool   updateNumLives;
    Couple numLivesAreaPos;
    Couple numLivesAreaSize;

    int    numAvailablePeppers;  // number of pepper shots available to player
    bool   updateNumAvailablePeppers;
    Couple numAvailablePeppersAreaPos;
    Couple numAvailablePeppersAreaSize;

    bool   updateLevelNo;
    Couple levelNoAreaPos;
    Couple levelNoAreaSize;

    Couple messagePos;  // position of text messages to the user


    Level theCurrentLevel;


    ///////////////////////////////////////////////////////////////////////////
    //
    //  IMPLEMENTATION FUNCTIONS
    //
    //
    void putSprite(const Sprite &s);
    void showInstructions();
    void initGameParameters();
    void initNextLevel(int levelNo = 0) throw(int);
    void startNewLife();
    void animateTemporarySprites(SpriteList &slist) const;
    void makePlayerWin();
    void makePlayerDie();
    void releaseAllCarriedEnemies();
    void detectEnemyCollisions(SpriteList &slist);
    void detectCollisionBetweenIngredientGroupAndEnemyList(
	    const IngredientGroup &aGroup, SpriteList &enemies);
    bool ingredientGroupCollidesWithSprite(
			const Couple groupPos, const Couple groupSize,
			const Sprite &s) const;
    void loadPixmaps() throw(PixmapLoadError);
    void loadLevel(int levelNo) throw(string);
    void displayErrorMessage(const string &msg) throw();
    int  initializeGraphicsContexts() throw();
    void initializeSprites() throw(PixmapLoadError);
    void initializeMisc() throw(string);
    //void readKeyboardCommands();
    void killSpritesInList(SpriteList &sl);
    IngredientGroup *findIngredientGroupRightBelow(
					const IngredientGroup &upperGroup);
    bool isIngredientSpriteOnFloor(const IngredientSprite &s) const;
    bool spriteTouchesIngredientGroup(
			    const Sprite &s, const IngredientGroup &g) const;
    size_t carryEnemies(IngredientGroup &g);
    size_t carryEnemiesInList(IngredientGroup &g, SpriteList &slist);
    size_t releaseCarriedEnemies(IngredientGroup &g);
    void createScoreSprites(long n, Couple center);
    void moveEnemyList(SpriteList &slist, int speedFactor, long score);
    Couple getDistanceToPerfectPos(const Sprite &s) const;
    Couple determineAllowedDirections(const Sprite &s,
				    int speedFactor, int tolerance,
				    bool allowedDirections[4]) const;
    Couple attemptMove(const Sprite &s, bool attemptLeft, bool attemptRight,
			bool attemptUp, bool attemptDown,
			int speedFactor) const;
    bool positionAllowsLeftMovement(Couple pos, Couple size) const;
    bool positionAllowsRightMovement(Couple pos, Couple size) const;
    bool spriteBottomCenterIsOnLadder(const Sprite &s) const;
    bool spriteBottomCenterIsOverLadder(const Sprite &s) const;
    void drawComplexEnemySprites(const SpriteList &slist, int oscBit);
    void addToScore(long n);
    void addToNumLives(int n);
    void initTimeForTreat();
    void displayPauseMessage(bool display);
    void displayStartMessage(bool display);


    /*	Forbidden operations:
    */
    BurgerSpaceEngine(const BurgerSpaceEngine &);
    BurgerSpaceEngine &operator = (const BurgerSpaceEngine &);
};


#endif  /* H_BurgerSpaceEngine */
