/* ==========================================================================
 * mod_auth_bsd - Apache module for BSD Authentication
 * --------------------------------------------------------------------------
 * Copyright (c) 2003, 2005, 2006  William Ahern
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to permit
 * persons to whom the Software is furnished to do so, subject to the
 * following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN
 * NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
 * USE OR OTHER DEALINGS IN THE SOFTWARE.
 * ==========================================================================
 */

#include "httpd.h"
#include "http_config.h"
#include "http_core.h"
#include "http_log.h"
#include "http_main.h"
#include "http_protocol.h"
#include "ap_md5.h"
#include "ap_sha1.h"

#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>

#include <limits.h>	/* NGROUPS_MAX */

#include <unistd.h>	/* access */

#include <sys/param.h>	/* MAXPATHLEN */

#include <string.h>	/* strerror, strlen, strcasecmp */
#include <errno.h>

#include <assert.h>


#define STRINGIFY_(x) #x
#define STRINGIFY(x)  STRINGIFY_(x)



static int trace_callback(void *data, const char *key, const char *val) {
	request_rec *r = (request_rec *)data;
	ap_rprintf(r, "Header Field %s' == %s'\n", key, val);
	return TRUE;
} /* trace_callback */


static const struct auth_config {
	int auth_enabled;
	int keep_pass;
	int require_ssl;
	int strict_require;
	char *auth_service;
	char *auth_style;
	char *auth_type;
} auth_config_initializer = {
	.auth_enabled	= 0,
	.keep_pass	= 0,
	.require_ssl	= 1,
	.strict_require	= 0,
	.auth_service	= NULL,
	.auth_style	= NULL,
	.auth_type	= NULL,
};


/*
 * Apache module parameter declaration
 */
module MODULE_VAR_EXPORT bsd_auth_module;


#include "hmac.c"	/* include it directly to keep everything private */
#include "authd.c"


/*
 * pool *p:	apache pool allocator
 * char *dn:	directory name
 */
static void *create_dir_config(pool *p, char *dn) {
	struct auth_config *cfg;

	cfg	= (struct auth_config *)ap_pcalloc(p, sizeof(*cfg));

	assert(cfg);

	*cfg	= auth_config_initializer;

	return cfg;
} /* create_dir_config */


#ifndef DEV_RANDOM
#error "DEV_RANDOM must be defined as a path to a randomness device"
#endif


static int cookie_fd	= -1;
static unsigned char cookie_secret[HMAC_BLOCK_SIZE];


static void module_init(server_rec *s, pool *p) {
	/* We're called twice (and unloaded inbetween). Only initialize on
	 * the second call. Checking if init(1) is our parent is a hack. 
	 * mod_ssl sets a context object with a counter using ap_ctx_set()
	 * that survives the restart.
	 */
	if (1 == getppid()) {
		authd_init(s);

		assert(-1 != (cookie_fd = open(STRINGIFY(DEV_RANDOM),O_RDONLY)));
	}

	return /* void */;
} /* module_init() */


static void child_init(server_rec *s, pool *p) {
	auth_child_init(s);

	assert((int)sizeof cookie_secret == read(cookie_fd,cookie_secret,sizeof(cookie_secret)));

	(void)close(cookie_fd);
	cookie_fd	= -1;

	return; /* void */
} /* child_init */


static int authenticate(request_rec *r) {
	struct auth_config *cfg			= ap_get_module_config(r->per_dir_config,&bsd_auth_module);
	const char *auth_type			= NULL;
	char user[256]				= { 0 };
	const char *pass			= NULL;
	unsigned char digest[HMAC_DIGEST_SIZE]	= { 0 };
	char mac[sizeof(digest)*2+1]		= { 0 };
	char auth[256]			= { 'B', 'a', 's', 'i', 'c',' ', '\0' };
	char log[256]			= { 0 };
	int i, ln, ret;
	FILE *p;

	/*ap_log_rerror(APLOG_MARK,APLOG_NOTICE|APLOG_NOERRNO,r,"[AuthBSD] In authenticate!");*/

	auth_type	= ap_auth_type(r);

	if (auth_type == NULL || strcasecmp(auth_type,"Basic") != 0) {
		if (cfg && cfg->auth_enabled) {
			ap_log_rerror(APLOG_MARK,APLOG_NOTICE|APLOG_NOERRNO,r,"[AuthBSD] Only Basic authentication supported");

			ret	= HTTP_INTERNAL_SERVER_ERROR;
		} else
			ret	= DECLINED;

		goto finish;
	}


	ret	= ap_get_basic_auth_pw(r,&pass);

	if (ret != OK)
		return ret;

	if (r->connection->user == NULL || *r->connection->user == '\0') {
		ap_note_basic_auth_failure(r);
		ap_log_reason("[AuthBSD] No username provided", r->uri, r);

		ret	= HTTP_UNAUTHORIZED;

		goto finish;
	}


	assert(cfg);

	if (!cfg->auth_enabled) {
		/*ap_log_rerror(APLOG_MARK,APLOG_DEBUG|APLOG_NOERRNO,r,"[AuthBSD] Not enabled for request");*/

		ret	= DECLINED;

		goto finish;
	}


	/* check for prior authorization (from internal redirect) */
	if (cfg->keep_pass == false) {
		hmac(digest,cookie_secret,sizeof(cookie_secret),r->connection->user,strlen(r->connection->user));

		for (i = 0; i < sizeof(digest); i++)
			(void)sprintf(&mac[i*2],"%.2x",digest[i]);

		if (strcmp(mac,pass) == 0) {
			ret	= OK;
			goto finish;
		}
	}


	/* check for SSL */
	if (cfg->require_ssl) {
		#ifdef EAPI
		if (ap_ctx_get(r->connection->client->ctx,"ssl") == NULL) {
			ap_note_basic_auth_failure(r);
			ap_log_reason("[AuthBSD] Will not authenticate without SSL connection", r->uri, r);

			ret	= HTTP_INTERNAL_SERVER_ERROR;

			goto finish;
		}
		#else
		ap_note_basic_auth_failure(r);
		ap_log_reason("[AuthBSD] Cannot determine if SSL is in use", r->uri, r);

		ret	= HTTP_INTERNAL_SERVER_ERROR;

		goto finish;
		#endif
	}


	if (cfg->auth_service && strcmp(cfg->auth_service,"login") != 0) {
		ap_note_basic_auth_failure(r);
		(void)snprintf(log,sizeof(log),"[AuthBSD] Unsupported service protocol: %s",cfg->auth_service);
		ap_log_reason(log, r->uri, r);

		ret	= HTTP_UNAUTHORIZED;

		goto finish;
	}


	switch (auth_child_userokay(r,r->connection->user,cfg->auth_style,cfg->auth_type,pass)) {
	case 1:
		ret	= OK;

		break;
	case 0:
		ap_note_basic_auth_failure(r);
		(void)snprintf(log,sizeof(log),"[AuthBSD] Authentication failed for user: %s",r->connection->user);
		ap_log_reason(log, r->uri, r);

		ret	= HTTP_UNAUTHORIZED;

		break;
	default:
		/* FALL THROUGH */
	case -1:
		ap_note_basic_auth_failure(r);
		(void)snprintf(log,sizeof(log),"[AuthBSD] Auth daemon communication failure");
		ap_log_reason(log, r->uri, r);

		ret	= HTTP_INTERNAL_SERVER_ERROR;

		break;
	}


	if (cfg->keep_pass == false) {
		hmac(digest,cookie_secret,sizeof(cookie_secret),r->connection->user,strlen(r->connection->user));

		for (i = 0; i < sizeof(digest); i++)
			(void)sprintf(&mac[i*2],"%.2x",digest[i]);

		ln	= snprintf(user,sizeof(user),"%s:%s",r->connection->user,mac);

		if (ln >= sizeof(user)) {
			ap_note_basic_auth_failure(r);
			ap_log_reason("[AuthBSD] Could not erase password", r->uri, r);

			ret	= HTTP_INTERNAL_SERVER_ERROR;

			goto finish;
		}

		/* XXX: "Basic " + auth */
		if (ap_base64encode_len(ln) >= (sizeof(auth) - strlen(auth))) {
			ap_note_basic_auth_failure(r);
			ap_log_reason("[AuthBSD] Could not erase password", r->uri, r);

			ret	= HTTP_INTERNAL_SERVER_ERROR;

			goto finish;
		}

		ap_base64encode(auth + strlen(auth),user,ln); /* ln comes from above */

		/* Password is stored in this MIME header */
		ap_table_set(r->headers_in,"Authorization",auth);
	}

finish:
	/*ap_table_do(trace_callback,r,r->headers_in,NULL);*/

	return ret;
} /* authenticate */


/*
 * Took cue from mod_auth_pam, but mod_auth_db and mod_auth_dbm have almost
 * the exact same logic. However, all those seem to return when they confirm
 * the first required directive. Can a request have multiple directives? 
 * I'll assume so.
 */
static int check_user(request_rec *r) {
	int restricted	= 0;
	struct auth_config *cfg;
	const array_header *reqs_arr;
	require_line *reqs;
	int i;

	assert(cfg = ap_get_module_config(r->per_dir_config,&bsd_auth_module));

	if (!cfg->auth_enabled)
		return DECLINED;

	if (!(reqs_arr = ap_requires(r)))
		return OK;

	assert(r->connection->user);

	reqs	= (require_line *)reqs_arr->elts;

	for (i = 0; i < reqs_arr->nelts; i++) {
		char *line, *type;

		/* Is this Require pertinent to the client's HTTP method */
		if (!(reqs[i].method_mask & (1 << r->method_number)))
			continue;	

		restricted	= 1;

		assert(line = reqs[i].requirement);
		assert(type = ap_getword(r->pool,(const char **)&line,' '));

		if (0 == strcmp(type,"valid-user")) {
			restricted	= 0;
		} else if (0 == strcmp(type,"user")) {
			char *user;

			while (restricted && *line) {
				assert(user = ap_getword_conf(r->pool,(const char **)&line));

				if (0 == strcmp(r->connection->user,user))
					restricted	= 0;
			}
		} else if (0 == strcmp(type,"group")) {
			char *groups[NGROUPS_MAX + 1];
			int ngroups	= sizeof groups / sizeof *groups;
			char *group;
			int n;

			if (-1 == auth_child_getgrouplist(r,r->connection->user,groups,&ngroups)) {
				ap_log_rerror(APLOG_MARK,APLOG_NOTICE|APLOG_NOERRNO,r,"[AuthBSD] -1 == auth_child_getgrouplist(ngroups=%d -> %d)",sizeof groups / sizeof *groups,ngroups);

				ngroups	= 0;
			}

			while (restricted && ngroups && *line) {
				assert(group = ap_getword_conf(r->pool,(const char **)&line));

				for (n = 0; n < ngroups; n++) {
					ap_log_rerror(APLOG_MARK,APLOG_DEBUG|APLOG_NOERRNO,r,"[AuthBSD] Required group %s == group %s?",group,groups[n]);

					if (0 == strcmp(groups[n],group)) {
						restricted	= 0;

						break;
					}
				}
			}
		} else {
			ap_log_rerror(APLOG_MARK,APLOG_NOTICE|APLOG_NOERRNO,r,"[AuthBSD] Unknown Require directive: %s",type);

			restricted	= 0;
		} /* Require valid-user || user || group */

		if (restricted && cfg->strict_require) {
			ap_log_rerror(APLOG_MARK,APLOG_NOTICE|APLOG_NOERRNO,r,"[AuthBSD] Access restricted by %s",type);

			break;	/* Bust out of loop early */
		} else if (!restricted && !cfg->strict_require) {
			break;	/* Bust out of loop early */
		}
	} /* for (i < reqs_arr->nelts) */

	if (restricted) {
		ap_note_basic_auth_failure(r);

		ap_log_reason("[AuthBSD] Access restricted by policy",r->uri,r);

		return HTTP_UNAUTHORIZED;
	}

	return OK;
} /* check_user() */


/*
 * Apache config command table
 */
static const command_rec ctable[] = {
	{
		"AuthBSDUser",
		authd_set_user,
		NULL,
		RSRC_CONF,
		TAKE1,
		"Auth daemon user: www | ..."
	},{
		"AuthBSDGroup",
		authd_set_group,
		NULL,
		RSRC_CONF,
		TAKE1,
		"Auth daemon group: www | ..."
	},{
		"AuthBSDFailureDelay",
		authd_set_fail_delay,
		NULL,
		RSRC_CONF,
		TAKE1,
		"Authentication delay enforced between failures",
	},{
		"AuthBSDFailureTolerance",
		authd_set_fail_tolerance,
		NULL,
		RSRC_CONF,
		TAKE1,
		"Number of failures tolerated before inserting delays",
	},{
		"AuthBSDCacheSize",
		authd_set_cache_size,
		NULL,
		RSRC_CONF,
		TAKE1,
		"Number of positive and negative credentials",
	},{
		"AuthBSDCacheTTL",
		authd_set_cache_ttl,
		NULL,
		RSRC_CONF,
		TAKE1,
		"Time to live of a credential in a cache",
	},{
		"AuthBSD",
		ap_set_flag_slot,
		(void *)XtOffsetOf(struct auth_config,auth_enabled),
		ACCESS_CONF,
		FLAG,
		"Toggle BSD Authentication: On | Off"
	},{
		"AuthBSDKeepPass",
		ap_set_flag_slot,
		(void *)XtOffsetOf(struct auth_config,keep_pass),
		ACCESS_CONF,
		FLAG,
		"Toggle password erasure: On | Off"
	},{
		"AuthBSDRequireSSL",
		ap_set_flag_slot,
		(void *)XtOffsetOf(struct auth_config,require_ssl),
		ACCESS_CONF,
		FLAG,
		"Toggle secure connection verification: On | Off"
	},{
		"AuthBSDStrictRequire",
		ap_set_flag_slot,
		(void *)XtOffsetOf(struct auth_config,strict_require),
		ACCESS_CONF,
		TAKE1,
		"Match all Require directives or at least one"
	},{
		"AuthBSDService",
		ap_set_string_slot,
		(void *)XtOffsetOf(struct auth_config,auth_service),
		ACCESS_CONF,
		TAKE1,
		"BSD Authentication protocol: login | challenge"
	},{
		"AuthBSDStyle",
		ap_set_string_slot,
		(void *)XtOffsetOf(struct auth_config,auth_style),
		ACCESS_CONF,
		TAKE1,
		"BSD Authentication style: passwd | radius | ..."
	},{
		"AuthBSDType",
		ap_set_string_slot,
		(void *)XtOffsetOf(struct auth_config,auth_type),
		ACCESS_CONF,
		TAKE1,
		"BSD Authentication type: auth-www | auth-webdav | ..."
	},{NULL}

};




/*
 * Apache module parameter definitions
 *
 * NOTE:
 * 	[N]: N is the order in which the routine is called, if called
 */
module MODULE_VAR_EXPORT bsd_auth_module = {
	STANDARD_MODULE_STUFF,
	module_init,		/* module initializer */
	create_dir_config,	/* per-directory config creator */
	NULL,			/* directory config merger */
	NULL,			/* server config creator */
	NULL,			/* server config merger */
	ctable,			/* command table */
	NULL,			/* [9] list of handlers */
	NULL,			/* [2] filename-to-URI translation */
	authenticate,		/* [5] check/validate user_id */
	check_user,		/* [6] check user_id is valid here */
	NULL,			/* [4] check access by host address */
	NULL,			/* [7] MIME type checker/settler */
	NULL,			/* [8] fixups */
	NULL,			/* [10] logger */
	NULL,			/* [3] header parser */
	child_init,		/* child_init */
	NULL,			/* child_exit */
	NULL,			/* [1] post read-request */
};

