/*
 * dstumbler v1.0 [menu.c]
 * by h1kari - (c) Dachb0den Labs 2001
 * with contributions from Kevin Kadow <dstumbler@msg.net>
 */

/*
 * Copyright (c) 2001 Dachb0den Labs.
 *      David Hulton <h1kari@dachb0den.com>.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by David Hulton.
 * 4. Neither the name of the author nor the names of any co-contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY David Hulton AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL David Hulton OR THE VOICES IN HIS HEAD
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>

#ifdef __OpenBSD__
#include <curses.h>
#else
#include <ncurses.h>
#endif

#include "dstumbler.h"
#include "screen.h"

/*
 * parse commands by the user
 */
void
parse_cmd(int c)
{
  int ans;
  char note_buf[NOTE_BUFFERLEN];

  switch(c)
  {
    /* for up key */
    case KEY_UP:
    case '[':
    case '8':
    case '+':
    case '=':
      APCHANGE(aps_cur > 0 ? aps_cur - 1 : aps_cur);
      REDRAWAP();
      REDRAWNODE();
      break;
    case KEY_DOWN:
    case ']':
    case '2':
    case '-':
      APCHANGE(aps_cur < (aps_len - 1) ? aps_cur + 1 : aps_cur);
      REDRAWAP();
      REDRAWNODE();
      break;
    case KEY_LEFT:
    case '<':
      if(!nodemode)
        break;

      NODECHANGE(aps[aps_cur]->node_cur > 0 ? aps[aps_cur]->node_cur - 1 :
       aps[aps_cur]->node_cur);
      REDRAWNODE();

      break;
    case KEY_RIGHT:
    case '>':
      if(!nodemode)
        break;

      NODECHANGE(aps[aps_cur]->node_cur < (aps[aps_cur]->node_len - 1) ?
       aps[aps_cur]->node_cur + 1 : aps[aps_cur]->node_cur);
      REDRAWNODE(); 

      break;
    case KEY_PPAGE:
    case 'u':
      APCHANGE(aps_cur > AP_SCR_LINES ? aps_cur - AP_SCR_LINES : 0);
      REDRAWAP();
      break;
    case KEY_NPAGE:
    case 'd':
      APCHANGE(aps_cur < (aps_len - (AP_SCR_LINES + 1)) ?
       aps_cur + AP_SCR_LINES : aps_len - 1);
      REDRAWAP();
      break;
    case KEY_END:
    case 'e':
      APCHANGE(aps_len > 0 ? aps_len - 1 : 0);
      REDRAWAP();
      break;
    case KEY_HOME:
    case 'h':
      APCHANGE(0);
      REDRAWAP();
      break;
    case 'n':
    case 'N':
      APCHANGE(aps_new);
      REDRAWAP();
      break;
    case 's':
    case 'S':
      sort_aplist();
      break;
    case 'a':
    case 'A':
      autosel++;
      autosel %= 2;

      redraw_toggle(autosel, 0, 'a');

      apchange++;
      break;
    case 'r':
    case 'R':
      resolvmfg++;
      resolvmfg %= 2;

      redraw_toggle(resolvmfg, 1, 'r');

      if(aps_len > 0)
        redraw_aps();

      if(nodemode && aps[aps_cur]->node_len > 0)
        redraw_nodes();

      break;
    case 'o':
    case 'O':
      if(!monmode && !nodevice)
        break;

      nodemode++;
      nodemode %= 2;

      redraw_toggle(nodemode, 2, 'o');
      refreshclean();
      break;
    case 'i':
    case 'I':
      audiomode++;
      audiomode %= 2;

      redraw_toggle(audiomode, 3, 'i');

      break;
    case 'c':
    case 'C':
      if(!monmode)
        break;

      chanlock++;
      chanlock %= 2;

      redraw_toggle(chanlock, 4, 'c');

      /* if we're in chanlock, signal an ap change so we can lock onto it */
      apchange++;
      break;
    case 'l':
    case 'L':
      load_aplist();
      break;
    case 'b':
    case 'B':
      backup_aplist();
      break;
    case 'q':
    case 'Q':
      ans = alert("Quit? (y/n)");

      if(ans == 'y' || ans == 'Y')
        exitclean(0);

      break;
    case '.':
    case ',':
      if(nodemode && aps[aps_cur]->node_len)
      {
        prompt(NOTE_PROMPT, note_buf, NOTE_BUFFERLEN);
        strcpy(aps[aps_cur]->node[aps[aps_cur]->node_cur]->note, note_buf);
      }
      else if(aps_len)
      {
        prompt(NOTE_PROMPT, note_buf, NOTE_BUFFERLEN);
        strcpy(aps[aps_cur]->note, note_buf);
      }

      PRINTBANNER();
      break;
    case 'm':
    case 'M':
      menumode++;
      menumode %= 2;

      redraw_toggle(nodemode, 5, 'm');
      refreshclean();
      break;
    case 12:
    case 'k':
    case 'K':
      refreshclean();
      break;
    default:
      break;
  }
}

/*
 * sort the aplist by the most recently seen aps
 */
void
sort_aplist(void)
{
  if(aps_len > 1)
  {
    /* sort out all the aps based on time */
    if(heapsort(aps, aps_len, sizeof(struct aps_s *), &aplistcmp) == -1)
    {
      alert("error: problem sorting aps: %s", strerror(errno));
      return;
    }

    APNEW(0);
    APCHANGE(0);

    REDRAWAP();
  }
}

/*
 * compare two ap pointers, this is kinda switched so we can sort decending
 * instead of ascending
 */
int
aplistcmp(const void *a1, const void *b1)
{
  struct aps_s *a, *b;

  a = (struct aps_s *)(*(void **)a1);
  b = (struct aps_s *)(*(void **)b1);

  /*
   * if a.sec == b.sec and a.usec == b.usec, then 0
   * if a.sec == b.sec and a.usec > b.usec, then 1
   * if a.sec == b.sec and a.usec < b.usec, then -1
   * if a.sec > b.sec, then 1
   * if a.sec < b.sec, then -1
   */
  return(CMPTVAL(a->lastseen, b->lastseen));
}

/*
 * draw menu of key options
 */
void
redraw_menu(void)
{
  int i, mode = 0;
  char menu[] = MENU;

  for(i = 0; i < strlen(menu); i++)
  {
    if(menu[i] == '[')
      mode = 1;
    else if(menu[i] == ']')
      mode = 0;

    if(mode && menu[i] != '[')
      waddch(menu_scr, menu[i] | MENU_COLOR_BOLD);
    else
      waddch(menu_scr, menu[i] | MENU_COLOR);
  }

  redraw.menu_scr++;
}

/*
 * draw various mode toggles on the border of the info screen in
 * the top right area of the screen.
 */
void
redraw_toggle(int toggle, int num, char chr)
{
  mvwaddch(top_right_border, num, 0,
   chr | (toggle ? BORDER_COLOR_BOLD : BORDER_COLOR));
  wrefresh(top_right_border);

  return;
}

/*
 * load ap list from a file
 */
#define DATABUF_LEN 256
void
load_aplist(void)
{
  int found, ap, node;
  u_char bssid[MACSIZE];
  char buffer[FILE_BUFFERLEN], databuf[DATABUF_LEN];
  struct aps_s new;
  struct node_s nodenew;
  FILE *fd;

  /*
   * first get the name of the file to import from
   */
  prompt(FILE_PROMPT, buffer, FILE_BUFFERLEN);

  /*
   * now loop through the file and import to the end of the aplist
   */
  if((fd = fopen(buffer, "r")) == NULL)
  {
    alert("error: unable to open specified file: %s", strerror(errno));
    return;
  }

  memset((char *)&new, 0, sizeof(struct aps_s));

  while(!feof(fd))
  {
    fgets(databuf, DATABUF_LEN - 1, fd);
    databuf[strlen(databuf) - 1] = '\0';

    if(strncmp(databuf, "#NODE ", 6) == 0)
    {
      unlog_node(fd, databuf, &nodenew, bssid);

      if((ap = find_node_ap(bssid)) == -1)
      {
        alert("error: unable to find ap for node in backup file");
        fclose(fd);
        exitclean(2);
      }

      if((node = find_node(ap, nodenew.mac)) > -1)
      {
        add_node_log(ap, aps[ap]->node[node], &nodenew);

        if(strlen(new.note))
          strcpy(aps[found]->note, new.note);
      }
      else
      {
        nodenew.firstseen.tv_sec = nodenew.lastseen.tv_sec;
        nodenew.firstseen.tv_usec = nodenew.lastseen.tv_usec;
        add_node(ap, &nodenew);

        memset((char *)aps[ap]->node[aps[ap]->node_len - 1]->quality, 0,
         sizeof(aps[ap]->node[aps[ap]->node_len - 1]->quality));

        memset((char *)&nodenew, 0, sizeof(struct node_s));
      }
    }
    else if(databuf[0] == '#')
      continue;
    else
    {
      unlog_ap(fd, databuf, &new);

      if((found = find_ap(new.ssid, new.bssid, new.chan)) > -1)
      {
        add_ap_log(aps[found], &new);

        if(strlen(new.note))
          strcpy(aps[found]->note, new.note);
      }
      else if(found == -1)
      {
        new.firstseen.tv_sec = new.lastseen.tv_sec;
        new.firstseen.tv_usec = new.lastseen.tv_usec;
        add_ap(&new);

        memset((char *)aps[aps_len - 1]->quality, 0,
         sizeof(aps[aps_len - 1]->quality));

        memset((char *)&new, 0, sizeof(struct aps_s));
      }
    }
  }
  
  fclose(fd);

  apchange++;
  REDRAWAP();
  PRINTBANNER();
}

/*
 * save ap list to a file
 */
#define DATEBUF_LEN 33
void
backup_aplist(void)
{
  int i, j, k;
  char buffer[FILE_BUFFERLEN];
  FILE *fd;

  /*
   * first we need the name of the file to export to
   */
  prompt(FILE_PROMPT, buffer, FILE_BUFFERLEN);

  if((fd = log_start(buffer)) == NULL)
    return;

  for(i = 0; i < aps_len; i++)
  {
    for(j = 0; j < aps[i]->log_len; j++)
      log_ap(fd, aps[i], aps[i]->log[j]);

    for(j = 0; j < aps[i]->node_len; j++)
    {
      for(k = 0; k < aps[i]->node[j]->log_len; k++)
        log_node(fd, aps[i], aps[i]->node[j], aps[i]->node[j]->log[k]);
    }
  }

  log_stop(fd); 
}

/*
 * prompt for filename in the alert screen
 */
#define MAX_PROMPT_BUFF_LEN (ALERT_SCR_COLS - strlen(prompt) - 4)
void
prompt(char *prompt, char *buffer, int len)
{
  len = MIN(MAX_PROMPT_BUFF_LEN, len);

  /*
   * first we need the name of the file to export to
   */
  werase(alert_scr);
  wattrset(alert_scr, PROMPT_COLOR);
  wprintw(alert_scr, "%s [ ", prompt);
  mvwaddch(alert_scr, 0, ALERT_SCR_COLS - 1, ']');
  redraw.alert_scr++;
  smart_redraw();   
    
  wattrset(alert_scr, PROMPT_COLOR_BOLD);
  echo();
  mvwgetnstr(alert_scr, 0, strlen(prompt) + 3, buffer, len - 1);
  noecho();

  PRINTBANNER();
}
