/*	$NetBSD: file.c,v 1.1 2007/11/09 21:18:25 plunky Exp $	*/

/*-
 * Copyright (c) 2007 Iain Hibbert
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <bluetooth.h>
#include <db.h>
#include <fcntl.h>
#include <stdbool.h>
#include <string.h>

#include "btkey.h"

static const char *key_file = "/var/db/bthcid.keys";

typedef struct {
	bdaddr_t laddr;
	bdaddr_t raddr;
} db_keyid_t;

/*
 * List keys file.
 */
bool
list_file(void)
{
	DB *db;
	DBT dbk, dbd;
	db_keyid_t id;
	bdaddr_t bdaddr;
	bool rv = false;
	int r;

	db = dbopen(key_file, O_RDONLY, 0600, DB_HASH, NULL);
	if (db == NULL)
		return false;

	memset(&dbk, 0, sizeof(dbk));
	memset(&dbd, 0, sizeof(dbd));
	for (r = db->seq(db, &dbk, &dbd, R_FIRST); r == 0;
	    r = db->seq(db, &dbk, &dbd, R_NEXT)) {
		if (dbk.size != sizeof(id))
			continue;

		if (dbd.size != HCI_KEY_SIZE)
			continue;

		memcpy(&id, dbk.data, dbk.size);
		if (!bdaddr_same(&id.laddr, &laddr))
			continue;

		printf("\n");
		print_addr("bdaddr", &id.raddr);
		print_key("file key", dbd.data);
	}

	db->close(db);
	return true;
}

/*
 * Read from keys file.
 */
bool
read_file(void)
{
	DB *db;
	DBT dbk, dbd;
	db_keyid_t id;
	bool rv = false;

	db = dbopen(key_file, O_RDONLY, 0600, DB_HASH, NULL);
	if (db == NULL)
		return false;

	memset(&id, 0, sizeof(id));
	bdaddr_copy(&id.laddr, &laddr);
	bdaddr_copy(&id.raddr, &raddr);
	memset(&dbk, 0, sizeof(dbk));
	dbk.data = &id;
	dbk.size = sizeof(id);
	memset(&dbd, 0, sizeof(dbd));
	if (db->get(db, &dbk, &dbd, 0) != 0)
		goto done;

	if (dbd.size != HCI_KEY_SIZE)
		goto done;

	memcpy(key, dbd.data, HCI_KEY_SIZE);
	rv = true;

done:
	(void)db->close(db);
	return rv;
}

/*
 * Write to keys file.
 */
bool
write_file(void)
{
	DB *db;
	DBT dbk, dbd;
	db_keyid_t id;
	bool rv = false;

	db = dbopen(key_file, O_RDWR, 0600, DB_HASH, NULL);
	if (db == NULL)
		return false;

	memset(&id, 0, sizeof(id));
	bdaddr_copy(&id.laddr, &laddr);
	bdaddr_copy(&id.raddr, &raddr);
	memset(&dbk, 0, sizeof(dbk));
	dbk.data = &id;
	dbk.size = sizeof(id);
	memset(&dbd, 0, sizeof(dbk));
	dbd.data = key;
	dbd.size = HCI_KEY_SIZE;
	if (db->put(db, &dbk, &dbd, 0) == 0)
		rv = true;

	(void)db->close(db);
	return rv;
}

/*
 * Clear from keys file.
 */
bool
clear_file(void)
{
	DB *db;
	DBT dbk;
	db_keyid_t id;
	bool rv = false;

	db = dbopen(key_file, O_RDWR, 0600, DB_HASH, NULL);
	if (db == NULL)
		return false;

	memset(&id, 0, sizeof(id));
	bdaddr_copy(&id.laddr, &laddr);
	bdaddr_copy(&id.raddr, &raddr);
	memset(&dbk, 0, sizeof(dbk));
	dbk.data = &id;
	dbk.size = sizeof(id);
	if (db->del(db, &dbk, 0) == 0)
		rv = true;

	(void)db->close(db);
	return rv;
}
