/*	$NetBSD: db.c,v 1.4 2007/08/17 17:59:16 pavel Exp $	*/

/*-
 * Copyright (c) 2006 Itronix Inc.
 * All rights reserved.
 *
 * Written by Iain Hibbert for Itronix Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of Itronix Inc. may not be used to endorse
 *    or promote products derived from this software without specific
 *    prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ITRONIX INC. ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL ITRONIX INC. BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <bluetooth.h>
#include <err.h>
#include <fcntl.h>
#include <stdlib.h>

#include <dev/bluetooth/btdev.h>

#include "btdevctl.h"

#define BTDEVCTL_DB		"/var/db/btdevctl"

static const char *db_pathvec[] = { BTDEVCTL_DB, NULL };

static struct {
	const char *name;
	uint16_t type;
	const char *description;
} svctype[] = {
	{ "HID",	BTDEV_HID,	"Human Interface Device" },
	{ "HSET",	BTDEV_HSET,	"Headset" }
};

static struct {
	const char *name;
	int mode;
} modename[] = {
	{ "none",	BTDEV_MODE_NONE },
	{ "auth",	BTDEV_MODE_AUTH },
	{ "encrypt",	BTDEV_MODE_ENCRYPT },
	{ "secure",	BTDEV_MODE_SECURE }
};

#define NUM(v)		(sizeof(v) / sizeof(v[0]))

typedef char db_key_t[32];

static int db_key(db_key_t, bdaddr_t *, bdaddr_t *, const char *);
static int db_cget(btdevctl_db_t, char *);
static int db_cget_hid(btdevctl_db_t, char *);
static int db_del(db_key_t);

/*
 * convert service name to bd_type number
 */
uint16_t
db_gettype(const char *name)
{
	int i;
	
	for (i = 0; i < NUM(svctype); i++) {
		if (strcmp(name, svctype[i].name) != 0)
			continue;

		return svctype[i].type;
	}

	printf("Known service names:\n");
	for (i = 0 ; i < NUM(svctype) ; i++)
		printf("\t%s\t%s\n", svctype[i].name, svctype[i].description);

	exit(EXIT_FAILURE);
}

/*
 * convert mode string to bd_mode number
 */
int
db_getmode(const char *name)
{
	int i;

	for (i = 0; i < NUM(modename); i++)
		if (strcmp(modename[i].name, name) == 0)
			return modename[i].mode;

	printf("Known mode names:\n");
	for (i = 0 ; i < NUM(svctype) ; i++)
		printf("\t%s\n", modename[i].name);

	exit(EXIT_FAILURE);
}

/*
 * convert bd_type number to service name
 */
const char *
db_typename(uint16_t type)
{
	int i;
	
	for (i = 0; i < NUM(svctype); i++)
		if (type == svctype[i].type)
			return svctype[i].name;

	printf("Known device types:\n");
	for (i = 0 ; i < NUM(svctype) ; i++)
		printf("\t%d\t%s (%s)\n", svctype[i].type,
		    svctype[i].description, svctype[i].name);

	exit(EXIT_FAILURE);
}

/*
 * convert mode number to mode name
 */
const char *
db_modename(int mode)
{
	int i;
	
	for (i = 0; i < NUM(modename); i++)
		if (mode == modename[i].mode)
			return modename[i].name;

	printf("Known modes:\n");
	for (i = 0 ; i < NUM(modename) ; i++)
		printf("\t%d\t%s\n", modename[i].mode,
		    modename[i].name);

	exit(EXIT_FAILURE);
}

/*
 * allocate a btdevctl database entry structure
 */
btdevctl_db_t
db_create(void)
{
	btdevctl_db_t db;

	if ((db = malloc(sizeof *db)) != NULL)
		memset(db, 0, sizeof *db);

	return db;
}

/*
 * generate a unique key
 */
static int
db_key(db_key_t key, bdaddr_t *laddr, bdaddr_t *raddr, const char *service)
{
	int l;

	l = snprintf(key, sizeof(db_key_t), "%02x%02x%02x%02x%02x%02x,"
	    "%02x%02x%02x%02x%02x%02x,%s", laddr->b[5], laddr->b[4],
	    laddr->b[3], laddr->b[2], laddr->b[1], laddr->b[0], raddr->b[5],
	    raddr->b[4], raddr->b[3], raddr->b[2], raddr->b[1], raddr->b[0],
	    service);
	return (l < sizeof(db_key_t));
}

/*
 * lookup laddr/raddr/service in database and return dictionary
 */
btdevctl_db_t
db_get(bdaddr_t *laddr, bdaddr_t *raddr, const char *service)
{
	btdevctl_db_t dev;
	db_key_t key;
	char *buf;

	if (!db_key(key, laddr, raddr, service))
		return NULL;

	if (cgetent(&buf, (char **)db_pathvec, key) != 0)
		return NULL;

	if ((dev = db_create()) == NULL) {
		free(buf);
		return NULL;
	}

	if (!db_cget(dev, buf))
		goto err;

	if (strcmp(service, "HID") == 0) {
		if (!db_cget_hid(dev, buf))
			goto err;
	} else
		goto err;

	free(buf);
	return dev;
 err:
	free(dev);
	free(buf);
	return NULL;
}

static int
db_cget(btdevctl_db_t dev, char *buf)
{
	char *str;
	int i;

	if (cgetustr(buf, "mode", &str) < 0)
		return 0;

	for (i = 0; i < NUM(modename); i++) {
		if (strcmp(modename[i].name, str) == 0) {
			dev->bd_mode = modename[i].mode;
			break;
		}
	}

	free(str);
	return i < NUM(modename);
}

static int
db_cget_hid(btdevctl_db_t dev, char *buf)
{
	long num;
	char *str;
	uint8_t *desc;
	uint16_t dlen;
	uint16_t i;

	if (cgetnum(buf, "flags", &num) != 0)
		return 0;
	dev->bd_hid.hid_flags = num;

	if (cgetnum(buf, "ctl-psm", &num) != 0)
		return 0;
	dev->bd_hid.hid_ctl = num;

	if (cgetnum(buf, "int-psm", &num) != 0)
		return 0;
	dev->bd_hid.hid_int = num;

	if (cgetustr(buf, "desc", &str) < 0)
		return 0;
	if (((num = strlen(str)) % 2) != 0 || num > 0xffff * 2) {
		free(str);
		return 0;
	}
	dlen = num / 2;
	if ((desc = malloc(dlen)) == NULL) {
		free(str);
		return 0;
	}
	for (i = 0; i < dlen; i++) {
		char xx[3];

		xx[0] = str[i*2+0];
		xx[1] = str[i*2+1];
		xx[2] = '\0';

		desc[i] = (uint8_t)strtol(xx, NULL, 16);
	}
	free(str);
	dev->bd_hid.hid_dlen = dlen;
	dev->bd_hid.hid_desc = desc;

	return 1;
}

/*
 * delete the named entry from the database
 */
static int
db_del(db_key_t key)
{
	char *dumpbuf;
	size_t dumplen;
	char *buf;
	int rv;
	int fd;

	dumpbuf = NULL;
	dumplen = 0;

	while ((rv = cgetnext(&buf, (char **)db_pathvec)) == 1) {
		char *newp;
		size_t buflen;

		if (cgetmatch(buf, key) == 0) {
			free(buf);
			continue;
		}

		buflen = strlen(buf);
		dumplen += buflen + 1;

		if ((newp = realloc(dumpbuf, dumplen)) == NULL) {
			free(buf);
			(void)cgetclose();
			if (dumpbuf != NULL)
				free(dumpbuf);
			return 0;
		}

		dumpbuf = newp;
		memcpy(dumpbuf + dumplen - buflen - 1, buf, buflen);
		memcpy(dumpbuf + dumplen - 1, "\n", 1);
		free(buf);
	}
	(void)cgetclose();

	if (rv != 0) {
		if (dumpbuf != NULL)
			free(dumpbuf);
		return 0;
	}

	if ((fd = open(BTDEVCTL_DB, O_WRONLY | O_TRUNC | O_CREAT,
	    0644)) == -1) {
		if (dumpbuf != NULL)
			free(dumpbuf);
		return 0;
	}

	buf = dumpbuf;
	while (dumplen > 0) {
		ssize_t len;

		len = write(fd, buf, dumplen);
		if (len < 1)
			err(1, "%s", BTDEVCTL_DB);

		buf += len;
		dumplen -= len;
	}

	if (dumpbuf != NULL)
		free(dumpbuf);
	return 1;
}

/*
 * store dictionary in database at laddr/raddr/service
 */
int
db_set(btdevctl_db_t dev, bdaddr_t *laddr, bdaddr_t *raddr, const char *service)
{
	char mode[32];
	db_key_t key;
	int i;
	int fd;
	FILE *db;

	if (!db_key(key, laddr, raddr, service))
		return 0;

	(void)db_del(key);

	for (i = 0; i < NUM(modename); i++) {
		if (modename[i].mode == dev->bd_mode) {
			strlcpy(mode, modename[i].name, sizeof mode);
			break;
		}
	}
	if (i == NUM(modename))
		snprintf(mode, sizeof mode, "%d", dev->bd_mode);

	if ((fd = open(BTDEVCTL_DB, O_WRONLY | O_APPEND | O_CREAT,
	    0644)) == -1)
		return 0;
	if ((db = fdopen(fd, "a")) == NULL) {
		close(fd);
		return 0;
	}

	fprintf(db, "%s:mode=%s:", key, mode);
	if (strcmp(service, "HID") == 0) {
		uint16_t dlen = dev->bd_hid.hid_dlen;
		uint16_t i;

		fprintf(db, "flags#0x%x:", dev->bd_hid.hid_flags);
		fprintf(db, "ctl-psm#%d:", dev->bd_hid.hid_ctl);
		fprintf(db, "int-psm#%d:", dev->bd_hid.hid_int);
		fprintf(db, "desc=");
		for (i = 0; i < dlen; i++)
			fprintf(db, "%02x",
			    ((uint8_t *)dev->bd_hid.hid_desc)[i]);
		fprintf(db, ":");
	}
	fprintf(db, "\n");

	(void)fclose(db);
	(void)close(fd);
	return 1;
}
