/*-
 * Copyright (c) 2006 Itronix Inc.
 * All rights reserved.
 *
 * Written by Iain Hibbert for Itronix Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of Itronix Inc. may not be used to endorse
 *    or promote products derived from this software without specific
 *    prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ITRONIX INC. ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL ITRONIX INC. BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/audioio.h>
#include <sys/ioctl.h>
#include <sys/time.h>

#include <bluetooth.h>
#include <err.h>
#include <errno.h>
#include <event.h>
#include <fcntl.h>
#include <signal.h>
#include <sdp.h>
#include <stdarg.h>
#include <stdlib.h>
#include <termios.h>
#include <unistd.h>

#include <netbt/rfcomm.h>
#include <netbt/sco.h>

#define RETRY_INTERVAL	30

int  main(int, char *[]);
void usage(void);
void do_signal(int, short, void *);

int init_kbd(void);
void reset_kbd(void);
void read_kbd(int, short, void *);

int listen_sco(void);
void accept_sco(int, short, void *);
void recv_sco(int, short, void *);
void read_audio(int, short, void *);
void close_audio(void);

int listen_rfcomm(void);
void accept_rfcomm(int, short, void *);
void open_rfcomm(int, short, void *);
int establish_rfcomm(void);
int send_rfcomm(const char *, ...);
void recv_rfcomm(int, short, void *);

void rc_error(char *);
void rc_ok(char *);
void rc_ring(char *);
void rc_no_carrier(char *);
void rc_busy(char *);
void rc_no_answer(char *);
void rc_delayed(char *);
void rc_blacklisted(char *);
void rc_brsf(char *);
void rc_bsir(char *);
void rc_btrh(char *);
void rc_bvra(char *);
void rc_ccwa(char *);
void rc_ciev(char *);
void rc_cind(char *);
void rc_clcc(char *);
void rc_cme_error(char *);
void rc_clip(char *);
void rc_cnum(char *);
void rc_vgm(char *);
void rc_vgs(char *);

int query_hfp_channel(void);

struct event sighup_ev;	/* bye bye */
struct event sigint_ev;	/* bye bye */
struct event sigterm_ev;/* bye bye */
struct event timer_ev;	/* reconnect timer */
struct event rfdata_ev;	/* gateway speaks */
struct event scdata_ev;	/* SCO audio data */
struct event audata_ev;	/* audio data ready (SIGIO) */

/* bdaddr information */
bdaddr_t laddr;		/* local bdaddr */
bdaddr_t raddr;		/* remote bdaddr */
int server_channel;	/* server channel */
int gateway_channel;	/* gateway channel */

/* audio buffers */
char *audio_buffer;
u_int blocksize;
uint16_t mtu;

/* file descriptors */
int rf;			/* rfcomm */
int sc;			/* SCO audio */
int au;			/* audio device */

/* stored termios */
struct termios tio;

/* options */
int verbose;		/* copy to stdout */
const char *audio;	/* audio device */
const char *mixer;	/* mixer device */

const char *number[10];	/* numbers */

/* state of establishment */
enum {	START,
	BRSF_SET,
	CIND_TEST,
	CIND_READ,
	CMER_SET,
	CLIP_SET,
	SERVICE
} state;

struct result_code {
	const char	*code;
	int		strlen;
	void		(*handler)(char *);
} result_codes[] = {
	{ "ERROR",		5,	rc_error	},
	{ "OK",			2,	rc_ok		},
	{ "RING",		4,	rc_ring		},
	{ "NO CARRIER",		10,	rc_no_carrier	},
	{ "BUSY",		4,	rc_busy		},
	{ "NO ANSWER",		9,	rc_no_answer	},
	{ "DELAYED",		7,	rc_delayed	},
	{ "BLACKLISTED",	11,	rc_blacklisted	},
	{ "+BRSF:",		6,	rc_brsf		},
	{ "+BSIR:",		6,	rc_bsir		},
	{ "+BTRH:",		6,	rc_btrh		},
	{ "+BVRA:",		6,	rc_bvra		},
	{ "+CCWA:",		6,	rc_ccwa		},
	{ "+CIEV:",		6,	rc_ciev		},
	{ "+CIND:",		6,	rc_cind		},
	{ "+CLCC:",		6,	rc_clcc		},
	{ "+CME ERROR:",	11,	rc_cme_error	},
	{ "+CLIP:",		6,	rc_clip		},
	{ "+CNUM:",		6,	rc_cnum		},
	{ "+VGM:",		5,	rc_vgm		},
	{ "+VGS:",		5,	rc_vgs		},
	{ NULL }
};

int
main(int ac, char *av[])
{
	int		ch;

	verbose = 0;
	audio = "/dev/audio";
	mixer = "/dev/mixer";
	memset(&laddr, 0, sizeof(laddr));
	memset(&raddr, 0, sizeof(raddr));
	server_channel = 0;
	gateway_channel = 0;
	rf = sc = au = -1;

	for (ch = 0 ; ch < 10 ; ch++)
		number[ch] = NULL;

	while ((ch = getopt(ac, av, "0:1:2:3:4:5:6:7:8:9:A:a:c:d:hM:s:v")) != EOF) {
		switch (ch) {
		case '0': case '1': case '2': case '3': case '4':
		case '5': case '6': case '7': case '8': case '9':
			number[ch - '0'] = optarg;
			break;

		case 'A':	/* Audio device */
			audio = optarg;
			break;

		case 'a':	/* remote bdaddr */
			if (!bt_aton(optarg, &raddr)) {
				struct hostent  *he = NULL;

				if ((he = bt_gethostbyname(optarg)) == NULL)
					errx(EXIT_FAILURE, "%s: %s",
						optarg, hstrerror(h_errno));

				bdaddr_copy(&raddr, (bdaddr_t *)he->h_addr);
			}
			break;

		case 'c':	/* gateway channel */
			gateway_channel = atoi(optarg);
			if (gateway_channel < RFCOMM_CHANNEL_MIN
			    || gateway_channel > RFCOMM_CHANNEL_MAX)
				errx(EXIT_FAILURE, "Bad channel number");;

			break;

		case 'd':	/* local device bdaddr */
			if (!bt_devaddr(optarg, &laddr))
				err(EXIT_FAILURE, "%s", optarg);

			break;

		case 'M':	/* Mixer device */
			mixer = optarg;
			break;

		case 's':	/* Server channel */
			server_channel = atoi(optarg);
			if (server_channel < RFCOMM_CHANNEL_MIN
			    || server_channel > RFCOMM_CHANNEL_MAX)
				errx(EXIT_FAILURE, "Bad channel number");;

			break;

		case 'v':	/* Verbose */
			verbose = 1;
			break;

		case 'h':	/* Help */
		default:
			usage();
		}
	}

	if (bdaddr_any(&raddr))
		usage();

	if (gateway_channel == 0
	    && query_hfp_channel() == 0)
		errx(EXIT_FAILURE, "Can't find Handsfree Audio Gateway channel");

	event_init();

	signal_set(&sighup_ev, SIGHUP, do_signal, NULL);
	if (signal_add(&sighup_ev, NULL) < 0)
		err(EXIT_FAILURE, "SIGHUP");

	signal_set(&sigint_ev, SIGINT, do_signal, NULL);
	if (signal_add(&sigint_ev, NULL) < 0)
		err(EXIT_FAILURE, "SIGINT");

	signal_set(&sigterm_ev, SIGTERM, do_signal, NULL);
	if (signal_add(&sigterm_ev, NULL) < 0)
		err(EXIT_FAILURE, "SIGTERM");

	if (init_kbd() < 0)
		err(EXIT_FAILURE, "init_kbd");

	if (listen_sco() < 0)
		err(EXIT_FAILURE, "sco");

	if (server_channel && listen_rfcomm() < 0)
		err(EXIT_FAILURE, "channel %d", server_channel);

	evtimer_set(&timer_ev, open_rfcomm, NULL);
	open_rfcomm(0, 0, NULL);

	event_dispatch();

	err(EXIT_FAILURE, "event_dispatch");
}

void
usage(void)
{
	extern char *__progname;

	fprintf(stderr,
		"usage: %s [-v] [-(0-9) number] [-A device] [-c channel] [-d bdaddr] [-M device] [-s channel] -a bdaddr\n"
		"Where:\n"
		"\t-(0-9) number set hotkey number\n"
		"\t-A device     audio device (default: /dev/audio)\n"
		"\t-a bdaddr     remote device bdaddr\n"
		"\t-c channel    specify remote channel\n"
		"\t-d bdaddr     local device bdaddr\n"
		"\t-M device     mixer device (default: /dev/mixer)\n"
		"\t-s channel    specify local channel (and register as Handsfree)\n"
		"\t-v            verbose output\n"
		"", __progname);

	exit(EXIT_FAILURE);
}

void
do_signal(int s, short ev, void *arg)
{

	switch (s) {
	case SIGHUP:
	case SIGINT:
	case SIGTERM:
	default:
		exit(EXIT_SUCCESS);
	}
}

/*
 * Set up the keyboard handler
 */
int
init_kbd(void)
{
	static struct event ev;
	struct termios t;

	/*
	 * Take a copy of the termios structure so that we can
	 * re-instate it at exit. Turn off echo and canonical
	 * line handling for stdin.
	 */
	if (tcgetattr(STDIN_FILENO, &t) < 0)
		return -1;

	memcpy(&tio, &t, sizeof(tio));
	t.c_lflag &= ~(ECHO | ICANON);

	if (tcsetattr(STDIN_FILENO, TCSANOW, &t) < 0)
		return -1;

	atexit(reset_kbd);

	event_set(&ev, STDIN_FILENO, EV_READ | EV_PERSIST, read_kbd, NULL);
	if (event_add(&ev, NULL) < 0)
		return -1;

	printf("Press ? for commands\n");
	return 0;
}

/*
 * Reset saved terminal state on stdin
 */
void
reset_kbd(void)
{

	tcsetattr(STDIN_FILENO, TCSAFLUSH, &tio);
}

/*
 * Keyboard event
 */
void
read_kbd(int fd, short ev, void *arg)
{
	char key;

	if (read(fd, &key, 1) != 1)
		err(EXIT_FAILURE, "read_kbd");

	switch (key) {
	case '0': case '1': case '2': case '3': case '4':
	case '5': case '6': case '7': case '8': case '9':
		if (number[key - '0'] != NULL)
			send_rfcomm("DT%s", number[key - '0']);

		break;

	case 'a': case 'A':	/* Answer */
		send_rfcomm("A");
		break;

	case 'c': case 'C':	/* Connect */
		if (rf == -1)
			open_rfcomm(0, 0, NULL);

		break;

	case 'h': case 'H':	/* Hangup */
		send_rfcomm("+CHUP");
		break;

	case 'q': case 'Q':	/* Quit */
		exit(EXIT_SUCCESS);

	case 'r': case 'R':	/* Redial */
		send_rfcomm("+BLDN");
		break;

	case '?':	/* Help */
		printf("Commands:\n"
			"a\tAnswer\n"
			"c\tConnect to phone\n"
			"h\tHanguup\n"
			"q\tQuit\n"
			"r\tRedial\n");

		for (key = '0' ; key <= '9' ; key++)
			if (number[key - '0'] != NULL)
				printf("%c\tdial \"%s\"\n", key, number[key - '0']);
		break;

	default:
		break;
	}
}


/*
 * Initialize the SCO listener
 */
int
listen_sco(void)
{
	static struct event ev;
	struct sockaddr_bt addr;
	int fd;

	fd = socket(PF_BLUETOOTH, SOCK_SEQPACKET, BTPROTO_SCO);
	if (fd < 0)
		return -1;

	memset(&addr, 0, sizeof(addr));
	addr.bt_len = sizeof(addr);
	addr.bt_family = AF_BLUETOOTH;
	bdaddr_copy(&addr.bt_bdaddr, &laddr);

	if (bind(fd, (struct sockaddr *)&addr, sizeof(addr)) < 0)
		return -1;

	if (listen(fd, 1) < 0)
		return -1;

	event_set(&ev, fd, EV_READ | EV_PERSIST, accept_sco, NULL);
	if (event_add(&ev, NULL) < 0)
		return -1;

	return 0;
}

/*
 * Incoming Audio connection
 */
void
accept_sco(int fd, short ev, void *arg)
{
	audio_info_t info;
	struct sockaddr_bt addr;
	socklen_t len;
	socklen_t size;
	int s;

	memset(&addr, 0, sizeof(addr));
	len = sizeof(addr);
	s = accept(fd, (struct sockaddr *)&addr, &len);
	if (s < 0) {
		perror("accept");
		return;
	}

	if (sc >= 0
	    || addr.bt_len != sizeof(addr)
	    || addr.bt_family != AF_BLUETOOTH
	    || !bdaddr_same(&addr.bt_bdaddr, &raddr)) {
		close(s);
		return;
	}
	sc = s;

	/* XXX might need to up the socket buffer size? */

	size = sizeof(mtu);
	if (getsockopt(sc, BTPROTO_SCO, SO_SCO_MTU, &mtu, &size) < 0) {
		close(sc);
		sc = -1;
		return;
	}

	if (fcntl(sc, F_SETFL, O_NONBLOCK) < 0) {
		close(sc);
		sc = -1;
		return;
	}

	event_set(&scdata_ev, sc, EV_READ | EV_PERSIST, recv_sco, NULL);
	if (event_add(&scdata_ev, NULL) < 0) {
		close(sc);
		close(au);
		sc = au = -1;
		return;
	}

	au = open(audio, O_RDWR, 0);
	if (au < 0) {
		close(sc);
		sc = -1;
		return;
	}

	if (fcntl(au, F_SETFL, O_NONBLOCK | O_ASYNC) < 0) {
		close(sc);
		close(au);
		sc = au = -1;
		return;
	}

	fd = 1;	/* set full duplex */
	if (ioctl(au, AUDIO_SETFD, &fd) < 0)
		err(EXIT_FAILURE, "Full Duplex");

	/*
	 * set play and record modes of 8kHz 16-bit slinear mono,
	 * and round the blocksize up so that our SCO packets are
	 * all the same size.
	 */
	if (ioctl(au, AUDIO_GETINFO, &info) < 0)
		err(EXIT_FAILURE, "AUDIO_GETINFO");

	info.play.sample_rate = 8000;
	info.play.channels = 1;
	info.play.precision = 16;
	info.play.encoding = AUDIO_ENCODING_SLINEAR_LE;

	info.record.sample_rate = 8000;
	info.record.channels = 1;
	info.record.precision = 16;
	info.record.encoding = AUDIO_ENCODING_SLINEAR_LE;

	info.mode = (AUMODE_PLAY | AUMODE_RECORD);

	blocksize = (info.blocksize / mtu + 1) * mtu;
	info.blocksize = blocksize;

	if (ioctl(au, AUDIO_SETINFO, &info) < 0)
		err(EXIT_FAILURE, "AUDIO_SETINFO");

	if (info.blocksize != blocksize)
		errx(EXIT_FAILURE, "blocksize = %d", info.blocksize);

	audio_buffer = malloc(blocksize);
	if (audio_buffer == NULL)
		err(EXIT_FAILURE, "malloc");

	event_set(&audata_ev, SIGIO, EV_SIGNAL | EV_PERSIST, read_audio, NULL);
	if (event_add(&audata_ev, NULL) < 0)
		err(EXIT_FAILURE, "SIGIO");

	printf("Audio Connection established\n");
}

/*
 * Receive Audio packets from SCO socket and copy to Audio Device
 */
void
recv_sco(int fd, short ev, void *arg)
{
	char buf[256];
	ssize_t len;

	for (;;) {
		len = recv(sc, buf, sizeof(buf), 0);
		if (len <= 0) {
			if (errno != EAGAIN)
				close_audio();

			break;
		}

		write(au, buf, len);
	}
}

/*
 * Receive Audio data from audio device, and copy to SCO socket
 */
void
read_audio(int fd, short ev, void *arg)
{
	char *p;
	ssize_t len;

	while (au != -1) {
		len = read(au, audio_buffer, blocksize);
		if (len <= 0) {
			if (errno != EAGAIN)
				close_audio();

			break;
		}

		p = audio_buffer;
		while (len > 0) {
			if (send(sc, p, mtu, 0) != mtu)
				break;

			len -= mtu;
			p += mtu;
		}
	}
}

/*
 * Shut down audio connection
 */
void
close_audio(void)
{

	ioctl(au, AUDIO_FLUSH, NULL);
	event_del(&scdata_ev);
	event_del(&audata_ev);
	close(sc);
	close(au);
	free(audio_buffer);
	sc = au = -1;
	printf("Audio Connection released\n");

	// XXX do something, change state?
}

/*
 * Initialise RFCOMM server socket
 */
int
listen_rfcomm(void)
{
	static struct event ev;
	struct sockaddr_bt addr;
	sdp_hf_profile_t hf;
	void *ss;
	int fd;

	fd = socket(PF_BLUETOOTH, SOCK_STREAM, BTPROTO_RFCOMM);
	if (fd < 0)
		return -1;

	memset(&addr, 0, sizeof(addr));
	addr.bt_len = sizeof(addr);
	addr.bt_family = AF_BLUETOOTH;
	bdaddr_copy(&addr.bt_bdaddr, &laddr);
	addr.bt_channel = server_channel;

	if (bind(fd, (struct sockaddr *)&addr, sizeof(addr)) < 0)
		return -1;

	if (listen(fd, 1) < 0)
		return -1;

	event_set(&ev, fd, EV_READ | EV_PERSIST, accept_rfcomm, NULL);
	if (event_add(&ev, NULL) < 0)
		return -1;

	/* Register as Handsfree */
	memset(&hf, 0, sizeof(hf));
	hf.server_channel = server_channel;
	hf.supported_features = 0x0000;

	ss = sdp_open_local(NULL);
	if (ss == NULL || (errno = sdp_error(ss)))
		return -1;

	if (sdp_register_service(ss,
			SDP_SERVICE_CLASS_HANDSFREE,
			&laddr, (uint8_t *)&hf, sizeof(hf), NULL) != 0) {
		errno = sdp_error(ss);
		sdp_close(ss);
		return -1;
	}

	return 0;
}

/*
 * Incoming connection from the Audio Gateway.
 */
void
accept_rfcomm(int fd, short ev, void *arg)
{
	struct sockaddr_bt addr;
	socklen_t len;
	int s;

	memset(&addr, 0, sizeof(addr));

	len = sizeof(addr);
	s = accept(fd, (struct sockaddr *)&addr, &len);
	if (s < 0)
		return;

	if (rf >= 0
	    || addr.bt_len != sizeof(addr)
	    || addr.bt_family != AF_BLUETOOTH
	    || !bdaddr_same(&addr.bt_bdaddr, &raddr)) {
		close(s);
		return;
	}

	rf = s;
	state = START;
	if (establish_rfcomm() < 0)
		err(EXIT_FAILURE, "establish_rfcomm");
}

/*
 * Open RFCOMM link. This is a self serving routine that triggers
 * itself when it fails.
 */
void
open_rfcomm(int s, short ev, void *arg)
{
	struct sockaddr_bt addr;
	static struct timeval tv = { RETRY_INTERVAL, 0 };

	if (rf != -1)
		return;

	rf = socket(PF_BLUETOOTH, SOCK_STREAM, BTPROTO_RFCOMM);
	if (rf < 0)
		return;

	memset(&addr, 0, sizeof(addr));
	addr.bt_len = sizeof(addr);
	addr.bt_family = AF_BLUETOOTH;
	bdaddr_copy(&addr.bt_bdaddr, &laddr);

	if (bind(rf, (struct sockaddr *)&addr, sizeof(addr)) < 0)
		return;

	printf("Connecting.. ");
	fflush(stdout);

	bdaddr_copy(&addr.bt_bdaddr, &raddr);
	addr.bt_channel = gateway_channel;

	if (connect(rf, (struct sockaddr *)&addr, sizeof(addr)) < 0) {
		close(rf);
		rf = -1;
		evtimer_add(&timer_ev, &tv);
		printf("failed, retry in %ds\n", RETRY_INTERVAL);
		return;
	}

	printf("ok\n");

	state = START;
	establish_rfcomm();
}

/*
 * Establish Service Level connection on RFCOMM link, this will
 * be called after each received OK.
 */
int
establish_rfcomm(void)
{

	switch (state) {
	case START:
		event_set(&rfdata_ev, rf, EV_READ | EV_PERSIST, recv_rfcomm, NULL);
		if (event_add(&rfdata_ev, NULL) < 0)
			return -1;

		state = BRSF_SET;
		/*
		 *	Bit	Feature
		 *	0	EC and/or NR function
		 *	1	Call waiting and 3-way calling
		 *	2	CLI presentation capability
		 *	3	Voice Recognition activation
		 *	4	Remote Volume Control
		 *	5	Enhanced call status
		 *	6	Enhanced call control
		 *	7-31	Reserved
		 */
		send_rfcomm("+BRSF=%u", (1 << 4) | (1 <<2));
		break;

	case BRSF_SET:
		state = CIND_TEST;
		send_rfcomm("+CIND=?");
		break;

	case CIND_TEST:
		state = CIND_READ;
		send_rfcomm("+CIND?");
		break;

	case CIND_READ:
		state = CMER_SET;
		send_rfcomm("+CMER=3,0,0,1");
		break;

	case CMER_SET:
		state = CLIP_SET;
		send_rfcomm("+CLIP=1");
		break;

	case CLIP_SET:
		state = SERVICE;
		printf("Service Level established\n");
		break;

	default:
		break;
	}

	return 0;
}

/*
 * Send a message to the Audio Gateway in appropriate wrapping.
 */
int
send_rfcomm(const char *msg, ...)
{
	char buf[256], fmt[256];
	va_list ap;
	int len;

	if (rf == -1)
		return 0;

	va_start(ap, msg);

	if (verbose) {
		snprintf(fmt, sizeof(fmt), "< AT%s\n", msg);
		vprintf(fmt, ap);
	}

	snprintf(fmt, sizeof(fmt), "AT%s\r", msg);
	vsnprintf(buf, sizeof(buf), fmt, ap);
	len = send(rf, buf, strlen(buf), 0);

	va_end(ap);
	return len;
}

/*
 * Receive a message from the Audio Gateway
 */
void
recv_rfcomm(int fd, short ev, void *arg)
{
	struct result_code *rc;
	char buf[256], *code, *next;
	size_t len;

	/* XXX what if buf is not large enough? */

	len = recv(rf, buf, sizeof(buf), 0);
	if (len < 0) {
		event_del(&rfdata_ev);
		close(rf);
		rf = -1;
		printf("Service Connection lost\n");
		open_rfcomm(0, 0, NULL);
		return;
	}

	/*
	 * Result Codes should be <cr><lf><code><cr><lf>, and we
	 * may get several stacked up at once.
	 */

	next = buf;

	for (;;) {
		code = next;
		if (len < 2 || (code[0] != '\r' && code[1] != '\n'))
			break;

		code += 2;
		len -= 2;

		next = code;
		while (len > 0 && next[0] != '\r')
			next++, len--;

		if (len < 2 || next[1] != '\n')
			break;

		next[0] = '\0';
		next += 2;
		len -= 2;

		if (verbose)
			printf("> %.*s\n", next - code, code);

		for (rc = result_codes ; rc->code != NULL ; rc++) {
			if (strncmp(code, rc->code, rc->strlen) == 0) {
				(*rc->handler)(code + rc->strlen);
				break;
			}
		}
	}
}

/*
 * Error
 */
void
rc_error(char *arg)
{

	switch (state) {
	case BRSF_SET:
		/*
		 * Older devices might not know BRSF command.
		 * (For these we should do a SDP query)
		 */
		establish_rfcomm();
		break;

	case CIND_TEST:
	case CIND_READ:
	case CMER_SET:
	case CLIP_SET:
	case SERVICE:
	default:
		break;
	}
}

/*
 * Ok
 */
void
rc_ok(char *arg)
{

	/*
	 * continue towards service level..
	 */
	if (state != SERVICE)
		establish_rfcomm();
}

/*
 * Ring
 */
void
rc_ring(char *arg)
{

	printf("Ring\n");
}

/*
 * No Carrier
 */
void
rc_no_carrier(char *arg)
{
}

/*
 * Busy
 */
void
rc_busy(char *arg)
{
}

/*
 * No Answer
 */
void
rc_no_answer(char *arg)
{
}

/*
 * Delayed
 */
void
rc_delayed(char *arg)
{
}

/*
 * Blacklisted
 */
void
rc_blacklisted(char *arg)
{
}

/*
 * Bluetooth Retrieve Supported Features
 *
 * +BRSF: <AG supported features bitmap>
 *
 *	Bit	Feature
 *	0	Three-way calling
 *	1	EC and/or NR function
 *	2	Voice recognition function
 *	3	In-band ring tone capability
 *	4	Attach a number to a voice tag
 *	5	Ability to reject a call
 *	6	Enhanced call status
 *	7	Enhanced call control
 *	8	Extended Error Result Codes
 *	9-31	Reserved
 */
void
rc_brsf(char *arg)
{
	uint16_t features;

	features = atoi(arg);
	printf("Features: 0x%4.4x\n", features);
}

/*
 * Bluetooth Setting of In-band Ring tone
 */
void
rc_bsir(char *arg)
{
}

/*
 * Bluetooth Response and Hold Feature
 */
void
rc_btrh(char *arg)
{
}

/*
 * Bluetooth Voice Recognition Activation
 */
void
rc_bvra(char *arg)
{
}

/*
 * Call Waiting notification
 */
void
rc_ccwa(char *arg)
{
}

/*
 * Indicator Events reporting
 */
void
rc_ciev(char *arg)
{
}

/*
 * Current phone indicators
 */
void
rc_cind(char *arg)
{

	switch (state) {
	case CIND_TEST: /* (<descr>,(<ind-list>))[,(<descr>,(<ind-list>))[,...]] */
		break;

	case CIND_READ:	/* <ind>[,<ind>[,...]] */
		break;

	default:
		break;
	}
}

/*
 * List Current Calls
 */
void
rc_clcc(char *arg)
{
}

/*
 * Communications Mobile Equipment Error
 */
void
rc_cme_error(char *arg)
{
}

/*
 * Calling Line Identification
 */
void
rc_clip(char *arg)
{
}

/*
 * Subscriber Number Information
 */
void
rc_cnum(char *arg)
{
}

/*
 * Gain of Microphone
 */
void
rc_vgm(char *arg)
{
}

/*
 * Gain of Speaker
 */
void
rc_vgs(char *arg)
{
}

/*
 * Query remote device to find the Headset channel
 */
int
query_hfp_channel(void)
{
	uint16_t	service;
	uint32_t	attr;
	sdp_attr_t	value;
	uint8_t		buffer[512];
	void		*ss;
	uint8_t		*ptr, *end;
	int32_t		type, len, uuid;

	value.flags = SDP_ATTR_INVALID;
	value.attr = 0;
	value.vlen = sizeof(buffer);
	value.value = buffer;

	service = SDP_SERVICE_CLASS_HANDSFREE_AUDIO_GATEWAY;

	attr = SDP_ATTR_RANGE(SDP_ATTR_PROTOCOL_DESCRIPTOR_LIST,
			      SDP_ATTR_PROTOCOL_DESCRIPTOR_LIST);

	ss = sdp_open(&laddr, &raddr);
	if (ss == NULL || (errno = sdp_error(ss)) != 0)
		return 0;
		

	if (sdp_search(ss, 1, &service, 1, &attr, 1, &value) != 0) {
		errno = sdp_error(ss);
		return 0;
	}

	sdp_close(ss);

	if (value.flags != SDP_ATTR_OK
	    || value.attr != SDP_ATTR_PROTOCOL_DESCRIPTOR_LIST)
		return 0;

	/*
	 * Parse protocol descriptor list for the RFCOMM channel
	 *
	 * seq8 len8				2
	 *	seq8 len8			2
	 *		uuid16 value16		3	L2CAP
	 *	seq8 len8			2
	 *		uuid16 value16		3	RFCOMM
	 *		uint8 value8		2	channel
	 *				      ===
	 *				       14
	 */

	ptr = value.value;
	end = value.value + value.vlen;

	if (end - ptr < 14)
		return 0;

	SDP_GET8(type, ptr);
	switch (type) {
	case SDP_DATA_SEQ8:
		SDP_GET8(len, ptr);
		break;

	case SDP_DATA_SEQ16:
		SDP_GET16(len, ptr);
		break;

	case SDP_DATA_SEQ32:
		SDP_GET32(len, ptr);
		break;

	default:
		return 0;
	}
	if (ptr + len > end)
		return 0;

	/* Protocol */
	SDP_GET8(type, ptr);
	switch (type) {
	case SDP_DATA_SEQ8:
		SDP_GET8(len, ptr);
		break;

	case SDP_DATA_SEQ16:
		SDP_GET16(len, ptr);
		break;

	case SDP_DATA_SEQ32:
		SDP_GET32(len, ptr);
		break;

	default:
		return 0;
	}
	if (ptr + len > end)
		return 0;

	/* UUID */
	if (ptr + 3 > end)
		return 0;
	SDP_GET8(type, ptr);
	switch (type) {
	case SDP_DATA_UUID16:
		SDP_GET16(uuid, ptr);
		if (uuid != SDP_UUID_PROTOCOL_L2CAP)
			return 0;
		break;

	default:
		return 0;
	}

	/* Protocol */
	SDP_GET8(type, ptr);
	switch (type) {
	case SDP_DATA_SEQ8:
		SDP_GET8(len, ptr);
		break;

	case SDP_DATA_SEQ16:
		SDP_GET16(len, ptr);
		break;

	case SDP_DATA_SEQ32:
		SDP_GET32(len, ptr);
		break;

	default:
		return 0;
	}
	if (ptr + len > end)
		return 0;

	/* UUID */
	if (ptr + 3 > end)
		return 0;
	SDP_GET8(type, ptr);
	switch (type) {
	case SDP_DATA_UUID16:
		SDP_GET16(uuid, ptr);
		if (uuid != SDP_UUID_PROTOCOL_RFCOMM)
			return 0;
		break;

	default:
		return 0;
	}

	/* channel */
	if (ptr + 2 > end)
		return 0;

	SDP_GET8(type, ptr);
	if (type != SDP_DATA_UINT8)
		return 0;

	SDP_GET8(gateway_channel, ptr);
	printf("Handsfree channel: %d\n", gateway_channel);
	return 1;
}
