/* Bluefish HTML Editor
 * spell.c - Check Spelling
 *
 * Copyright (C)2000 Pablo De Napoli (for this module)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Note: some code has been taken from Lyx (wich is also covered by the
 * GNU Public Licence).
*/


#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif


#ifdef WITH_SPC

/* for the select function */
#define _GNU_SOURCE

#include "intl.h"
#include <glib.h>
#include <gtk/gtk.h>

#include "spell.h"
#include "bluefish.h"
#include "debug.h"
#include "gtk_easy.h"

#include <unistd.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <ctype.h>

#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif

#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif

#define CURRENT_DOCUMENT_TEXT main_v->current_document->textbox



/* Spellchecker status */
enum {
	ISP_OK = 1,
	ISP_ROOT,
	ISP_COMPOUNDWORD,
	ISP_UNKNOWN,
	ISP_MISSED,
	ISP_IGNORE
};

static FILE *in, *out;  /* streams to communicate with ispell */
pid_t isp_pid = -1; /* pid for the `ispell' process */

static int isp_fd;

/* void sigchldhandler(int sig); */
void sigchldhandler(pid_t pid, int *status);

/* extern void sigchldchecker(int sig); */
extern void sigchldchecker(pid_t pid, int *status);

typedef  struct  {
	gint flag;
	gint count;
	gchar **misses;
	} isp_result;

gint spc_message;

/*  flag to know if we reach the end of text */
static gboolean end_of_text;

static gchar get_next_char()
/* FIXME: a more eficient implementation? */
/* use an HTML parser like gnome-xml ? */
{
  gint current_position;
  gchar c ;
  current_position = gtk_text_get_point(GTK_TEXT(CURRENT_DOCUMENT_TEXT));
  c = GTK_TEXT_INDEX(GTK_TEXT(CURRENT_DOCUMENT_TEXT) ,current_position);
  /* macro to get a single character */
  if (current_position < gtk_text_get_length(GTK_TEXT(CURRENT_DOCUMENT_TEXT)))
  {
   current_position++;
   gtk_text_set_point(GTK_TEXT(CURRENT_DOCUMENT_TEXT),current_position);
   end_of_text = 0;
  }
  else
   end_of_text = 1;
  DEBUG_MSG("next_char ='%c'\n",c);
  return (c);
 }

typedef struct 
{
 gchar* text;
 gint begin;
 gint end;
} Tword;

inline void free_word (Tword* word)
{
 DEBUG_MSG("Free word \" %s \" \n",word->text);
 g_free(word->text);
 g_free(word);
}

static Tword* get_next_word(void)
{
 Tword* next_word;
 gchar c;
 DEBUG_MSG("getting next word\n");
 next_word = (Tword*) g_malloc(sizeof(Tword));
 if (next_word == NULL)
   {
    DEBUG_MSG("out of memory in get_next_word!");
    return next_word;
   };
 next_word->begin = gtk_text_get_point(GTK_TEXT(CURRENT_DOCUMENT_TEXT));
 while (((c=get_next_char())!=' ') && (!end_of_text) && (c!='<') && (c!='\n'));
 /* at the end of text , do not substract 1 */
 next_word->end = gtk_text_get_point(GTK_TEXT(CURRENT_DOCUMENT_TEXT))-(!end_of_text);
 next_word->text = gtk_editable_get_chars(GTK_EDITABLE(CURRENT_DOCUMENT_TEXT),next_word->begin,next_word->end);
 DEBUG_MSG("next word= \"%s\" \n",next_word-> text);
 /* ignore html tags */
 if (c=='<')
 {
   DEBUG_MSG("ignoring html tag\n");
   while (((c=get_next_char())!='>') && (!end_of_text));
 }
 return next_word;
}


static
void create_ispell_pipe()
{
	static char o_buf[BUFSIZ];  
	int pipein[2], pipeout[2];
	char * argv[14];
	int argc;
	int i;
	
	gchar buf[2048];
        fd_set infds;
	struct timeval tv; 
	int retval = 0;

        DEBUG_MSG ("Creating ispell pipe ...");
        
	isp_pid = -1;

	if(pipe(pipein)==-1 || pipe(pipeout)==-1) {
		g_printerr("Bluefish: Can't create pipe for spellchecker!");
		return;
	}

	if ((out = fdopen(pipein[1], "w"))==NULL) {
		g_printerr("Bluefish: Can't create stream for pipe for spellchecker!");
		return;
	}

	if ((in = fdopen(pipeout[0], "r"))==NULL) {
		g_printerr("Bluefish: Can't create stream for pipe for spellchecker!");
		return;
	}

	setvbuf(out, o_buf, _IOLBF, BUFSIZ);

	isp_fd = pipeout[0];

	isp_pid = fork();

	if(isp_pid==-1) {
		g_printerr("Bluefish: Can't create child process for spellchecker!");
		return;
	}

	if(isp_pid==0) {        
		/* child process */
		
		DEBUG_MSG ("spell-checker child process \n");
		
		dup2(pipein[0], STDIN_FILENO);
		dup2(pipeout[1], STDOUT_FILENO);
		close(pipein[0]);
		close(pipein[1]);
		close(pipeout[0]);
		close(pipeout[1]);

		argc = 0;
		argv[argc++] = main_v->props.cfg_spc_cline ;
		argv[argc++] = g_strdup("-a"); // "Pipe" mode

		if (main_v -> props.cfg_spc_lang != "default") {
			argv[argc++] = g_strdup("-d"); // Dictionary file
			argv[argc++] = main_v -> props.cfg_spc_lang;
		}

		if (main_v-> props.spc_accept_compound)
			/* Consider run-together words as legal compounds */
			argv[argc++] = g_strdup("-C"); 
		else
			/* Report run-together words with
			  missing blanks as errors */
			argv[argc++] = g_strdup("-B"); 

		if (main_v->props.spc_use_esc_chars) {
			/* Specify additional characters that
			   can be part of a word */
			argv[argc++] = g_strdup("-w"); 
			/* Put the escape chars in ""s */
			argv[argc++] = g_strconcat("\"",main_v->props.spc_esc_chars
			                ,"\"");
		}
		if (main_v->props.spc_use_pers_dict) {
			/* Specify an alternate personal dictionary */
			argv[argc++] = g_strdup("-p"); 
			argv[argc++] = main_v->props.spc_pers_dict;
		}
		/*
		if (spc_use_input_encoding &&
        	    current_view->currentBuffer()->params.inputenc != "default") {
			argv[argc++] = g_strdup("-T"); // Input encoding
			argv[argc++] = current_view->currentBuffer()->params.inputenc.copy();
		}
		*/

		argv[argc++] = NULL;

		execvp(main_v->props.cfg_spc_cline , (char * const *) argv);
		
		
		// free the memory used 
		for (i=0; i < argc -1; i++)
			g_free(argv[i]);
		
		g_printerr("Bluefish: Failed to start ispell!\n");
		_exit(0);
	}

	/* Parent process: Read ispells identification message 
	   Hmm...what are we using this id msg for? Nothing? (Lgb)
	   Actually I used it to tell if it's truly Ispell or if it's
	   aspell -- (kevinatk@home.com) */
	
#ifdef WITH_WARNINGS
#warning verify that this works.
#endif
	
	FD_ZERO(&infds);
	FD_SET(pipeout[0], &infds);
	tv.tv_sec = 15; /*  fifteen second timeout. Probably too much,
			 but it can't really hurt. */
	tv.tv_usec = 0;

	/* Configure provides us with macros which are supposed to do
	   the right typecast. */
	   
	retval = select(( SELECT_TYPE_ARG1 ) (pipeout[0]+1), 
			SELECT_TYPE_ARG234 (&infds), 
			0, 
			0, 
			SELECT_TYPE_ARG5 (&tv));

	if (retval > 0) {
		/* Ok, do the reading. We don't have to FD_ISSET since
		   there is only one fd in infds. */
		fgets(buf, 2048, in);
      DEBUG_MSG ("Ispell pipe created\n");
      g_print ("%s\n",buf);
		
	} else if (retval == 0) {
		/* timeout. Give nice message to user. */
		g_printerr("Ispell read timed out, what now?\n");
#ifdef WITH_WARNINGS
#warning Is this the correct thing to do?
#endif
		isp_pid = -1;
		close(pipeout[0]); close(pipeout[1]);
		close(pipein[0]); close(pipein[1]);
		isp_fd = -1;
	} else {
		/* Select returned error */
		g_printerr("Select on ispell returned error, what now?\n");
	}
}


static inline void ispell_terse_mode(void)
{
	fputs("!\n", out); /* Set terse mode (silently accept correct words) */
}


static inline void ispell_insert_word(char const *word)
{
	fputc('*', out); /* Insert word in personal dictionary */
	fputs(word, out);
	fputc('\n', out);
}


static
inline void ispell_accept_word(char const *word)
{
	fputc('@', out); /* Accept in this session */
	fputs(word, out);
	fputc('\n', out);
}

static inline void ispell_terminate(void)
{

	fputs("#\n", out); /* Save personal dictionary */ 
	fflush(out);
	fclose(out);
}

/* Send word to ispell and get reply */
static isp_result * ispell_check_word (gchar *word)
{
	isp_result *result;
	gchar buf[1024], *p, *np;
   gchar *nb;
	gint count;

   #ifdef DEBUG
   gint i;
   #endif

   /* Coment from the original author:
   I think we have to check if ispell is still alive here */
	
   if (isp_pid == -1) return (isp_result *) NULL;

   DEBUG_MSG("Sending word \"%s\" to ispell\n",word);
	fputs(word, out);
	fputc('\n', out);

   fgets(buf,1024,in);

   DEBUG_MSG("Message from ispell=\"%s\"\n",buf);

	result = g_malloc(sizeof(isp_result));

	switch (*buf) {
	case '*': /* Word found */
		result->flag = ISP_OK;
                DEBUG_MSG("Ispell: Word found\n");
		break;
	case '+': /* Word found through affix removal */
		result->flag = ISP_ROOT;
                DEBUG_MSG("Ispell: Word found through affix removal\n");
		break;
	case '-': /* Word found through compound formation */
		result->flag = ISP_COMPOUNDWORD;
                DEBUG_MSG("Ispell: Word found through compound formation\n");
		break;
	case '\n': /* Number or when in terse mode: no problems */
		result->flag = ISP_IGNORE;
                DEBUG_MSG("Ispell: Ignore Word\n");
		break;
	case '#': /* Not found, no near misses and guesses */
		result->flag = ISP_UNKNOWN;
                DEBUG_MSG("Ispell: Unknown Word\n");
		break;
	case '?': /* Not found, and no near misses, but guesses (guesses are ignored) */
	case '&': /* Not found, but we have near misses */
	{
		result->flag = ISP_MISSED;
      DEBUG_MSG("Ispell: Near Misses\n");
      /* FIXME: why duplicate buffer ?*/
      nb = g_strdup(buf);
		p = strpbrk(nb+2, " ");
		sscanf(p, "%d", &count); /* Get near misses count */
		result->count = count;
		if (count)
                {
		  p = strpbrk(nb, ":");
		  p +=2;

        /* remove the last '\n' from ispell output */
        DEBUG_MSG("Removing end-of-line \n");
        np = p;
        while (*np!='\0')np++;
        np--;
        *np = '\0';

        DEBUG_MSG  ("misses list \"%s\" \n",p);

		  result->misses = g_strsplit(p,", ",count);
        g_free(nb);
        #ifdef DEBUG
          for(i=0;i<count;i++)
           DEBUG_MSG("near_miss = \"%s\" \n", result-> misses[i]);
        #endif
		}


		break;
	}
	default: /* This shouldn't happend, but you know Murphy */
		result->flag = ISP_UNKNOWN;
                DEBUG_MSG("Ispell: Unknown\n");
	}

	*buf = 0;
	if (result->flag!=ISP_IGNORE) {
		while (*buf!='\n') fgets(buf, 255, in); /* wait for ispell to finish */
	}
	return result;
}


static inline void select_word (Tword* word)
{
  gtk_editable_select_region (GTK_EDITABLE(CURRENT_DOCUMENT_TEXT),word->begin,word->end);

}

/*for gtk_text_insert*/
#define ALL_THE_STRING -1

static void replace_word_with (Tword *word,gchar* replace_with)
{
  DEBUG_MSG("Deleting word \"%s\" \n",word->text);
  gtk_editable_delete_selection(GTK_EDITABLE(CURRENT_DOCUMENT_TEXT));
  DEBUG_MSG("inserting replace text =\"%s\" \n",replace_with);
  gtk_editable_insert_text(GTK_EDITABLE(CURRENT_DOCUMENT_TEXT), replace_with,strlen(replace_with),&(word->begin));
  /* DEBUG_MSG("setting new point\n");
  gtk_text_set_point(GTK_TEXT(CURRENT_DOCUMENT_TEXT), word->begin + strlen(replace_with)-1); */
}

static void correct_word(Tword* word)
/* ask the user how to correct a word, and do it */
{
 gboolean word_corrected = 0;
 select_word(word);
 do {
     spc_message = SPC_NONE;

     while (gtk_events_pending())
           gtk_main_iteration();
     switch (spc_message)
     {
     case SPC_INSERT:      ispell_insert_word(word->text);
                           word_corrected = 1;
                           break;
     case SPC_ACCEPT:      ispell_accept_word(word->text);
                           word_corrected = 1;
			                  break;
     case SPC_IGNORE:      word_corrected = 1;
                           break;
     case SPC_REPLACE:     replace_word_with (word,gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(spc_gui.replace_combo)->entry)));
                           word_corrected = 1;
                           break;
     /* FIXME: case SPC_STOP */

     }
 } while (!word_corrected);
 /* gtk_editable_delete_selection (GTK_EDITABLE(CURRENT_DOCUMENT_TEXT));*/
 DEBUG_MSG ("Word corrected\n");
};


static void check_word (Tword* word)
{
 isp_result *result;
 gint i;
 GList* near_misses_list = NULL;
 GList* tmp = NULL;
 spc_message = SPC_NONE;
 while (gtk_events_pending())
        gtk_main_iteration();
 DEBUG_MSG ("checking word \"%s\" \n",word->text);
 result = ispell_check_word (word->text);
 switch (result->flag) {
   case ISP_MISSED:  /* creates near misses list */
                     DEBUG_MSG("Creating misses list\n");
                     for (i=0; i<result->count; i++)
                     /* FIXME: use g_list_insert_sorted instead ? */
                     {
                         DEBUG_MSG("Adding \"%s\" to the near_misses_list\n",result->misses[i]);
                         near_misses_list = g_list_append(near_misses_list,result->misses[i]);
                     };
                     gtk_combo_set_popdown_strings (GTK_COMBO(spc_gui.replace_combo),near_misses_list);
   case ISP_UNKNOWN: gtk_entry_set_text (GTK_ENTRY(spc_gui.word_entry),word->text);
                     gtk_entry_set_text (GTK_ENTRY(GTK_COMBO(spc_gui.replace_combo)->entry),word->text);
                     correct_word(word);
   }; /* end switch */
  if (result->flag ==ISP_MISSED)
                                {
                                  DEBUG_MSG("free near_misses_lists ...\n");
                                  tmp = near_misses_list;
                                  while (tmp != NULL) {
                                        g_free (tmp->data);
                                        tmp = g_list_next (tmp);
                                        };
                                  DEBUG_MSG("free near_misses_list elements done\n");
                                  g_list_free(near_misses_list);
                                  DEBUG_MSG("free near_misses_list done\n");
                                 }; /*end if */
  g_free(result);
  DEBUG_MSG("word checked\n");
} /* end check_word */

/* update slider if and only if value has changed
		newvalue = int(100.0*newval);
		if(newvalue!=oldval) {
			oldval = newvalue;
			fl_set_slider_value(fd_form_spell_check->slider, oldval);
		} */

/* Perform an ispell session */
gboolean run_spell_checker (const gchar* lang)
{
	Tword *word;
	gint oldval; /* , newvalue; */
	gfloat newval;
   unsigned int word_count = 0;
   oldval = 0;  /* used for updating slider only when needed */
	newval = 0.0;

   gtk_text_freeze(GTK_TEXT(CURRENT_DOCUMENT_TEXT));
    
     /* create ispell process */
	
    create_ispell_pipe();
    gtk_text_set_point(GTK_TEXT(CURRENT_DOCUMENT_TEXT),0);

    if (isp_pid == -1) {
		                 error_dialog(_("Bluefish Spell Checker Error"),
			              _("\n\n"
			              "The ispell-process has died for some reason. *One* possible reason\n"
			              "could be that you do not have a dictionary file\n"
			              "for the language of this document installed.\n"
			              "Check /usr/lib/ispell or set another\n"
			              "dictionary in the Spellchecker Options menu."));
		                 fclose(out);
		                 return 1;
                     }; /*end if*/

	/* Put ispell in terse mode to improve speed */
	/* ispell_terse_mode();*/
	do
   {
      word = get_next_word();
      word_count++;
      check_word(word);
      /* FIXME: update_slider() */
      free_word (word);
      if(spc_message==SPC_DONE)
                      {
		                ispell_terminate();
                      DEBUG_MSG("Ispell died\n");
                      break;
                      };
   } while (!end_of_text);
   DEBUG_MSG("Spell checking done\n");

  gtk_text_thaw(GTK_TEXT(CURRENT_DOCUMENT_TEXT));
  destroy_spc_window();

return (spc_message != SPC_DONE);
} /* end run_spell_checker */

#endif
