/* Bluefish HTML Editor
 * interface.c - this file contains the UI code and some init code
 *
 * Copyright (C) 1998-1999 Olivier Sessink and Chris Mazuc
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/* Changes by Antti-Juhani Kaijanaho <gaia@iki.fi> on 1999-10-20  */
#include "default_include.h"

#include <unistd.h>
#include <string.h>
#include <stdlib.h>

#include <locale.h>

#include "bluefish.h"
#include "interface.h"
#include "init.h"
#include "undo.h"
#include "toolbars.h"
#include "callbacks.h"
#include "rpopup.h"
#include "document.h"
#include "menu.h"
#include "gtk_easy.h"
#include "network.h"

#include "pixmaps/bluefish_icon.xpm"
#include "pixmaps/bluefish.xpm"

enum {
	TARGET_STRING,
};



GtkWidget *cust_handle_box, *handle_box, *html_handle_box;

/********************************************************************/



/********************************************************************/


/* local "global" variables */
static gint context_id = 0;

static GtkTargetEntry target_table[] = {
	{"STRING", 0, TARGET_STRING},
#define NUM_TARGETS (sizeof(target_table)/sizeof(GtkTargetEntry))
};


/* 
 * start of the functions
 * 
 */

gint statusbar_remove(gpointer message_id)
{
	gtk_statusbar_remove(GTK_STATUSBAR(main_v->statusbar), context_id, GPOINTER_TO_INT(message_id));
	return 0;
}


void statusbar_message(gpointer message, gint time)
{

	gint count;

	count = gtk_statusbar_push(GTK_STATUSBAR(main_v->statusbar), context_id, (gchar *) message);
	gtk_timeout_add(time, statusbar_remove, GINT_TO_POINTER(count));
	flush_queue();
}

gint bf_statusbar_message(gpointer message)
{

	gint count;
	count = gtk_statusbar_push(GTK_STATUSBAR(main_v->statusbar), context_id, (gchar *) message);
	return count;
}


void switch_to_document(gpointer data)
{
	gint index;

	index = g_list_index(main_v->documentlist, data);
	DEBUG_MSG("switch_to_document, index=%d\n", index);
	gtk_notebook_set_page(GTK_NOTEBOOK(main_v->notebook), (index));
	notebook_changed();
}

void notebook_next_document_cb(GtkWidget *widget, gpointer data) {
	gtk_notebook_next_page(GTK_NOTEBOOK(main_v->notebook));
	notebook_changed();
}

void notebook_prev_document_cb(GtkWidget *widget, gpointer data) {
	gtk_notebook_prev_page(GTK_NOTEBOOK(main_v->notebook));
	notebook_changed();
}

void notebook_changed(void)
{
	gint cur;

	/* This one is called when you click on the notebook
	   it _should_ be called also when you use the keys to change the page */
	cur = (gint) gtk_notebook_current_page(GTK_NOTEBOOK(main_v->notebook));
	if ((main_v->last_notebook_page == cur) 
		&& (main_v->current_document != NULL)
		&& (main_v->current_document == g_list_nth_data(main_v->documentlist, cur))) {
		DEBUG_MSG("notebook_changed, it didn't change to a new document (cur=%d, current_document=%p)\n", cur, main_v->current_document);
		return;
	} else {
		DEBUG_MSG("notebook_changed, it did really change to a new document, cur=%d\n", cur);
		main_v->last_notebook_page = cur;
		main_v->current_document = NULL;
		main_v->current_document = g_list_nth_data(main_v->documentlist, cur);
		DEBUG_MSG("notebook_changed, finished, main_v->current_document=%p\n", main_v->current_document);
		if (main_v->current_document == NULL) {
#ifdef DEBUG
			DEBUG_MSG("notebook_changed, finished, main_v->current_document == NULL\n");
#endif
		} else {
			DEBUG_MSG("notebook_changed, grabbing document %p textbox=%p\n", main_v->current_document, main_v->current_document->textbox);
			gtk_widget_grab_focus(GTK_WIDGET(main_v->current_document->textbox));
			DEBUG_MSG("notebook_changed, setting up toggle item\n");
			setup_toggle_item(gtk_item_factory_from_widget(main_v->menubar), N_("/View/Highlight syntax"),
							  main_v->current_document->highlightstate);
		}
	}
}


void notebook_set_tab_pos(gchar * position)
{
	if (position) {
		if (strcmp(position, "left") == 0) {
			gtk_notebook_set_tab_pos(GTK_NOTEBOOK(main_v->notebook), GTK_POS_LEFT);
		} else if (strcmp(position, "right") == 0) {
			gtk_notebook_set_tab_pos(GTK_NOTEBOOK(main_v->notebook), GTK_POS_RIGHT);
		} else if (strcmp(position, "top") == 0) {
			gtk_notebook_set_tab_pos(GTK_NOTEBOOK(main_v->notebook), GTK_POS_TOP);
		} else {
			gtk_notebook_set_tab_pos(GTK_NOTEBOOK(main_v->notebook), GTK_POS_BOTTOM);
		}
	}
}


/* 
   Passes drag-data to file_open or open_web if action== GDK_ACTION_COPY,
   else it inserts an <a href=""> or <img src=""> according to the filename
   extension.
 */


static void on_drag_data_cb(GtkWidget * widget, GdkDragContext * context, gint x, gint y, GtkSelectionData * data, guint info, guint time,
							void *client)
{
	char *filename, *temp, *close_tag;
	int mode = 0, len;

	if ((data->length == 0) || (data->format != 8) || (info != TARGET_STRING)) {
		DEBUG_MSG("wrong drag-object\n");
		gtk_drag_finish(context, FALSE, TRUE, time);
	}
	DEBUG_MSG("item dragged onto window\n");

	len = data->length;
	filename = g_malloc(len + 6);
	close_tag = g_malloc(10);

	temp = g_malloc(len + 30);

	filename[len] = '\0';
	strncpy(filename, data->data, len);
	if ((filename[len - 1] == '\n') || (filename[len - 1] == '\r')) {
		filename[len - 1] = '\0';
	}
	if ((filename[len - 2] == '\n') || (filename[len - 2] == '\r')) {
		filename[len - 2] = '\0';
	}
	DEBUG_MSG("Dragged item-context was: '%s'\n", filename);

	strncpy(temp, filename, 7);
	temp[7] = '\0';
	if (context->action != GDK_ACTION_MOVE) {
		if (strcmp(temp, "http://") == 0) {
			DEBUG_MSG("started with 'http://',opening from web...\n");
			open_web(filename);
		} else {
			strncpy(temp, filename, 5);
			temp[5] = '\0';
			if (strcmp(temp, "file:") == 0) {
				temp = g_strndup(&filename[5], strlen(filename) - 5);
				strcpy(filename, temp);
			}
			file_open(filename);
		}
	} else {
		strcpy(temp, strrchr(filename, '.'));
		DEBUG_MSG(" suffix: %s\n", temp);
		if ((strcmp(temp, ".htm") == 0) || (strcmp(temp, ".html") == 0)) {
			DEBUG_MSG(" ends in .htm[l]\n");
			strcpy(temp, "<a href=\"");
			strcat(temp, filename);
			strcat(temp, "\">");
			strcpy(close_tag, "</a>");
		} else if ((strcmp(temp, ".gif") == 0) || (strcmp(temp, ".jpg") == 0) || (strcmp(temp, ".jpeg") == 0) || (strcmp(temp, ".png") == 0)) {

			DEBUG_MSG(" ends in .gif,.jpg,.jpeg,.png\n");
			strcpy(temp, "<img src=\"");
			strcat(temp, filename);
			strcat(temp, "\" border=\"0\" alt=\"\">");
			strcpy(close_tag, "");
		} else {
			DEBUG_MSG(" unknown file format\n");
			strcpy(temp, filename);
			strcpy(close_tag, "");
		}
		gtk_text_set_point(GTK_TEXT(main_v->current_document->textbox),
						   gtk_editable_get_position(GTK_EDITABLE(main_v->current_document->textbox)));
		insert_text(temp, close_tag);
	}
	gtk_drag_finish(context, TRUE, (mode == GDK_ACTION_COPY), time);
	g_free(filename);
	g_free(close_tag);
	g_free(temp);
}




static void make_main_window(void)
{
	GtkWidget *vbox;

	/* main window */
	DEBUG_MSG("make_main_window, started\n");
	main_v->main_window = window_full(CURRENT_VERSION_NAME, GTK_WIN_POS_CENTER, GTK_WINDOW_TOPLEVEL, 0, cya_later, NULL);
	DEBUG_MSG("make_main_window, main_v->main_window(%p)\n", main_v->main_window);

	DEBUG_MSG("make_main_window, gdk_screen_height()=%d, gdk_screen_width()=%d\n", gdk_screen_height(), gdk_screen_width());
	if ((main_v->props.main_window_h < gdk_screen_height()) && (main_v->props.main_window_w < gdk_screen_width())) {
		DEBUG_MSG("make_main_window, , set window height and width\n");
		gtk_window_set_policy(GTK_WINDOW(main_v->main_window), FALSE, FALSE, FALSE);
		gtk_widget_set_usize(GTK_WIDGET(main_v->main_window), main_v->props.main_window_w, main_v->props.main_window_h);
		gtk_window_set_policy(GTK_WINDOW(main_v->main_window), TRUE, TRUE, FALSE);
	}

	gtk_window_set_wmclass(GTK_WINDOW(main_v->main_window), "Bluefish", "bluefish");
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(main_v->main_window), vbox);
	gtk_widget_show(vbox);
	/* menu bar */
	get_main_menu(main_v->main_window, &main_v->menubar);
	gtk_box_pack_start(GTK_BOX(vbox), main_v->menubar, FALSE, TRUE, 0);
	gtk_widget_show(main_v->menubar);

	/* the pixmaps can't be created without a realise on the window, the
	   error Gdk-WARNING **: Creating pixmap from xpm with NULL window and 
	   colormap is removed by adding this line */
	gtk_widget_realize(main_v->main_window);

	/* drag n drop support */
	gtk_drag_dest_set(main_v->main_window, GTK_DEST_DEFAULT_ALL, target_table, NUM_TARGETS, GDK_ACTION_COPY|GDK_ACTION_MOVE);
	gtk_signal_connect(GTK_OBJECT(main_v->main_window), "drag_data_received", GTK_SIGNAL_FUNC(on_drag_data_cb), NULL);


	/* pack the main toolbar handle box on the right place */
	handle_box = gtk_handle_box_new();
	gtk_box_pack_start(GTK_BOX(vbox), handle_box, FALSE, FALSE, 0);
	if (main_v->props.v_main_tb)
		show_main_toolbar_cb(NULL, NULL);

	/* pack the HTML toolbar handle box on the right place */
	html_handle_box = gtk_handle_box_new();
	gtk_box_pack_start(GTK_BOX(vbox), GTK_WIDGET(html_handle_box), FALSE, FALSE, 0);
	if (main_v->props.v_html_tb)
		show_html_toolbar_cb(NULL, NULL);

	/* pack the custom toolbar handle box on the right place */
	cust_handle_box = gtk_handle_box_new();
	gtk_box_pack_start(GTK_BOX(vbox), cust_handle_box, FALSE, FALSE, 0);
	if (main_v->props.v_custom_tb)
		show_cust_menubar_cb(NULL, NULL);

	/* notebook with the text widget in there */
	main_v->notebook = gtk_notebook_new();
	notebook_set_tab_pos(main_v->props.cfg_tab_pos);
	gtk_notebook_set_show_tabs(GTK_NOTEBOOK(main_v->notebook), TRUE);
	gtk_notebook_set_show_border(GTK_NOTEBOOK(main_v->notebook), TRUE);
	gtk_notebook_set_tab_border(GTK_NOTEBOOK(main_v->notebook), 1);

	/* We have to know when the notebook changes */
	gtk_signal_connect(GTK_OBJECT(main_v->notebook), "button_release_event", notebook_changed, NULL);
	gtk_signal_connect_after(GTK_OBJECT(main_v->notebook), "key_press_event", notebook_changed, NULL);

	gtk_notebook_set_page(GTK_NOTEBOOK(main_v->notebook), 0);
	gtk_box_pack_start(GTK_BOX(vbox), main_v->notebook, TRUE, TRUE, 0);
	gtk_notebook_set_scrollable(GTK_NOTEBOOK(main_v->notebook), TRUE);
	gtk_widget_show(main_v->notebook);

	/* status bar */
	main_v->statusbar = gtk_statusbar_new();
	gtk_box_pack_start(GTK_BOX(vbox), main_v->statusbar, FALSE, TRUE, 0);
	gtk_widget_show(main_v->statusbar);

	DEBUG_MSG("make_main_window, end\n");
}

void startup(void)
{
	GdkPixmap *icon = NULL;
#ifndef NOSPLASH
	GtkWidget *vbox, *splash_window;
	GtkWidget *wpixmap;
	GdkPixmap *pixm;
#endif

	/* splash screen */
#ifndef NOSPLASH
	splash_window = window_with_title(CURRENT_VERSION_NAME, GTK_WIN_POS_CENTER, GTK_WINDOW_DIALOG, 0);
	gtk_widget_realize(splash_window);
	gdk_window_set_decorations(GTK_WIDGET(splash_window)->window, 0);
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(splash_window), vbox);
	gtk_widget_show(vbox);
	gtk_widget_realize(splash_window);
	pixm = gdk_pixmap_create_from_xpm_d(splash_window->window, NULL, NULL, bluefish_xpm);
	wpixmap = gtk_pixmap_new(pixm, NULL);
	gdk_pixmap_unref(pixm);
	gtk_box_pack_start(GTK_BOX(vbox), wpixmap, FALSE, FALSE, 0);
	gtk_widget_show(wpixmap);
	gtk_widget_realize(wpixmap);
	gtk_widget_show(splash_window);
	flush_queue();
#endif
	check_files_links_directories();
	parse_bluefish_rcfile();
	make_main_window();
	gtk_widget_realize(main_v->main_window);
	flush_queue();
	/* make external commands and filters ready */
	make_external_menu_entries(1);
	make_external_menu_entries(2);
	context_id = gtk_statusbar_get_context_id(GTK_STATUSBAR(main_v->statusbar), "Bluefish");
	DEBUG_MSG("startup, main_window is ready\n");
	/* add icon to the window when in minimized state */
	icon = gdk_pixmap_create_from_xpm_d(main_v->main_window->window, NULL, NULL, (gchar **) bluefish_icon_xpm);
	gdk_window_set_icon(main_v->main_window->window, NULL, icon, NULL);
	gdk_window_set_icon_name(main_v->main_window->window, CURRENT_VERSION_NAME);
	DEBUG_MSG("startup, icon is there\n");
	load_menu_shortcuts();	
#ifndef NOSPLASH
	/* kill the splash window when the main window is going up */
	flush_queue();
	sleep(1);
	window_destroy(splash_window);
#endif
	gtk_widget_show(main_v->main_window);
	if (g_list_length(main_v->documentlist) < 1) {
		main_v->current_document = new_document();
	}
	gtk_window_set_policy(GTK_WINDOW(main_v->main_window), TRUE, TRUE, FALSE);
}

void show_about_window_cb(GtkWidget * widget, gpointer data)
{
	GtkWidget *wpixmap, *window, *vbox, *button;
	GdkPixmap *pixm;
	GtkStyle *style;

	DEBUG_MSG("about, started\n");
	window = window_with_title(_("About Bluefish"), GTK_WIN_POS_MOUSE, GTK_WINDOW_DIALOG, 10);

	style = gtk_style_new();
	style->bg->red = 65535;
	style->bg->blue = 65535;
	style->bg->green = 65535;
	gtk_widget_push_style(style);
	gtk_widget_set_style(GTK_WIDGET(window), style);
	gtk_widget_pop_style();

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_widget_realize(window);
	pixm = gdk_pixmap_create_from_xpm_d(window->window, NULL, NULL, bluefish_xpm);
	wpixmap = gtk_pixmap_new(pixm, NULL);
	gdk_pixmap_unref(pixm);
	gtk_box_pack_start(GTK_BOX(vbox), wpixmap, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), gtk_label_new(CURRENT_VERSION_NAME), FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox),
					   gtk_label_new(_
									 ("(C) Olivier Sessink\n\ndeveloped by the bluefish development team\nsee AUTHORS for more information\nhttp://bluefish.openoffice.nl/")),
					   FALSE, FALSE, 0);
	button = gtk_button_new_with_label(_("Dismiss"));
	gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 0);
	gtk_signal_connect_object(GTK_OBJECT(button), "clicked", gtk_widget_destroy, GTK_OBJECT(window));
	gtk_container_add(GTK_CONTAINER(window), vbox);
	gtk_widget_show_all(window);
	DEBUG_MSG("about, finished\n");
}
