/* Bluefish HTML Editor
 * (C) 1998-2000 Olivier Sessink
 *
 * highlight.c - syntax highlighting code
 * Copyright (C) 2000 Roland Steinbach, Gero Takke and Olivier Sessink
 * 
 * incorporates small pieces of code from David Vogler and Adrian Reber
 *
 * This is the code to do syntax-highlighting in the standard gtk_text widget.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include "default_include.h"

#include <glib.h>
#include <gdk/gdk.h>

#include <stdio.h>
#include <string.h>
#include <string.h>
#include <sys/types.h>  /* _before_ regex.h for freeBSD */
#include <regex.h>


#include <locale.h>

#include "bluefish.h"
#include "document.h"
#include "stringlist.h" /* edit stringlist */
#include "highlight.h"
#include "gtk_easy.h"
#include "undo.h"
#include "interface.h"

/* Maximal store-size for regular expressions...
 * If anybody figured out how to get multiple matches
 * in one regex, it should be increased ;-)
 */
#define NMAX 1

/***********************************************************************/

typedef struct {
	GdkColor color;
	regex_t reg_comp_pat;
} Thighlight_pattern;

/***********************************************************************/

void reset_syntax_highlighting_lcb(GtkWidget * widget, gpointer data)
{
	free_stringlist(main_v->props.syntax_configstrings);
	main_v->props.syntax_configstrings = NULL;

	main_v->props.syntax_configstrings = add_to_stringlist(main_v->props.syntax_configstrings, "#000080 <[^(!|%|?)][^>]*>");
	main_v->props.syntax_configstrings = add_to_stringlist(main_v->props.syntax_configstrings, "#ee0000 (<\\\?)[^?]*(\\\?>)");
	main_v->props.syntax_configstrings = add_to_stringlist(main_v->props.syntax_configstrings, "#ee0000 (<\\\?)[^>]*(\\\?>)");
	main_v->props.syntax_configstrings = add_to_stringlist(main_v->props.syntax_configstrings, "#bd0000 (<[/]*script[^>]*>)");
	main_v->props.syntax_configstrings =
		add_to_stringlist(main_v->props.syntax_configstrings, "#00AAAA (<[/]*table[^>]*>)|(<[/]*td[^>]*>)|(<[/]*tr[^>]*>)|(<[/]*th[^>]*>)");
	main_v->props.syntax_configstrings =
		add_to_stringlist(main_v->props.syntax_configstrings,
						  "#4080bf (<[/]*form[^>]*>)|(<input[^>]*>)|(<[/]*select[^>]*>)|(<[/]*textarea[^>]*>)|(<[/]*option[^>]*>)");
	main_v->props.syntax_configstrings = add_to_stringlist(main_v->props.syntax_configstrings, "#eead0e (<[/]*a[^>]*>)|(<[/]*img[^>]*>)");
	main_v->props.syntax_configstrings = add_to_stringlist(main_v->props.syntax_configstrings, "#969696 (<!--)[^>]*-->");
	main_v->props.syntax_configstrings = add_to_stringlist(main_v->props.syntax_configstrings, "#22AA00 \"[^\"]*\"");
}


/* checkreg
 *
 * returns a regmatch_t struct including the
 * first match from regex in htmlstring
 *
 */
static regmatch_t checkreg(char *htmlstring, regex_t * pregp)
{
	size_t siz = NMAX;
	regmatch_t pmatch[NMAX];
	gint i;

	if (!(i = regexec(pregp, htmlstring, siz, pmatch, 0)) == 0) {
		pmatch[0].rm_so = -1;
		pmatch[0].rm_eo = -1;
	}
	return pmatch[0];
}

/* hlregex
 * the "core"...
 * it highlights the given regex in the given color fore
 *
 */
#define CRAPPY_FIX

static void hlregex(GtkWidget * textbox, Thighlight_pattern * hipa, gchar * fulltext)
{
	regmatch_t tagmatch;
	gshort loop = 1;
	gchar *tmpbuf;
	gint pos = 0;
	gint maxlen;

	maxlen = strlen(fulltext);
	while (loop == 1) {
		tagmatch = checkreg(&fulltext[pos], &hipa->reg_comp_pat);

		if (tagmatch.rm_so == -1) {
			/* DEBUG_MSG ("EXIT LOOP (REGULAR)\n"); */
			loop = 0;
		} else {
#ifdef DEBUG1
			g_print
				("position in text is=%d, match start=%d, match end=%d, match length=%d\n",
				 pos, pos + tagmatch.rm_so, pos + tagmatch.rm_eo, tagmatch.rm_eo - tagmatch.rm_so);
#endif
#ifdef CRAPPY_FIX_OLD
#if GTK_GT_7 == 0
			gtk_text_thaw(GTK_TEXT(textbox));
#endif
#endif
			gtk_text_set_point(GTK_TEXT(textbox), pos + tagmatch.rm_so);
#ifdef CRAPPY_FIX_OLD
#if GTK_GT_7 == 0
			gtk_text_freeze(GTK_TEXT(textbox));
#endif
#endif
			gtk_text_forward_delete(GTK_TEXT(textbox)
									, tagmatch.rm_eo - tagmatch.rm_so);
			tmpbuf = g_strndup(&fulltext[pos + tagmatch.rm_so]
							   , tagmatch.rm_eo - tagmatch.rm_so);
			gtk_text_insert(GTK_TEXT(textbox), NULL, &hipa->color, NULL, tmpbuf, -1);
			g_free(tmpbuf);
			pos = pos + tagmatch.rm_eo;
			if (pos >= maxlen) {
				loop = 0;
			}
		}

	}
}

static void hlregex_line(GtkWidget * textbox, Thighlight_pattern * hipa, gint startol, gint endol, gchar * line)
{
	regmatch_t tagmatch;
	gshort loop = 1;
	gchar *tmpbuf;
	guint smallpos = 0;
	gint maxlen, buflen;

	maxlen = endol - startol;

	while (loop == 1) {
		tagmatch = checkreg(&line[smallpos], &hipa->reg_comp_pat);

		if (tagmatch.rm_so == -1) {
			loop = 0;
		} else {
			buflen = tagmatch.rm_eo - tagmatch.rm_so;
			gtk_text_set_point(GTK_TEXT(textbox), startol + smallpos + tagmatch.rm_so);
			gtk_text_forward_delete(GTK_TEXT(textbox) ,buflen);
			tmpbuf = g_strndup(&line[smallpos + tagmatch.rm_so]
							   ,buflen);
			gtk_text_insert(GTK_TEXT(textbox), NULL, &hipa->color, NULL, tmpbuf, buflen);

#ifdef CRAPPY_FIX
#if GTK_GT_7 == 0
			gtk_text_thaw(GTK_TEXT(textbox));
			gtk_text_freeze(GTK_TEXT(textbox));
#endif
#endif

			g_free(tmpbuf);
			smallpos = smallpos + tagmatch.rm_eo;
			if (smallpos >= maxlen) {
				loop = 0;
			}
		}
	}
}


static void start_highlighting(GtkWidget * textbox)
{
	GList *tmplist;
	gchar *fulltext;


	DEBUG_MSG("start_highlighting, started\n");
	fulltext = gtk_editable_get_chars(GTK_EDITABLE(textbox), 0, -1);

	gtk_text_freeze(GTK_TEXT(textbox));

	tmplist = g_list_first(main_v->syntax_structs);
	while (tmplist) {
		hlregex(textbox, (Thighlight_pattern *) tmplist->data, fulltext);
#ifdef CRAPPY_FIX
#if GTK_GT_7 == 0
		gtk_text_thaw(GTK_TEXT(textbox));
		gtk_text_freeze(GTK_TEXT(textbox));
#endif
#endif
		tmplist = g_list_next(tmplist);
	}
	g_free(fulltext);

	gtk_text_thaw(GTK_TEXT(textbox));

}

void line_highlighting(GtkWidget * textbox)
{
	GList *tmplist;
	gchar *line, curchar;
	guint pos,linelen;
	gint startol, endol, max;

	DEBUG_MSG("line_highlighting textbox=%p\n", textbox);
	gtk_text_freeze(GTK_TEXT(textbox));
	
	pos = gtk_editable_get_position(GTK_EDITABLE(textbox));
	max = gtk_text_get_length(GTK_TEXT(textbox));

	startol = pos;
	curchar = GTK_TEXT_INDEX(GTK_TEXT(textbox), startol);
	if (curchar == '\n') {
		startol--;
		curchar = GTK_TEXT_INDEX(GTK_TEXT(textbox), startol);
	}
	while ((curchar != '\n') && (startol != 0)){
		startol--;
		curchar = GTK_TEXT_INDEX(GTK_TEXT(textbox), startol);
	}
	endol = pos;
	curchar = GTK_TEXT_INDEX(GTK_TEXT(textbox), endol);
	while ((curchar != '\n') && (endol != max)) {
		endol++;
		curchar = GTK_TEXT_INDEX(GTK_TEXT(textbox), endol);
	}

	linelen = endol - startol;
	DEBUG_MSG("line_highlighting, startol=%d, endol=%d, linelen=%d\n", startol, endol, linelen);
	
	line = gtk_editable_get_chars(GTK_EDITABLE(textbox), startol, endol);
	if (!line) {
		g_free(line);
		gtk_text_thaw (GTK_TEXT(textbox));
		return;
	}

	gtk_text_set_point(GTK_TEXT(textbox), startol);
	gtk_text_forward_delete(GTK_TEXT(textbox), linelen);
	DEBUG_MSG("line_highlighting, startol=%d, endol=%d, inserting %s\n", startol, endol, line);
	gtk_text_insert(GTK_TEXT(textbox), NULL, NULL, NULL, line, linelen);

	tmplist = g_list_first(main_v->syntax_structs);
	while (tmplist) {
		hlregex_line(textbox, (Thighlight_pattern *) tmplist->data, startol,endol,line);
#ifdef CRAPPY_FIX
#if GTK_GT_7 == 0
		gtk_text_thaw(GTK_TEXT(textbox));
		gtk_text_freeze(GTK_TEXT(textbox));
#endif
#endif
		tmplist = g_list_next(tmplist);
	}
	g_free(line);

	gtk_text_thaw(GTK_TEXT(textbox));
}


static void stop_highlighting(GtkWidget * textbox)
{
	gchar *buffer;
	guint buflen;
	guint scrollp;
	DEBUG_MSG("stop_highlighting, started\n");
	gtk_text_freeze(GTK_TEXT(textbox));
	scrollp = gtk_editable_get_position(GTK_EDITABLE(textbox));
	buffer = gtk_editable_get_chars(GTK_EDITABLE(textbox), 0, -1);
	buflen = strlen(buffer);
	if (buflen > 0) {
		DEBUG_MSG("stop_highlighting, buflen=%d\n", buflen);
		gtk_text_set_point(GTK_TEXT(textbox), 0);
		DEBUG_MSG("stop_highlighting, point set to %d\n", gtk_text_get_point(GTK_TEXT(textbox)));
		gtk_text_forward_delete(GTK_TEXT(textbox), buflen);
#ifdef CRAPPY_FIX
#if GTK_GT_7 == 0
		gtk_text_thaw(GTK_TEXT(textbox));
		gtk_text_freeze(GTK_TEXT(textbox));
#endif
#endif
		gtk_text_insert(GTK_TEXT(textbox), NULL, &textbox->style->black, NULL, buffer, buflen);
	}
	g_free(buffer);
	gtk_text_thaw(GTK_TEXT(textbox));
	DEBUG_MSG("stop_highlighting, set scrollp to %d\n", scrollp);
	gtk_text_set_point(GTK_TEXT(textbox), scrollp);
}

void stop_hi_cb(GtkWidget * widget, gpointer data)
{
	Ttext_positions positions;
	Tdocument *whichdoc;
	DEBUG_MSG("stop_hi_cb, started\n");
	if (data == NULL) {
		whichdoc = main_v->current_document;
	} else {
		whichdoc = data;
	}
	positions = get_positions(whichdoc->textbox);
	gtk_signal_disconnect(GTK_OBJECT(whichdoc->textbox), whichdoc->ins_txt_id);
	gtk_signal_disconnect(GTK_OBJECT(whichdoc->textbox), whichdoc->del_txt_id);
	stop_highlighting(whichdoc->textbox);
	whichdoc->ins_txt_id = gtk_signal_connect(GTK_OBJECT(whichdoc->textbox), "insert_text", GTK_SIGNAL_FUNC(doc_insert_text_cb), whichdoc);
	whichdoc->del_txt_id = gtk_signal_connect(GTK_OBJECT(whichdoc->textbox), "delete_text", GTK_SIGNAL_FUNC(doc_delete_text_cb), whichdoc);
	restore_positions(positions, whichdoc->textbox);
}

void start_hi_cb(GtkWidget * widget, gpointer data)
{
	Ttext_positions positions;
	Tdocument *whichdoc;
	DEBUG_MSG("start_hi_cb, started\n");
	if (data == NULL) {
		whichdoc = main_v->current_document;
	} else {
		whichdoc = (Tdocument *) data;
	}
	positions = get_positions(whichdoc->textbox);
	gtk_signal_disconnect(GTK_OBJECT(whichdoc->textbox), whichdoc->ins_txt_id);
	gtk_signal_disconnect(GTK_OBJECT(whichdoc->textbox), whichdoc->del_txt_id);
	start_highlighting(whichdoc->textbox);
	whichdoc->ins_txt_id = gtk_signal_connect(GTK_OBJECT(whichdoc->textbox), "insert_text", GTK_SIGNAL_FUNC(doc_insert_text_cb), whichdoc);
	whichdoc->del_txt_id = gtk_signal_connect(GTK_OBJECT(whichdoc->textbox), "delete_text", GTK_SIGNAL_FUNC(doc_delete_text_cb), whichdoc);
	restore_positions(positions, whichdoc->textbox);
}

void refresh_hi_cb(GtkWidget * widget, gpointer data)
{
	Ttext_positions positions;
	Tdocument *whichdoc;
	DEBUG_MSG("refresh_hi_cb, started, data=%p, main_v->current_document=%p\n", data, main_v->current_document);
	if (data == NULL) {
		whichdoc = main_v->current_document;
	} else {
		whichdoc = (Tdocument *) data;
	}
	positions = get_positions(whichdoc->textbox);
	gtk_signal_disconnect(GTK_OBJECT(whichdoc->textbox), whichdoc->ins_txt_id);
	gtk_signal_disconnect(GTK_OBJECT(whichdoc->textbox), whichdoc->del_txt_id);
	stop_highlighting(whichdoc->textbox);
	start_highlighting(whichdoc->textbox);
	whichdoc->ins_txt_id = gtk_signal_connect(GTK_OBJECT(whichdoc->textbox), "insert_text", GTK_SIGNAL_FUNC(doc_insert_text_cb), whichdoc);
	whichdoc->del_txt_id = gtk_signal_connect(GTK_OBJECT(whichdoc->textbox), "delete_text", GTK_SIGNAL_FUNC(doc_delete_text_cb), whichdoc);
	restore_positions(positions, whichdoc->textbox);
}

void line_hi_cb(GtkWidget * widget, gpointer data)
{
	Ttext_positions positions;
	Tdocument *whichdoc;
	DEBUG_MSG("line_hi_cb, started, data=%p, main_v->current_document=%p\n", data, main_v->current_document);

	if (data == NULL) {
		whichdoc = main_v->current_document;
	} else {
		whichdoc = (Tdocument *) data;
	}
	positions = get_positions(whichdoc->textbox);
	gtk_signal_disconnect(GTK_OBJECT(whichdoc->textbox), whichdoc->ins_txt_id);
	gtk_signal_disconnect(GTK_OBJECT(whichdoc->textbox), whichdoc->del_txt_id);
	line_highlighting(whichdoc->textbox);

	whichdoc->ins_txt_id = gtk_signal_connect(GTK_OBJECT(whichdoc->textbox), "insert_text", GTK_SIGNAL_FUNC(doc_insert_text_cb), whichdoc);
	whichdoc->del_txt_id = gtk_signal_connect(GTK_OBJECT(whichdoc->textbox), "delete_text", GTK_SIGNAL_FUNC(doc_delete_text_cb), whichdoc);
	restore_positions(positions, whichdoc->textbox);
}


void toggle_highlight_syntax_cb(GtkWidget * w, gpointer data)
{
	DEBUG_MSG("toggle_highlight_syntax, started\n");

	main_v->current_document->highlightstate = 1 - main_v->current_document->highlightstate;
	if (main_v->current_document->highlightstate == 0) {
		stop_hi_cb(NULL, main_v->current_document);
	} else {
		start_hi_cb(NULL, main_v->current_document);
	}
	setup_toggle_item(gtk_item_factory_from_widget(main_v->menubar),
					  N_("/View/Highlight syntax"), main_v->current_document->highlightstate);
}

void make_syntax_struct_list(void)
{

	GList *tmplist;
	gchar **splittedstring;
	gint ret;
	GdkColormap *colormap;
	Thighlight_pattern *hipa;

	DEBUG_MSG("apply_highlighting_config, started\n");
	colormap = gdk_colormap_get_system();

	tmplist = g_list_first(main_v->syntax_structs);
	while (tmplist) {
		hipa = (Thighlight_pattern *) tmplist->data;
		regfree(&hipa->reg_comp_pat);
		gdk_colormap_free_colors(colormap, &hipa->color, 1);
		g_free(hipa);
		tmplist = g_list_next(tmplist);
	}
	g_list_free(main_v->syntax_structs);
	main_v->syntax_structs = NULL;

	tmplist = g_list_first(main_v->props.syntax_configstrings);
	while (tmplist) {
		splittedstring = g_strsplit((gchar *) tmplist->data, " ", 2);
		DEBUG_MSG("apply_highlighting_config, splittedstring[0]=%s\n", splittedstring[0]);
		DEBUG_MSG("apply_highlighting_config, splittedstring[1]=%s\n", splittedstring[1]);
		hipa = g_malloc0(sizeof(Thighlight_pattern));
		ret = gdk_color_parse(splittedstring[0], &hipa->color);
		if (ret == 1) {
			regcomp(&hipa->reg_comp_pat, splittedstring[1], REG_EXTENDED | REG_ICASE);
			gdk_colormap_alloc_color(colormap, &hipa->color, TRUE, TRUE);
			main_v->syntax_structs = g_list_append(main_v->syntax_structs, hipa);
		} else {
			DEBUG_MSG("apply_highlighting_config, ret=%d !!!\n", ret);
			g_free(hipa);
		}
		g_strfreev(splittedstring);
		tmplist = g_list_next(tmplist);
	}
	DEBUG_MSG("apply_highlighting_config, ended\n");
}
