/* Bluefish HTML Editor
 *
 * Copyright (C) 1999-2000 Olivier Sessink
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "default_include.h"

#include <locale.h>

#include "bluefish.h"
#include "pixmaps.h"

/*
 * Function: error_dialog
 * Arguments:
 * 	windiw_title - the title of the error window
 * 	error_string - the description of the error
 * Return value:
 * 	void
 * Description:
 * 	displays an error dialog
 */

void error_dialog(gchar * window_title, gchar * error_string)
{
	GtkWidget *dialog, *okbutton;

	dialog = gtk_dialog_new();
	gtk_window_set_title(GTK_WINDOW(dialog), window_title);
	gtk_window_position(GTK_WINDOW(dialog), GTK_WIN_POS_MOUSE);
	gtk_container_border_width(GTK_CONTAINER(dialog), 15);

	okbutton = gtk_button_new_with_label(_("OK"));
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area), okbutton, TRUE, TRUE, 0);
	GTK_WIDGET_SET_FLAGS(okbutton, GTK_CAN_DEFAULT);
	gtk_widget_grab_default(okbutton);
	gtk_signal_connect_object(GTK_OBJECT(okbutton), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(dialog));
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), gtk_label_new(error_string), TRUE, TRUE, 0);
	gtk_widget_show_all(dialog);
}



/*
 * Function: setup_toggle_item
 * Arguments:
 * 	ifactory - the itemfactory where the toggle item is in defined
 * 	path - the path in the itemfactory to find the toggle item
 * 	state - the state to put it in (0 or 1)
 * Return value:
 * 	void
 * Description:
 * 	change the setting of a toggle item in a menu
 */
void setup_toggle_item(GtkItemFactory * ifactory, gchar * path, gint state)
{

	GtkWidget *toggle;

	toggle = gtk_item_factory_get_widget(ifactory, path);
	if (!toggle) {
		DEBUG_MSG("warning, this menuitem (%s) does NOT exist!", path);
		return;
	}
	if ((GTK_CHECK_MENU_ITEM(toggle)->active) && state == 0) {
		GTK_CHECK_MENU_ITEM(toggle)->active = state;
		return;
	}
	if ((!GTK_CHECK_MENU_ITEM(toggle)->active) && state == 1) {
		GTK_CHECK_MENU_ITEM(toggle)->active = state;
		return;
	}
}

/*
 * Function: string_apply
 * Arguments:
 * 	config_var - a pointer to a gchar pointer with where it should be placed
 * 	entry - the entry to read
 * Return value:
 * 	void
 * Description:
 * 	free the old string, and replace it with the new string
 */
void string_apply(gchar ** config_var, GtkWidget * entry)
{
	gchar *tmpstring;
	DEBUG_MSG("string_apply, start\n");

	tmpstring = gtk_editable_get_chars(GTK_EDITABLE(entry), 0, -1);
	DEBUG_MSG("string_apply, tmpstring=%s\n", tmpstring);
	if (tmpstring) {
		if (config_var != NULL) {
			g_free(*config_var);
		}
		*config_var = tmpstring;
	}
}

/*
 * Function: combo_with_popdown
 * Arguments:
 * 	setstring - string to setb in textbox, if NULL set ""
 * 	which_list - GList to set in popdown widget
 * 	editable - if editable (1 or 0)
 * Return value:
 * 	Pointer to create combo
 * Description:
 * 	Create new combo and preset some values
 */
GtkWidget *combo_with_popdown(gchar * setstring, GList * which_list, gint editable)
{

	GtkWidget *returnwidget;

	returnwidget = gtk_combo_new();
	if (which_list != NULL) {
		gtk_combo_set_popdown_strings(GTK_COMBO(returnwidget), which_list);
	}
	if (setstring != NULL) {
		gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(returnwidget)->entry), setstring);
	} else {
		gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(returnwidget)->entry), "");
	}
	if (editable == 0) {
		gtk_editable_set_editable(GTK_EDITABLE(GTK_ENTRY(GTK_COMBO(returnwidget)->entry)), FALSE);
	}
	return returnwidget;
}

GtkWidget *boxed_combo_with_popdown(gchar * setstring, GList * which_list, gint editable, GtkWidget *box) {

	GtkWidget *returnwidget;
	
	returnwidget = combo_with_popdown(setstring, which_list, editable);
	gtk_box_pack_start(GTK_BOX(box), returnwidget, TRUE, TRUE, 3);
	return returnwidget;
}

/*
 * Function: entry_with_text
 * Arguments:
 * 	setstring - if !NULL set this text
 * 	max_lenght - max. characters in the entry
 * Return value:
 * 	Pointer to the new entry widget
 * Description:
 * 	Create new entry with some preset values
 */

GtkWidget *entry_with_text(gchar * setstring, gint max_lenght)
{

	GtkWidget *returnwidget;

	if (max_lenght) {
		returnwidget = gtk_entry_new_with_max_length(max_lenght);
	} else {
		returnwidget = gtk_entry_new();
	}
	if (setstring) {
		gtk_entry_set_text(GTK_ENTRY(returnwidget), setstring);
	}
	return returnwidget;

}

GtkWidget *boxed_entry_with_text(gchar * setstring, gint max_lenght, GtkWidget *box) {
	GtkWidget *returnwidget;
	
	returnwidget = entry_with_text(setstring, max_lenght);
	gtk_box_pack_start(GTK_BOX(box), returnwidget, TRUE, TRUE, 3);
	return returnwidget;

}

GtkWidget *checkbut_with_value(gchar *labeltext, gint which_config_int) {
	GtkWidget *returnwidget;

	returnwidget = gtk_check_button_new_with_label(labeltext);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(returnwidget), which_config_int);
	return returnwidget;
}

GtkWidget *boxed_checkbut_with_value(gchar *labeltext, gint which_config_int, GtkWidget * box) {
	GtkWidget *returnwidget;

	returnwidget = checkbut_with_value(labeltext, which_config_int);
	gtk_box_pack_start(GTK_BOX(box), returnwidget, FALSE, FALSE, 0);
	return returnwidget;
}

GtkWidget *radiobut_with_value(gchar *labeltext, gint enabled, GtkRadioButton *prevbut) {
	GtkWidget *returnwidget;
	GSList *group=NULL;

	if (prevbut) {
		group = gtk_radio_button_group(prevbut);
	}	
	returnwidget = gtk_radio_button_new_with_label(group, labeltext);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(returnwidget), enabled);
	return returnwidget;
}

GtkWidget *boxed_radiobut_with_value(gchar *labeltext, gint enabled, GtkRadioButton *prevbut, GtkWidget *box) {
	GtkWidget *returnwidget;

	returnwidget = radiobut_with_value(labeltext, enabled, prevbut);
	gtk_box_pack_start(GTK_BOX(box), returnwidget, FALSE, FALSE, 0);
	return returnwidget;
}

/*
 * Function: window_with_title
 * Arguments:
 * 	title - the title string
 * 	position - the gtk window position
 * 	type - the gtk window type
 * Return value:
 * 	Pointer to created windpw
 * Description:
 * 	Create new window with title and some more settings
 */
GtkWidget *window_with_title(gchar * title, GtkWindowPosition position, GtkWindowType type, gint borderwidth)
{

	GtkWidget *returnwidget;

	returnwidget = gtk_window_new(type);
	if (title) {
		gtk_window_set_title(GTK_WINDOW(returnwidget), title);
	}
	if (borderwidth) {
		gtk_container_set_border_width(GTK_CONTAINER(returnwidget), borderwidth);
	}
	if (position) {
		gtk_window_set_position(GTK_WINDOW(returnwidget), position);
	}
	return returnwidget;
}

/*
 * Function: window_full
 * Arguments:
 * 	title - the title string
 * 	position - the gtk window position
 * 	type - the gtk window type
 * 	close_func - the callback function when closing the window
 * 	close_data - the gpointer data passed to the closefunc
 * Return value:
 * 	Pointer to created windpw
 * Description:
 * 	Create new window with title, callback functions and some more settings
 */
GtkWidget *window_full(gchar * title, GtkWindowPosition position, GtkWindowType type, gint borderwidth, GtkSignalFunc close_func,
					   gpointer close_data)
{

	GtkWidget *returnwidget;

	returnwidget = window_with_title(title, position, type, borderwidth);
	gtk_signal_connect(GTK_OBJECT(returnwidget), "delete_event", GTK_SIGNAL_FUNC(close_func), close_data);
	gtk_signal_connect(GTK_OBJECT(returnwidget), "destroy", GTK_SIGNAL_FUNC(close_func), close_data);

	return returnwidget;
}

/*
 * Function: window_destroy
 * Arguments:
 * 	windowname - a pointer to the window widget to destroy
 * Description:
 * 	Remove grab and signals and then destroy window
 */
void window_destroy(GtkWidget * windowname)
{
	DEBUG_MSG("window_destroy, windowname=%p\n", windowname);
	gtk_signal_handlers_destroy(GTK_OBJECT(windowname));
	gtk_grab_remove(windowname);
	gtk_widget_destroy(windowname);
}

/*
 * Function: window_close_by_widget_cb
 * Arguments:
 * 	widget - the parent window of this widget will be destroyed
 * 	data - ignored
 * Description:
 * 	destroy the toplevel window for widget
 */
void window_close_by_widget_cb(GtkWidget * widget, gpointer data)
{
	DEBUG_MSG("window_close_by_data_cb, widget=%p\n", widget);
	window_destroy(gtk_widget_get_toplevel(widget));
}

/*
 * Function: window_close_by_data_cb
 * Arguments:
 * 	widget - ignored
 * 	data - a pointer to a gtk-window which will be detroyed
 * Description:
 * 	destroy a window using the gpointer data
 */
void window_close_by_data_cb(GtkWidget * widget, gpointer data)
{
	DEBUG_MSG("window_close_by_data_cb, data=%p\n", data);
	window_destroy(GTK_WIDGET(data));
}

/*
 * Function: apply_font_style
 * Arguments:
 * 	this_widget - the widget to apply the new fontstyle
 * 	fontstring - the fontset which will be loaded
 * Return value:
 * 	this_widget
 * Description:
 * 	Change the font of a widget with a certain font in a string
 */
GtkWidget *apply_font_style(GtkWidget * this_widget, gchar * fontstring)
{
	GdkFont *tmpfont;
	GtkStyle *style;

	DEBUG_MSG("apply_font_style, started\n");
	if ((style = gtk_rc_get_style(main_v->main_window)) == NULL) {
		style = gtk_style_new();
	}
	tmpfont = style->font;

	if (main_v->props.fontset) {
		style->font = gdk_fontset_load(fontstring);
	} else {
		style->font = gdk_font_load(fontstring);
	}
	if (style->font) {
		DEBUG_MSG("apply_font_style, loaded font is ok\n");
		gdk_font_unref(tmpfont);
		gtk_widget_push_style(style);
		gtk_widget_set_style(GTK_WIDGET(this_widget), style);
		gtk_widget_pop_style();
		DEBUG_MSG("apply_font_style, style applied\n");
	} else {
		DEBUG_MSG("apply_font_style, font = NULL, fontstring=%s\n", fontstring);
		style->font = tmpfont;
		gtk_style_unref(style);
	}
	return this_widget;
}

/*
 * Function: bf_stock_button
 * Arguments:
 * 	Text - label string
 * 	func - pointer to signal handler
 * 	func_data - data for signal handler
 * Return value:
 * 	Pointer to create button
 * Description:
 * 	Create new button and link button with "clicked" signal handler
 */
GtkWidget *bf_stock_button(const gchar * Text, GtkSignalFunc func, gpointer func_data)
{
	GtkWidget *button;

	button = gtk_button_new_with_label(Text);
	g_return_val_if_fail(button, NULL);
	gtk_signal_connect(GTK_OBJECT(button), "clicked", func, func_data);
	DEBUG_MSG("bf_stock_button, func_data=%p\n", func_data);
	return button;
};

/*
 * Function: bf_stock_ok_button
 * Arguments:
 * 	func - pointer to signal handler
 * 	func_data - data for signal handler
 * Return value:
 * 	Pointer to create button
 * Description:
 * 	Create new "Ok" button
 */
GtkWidget *bf_stock_ok_button(GtkSignalFunc func, gpointer func_data)
{
	return bf_stock_button(_(" Ok "), func, func_data);
};

/*
 * Function: bf_stock_cancel_button
 * Arguments:
 * 	func - pointer to signal handler
 * 	func_data - data for signal handler
 * Return value:
 * 	Pointer to create button
 * Description:
 * 	Create new "Cancel" button
 */
GtkWidget *bf_stock_cancel_button(GtkSignalFunc func, gpointer func_data)
{
	return bf_stock_button(_(" Cancel "), func, func_data);
};


/************************************************************************/

/*
 * Function: file_but_new
 * Arguments:
 * 	which_entry - GTK_ENTRY where to put the filename
 * 	win - the GTK_WINDOW where the button is placed
 * 	full_pathname - 1 or 0, if you want the full path or a relative path
 * Return value:
 * 	Pointer to file button
 * Description:
 * 	Create new "open file" button
 */

static void file_but_clicked_lcb(GtkWidget * widget, GtkWidget * which_entry)
{
	gchar *tmpstring, *tmp2string;

	tmpstring = return_file(NULL);
	if (tmpstring != NULL) {
		if (main_v->current_document->filename != NULL) {
/*			tmp2string = strip_common_path(tmpstring, main_v->current_document->filename); */
			tmp2string = create_relative_link_to(main_v->current_document->filename, tmpstring);
			g_free(tmpstring);
			tmpstring = tmp2string;
		}
		gtk_entry_set_text(GTK_ENTRY(which_entry), tmpstring);
		gtk_signal_emit_by_name(GTK_OBJECT(which_entry), "activate");
		g_free(tmpstring);
	}
}


static void file_but_clicked_full_lcb(GtkWidget * widget, GtkWidget * which_entry)
{
	gchar *tmpstring;

	tmpstring = return_file(gtk_entry_get_text(GTK_ENTRY(which_entry)));
	if (tmpstring != NULL) {
		gtk_entry_set_text(GTK_ENTRY(which_entry), tmpstring);
		g_free(tmpstring);
	}
}

GtkWidget *file_but_new(GtkWidget * which_entry, GtkWidget * win, gint full_pathname)
{
	GtkWidget *pixmap, *file_but;

	file_but = gtk_button_new();
	pixmap = new_pixmap(177, win->window, NULL);
	gtk_widget_show(pixmap);
	gtk_container_add(GTK_CONTAINER(file_but), pixmap);
	if (full_pathname == 1) {
		gtk_signal_connect(GTK_OBJECT(file_but), "clicked", file_but_clicked_full_lcb, which_entry);
	} else {
		gtk_signal_connect(GTK_OBJECT(file_but), "clicked", file_but_clicked_lcb, which_entry);
	}
	gtk_widget_show(file_but);
	return file_but;
}

/************************************************************************/

/*
 * Function: close_save_cancel_dialog
 * Arguments:
 * 	title - title string
 * 	label - label string
 * Return value:
 * 	1 - close, 2 - save, 3 - cancel
 * Description:
 * 	create dialog which returns a close, save or cancel
 */
static void close_clicked_lcb(GtkWidget * widget, gpointer data)
{
	*(gint *) data = 1;
	gtk_main_quit();
}

static void save_clicked_lcb(GtkWidget * widget, gpointer data)
{
	*(gint *) data = 2;
	gtk_main_quit();
}

static void cancel_clicked_lcb(GtkWidget * widget, gpointer data)
{
	*(gint *) data = 3;
	gtk_main_quit();
}

gint close_save_cancel_dialog(gchar * title, gchar * label)
{

	GtkWidget *win, *but1, *but2, *but3;
	GtkWidget *vbox, *hbox;
	gint retval = 0;

	win = window_with_title(title, GTK_WIN_POS_CENTER, GTK_WINDOW_DIALOG, 5);
	gtk_grab_add(win);
	gtk_signal_connect(GTK_OBJECT(win), "delete_event", GTK_SIGNAL_FUNC(cancel_clicked_lcb), &retval);
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(win), vbox);
	gtk_box_pack_start(GTK_BOX(vbox), gtk_label_new(label), FALSE, FALSE, 0);
	hbox = gtk_hbox_new(TRUE, 5);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	but1 = bf_stock_button("Close", close_clicked_lcb, (gpointer) & retval);
	but2 = bf_stock_button("Save", save_clicked_lcb, (gpointer) & retval);
	but3 = bf_stock_button("Cancel", cancel_clicked_lcb, (gpointer) & retval);
	GTK_WIDGET_SET_FLAGS(but1, GTK_CAN_DEFAULT);
	GTK_WIDGET_SET_FLAGS(but2, GTK_CAN_DEFAULT);
	GTK_WIDGET_SET_FLAGS(but3, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(hbox), but1, TRUE, TRUE, 5);
	gtk_box_pack_start(GTK_BOX(hbox), but2, TRUE, TRUE, 5);
	gtk_box_pack_start(GTK_BOX(hbox), but3, TRUE, TRUE, 5);
	gtk_widget_show_all(win);
	gtk_main();
	gtk_signal_handlers_destroy(GTK_OBJECT(win));
	gtk_widget_destroy(win);
	return retval;
}

/************************************************************************/
